/********************************************************************************
 * Copyright (C) 2018 Google and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/

import { interfaces } from 'inversify';
import { createPreferenceProxy, PreferenceProxy, PreferenceService, PreferenceContribution, PreferenceSchema } from './preferences';
import { SUPPORTED_ENCODINGS } from './supported-encodings';
import { FrontendApplicationConfigProvider } from './frontend-application-config-provider';
import { isOSX } from '../common/os';
import { nls } from '../common/nls';

export const corePreferenceSchema: PreferenceSchema = {
    'type': 'object',
    properties: {
        'application.confirmExit': {
            type: 'string',
            enum: [
                'never',
                'ifRequired',
                'always',
            ],
            default: 'ifRequired',
            description: nls.localize('vscode/workbench.contribution/confirmBeforeCloseWeb', 'When to confirm before closing the application window.'),
        },
        'breadcrumbs.enabled': {
            'type': 'boolean',
            'default': true,
            'description': nls.localize('vscode/breadcrumbs/enabled', 'Enable/disable navigation breadcrumbs.'),
            'scope': 'application'
        },
        'files.encoding': {
            'type': 'string',
            'enum': Object.keys(SUPPORTED_ENCODINGS),
            'default': 'utf8',
            'description': nls.localize('vscode/files.contribution/encoding',
                'The default character set encoding to use when reading and writing files. This setting can also be configured per language.'),
            'scope': 'language-overridable',
            'enumDescriptions': Object.keys(SUPPORTED_ENCODINGS).map(key => SUPPORTED_ENCODINGS[key].labelLong),
            'included': Object.keys(SUPPORTED_ENCODINGS).length > 1
        },
        'keyboard.dispatch': {
            type: 'string',
            enum: [
                'code',
                'keyCode',
            ],
            default: 'code',
            description: nls.localize('vscode/keybindingService/dispatch',
                'Whether to interpret keypresses by the `code` of the physical key, or by the `keyCode` provided by the OS.')
        },
        'window.menuBarVisibility': {
            type: 'string',
            enum: ['classic', 'visible', 'hidden', 'compact'],
            markdownEnumDescriptions: [
                nls.localize('vscode/workbench.contribution/window.menuBarVisibility.default', 'Menu is displayed at the top of the window and only hidden in full screen mode.'),
                nls.localize('vscode/workbench.contribution/window.menuBarVisibility.visible', 'Menu is always visible at the top of the window even in full screen mode.'),
                nls.localize('vscode/workbench.contribution/window.menuBarVisibility.hidden', 'Menu is always hidden.'),
                nls.localize('vscode/workbench.contribution/window.menuBarVisibility.compact', 'Menu is displayed as a compact button in the sidebar.')
            ],
            default: 'classic',
            scope: 'application',
            markdownDescription: nls.localize('vscode/workbench.contribution/menuBarVisibility', `Control the visibility of the menu bar.
            A setting of 'compact' will move the menu into the sidebar.`),
            included: !isOSX
        },
        'workbench.list.openMode': {
            type: 'string',
            enum: [
                'singleClick',
                'doubleClick'
            ],
            default: 'singleClick',
            description: nls.localize('vscode/listService/openModeModifier', 'Controls how to open items in trees using the mouse.')
        },
        'workbench.editor.highlightModifiedTabs': {
            'type': 'boolean',
            'description': nls.localize('vscode/workbench.contribution/highlightModifiedTabs', 'Controls whether a top border is drawn on modified (dirty) editor tabs or not.'),
            'default': false
        },
        'workbench.editor.closeOnFileDelete': {
            'type': 'boolean',
            // eslint-disable-next-line max-len
            'description': nls.localize('vscode/workbench.contribution/closeOnFileDelete', 'Controls whether editors showing a file that was opened during the session should close automatically when getting deleted or renamed by some other process. Disabling this will keep the editor open  on such an event. Note that deleting from within the application will always close the editor and that dirty files will never close to preserve your data.'),
            'default': false
        },
        'workbench.commandPalette.history': {
            type: 'number',
            default: 50,
            minimum: 0,
            // eslint-disable-next-line max-len
            description: nls.localize('vscode/workbench.contribution/commandHistory', 'Controls the number of recently used commands to keep in history for the command palette. Set to 0 to disable command history.')
        },
        'workbench.colorTheme': {
            type: 'string',
            default: FrontendApplicationConfigProvider.get().defaultTheme,
            description: nls.localize('vscode/themeConfiguration/colorTheme', 'Specifies the color theme used in the workbench.')
        },
        'workbench.iconTheme': {
            type: ['string', 'null'],
            default: FrontendApplicationConfigProvider.get().defaultIconTheme,
            description: nls.localize('vscode/themeConfiguration/iconTheme', "Specifies the icon theme used in the workbench or 'null' to not show any file icons.")
        },
        'workbench.silentNotifications': {
            type: 'boolean',
            default: false,
            description: nls.localize('vscode/workbench.contribution/zenMode.silentNotifications', 'Controls whether to suppress notification popups.')
        },
        'workbench.statusBar.visible': {
            type: 'boolean',
            default: true,
            description: 'Controls the visibility of the status bar at the bottom of the workbench.'
        },
        'workbench.tree.renderIndentGuides': {
            type: 'string',
            enum: ['onHover', 'none', 'always'],
            default: 'onHover',
            description: nls.localize('vscode/listService/render tree indent guides', 'Controls whether the tree should render indent guides.')
        },
        'workbench.hover.delay': {
            type: 'number',
            default: isOSX ? 1500 : 500,
            description: 'Controls the delay in milliseconds after which the hover is shown for workbench items (ex. some extension provided tree view items).'
        },
    }
};

export interface CoreConfiguration {
    'application.confirmExit': 'never' | 'ifRequired' | 'always';
    'breadcrumbs.enabled': boolean;
    'files.encoding': string
    'keyboard.dispatch': 'code' | 'keyCode';
    'window.menuBarVisibility': 'classic' | 'visible' | 'hidden' | 'compact';
    'workbench.list.openMode': 'singleClick' | 'doubleClick';
    'workbench.commandPalette.history': number;
    'workbench.editor.highlightModifiedTabs': boolean;
    'workbench.editor.closeOnFileDelete': boolean;
    'workbench.colorTheme': string;
    'workbench.iconTheme': string | null;
    'workbench.silentNotifications': boolean;
    'workbench.statusBar.visible': boolean;
    'workbench.tree.renderIndentGuides': 'onHover' | 'none' | 'always';
    'workbench.hover.delay': number;
}

export const CorePreferenceContribution = Symbol('CorePreferenceContribution');
export const CorePreferences = Symbol('CorePreferences');
export type CorePreferences = PreferenceProxy<CoreConfiguration>;

export function createCorePreferences(preferences: PreferenceService, schema: PreferenceSchema = corePreferenceSchema): CorePreferences {
    return createPreferenceProxy(preferences, schema);
}

export function bindCorePreferences(bind: interfaces.Bind): void {
    bind(CorePreferences).toDynamicValue(ctx => {
        const preferences = ctx.container.get<PreferenceService>(PreferenceService);
        const contribution = ctx.container.get<PreferenceContribution>(CorePreferenceContribution);
        return createCorePreferences(preferences, contribution.schema);
    }).inSingletonScope();
    bind(CorePreferenceContribution).toConstantValue({ schema: corePreferenceSchema });
    bind(PreferenceContribution).toService(CorePreferenceContribution);
}
