"use strict";
/********************************************************************************
 * Copyright (C) 2018 Red Hat, Inc. and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.DebugAdapterSessionImpl = void 0;
const disposable_1 = require("@theia/core/lib/common/disposable");
/**
 * [DebugAdapterSession](#DebugAdapterSession) implementation.
 */
class DebugAdapterSessionImpl {
    constructor(id, communicationProvider) {
        this.id = id;
        this.communicationProvider = communicationProvider;
        this.toDispose = new disposable_1.DisposableCollection();
        this.toDispose.pushAll([
            this.communicationProvider,
            disposable_1.Disposable.create(() => this.write(JSON.stringify({ seq: -1, type: 'request', command: 'disconnect' }))),
            disposable_1.Disposable.create(() => this.write(JSON.stringify({ seq: -1, type: 'request', command: 'terminate' })))
        ]);
        this.communicationProvider.onMessageReceived((message) => this.send(message));
        this.communicationProvider.onClose(() => this.onDebugAdapterExit(1, undefined)); // FIXME pass a proper exit code
        this.communicationProvider.onError(error => this.onDebugAdapterError(error));
    }
    async start(channel) {
        if (this.channel) {
            throw new Error('The session has already been started, id: ' + this.id);
        }
        this.channel = channel;
        this.channel.onMessage((message) => this.write(message));
        this.channel.onClose(() => this.channel = undefined);
    }
    onDebugAdapterExit(exitCode, signal) {
        const event = {
            type: 'event',
            event: 'exited',
            seq: -1,
            body: {
                exitCode
            }
        };
        this.send(JSON.stringify(event));
    }
    onDebugAdapterError(error) {
        const event = {
            type: 'event',
            event: 'error',
            seq: -1,
            body: error
        };
        this.send(JSON.stringify(event));
    }
    send(message) {
        if (this.channel) {
            this.channel.send(message);
        }
    }
    write(message) {
        this.communicationProvider.send(message);
    }
    async stop() {
        this.toDispose.dispose();
    }
}
exports.DebugAdapterSessionImpl = DebugAdapterSessionImpl;
//# sourceMappingURL=debug-adapter-session.js.map