"use strict";
/********************************************************************************
 * Copyright (C) 2017 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var QuickFileOpenService_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QuickFileOpenService = exports.quickFileOpen = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const browser_1 = require("@theia/core/lib/browser");
const workspace_service_1 = require("@theia/workspace/lib/browser/workspace-service");
const uri_1 = require("@theia/core/lib/common/uri");
const file_search_service_1 = require("../common/file-search-service");
const common_1 = require("@theia/core/lib/common");
const label_provider_1 = require("@theia/core/lib/browser/label-provider");
const navigation_location_service_1 = require("@theia/editor/lib/browser/navigation/navigation-location-service");
const fuzzy = require("@theia/core/shared/fuzzy");
const message_service_1 = require("@theia/core/lib/common/message-service");
const browser_2 = require("@theia/filesystem/lib/browser");
const browser_3 = require("@theia/editor/lib/browser");
const quick_input_service_1 = require("@theia/core/lib/browser/quick-input/quick-input-service");
exports.quickFileOpen = common_1.Command.toLocalizedCommand({
    id: 'file-search.openFile',
    category: browser_1.CommonCommands.FILE_CATEGORY,
    label: 'Open File...'
}, 'vscode/workspaceActions/openFile', browser_1.CommonCommands.FILE_CATEGORY_KEY);
// Supports patterns of <path><#|:><line><#|:|,><col?>
const LINE_COLON_PATTERN = /\s?[#:\(](?:line )?(\d*)(?:[#:,](\d*))?\)?\s*$/;
let QuickFileOpenService = QuickFileOpenService_1 = class QuickFileOpenService {
    constructor() {
        /**
         * Whether to hide .gitignored (and other ignored) files.
         */
        this.hideIgnoredFiles = true;
        /**
         * Whether the dialog is currently open.
         */
        this.isOpen = false;
        this.updateIsOpen = true;
        this.filterAndRangeDefault = { filter: '', range: undefined };
        /**
         * Tracks the user file search filter and location range e.g. fileFilter:line:column or fileFilter:line,column
         */
        this.filterAndRange = this.filterAndRangeDefault;
    }
    registerQuickAccessProvider() {
        this.quickAccessRegistry.registerQuickAccessProvider({
            getInstance: () => this,
            prefix: QuickFileOpenService_1.PREFIX,
            placeholder: this.getPlaceHolder(),
            helpEntries: [{ description: 'Open File', needsEditor: false }]
        });
    }
    init() {
        var _a;
        (_a = this.quickInputService) === null || _a === void 0 ? void 0 : _a.onHide(() => {
            if (this.updateIsOpen) {
                this.isOpen = false;
            }
            else {
                this.updateIsOpen = true;
            }
        });
    }
    isEnabled() {
        return this.workspaceService.opened;
    }
    open() {
        var _a;
        // Triggering the keyboard shortcut while the dialog is open toggles
        // showing the ignored files.
        if (this.isOpen) {
            this.hideIgnoredFiles = !this.hideIgnoredFiles;
            this.hideQuickPick();
        }
        else {
            this.hideIgnoredFiles = true;
            this.filterAndRange = this.filterAndRangeDefault;
            this.isOpen = true;
        }
        (_a = this.quickInputService) === null || _a === void 0 ? void 0 : _a.open(this.filterAndRange.filter);
    }
    hideQuickPick() {
        var _a;
        this.updateIsOpen = false;
        (_a = this.quickInputService) === null || _a === void 0 ? void 0 : _a.hide();
    }
    /**
     * Get a string (suitable to show to the user) representing the keyboard
     * shortcut used to open the quick file open menu.
     */
    getKeyCommand() {
        const keyCommand = this.keybindingRegistry.getKeybindingsForCommand(exports.quickFileOpen.id);
        if (keyCommand) {
            // We only consider the first keybinding.
            const accel = this.keybindingRegistry.acceleratorFor(keyCommand[0], '+');
            return accel.join(' ');
        }
        return undefined;
    }
    async getPicks(filter, token) {
        const roots = this.workspaceService.tryGetRoots();
        this.filterAndRange = this.splitFilterAndRange(filter);
        const fileFilter = this.filterAndRange.filter;
        const alreadyCollected = new Set();
        const recentlyUsedItems = [];
        const locations = [...this.navigationLocationService.locations()].reverse();
        for (const location of locations) {
            const uriString = location.uri.toString();
            if (location.uri.scheme === 'file' && !alreadyCollected.has(uriString) && fuzzy.test(fileFilter, uriString)) {
                if (recentlyUsedItems.length === 0) {
                    recentlyUsedItems.push({ type: 'separator', label: 'recently opened' });
                }
                const item = this.toItem(fileFilter, location.uri);
                recentlyUsedItems.push(item);
                alreadyCollected.add(uriString);
            }
        }
        if (fileFilter.length > 0) {
            const handler = async (results) => {
                if (token.isCancellationRequested || results.length <= 0) {
                    return [];
                }
                const result = [...recentlyUsedItems];
                const fileSearchResultItems = [];
                for (const fileUri of results) {
                    if (!alreadyCollected.has(fileUri)) {
                        const item = this.toItem(fileFilter, fileUri);
                        fileSearchResultItems.push(item);
                        alreadyCollected.add(fileUri);
                    }
                }
                // Create a copy of the file search results and sort.
                const sortedResults = fileSearchResultItems.slice();
                sortedResults.sort((a, b) => this.compareItems(a, b));
                if (sortedResults.length > 0) {
                    result.unshift({ type: 'separator', label: 'file results' });
                    result.push(...sortedResults);
                }
                // Return the recently used items, followed by the search results.
                return result;
            };
            return this.fileSearchService.find(fileFilter, {
                rootUris: roots.map(r => r.resource.toString()),
                fuzzyMatch: true,
                limit: 200,
                useGitIgnore: this.hideIgnoredFiles,
                excludePatterns: this.hideIgnoredFiles
                    ? Object.keys(this.fsPreferences['files.exclude'])
                    : undefined,
            }, token).then(handler);
        }
        else {
            return roots.length !== 0 ? recentlyUsedItems : [];
        }
    }
    compareItems(left, right) {
        /**
         * Score a given string.
         *
         * @param str the string to score on.
         * @returns the score.
         */
        function score(str) {
            if (!str) {
                return 0;
            }
            // Adjust for whitespaces in the query.
            const querySplit = query.split(file_search_service_1.WHITESPACE_QUERY_SEPARATOR);
            const queryJoin = querySplit.join('');
            // Check exact and partial exact matches.
            let exactMatch = true;
            let partialMatch = false;
            querySplit.forEach(part => {
                const partMatches = str.includes(part);
                exactMatch = exactMatch && partMatches;
                partialMatch = partialMatch || partMatches;
            });
            // Check fuzzy matches.
            const fuzzyMatch = fuzzy.match(queryJoin, str);
            let matchScore = 0;
            // eslint-disable-next-line no-null/no-null
            if (!!fuzzyMatch && matchScore !== null) {
                matchScore = (fuzzyMatch.score === Infinity) ? QuickFileOpenService_1.Scores.max : fuzzyMatch.score;
            }
            // Prioritize exact matches, then partial exact matches, then fuzzy matches.
            if (exactMatch) {
                return matchScore + QuickFileOpenService_1.Scores.exact;
            }
            else if (partialMatch) {
                return matchScore + QuickFileOpenService_1.Scores.partial;
            }
            else {
                // eslint-disable-next-line no-null/no-null
                return (fuzzyMatch === null) ? 0 : matchScore;
            }
        }
        const query = normalize(this.filterAndRange.filter);
        const compareByLabelScore = (l, r) => score(r.label) - score(l.label);
        const compareByLabelIndex = (l, r) => r.label.indexOf(query) - l.label.indexOf(query);
        const compareByLabel = (l, r) => r.label.localeCompare(l.label);
        const compareByPathScore = (l, r) => score(r.uri.path.toString()) - score(l.uri.path.toString());
        const compareByPathIndex = (l, r) => r.uri.path.toString().indexOf(query) - l.uri.path.toString().indexOf(query);
        const compareByPathLabel = (l, r) => r.uri.path.toString().localeCompare(l.uri.path.toString());
        return compareWithDiscriminators(left, right, compareByLabelScore, compareByLabelIndex, compareByLabel, compareByPathScore, compareByPathIndex, compareByPathLabel);
    }
    openFile(uri) {
        const options = this.buildOpenerOptions();
        const closedEditor = this.navigationLocationService.closedEditorsStack.find(editor => editor.uri.path.toString() === uri.path.toString());
        this.openerService.getOpener(uri, options)
            .then(opener => opener.open(uri, options))
            .then(widget => {
            // Attempt to restore the editor state if it exists, and no selection is explicitly requested.
            if (widget instanceof browser_3.EditorWidget && closedEditor && !options.selection) {
                widget.editor.restoreViewState(closedEditor.viewState);
            }
        })
            .catch(error => this.messageService.error(error));
    }
    buildOpenerOptions() {
        return { selection: this.filterAndRange.range };
    }
    toItem(lookFor, uriOrString) {
        const uri = uriOrString instanceof uri_1.default ? uriOrString : new uri_1.default(uriOrString);
        const label = this.labelProvider.getName(uri);
        const description = this.getItemDescription(uri);
        const iconClasses = this.getItemIconClasses(uri);
        return {
            label,
            description,
            highlights: {
                label: quick_input_service_1.findMatches(label, lookFor),
                description: quick_input_service_1.findMatches(description, lookFor)
            },
            iconClasses,
            uri,
            execute: () => this.openFile(uri)
        };
    }
    getItemIconClasses(uri) {
        const icon = this.labelProvider.getIcon(uri);
        return icon !== '' ? [icon + ' file-icon'] : [];
    }
    getItemDescription(uri) {
        const resourcePathLabel = this.labelProvider.getLongName(uri.parent);
        const rootUri = this.workspaceService.getWorkspaceRootUri(uri);
        // Display the path normally if outside any workspace root.
        if (!rootUri) {
            return resourcePathLabel;
        }
        // Compute the multi-root label for the resource.
        if (this.workspaceService.isMultiRootWorkspaceOpened) {
            const rootLabel = this.labelProvider.getName(rootUri);
            return rootUri.path.toString() === uri.parent.path.toString() ? `${rootLabel}` : `${rootLabel} • ${resourcePathLabel}`;
        }
        else {
            return rootUri.toString() === uri.parent.toString() ? '' : resourcePathLabel;
        }
    }
    getPlaceHolder() {
        let placeholder = 'File name to search (append : to go to line).';
        const keybinding = this.getKeyCommand();
        if (keybinding) {
            placeholder += ` (Press ${keybinding} to show/hide ignored files)`;
        }
        return placeholder;
    }
    /**
     * Splits the given expression into a structure of search-file-filter and
     * location-range.
     *
     * @param expression patterns of <path><#|:><line><#|:|,><col?>
     */
    splitFilterAndRange(expression) {
        var _a, _b;
        let filter = expression;
        let range = undefined;
        // Find line and column number from the expression using RegExp.
        const patternMatch = LINE_COLON_PATTERN.exec(expression);
        if (patternMatch) {
            const line = parseInt((_a = patternMatch[1]) !== null && _a !== void 0 ? _a : '', 10);
            if (Number.isFinite(line)) {
                const lineNumber = line > 0 ? line - 1 : 0;
                const column = parseInt((_b = patternMatch[2]) !== null && _b !== void 0 ? _b : '', 10);
                const startColumn = Number.isFinite(column) && column > 0 ? column - 1 : 0;
                const position = browser_3.Position.create(lineNumber, startColumn);
                filter = expression.substr(0, patternMatch.index);
                range = browser_3.Range.create(position, position);
            }
        }
        return { filter, range };
    }
};
QuickFileOpenService.PREFIX = '';
/**
 * The score constants when comparing file search results.
 */
QuickFileOpenService.Scores = {
    max: 1000,
    exact: 500,
    partial: 250 // represents the score assigned to partial matching.
};
__decorate([
    inversify_1.inject(browser_1.KeybindingRegistry),
    __metadata("design:type", browser_1.KeybindingRegistry)
], QuickFileOpenService.prototype, "keybindingRegistry", void 0);
__decorate([
    inversify_1.inject(workspace_service_1.WorkspaceService),
    __metadata("design:type", workspace_service_1.WorkspaceService)
], QuickFileOpenService.prototype, "workspaceService", void 0);
__decorate([
    inversify_1.inject(browser_1.OpenerService),
    __metadata("design:type", Object)
], QuickFileOpenService.prototype, "openerService", void 0);
__decorate([
    inversify_1.inject(quick_input_service_1.QuickInputService),
    inversify_1.optional(),
    __metadata("design:type", Object)
], QuickFileOpenService.prototype, "quickInputService", void 0);
__decorate([
    inversify_1.inject(browser_1.QuickAccessRegistry),
    __metadata("design:type", Object)
], QuickFileOpenService.prototype, "quickAccessRegistry", void 0);
__decorate([
    inversify_1.inject(file_search_service_1.FileSearchService),
    __metadata("design:type", Object)
], QuickFileOpenService.prototype, "fileSearchService", void 0);
__decorate([
    inversify_1.inject(label_provider_1.LabelProvider),
    __metadata("design:type", label_provider_1.LabelProvider)
], QuickFileOpenService.prototype, "labelProvider", void 0);
__decorate([
    inversify_1.inject(navigation_location_service_1.NavigationLocationService),
    __metadata("design:type", navigation_location_service_1.NavigationLocationService)
], QuickFileOpenService.prototype, "navigationLocationService", void 0);
__decorate([
    inversify_1.inject(message_service_1.MessageService),
    __metadata("design:type", message_service_1.MessageService)
], QuickFileOpenService.prototype, "messageService", void 0);
__decorate([
    inversify_1.inject(browser_2.FileSystemPreferences),
    __metadata("design:type", Object)
], QuickFileOpenService.prototype, "fsPreferences", void 0);
__decorate([
    inversify_1.postConstruct(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", void 0)
], QuickFileOpenService.prototype, "init", null);
QuickFileOpenService = QuickFileOpenService_1 = __decorate([
    inversify_1.injectable()
], QuickFileOpenService);
exports.QuickFileOpenService = QuickFileOpenService;
/**
 * Normalize a given string.
 *
 * @param str the raw string value.
 * @returns the normalized string value.
 */
function normalize(str) {
    return str.trim().toLowerCase();
}
function compareWithDiscriminators(left, right, ...discriminators) {
    let comparisonValue = 0;
    let i = 0;
    while (comparisonValue === 0 && i < discriminators.length) {
        comparisonValue = discriminators[i](left, right);
        i++;
    }
    return comparisonValue;
}
//# sourceMappingURL=quick-file-open.js.map