"use strict";
/********************************************************************************
 * Copyright (C) 2017 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileNavigatorWidget = exports.CLASS = exports.LABEL = exports.FILE_NAVIGATOR_ID = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const uri_1 = require("@theia/core/lib/common/uri");
const common_1 = require("@theia/core/lib/common");
const browser_1 = require("@theia/core/lib/browser");
const browser_2 = require("@theia/filesystem/lib/browser");
const browser_3 = require("@theia/workspace/lib/browser");
const application_shell_1 = require("@theia/core/lib/browser/shell/application-shell");
const navigator_tree_1 = require("./navigator-tree");
const navigator_model_1 = require("./navigator-model");
const core_1 = require("@theia/core");
const React = require("@theia/core/shared/react");
const navigator_context_key_service_1 = require("./navigator-context-key-service");
const navigator_contribution_1 = require("./navigator-contribution");
const nls_1 = require("@theia/core/lib/common/nls");
exports.FILE_NAVIGATOR_ID = 'files';
exports.LABEL = nls_1.nls.localize('vscode/emptyView/noWorkspace', 'No folder opened');
exports.CLASS = 'theia-Files';
let FileNavigatorWidget = class FileNavigatorWidget extends browser_2.FileTreeWidget {
    constructor(props, model, contextMenuRenderer, commandService, selectionService, workspaceService, shell) {
        super(props, model, contextMenuRenderer);
        this.props = props;
        this.model = model;
        this.commandService = commandService;
        this.selectionService = selectionService;
        this.workspaceService = workspaceService;
        this.shell = shell;
        this.canOpenWorkspaceFileAndFolder = core_1.isOSX || !core_1.environment.electron.is();
        this.openWorkspace = () => this.doOpenWorkspace();
        this.openFolder = () => this.doOpenFolder();
        this.addFolder = () => this.doAddFolder();
        this.keyUpHandler = (e) => {
            if (browser_1.Key.ENTER.keyCode === e.keyCode) {
                e.target.click();
            }
        };
        this.id = exports.FILE_NAVIGATOR_ID;
        this.addClass(exports.CLASS);
    }
    init() {
        super.init();
        // This ensures that the context menu command to hide this widget receives the label 'Folders'
        // regardless of the name of workspace. See ViewContainer.updateToolbarItems.
        const dataset = Object.assign(Object.assign({}, this.title.dataset), { visibilityCommandLabel: nls_1.nls.localize('vscode/explorerViewlet/folders', 'Folders') });
        this.title.dataset = dataset;
        this.updateSelectionContextKeys();
        this.toDispose.pushAll([
            this.model.onSelectionChanged(() => this.updateSelectionContextKeys()),
            this.model.onExpansionChanged(node => {
                if (node.expanded && node.children.length === 1) {
                    const child = node.children[0];
                    if (browser_1.ExpandableTreeNode.is(child) && !child.expanded) {
                        this.model.expandNode(child);
                    }
                }
            })
        ]);
    }
    doUpdateRows() {
        super.doUpdateRows();
        this.title.label = exports.LABEL;
        if (navigator_tree_1.WorkspaceNode.is(this.model.root)) {
            if (this.model.root.name === navigator_tree_1.WorkspaceNode.name) {
                const rootNode = this.model.root.children[0];
                if (navigator_tree_1.WorkspaceRootNode.is(rootNode)) {
                    this.title.label = this.toNodeName(rootNode);
                    this.title.caption = this.labelProvider.getLongName(rootNode.uri);
                }
            }
            else {
                this.title.label = this.toNodeName(this.model.root);
                this.title.caption = this.title.label;
            }
        }
        else {
            this.title.caption = this.title.label;
        }
    }
    enableDndOnMainPanel() {
        const mainPanelNode = this.shell.mainPanel.node;
        this.addEventListener(mainPanelNode, 'drop', async ({ dataTransfer }) => {
            var _a;
            const treeNodes = dataTransfer && this.getSelectedTreeNodesFromData(dataTransfer) || [];
            if (treeNodes.length > 0) {
                treeNodes.filter(browser_2.FileNode.is).forEach(treeNode => {
                    if (!browser_1.SelectableTreeNode.isSelected(treeNode)) {
                        this.model.toggleNode(treeNode);
                    }
                });
                this.commandService.executeCommand(navigator_contribution_1.FileNavigatorCommands.OPEN.id);
            }
            else if (dataTransfer && ((_a = dataTransfer.files) === null || _a === void 0 ? void 0 : _a.length) > 0) {
                // the files were dragged from the outside the workspace
                Array.from(dataTransfer.files).forEach(async (file) => {
                    const fileUri = new uri_1.default(file.path);
                    const opener = await this.openerService.getOpener(fileUri);
                    opener.open(fileUri);
                });
            }
        });
        const handler = (e) => {
            if (e.dataTransfer) {
                e.dataTransfer.dropEffect = 'link';
                e.preventDefault();
            }
        };
        this.addEventListener(mainPanelNode, 'dragover', handler);
        this.addEventListener(mainPanelNode, 'dragenter', handler);
    }
    getContainerTreeNode() {
        const root = this.model.root;
        if (this.workspaceService.isMultiRootWorkspaceOpened) {
            return root;
        }
        if (navigator_tree_1.WorkspaceNode.is(root)) {
            return root.children[0];
        }
        return undefined;
    }
    renderTree(model) {
        if (this.model.root && this.isEmptyMultiRootWorkspace(model)) {
            return this.renderEmptyMultiRootWorkspace();
        }
        return super.renderTree(model);
    }
    renderTailDecorations(node, props) {
        const tailDecorations = this.getDecorationData(node, 'tailDecorations').filter(common_1.notEmpty).reduce((acc, current) => acc.concat(current), []);
        if (tailDecorations.length === 0) {
            return;
        }
        // Handle rendering of directories versus file nodes.
        if (browser_2.FileStatNode.is(node) && node.fileStat.isDirectory) {
            return this.renderTailDecorationsForDirectoryNode(node, props, tailDecorations);
        }
        else {
            return this.renderTailDecorationsForNode(node, props, tailDecorations);
        }
    }
    renderTailDecorationsForDirectoryNode(node, props, tailDecorations) {
        // If the node represents a directory, we just want to use the decorationData with the highest priority (last element).
        const decoration = tailDecorations[tailDecorations.length - 1];
        const { tooltip, fontData } = decoration;
        const color = decoration.color;
        const className = [browser_1.TREE_NODE_SEGMENT_CLASS, browser_1.TREE_NODE_TAIL_CLASS].join(' ');
        const style = fontData ? this.applyFontStyles({}, fontData) : color ? { color } : undefined;
        const content = React.createElement("span", { className: this.getIconClass('circle', [browser_1.TreeDecoration.Styles.DECORATOR_SIZE_CLASS]) });
        return React.createElement("div", { className: className, style: style, title: tooltip }, content);
    }
    shouldShowWelcomeView() {
        return this.model.root === undefined;
    }
    onAfterAttach(msg) {
        super.onAfterAttach(msg);
        this.addClipboardListener(this.node, 'copy', e => this.handleCopy(e));
        this.addClipboardListener(this.node, 'paste', e => this.handlePaste(e));
        this.enableDndOnMainPanel();
    }
    handleCopy(event) {
        const uris = this.model.selectedFileStatNodes.map(node => node.uri.toString());
        if (uris.length > 0 && event.clipboardData) {
            event.clipboardData.setData('text/plain', uris.join('\n'));
            event.preventDefault();
        }
    }
    handlePaste(event) {
        if (event.clipboardData) {
            const raw = event.clipboardData.getData('text/plain');
            if (!raw) {
                return;
            }
            const target = this.model.selectedFileStatNodes[0];
            if (!target) {
                return;
            }
            for (const file of raw.split('\n')) {
                event.preventDefault();
                const source = new uri_1.default(file);
                this.model.copy(source, target);
            }
        }
    }
    doOpenWorkspace() {
        this.commandService.executeCommand(browser_3.WorkspaceCommands.OPEN_WORKSPACE.id);
    }
    doOpenFolder() {
        this.commandService.executeCommand(browser_3.WorkspaceCommands.OPEN_FOLDER.id);
    }
    doAddFolder() {
        this.commandService.executeCommand(browser_3.WorkspaceCommands.ADD_FOLDER.id);
    }
    /**
     * When a multi-root workspace is opened, a user can remove all the folders from it.
     * Instead of displaying an empty navigator tree, this will show a button to add more folders.
     */
    renderEmptyMultiRootWorkspace() {
        // TODO: @msujew Implement a markdown renderer and use vscode/explorerViewlet/noFolderHelp
        return React.createElement("div", { className: 'theia-navigator-container' },
            React.createElement("div", { className: 'center' }, "You have not yet added a folder to the workspace."),
            React.createElement("div", { className: 'open-workspace-button-container' },
                React.createElement("button", { className: 'theia-button open-workspace-button', title: 'Add a folder to your workspace', onClick: this.addFolder, onKeyUp: this.keyUpHandler }, "Add Folder")));
    }
    isEmptyMultiRootWorkspace(model) {
        return navigator_tree_1.WorkspaceNode.is(model.root) && model.root.children.length === 0;
    }
    handleClickEvent(node, event) {
        const modifierKeyCombined = core_1.isOSX ? (event.shiftKey || event.metaKey) : (event.shiftKey || event.ctrlKey);
        if (!modifierKeyCombined && node && this.corePreferences['workbench.list.openMode'] === 'singleClick') {
            this.model.previewNode(node);
        }
        super.handleClickEvent(node, event);
    }
    onAfterShow(msg) {
        super.onAfterShow(msg);
        this.contextKeyService.explorerViewletVisible.set(true);
    }
    onAfterHide(msg) {
        super.onAfterHide(msg);
        this.contextKeyService.explorerViewletVisible.set(false);
    }
    updateSelectionContextKeys() {
        this.contextKeyService.explorerResourceIsFolder.set(browser_2.DirNode.is(this.model.selectedNodes[0]));
    }
};
__decorate([
    inversify_1.inject(browser_1.CorePreferences),
    __metadata("design:type", Object)
], FileNavigatorWidget.prototype, "corePreferences", void 0);
__decorate([
    inversify_1.inject(navigator_context_key_service_1.NavigatorContextKeyService),
    __metadata("design:type", navigator_context_key_service_1.NavigatorContextKeyService)
], FileNavigatorWidget.prototype, "contextKeyService", void 0);
__decorate([
    inversify_1.inject(browser_1.OpenerService),
    __metadata("design:type", Object)
], FileNavigatorWidget.prototype, "openerService", void 0);
__decorate([
    inversify_1.postConstruct(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", void 0)
], FileNavigatorWidget.prototype, "init", null);
FileNavigatorWidget = __decorate([
    inversify_1.injectable(),
    __param(0, inversify_1.inject(browser_1.TreeProps)),
    __param(1, inversify_1.inject(navigator_model_1.FileNavigatorModel)),
    __param(2, inversify_1.inject(browser_1.ContextMenuRenderer)),
    __param(3, inversify_1.inject(common_1.CommandService)),
    __param(4, inversify_1.inject(common_1.SelectionService)),
    __param(5, inversify_1.inject(browser_3.WorkspaceService)),
    __param(6, inversify_1.inject(application_shell_1.ApplicationShell)),
    __metadata("design:paramtypes", [Object, navigator_model_1.FileNavigatorModel,
        browser_1.ContextMenuRenderer, Object, common_1.SelectionService,
        browser_3.WorkspaceService,
        application_shell_1.ApplicationShell])
], FileNavigatorWidget);
exports.FileNavigatorWidget = FileNavigatorWidget;
//# sourceMappingURL=navigator-widget.js.map