/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
import { CoreNavigationCommands } from '../controller/coreCommands';
import { Position } from '../../common/core/position';
import * as platform from '../../../base/common/platform';
var ViewController = /** @class */ (function () {
    function ViewController(configuration, viewModel, userInputEvents, commandDelegate) {
        this.configuration = configuration;
        this.viewModel = viewModel;
        this.userInputEvents = userInputEvents;
        this.commandDelegate = commandDelegate;
    }
    ViewController.prototype.paste = function (text, pasteOnNewLine, multicursorText, mode) {
        this.commandDelegate.paste(text, pasteOnNewLine, multicursorText, mode);
    };
    ViewController.prototype.type = function (text) {
        this.commandDelegate.type(text);
    };
    ViewController.prototype.compositionType = function (text, replacePrevCharCnt, replaceNextCharCnt, positionDelta) {
        this.commandDelegate.compositionType(text, replacePrevCharCnt, replaceNextCharCnt, positionDelta);
    };
    ViewController.prototype.compositionStart = function () {
        this.commandDelegate.startComposition();
    };
    ViewController.prototype.compositionEnd = function () {
        this.commandDelegate.endComposition();
    };
    ViewController.prototype.cut = function () {
        this.commandDelegate.cut();
    };
    ViewController.prototype.setSelection = function (modelSelection) {
        CoreNavigationCommands.SetSelection.runCoreEditorCommand(this.viewModel, {
            source: 'keyboard',
            selection: modelSelection
        });
    };
    ViewController.prototype._validateViewColumn = function (viewPosition) {
        var minColumn = this.viewModel.getLineMinColumn(viewPosition.lineNumber);
        if (viewPosition.column < minColumn) {
            return new Position(viewPosition.lineNumber, minColumn);
        }
        return viewPosition;
    };
    ViewController.prototype._hasMulticursorModifier = function (data) {
        switch (this.configuration.options.get(64 /* multiCursorModifier */)) {
            case 'altKey':
                return data.altKey;
            case 'ctrlKey':
                return data.ctrlKey;
            case 'metaKey':
                return data.metaKey;
            default:
                return false;
        }
    };
    ViewController.prototype._hasNonMulticursorModifier = function (data) {
        switch (this.configuration.options.get(64 /* multiCursorModifier */)) {
            case 'altKey':
                return data.ctrlKey || data.metaKey;
            case 'ctrlKey':
                return data.altKey || data.metaKey;
            case 'metaKey':
                return data.ctrlKey || data.altKey;
            default:
                return false;
        }
    };
    ViewController.prototype.dispatchMouse = function (data) {
        var options = this.configuration.options;
        var selectionClipboardIsOn = (platform.isLinux && options.get(91 /* selectionClipboard */));
        var columnSelection = options.get(15 /* columnSelection */);
        if (data.middleButton && !selectionClipboardIsOn) {
            this._columnSelect(data.position, data.mouseColumn, data.inSelectionMode);
        }
        else if (data.startedOnLineNumbers) {
            // If the dragging started on the gutter, then have operations work on the entire line
            if (this._hasMulticursorModifier(data)) {
                if (data.inSelectionMode) {
                    this._lastCursorLineSelect(data.position);
                }
                else {
                    this._createCursor(data.position, true);
                }
            }
            else {
                if (data.inSelectionMode) {
                    this._lineSelectDrag(data.position);
                }
                else {
                    this._lineSelect(data.position);
                }
            }
        }
        else if (data.mouseDownCount >= 4) {
            this._selectAll();
        }
        else if (data.mouseDownCount === 3) {
            if (this._hasMulticursorModifier(data)) {
                if (data.inSelectionMode) {
                    this._lastCursorLineSelectDrag(data.position);
                }
                else {
                    this._lastCursorLineSelect(data.position);
                }
            }
            else {
                if (data.inSelectionMode) {
                    this._lineSelectDrag(data.position);
                }
                else {
                    this._lineSelect(data.position);
                }
            }
        }
        else if (data.mouseDownCount === 2) {
            if (this._hasMulticursorModifier(data)) {
                this._lastCursorWordSelect(data.position);
            }
            else {
                if (data.inSelectionMode) {
                    this._wordSelectDrag(data.position);
                }
                else {
                    this._wordSelect(data.position);
                }
            }
        }
        else {
            if (this._hasMulticursorModifier(data)) {
                if (!this._hasNonMulticursorModifier(data)) {
                    if (data.shiftKey) {
                        this._columnSelect(data.position, data.mouseColumn, true);
                    }
                    else {
                        // Do multi-cursor operations only when purely alt is pressed
                        if (data.inSelectionMode) {
                            this._lastCursorMoveToSelect(data.position);
                        }
                        else {
                            this._createCursor(data.position, false);
                        }
                    }
                }
            }
            else {
                if (data.inSelectionMode) {
                    if (data.altKey) {
                        this._columnSelect(data.position, data.mouseColumn, true);
                    }
                    else {
                        if (columnSelection) {
                            this._columnSelect(data.position, data.mouseColumn, true);
                        }
                        else {
                            this._moveToSelect(data.position);
                        }
                    }
                }
                else {
                    this.moveTo(data.position);
                }
            }
        }
    };
    ViewController.prototype._usualArgs = function (viewPosition) {
        viewPosition = this._validateViewColumn(viewPosition);
        return {
            source: 'mouse',
            position: this._convertViewToModelPosition(viewPosition),
            viewPosition: viewPosition
        };
    };
    ViewController.prototype.moveTo = function (viewPosition) {
        CoreNavigationCommands.MoveTo.runCoreEditorCommand(this.viewModel, this._usualArgs(viewPosition));
    };
    ViewController.prototype._moveToSelect = function (viewPosition) {
        CoreNavigationCommands.MoveToSelect.runCoreEditorCommand(this.viewModel, this._usualArgs(viewPosition));
    };
    ViewController.prototype._columnSelect = function (viewPosition, mouseColumn, doColumnSelect) {
        viewPosition = this._validateViewColumn(viewPosition);
        CoreNavigationCommands.ColumnSelect.runCoreEditorCommand(this.viewModel, {
            source: 'mouse',
            position: this._convertViewToModelPosition(viewPosition),
            viewPosition: viewPosition,
            mouseColumn: mouseColumn,
            doColumnSelect: doColumnSelect
        });
    };
    ViewController.prototype._createCursor = function (viewPosition, wholeLine) {
        viewPosition = this._validateViewColumn(viewPosition);
        CoreNavigationCommands.CreateCursor.runCoreEditorCommand(this.viewModel, {
            source: 'mouse',
            position: this._convertViewToModelPosition(viewPosition),
            viewPosition: viewPosition,
            wholeLine: wholeLine
        });
    };
    ViewController.prototype._lastCursorMoveToSelect = function (viewPosition) {
        CoreNavigationCommands.LastCursorMoveToSelect.runCoreEditorCommand(this.viewModel, this._usualArgs(viewPosition));
    };
    ViewController.prototype._wordSelect = function (viewPosition) {
        CoreNavigationCommands.WordSelect.runCoreEditorCommand(this.viewModel, this._usualArgs(viewPosition));
    };
    ViewController.prototype._wordSelectDrag = function (viewPosition) {
        CoreNavigationCommands.WordSelectDrag.runCoreEditorCommand(this.viewModel, this._usualArgs(viewPosition));
    };
    ViewController.prototype._lastCursorWordSelect = function (viewPosition) {
        CoreNavigationCommands.LastCursorWordSelect.runCoreEditorCommand(this.viewModel, this._usualArgs(viewPosition));
    };
    ViewController.prototype._lineSelect = function (viewPosition) {
        CoreNavigationCommands.LineSelect.runCoreEditorCommand(this.viewModel, this._usualArgs(viewPosition));
    };
    ViewController.prototype._lineSelectDrag = function (viewPosition) {
        CoreNavigationCommands.LineSelectDrag.runCoreEditorCommand(this.viewModel, this._usualArgs(viewPosition));
    };
    ViewController.prototype._lastCursorLineSelect = function (viewPosition) {
        CoreNavigationCommands.LastCursorLineSelect.runCoreEditorCommand(this.viewModel, this._usualArgs(viewPosition));
    };
    ViewController.prototype._lastCursorLineSelectDrag = function (viewPosition) {
        CoreNavigationCommands.LastCursorLineSelectDrag.runCoreEditorCommand(this.viewModel, this._usualArgs(viewPosition));
    };
    ViewController.prototype._selectAll = function () {
        CoreNavigationCommands.SelectAll.runCoreEditorCommand(this.viewModel, { source: 'mouse' });
    };
    // ----------------------
    ViewController.prototype._convertViewToModelPosition = function (viewPosition) {
        return this.viewModel.coordinatesConverter.convertViewPositionToModelPosition(viewPosition);
    };
    ViewController.prototype.emitKeyDown = function (e) {
        this.userInputEvents.emitKeyDown(e);
    };
    ViewController.prototype.emitKeyUp = function (e) {
        this.userInputEvents.emitKeyUp(e);
    };
    ViewController.prototype.emitContextMenu = function (e) {
        this.userInputEvents.emitContextMenu(e);
    };
    ViewController.prototype.emitMouseMove = function (e) {
        this.userInputEvents.emitMouseMove(e);
    };
    ViewController.prototype.emitMouseLeave = function (e) {
        this.userInputEvents.emitMouseLeave(e);
    };
    ViewController.prototype.emitMouseUp = function (e) {
        this.userInputEvents.emitMouseUp(e);
    };
    ViewController.prototype.emitMouseDown = function (e) {
        this.userInputEvents.emitMouseDown(e);
    };
    ViewController.prototype.emitMouseDrag = function (e) {
        this.userInputEvents.emitMouseDrag(e);
    };
    ViewController.prototype.emitMouseDrop = function (e) {
        this.userInputEvents.emitMouseDrop(e);
    };
    ViewController.prototype.emitMouseDropCanceled = function () {
        this.userInputEvents.emitMouseDropCanceled();
    };
    ViewController.prototype.emitMouseWheel = function (e) {
        this.userInputEvents.emitMouseWheel(e);
    };
    return ViewController;
}());
export { ViewController };
