/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
import { CursorColumns, SingleCursorState } from './cursorCommon';
import { Position } from '../core/position';
import { Range } from '../core/range';
import * as strings from '../../../base/common/strings';
import { AtomicTabMoveOperations } from './cursorAtomicMoveOperations';
var CursorPosition = /** @class */ (function () {
    function CursorPosition(lineNumber, column, leftoverVisibleColumns) {
        this.lineNumber = lineNumber;
        this.column = column;
        this.leftoverVisibleColumns = leftoverVisibleColumns;
    }
    return CursorPosition;
}());
export { CursorPosition };
var MoveOperations = /** @class */ (function () {
    function MoveOperations() {
    }
    MoveOperations.leftPosition = function (model, lineNumber, column) {
        if (column > model.getLineMinColumn(lineNumber)) {
            column = column - strings.prevCharLength(model.getLineContent(lineNumber), column - 1);
        }
        else if (lineNumber > 1) {
            lineNumber = lineNumber - 1;
            column = model.getLineMaxColumn(lineNumber);
        }
        return new Position(lineNumber, column);
    };
    MoveOperations.leftPositionAtomicSoftTabs = function (model, lineNumber, column, tabSize) {
        var minColumn = model.getLineMinColumn(lineNumber);
        var lineContent = model.getLineContent(lineNumber);
        var newPosition = AtomicTabMoveOperations.atomicPosition(lineContent, column - 1, tabSize, 0 /* Left */);
        if (newPosition === -1 || newPosition + 1 < minColumn) {
            return this.leftPosition(model, lineNumber, column);
        }
        return new Position(lineNumber, newPosition + 1);
    };
    MoveOperations.left = function (config, model, lineNumber, column) {
        var pos = config.stickyTabStops
            ? MoveOperations.leftPositionAtomicSoftTabs(model, lineNumber, column, config.tabSize)
            : MoveOperations.leftPosition(model, lineNumber, column);
        return new CursorPosition(pos.lineNumber, pos.column, 0);
    };
    MoveOperations.moveLeft = function (config, model, cursor, inSelectionMode, noOfColumns) {
        var lineNumber, column;
        if (cursor.hasSelection() && !inSelectionMode) {
            // If we are in selection mode, move left without selection cancels selection and puts cursor at the beginning of the selection
            lineNumber = cursor.selection.startLineNumber;
            column = cursor.selection.startColumn;
        }
        else {
            var r = MoveOperations.left(config, model, cursor.position.lineNumber, cursor.position.column - (noOfColumns - 1));
            lineNumber = r.lineNumber;
            column = r.column;
        }
        return cursor.move(inSelectionMode, lineNumber, column, 0);
    };
    MoveOperations.rightPosition = function (model, lineNumber, column) {
        if (column < model.getLineMaxColumn(lineNumber)) {
            column = column + strings.nextCharLength(model.getLineContent(lineNumber), column - 1);
        }
        else if (lineNumber < model.getLineCount()) {
            lineNumber = lineNumber + 1;
            column = model.getLineMinColumn(lineNumber);
        }
        return new Position(lineNumber, column);
    };
    MoveOperations.rightPositionAtomicSoftTabs = function (model, lineNumber, column, tabSize, indentSize) {
        var lineContent = model.getLineContent(lineNumber);
        var newPosition = AtomicTabMoveOperations.atomicPosition(lineContent, column - 1, tabSize, 1 /* Right */);
        if (newPosition === -1) {
            return this.rightPosition(model, lineNumber, column);
        }
        return new Position(lineNumber, newPosition + 1);
    };
    MoveOperations.right = function (config, model, lineNumber, column) {
        var pos = config.stickyTabStops
            ? MoveOperations.rightPositionAtomicSoftTabs(model, lineNumber, column, config.tabSize, config.indentSize)
            : MoveOperations.rightPosition(model, lineNumber, column);
        return new CursorPosition(pos.lineNumber, pos.column, 0);
    };
    MoveOperations.moveRight = function (config, model, cursor, inSelectionMode, noOfColumns) {
        var lineNumber, column;
        if (cursor.hasSelection() && !inSelectionMode) {
            // If we are in selection mode, move right without selection cancels selection and puts cursor at the end of the selection
            lineNumber = cursor.selection.endLineNumber;
            column = cursor.selection.endColumn;
        }
        else {
            var r = MoveOperations.right(config, model, cursor.position.lineNumber, cursor.position.column + (noOfColumns - 1));
            lineNumber = r.lineNumber;
            column = r.column;
        }
        return cursor.move(inSelectionMode, lineNumber, column, 0);
    };
    MoveOperations.down = function (config, model, lineNumber, column, leftoverVisibleColumns, count, allowMoveOnLastLine) {
        var currentVisibleColumn = CursorColumns.visibleColumnFromColumn(model.getLineContent(lineNumber), column, config.tabSize) + leftoverVisibleColumns;
        var lineCount = model.getLineCount();
        var wasOnLastPosition = (lineNumber === lineCount && column === model.getLineMaxColumn(lineNumber));
        lineNumber = lineNumber + count;
        if (lineNumber > lineCount) {
            lineNumber = lineCount;
            if (allowMoveOnLastLine) {
                column = model.getLineMaxColumn(lineNumber);
            }
            else {
                column = Math.min(model.getLineMaxColumn(lineNumber), column);
            }
        }
        else {
            column = CursorColumns.columnFromVisibleColumn2(config, model, lineNumber, currentVisibleColumn);
        }
        if (wasOnLastPosition) {
            leftoverVisibleColumns = 0;
        }
        else {
            leftoverVisibleColumns = currentVisibleColumn - CursorColumns.visibleColumnFromColumn(model.getLineContent(lineNumber), column, config.tabSize);
        }
        return new CursorPosition(lineNumber, column, leftoverVisibleColumns);
    };
    MoveOperations.moveDown = function (config, model, cursor, inSelectionMode, linesCount) {
        var lineNumber, column;
        if (cursor.hasSelection() && !inSelectionMode) {
            // If we are in selection mode, move down acts relative to the end of selection
            lineNumber = cursor.selection.endLineNumber;
            column = cursor.selection.endColumn;
        }
        else {
            lineNumber = cursor.position.lineNumber;
            column = cursor.position.column;
        }
        var r = MoveOperations.down(config, model, lineNumber, column, cursor.leftoverVisibleColumns, linesCount, true);
        return cursor.move(inSelectionMode, r.lineNumber, r.column, r.leftoverVisibleColumns);
    };
    MoveOperations.translateDown = function (config, model, cursor) {
        var selection = cursor.selection;
        var selectionStart = MoveOperations.down(config, model, selection.selectionStartLineNumber, selection.selectionStartColumn, cursor.selectionStartLeftoverVisibleColumns, 1, false);
        var position = MoveOperations.down(config, model, selection.positionLineNumber, selection.positionColumn, cursor.leftoverVisibleColumns, 1, false);
        return new SingleCursorState(new Range(selectionStart.lineNumber, selectionStart.column, selectionStart.lineNumber, selectionStart.column), selectionStart.leftoverVisibleColumns, new Position(position.lineNumber, position.column), position.leftoverVisibleColumns);
    };
    MoveOperations.up = function (config, model, lineNumber, column, leftoverVisibleColumns, count, allowMoveOnFirstLine) {
        var currentVisibleColumn = CursorColumns.visibleColumnFromColumn(model.getLineContent(lineNumber), column, config.tabSize) + leftoverVisibleColumns;
        var wasOnFirstPosition = (lineNumber === 1 && column === 1);
        lineNumber = lineNumber - count;
        if (lineNumber < 1) {
            lineNumber = 1;
            if (allowMoveOnFirstLine) {
                column = model.getLineMinColumn(lineNumber);
            }
            else {
                column = Math.min(model.getLineMaxColumn(lineNumber), column);
            }
        }
        else {
            column = CursorColumns.columnFromVisibleColumn2(config, model, lineNumber, currentVisibleColumn);
        }
        if (wasOnFirstPosition) {
            leftoverVisibleColumns = 0;
        }
        else {
            leftoverVisibleColumns = currentVisibleColumn - CursorColumns.visibleColumnFromColumn(model.getLineContent(lineNumber), column, config.tabSize);
        }
        return new CursorPosition(lineNumber, column, leftoverVisibleColumns);
    };
    MoveOperations.moveUp = function (config, model, cursor, inSelectionMode, linesCount) {
        var lineNumber, column;
        if (cursor.hasSelection() && !inSelectionMode) {
            // If we are in selection mode, move up acts relative to the beginning of selection
            lineNumber = cursor.selection.startLineNumber;
            column = cursor.selection.startColumn;
        }
        else {
            lineNumber = cursor.position.lineNumber;
            column = cursor.position.column;
        }
        var r = MoveOperations.up(config, model, lineNumber, column, cursor.leftoverVisibleColumns, linesCount, true);
        return cursor.move(inSelectionMode, r.lineNumber, r.column, r.leftoverVisibleColumns);
    };
    MoveOperations.translateUp = function (config, model, cursor) {
        var selection = cursor.selection;
        var selectionStart = MoveOperations.up(config, model, selection.selectionStartLineNumber, selection.selectionStartColumn, cursor.selectionStartLeftoverVisibleColumns, 1, false);
        var position = MoveOperations.up(config, model, selection.positionLineNumber, selection.positionColumn, cursor.leftoverVisibleColumns, 1, false);
        return new SingleCursorState(new Range(selectionStart.lineNumber, selectionStart.column, selectionStart.lineNumber, selectionStart.column), selectionStart.leftoverVisibleColumns, new Position(position.lineNumber, position.column), position.leftoverVisibleColumns);
    };
    MoveOperations._isBlankLine = function (model, lineNumber) {
        if (model.getLineFirstNonWhitespaceColumn(lineNumber) === 0) {
            // empty or contains only whitespace
            return true;
        }
        return false;
    };
    MoveOperations.moveToPrevBlankLine = function (config, model, cursor, inSelectionMode) {
        var lineNumber = cursor.position.lineNumber;
        // If our current line is blank, move to the previous non-blank line
        while (lineNumber > 1 && this._isBlankLine(model, lineNumber)) {
            lineNumber--;
        }
        // Find the previous blank line
        while (lineNumber > 1 && !this._isBlankLine(model, lineNumber)) {
            lineNumber--;
        }
        return cursor.move(inSelectionMode, lineNumber, model.getLineMinColumn(lineNumber), 0);
    };
    MoveOperations.moveToNextBlankLine = function (config, model, cursor, inSelectionMode) {
        var lineCount = model.getLineCount();
        var lineNumber = cursor.position.lineNumber;
        // If our current line is blank, move to the next non-blank line
        while (lineNumber < lineCount && this._isBlankLine(model, lineNumber)) {
            lineNumber++;
        }
        // Find the next blank line
        while (lineNumber < lineCount && !this._isBlankLine(model, lineNumber)) {
            lineNumber++;
        }
        return cursor.move(inSelectionMode, lineNumber, model.getLineMinColumn(lineNumber), 0);
    };
    MoveOperations.moveToBeginningOfLine = function (config, model, cursor, inSelectionMode) {
        var lineNumber = cursor.position.lineNumber;
        var minColumn = model.getLineMinColumn(lineNumber);
        var firstNonBlankColumn = model.getLineFirstNonWhitespaceColumn(lineNumber) || minColumn;
        var column;
        var relevantColumnNumber = cursor.position.column;
        if (relevantColumnNumber === firstNonBlankColumn) {
            column = minColumn;
        }
        else {
            column = firstNonBlankColumn;
        }
        return cursor.move(inSelectionMode, lineNumber, column, 0);
    };
    MoveOperations.moveToEndOfLine = function (config, model, cursor, inSelectionMode, sticky) {
        var lineNumber = cursor.position.lineNumber;
        var maxColumn = model.getLineMaxColumn(lineNumber);
        return cursor.move(inSelectionMode, lineNumber, maxColumn, sticky ? 1073741824 /* MAX_SAFE_SMALL_INTEGER */ - maxColumn : 0);
    };
    MoveOperations.moveToBeginningOfBuffer = function (config, model, cursor, inSelectionMode) {
        return cursor.move(inSelectionMode, 1, 1, 0);
    };
    MoveOperations.moveToEndOfBuffer = function (config, model, cursor, inSelectionMode) {
        var lastLineNumber = model.getLineCount();
        var lastColumn = model.getLineMaxColumn(lastLineNumber);
        return cursor.move(inSelectionMode, lastLineNumber, lastColumn, 0);
    };
    return MoveOperations;
}());
export { MoveOperations };
