/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
import { DisposableStore } from '../../../base/common/lifecycle';
var OvertypingCapturer = /** @class */ (function () {
    function OvertypingCapturer(editor, suggestModel) {
        var _this = this;
        this._disposables = new DisposableStore();
        this._lastOvertyped = [];
        this._empty = true;
        this._disposables.add(editor.onWillType(function () {
            if (!_this._empty) {
                return;
            }
            if (!editor.hasModel()) {
                return;
            }
            var selections = editor.getSelections();
            var selectionsLength = selections.length;
            // Check if it will overtype any selections
            var willOvertype = false;
            for (var i = 0; i < selectionsLength; i++) {
                if (!selections[i].isEmpty()) {
                    willOvertype = true;
                    break;
                }
            }
            if (!willOvertype) {
                return;
            }
            _this._lastOvertyped = [];
            var model = editor.getModel();
            for (var i = 0; i < selectionsLength; i++) {
                var selection = selections[i];
                // Check for overtyping capturer restrictions
                if (model.getValueLengthInRange(selection) > OvertypingCapturer._maxSelectionLength) {
                    return;
                }
                _this._lastOvertyped[i] = { value: model.getValueInRange(selection), multiline: selection.startLineNumber !== selection.endLineNumber };
            }
            _this._empty = false;
        }));
        this._disposables.add(suggestModel.onDidCancel(function (e) {
            if (!_this._empty && !e.retrigger) {
                _this._empty = true;
            }
        }));
    }
    OvertypingCapturer.prototype.getLastOvertypedInfo = function (idx) {
        if (!this._empty && idx >= 0 && idx < this._lastOvertyped.length) {
            return this._lastOvertyped[idx];
        }
        return undefined;
    };
    OvertypingCapturer.prototype.dispose = function () {
        this._disposables.dispose();
    };
    OvertypingCapturer._maxSelectionLength = 51200;
    return OvertypingCapturer;
}());
export { OvertypingCapturer };
