/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
import './inspectTokens.css';
import { $, append, reset } from '../../../../base/browser/dom';
import { Color } from '../../../../base/common/color';
import { Disposable } from '../../../../base/common/lifecycle';
import { EditorAction, registerEditorAction, registerEditorContribution } from '../../../browser/editorExtensions';
import { TokenMetadata, TokenizationRegistry } from '../../../common/modes';
import { NULL_STATE, nullTokenize, nullTokenize2 } from '../../../common/modes/nullMode';
import { IModeService } from '../../../common/services/modeService';
import { IStandaloneThemeService } from '../../common/standaloneThemeService';
import { editorHoverBackground, editorHoverBorder, editorHoverForeground } from '../../../../platform/theme/common/colorRegistry';
import { registerThemingParticipant } from '../../../../platform/theme/common/themeService';
import { InspectTokensNLS } from '../../../common/standaloneStrings';
import { ColorScheme } from '../../../../platform/theme/common/theme';
var InspectTokensController = /** @class */ (function (_super) {
    __extends(InspectTokensController, _super);
    function InspectTokensController(editor, standaloneColorService, modeService) {
        var _this = _super.call(this) || this;
        _this._editor = editor;
        _this._modeService = modeService;
        _this._widget = null;
        _this._register(_this._editor.onDidChangeModel(function (e) { return _this.stop(); }));
        _this._register(_this._editor.onDidChangeModelLanguage(function (e) { return _this.stop(); }));
        _this._register(TokenizationRegistry.onDidChange(function (e) { return _this.stop(); }));
        _this._register(_this._editor.onKeyUp(function (e) { return e.keyCode === 9 /* Escape */ && _this.stop(); }));
        return _this;
    }
    InspectTokensController.get = function (editor) {
        return editor.getContribution(InspectTokensController.ID);
    };
    InspectTokensController.prototype.dispose = function () {
        this.stop();
        _super.prototype.dispose.call(this);
    };
    InspectTokensController.prototype.launch = function () {
        if (this._widget) {
            return;
        }
        if (!this._editor.hasModel()) {
            return;
        }
        this._widget = new InspectTokensWidget(this._editor, this._modeService);
    };
    InspectTokensController.prototype.stop = function () {
        if (this._widget) {
            this._widget.dispose();
            this._widget = null;
        }
    };
    InspectTokensController.ID = 'editor.contrib.inspectTokens';
    InspectTokensController = __decorate([
        __param(1, IStandaloneThemeService),
        __param(2, IModeService)
    ], InspectTokensController);
    return InspectTokensController;
}(Disposable));
var InspectTokens = /** @class */ (function (_super) {
    __extends(InspectTokens, _super);
    function InspectTokens() {
        return _super.call(this, {
            id: 'editor.action.inspectTokens',
            label: InspectTokensNLS.inspectTokensAction,
            alias: 'Developer: Inspect Tokens',
            precondition: undefined
        }) || this;
    }
    InspectTokens.prototype.run = function (accessor, editor) {
        var controller = InspectTokensController.get(editor);
        if (controller) {
            controller.launch();
        }
    };
    return InspectTokens;
}(EditorAction));
function renderTokenText(tokenText) {
    var result = '';
    for (var charIndex = 0, len = tokenText.length; charIndex < len; charIndex++) {
        var charCode = tokenText.charCodeAt(charIndex);
        switch (charCode) {
            case 9 /* Tab */:
                result += '\u2192'; // &rarr;
                break;
            case 32 /* Space */:
                result += '\u00B7'; // &middot;
                break;
            default:
                result += String.fromCharCode(charCode);
        }
    }
    return result;
}
function getSafeTokenizationSupport(languageIdentifier) {
    var tokenizationSupport = TokenizationRegistry.get(languageIdentifier.language);
    if (tokenizationSupport) {
        return tokenizationSupport;
    }
    return {
        getInitialState: function () { return NULL_STATE; },
        tokenize: function (line, hasEOL, state, deltaOffset) { return nullTokenize(languageIdentifier.language, line, state, deltaOffset); },
        tokenize2: function (line, hasEOL, state, deltaOffset) { return nullTokenize2(languageIdentifier.id, line, state, deltaOffset); }
    };
}
var InspectTokensWidget = /** @class */ (function (_super) {
    __extends(InspectTokensWidget, _super);
    function InspectTokensWidget(editor, modeService) {
        var _this = _super.call(this) || this;
        // Editor.IContentWidget.allowEditorOverflow
        _this.allowEditorOverflow = true;
        _this._editor = editor;
        _this._modeService = modeService;
        _this._model = _this._editor.getModel();
        _this._domNode = document.createElement('div');
        _this._domNode.className = 'tokens-inspect-widget';
        _this._tokenizationSupport = getSafeTokenizationSupport(_this._model.getLanguageIdentifier());
        _this._compute(_this._editor.getPosition());
        _this._register(_this._editor.onDidChangeCursorPosition(function (e) { return _this._compute(_this._editor.getPosition()); }));
        _this._editor.addContentWidget(_this);
        return _this;
    }
    InspectTokensWidget.prototype.dispose = function () {
        this._editor.removeContentWidget(this);
        _super.prototype.dispose.call(this);
    };
    InspectTokensWidget.prototype.getId = function () {
        return InspectTokensWidget._ID;
    };
    InspectTokensWidget.prototype._compute = function (position) {
        var data = this._getTokensAtLine(position.lineNumber);
        var token1Index = 0;
        for (var i = data.tokens1.length - 1; i >= 0; i--) {
            var t = data.tokens1[i];
            if (position.column - 1 >= t.offset) {
                token1Index = i;
                break;
            }
        }
        var token2Index = 0;
        for (var i = (data.tokens2.length >>> 1); i >= 0; i--) {
            if (position.column - 1 >= data.tokens2[(i << 1)]) {
                token2Index = i;
                break;
            }
        }
        var lineContent = this._model.getLineContent(position.lineNumber);
        var tokenText = '';
        if (token1Index < data.tokens1.length) {
            var tokenStartIndex = data.tokens1[token1Index].offset;
            var tokenEndIndex = token1Index + 1 < data.tokens1.length ? data.tokens1[token1Index + 1].offset : lineContent.length;
            tokenText = lineContent.substring(tokenStartIndex, tokenEndIndex);
        }
        reset(this._domNode, $('h2.tm-token', undefined, renderTokenText(tokenText), $('span.tm-token-length', undefined, tokenText.length + " " + (tokenText.length === 1 ? 'char' : 'chars'))));
        append(this._domNode, $('hr.tokens-inspect-separator', { 'style': 'clear:both' }));
        var metadata = (token2Index << 1) + 1 < data.tokens2.length ? this._decodeMetadata(data.tokens2[(token2Index << 1) + 1]) : null;
        append(this._domNode, $('table.tm-metadata-table', undefined, $('tbody', undefined, $('tr', undefined, $('td.tm-metadata-key', undefined, 'language'), $('td.tm-metadata-value', undefined, "" + (metadata ? metadata.languageIdentifier.language : '-?-'))), $('tr', undefined, $('td.tm-metadata-key', undefined, 'token type'), $('td.tm-metadata-value', undefined, "" + (metadata ? this._tokenTypeToString(metadata.tokenType) : '-?-'))), $('tr', undefined, $('td.tm-metadata-key', undefined, 'font style'), $('td.tm-metadata-value', undefined, "" + (metadata ? this._fontStyleToString(metadata.fontStyle) : '-?-'))), $('tr', undefined, $('td.tm-metadata-key', undefined, 'foreground'), $('td.tm-metadata-value', undefined, "" + (metadata ? Color.Format.CSS.formatHex(metadata.foreground) : '-?-'))), $('tr', undefined, $('td.tm-metadata-key', undefined, 'background'), $('td.tm-metadata-value', undefined, "" + (metadata ? Color.Format.CSS.formatHex(metadata.background) : '-?-'))))));
        append(this._domNode, $('hr.tokens-inspect-separator'));
        if (token1Index < data.tokens1.length) {
            append(this._domNode, $('span.tm-token-type', undefined, data.tokens1[token1Index].type));
        }
        this._editor.layoutContentWidget(this);
    };
    InspectTokensWidget.prototype._decodeMetadata = function (metadata) {
        var colorMap = TokenizationRegistry.getColorMap();
        var languageId = TokenMetadata.getLanguageId(metadata);
        var tokenType = TokenMetadata.getTokenType(metadata);
        var fontStyle = TokenMetadata.getFontStyle(metadata);
        var foreground = TokenMetadata.getForeground(metadata);
        var background = TokenMetadata.getBackground(metadata);
        return {
            languageIdentifier: this._modeService.getLanguageIdentifier(languageId),
            tokenType: tokenType,
            fontStyle: fontStyle,
            foreground: colorMap[foreground],
            background: colorMap[background]
        };
    };
    InspectTokensWidget.prototype._tokenTypeToString = function (tokenType) {
        switch (tokenType) {
            case 0 /* Other */: return 'Other';
            case 1 /* Comment */: return 'Comment';
            case 2 /* String */: return 'String';
            case 4 /* RegEx */: return 'RegEx';
            default: return '??';
        }
    };
    InspectTokensWidget.prototype._fontStyleToString = function (fontStyle) {
        var r = '';
        if (fontStyle & 1 /* Italic */) {
            r += 'italic ';
        }
        if (fontStyle & 2 /* Bold */) {
            r += 'bold ';
        }
        if (fontStyle & 4 /* Underline */) {
            r += 'underline ';
        }
        if (r.length === 0) {
            r = '---';
        }
        return r;
    };
    InspectTokensWidget.prototype._getTokensAtLine = function (lineNumber) {
        var stateBeforeLine = this._getStateBeforeLine(lineNumber);
        var tokenizationResult1 = this._tokenizationSupport.tokenize(this._model.getLineContent(lineNumber), true, stateBeforeLine, 0);
        var tokenizationResult2 = this._tokenizationSupport.tokenize2(this._model.getLineContent(lineNumber), true, stateBeforeLine, 0);
        return {
            startState: stateBeforeLine,
            tokens1: tokenizationResult1.tokens,
            tokens2: tokenizationResult2.tokens,
            endState: tokenizationResult1.endState
        };
    };
    InspectTokensWidget.prototype._getStateBeforeLine = function (lineNumber) {
        var state = this._tokenizationSupport.getInitialState();
        for (var i = 1; i < lineNumber; i++) {
            var tokenizationResult = this._tokenizationSupport.tokenize(this._model.getLineContent(i), true, state, 0);
            state = tokenizationResult.endState;
        }
        return state;
    };
    InspectTokensWidget.prototype.getDomNode = function () {
        return this._domNode;
    };
    InspectTokensWidget.prototype.getPosition = function () {
        return {
            position: this._editor.getPosition(),
            preference: [2 /* BELOW */, 1 /* ABOVE */]
        };
    };
    InspectTokensWidget._ID = 'editor.contrib.inspectTokensWidget';
    return InspectTokensWidget;
}(Disposable));
registerEditorContribution(InspectTokensController.ID, InspectTokensController);
registerEditorAction(InspectTokens);
registerThemingParticipant(function (theme, collector) {
    var border = theme.getColor(editorHoverBorder);
    if (border) {
        var borderWidth = theme.type === ColorScheme.HIGH_CONTRAST ? 2 : 1;
        collector.addRule(".monaco-editor .tokens-inspect-widget { border: " + borderWidth + "px solid " + border + "; }");
        collector.addRule(".monaco-editor .tokens-inspect-widget .tokens-inspect-separator { background-color: " + border + "; }");
    }
    var background = theme.getColor(editorHoverBackground);
    if (background) {
        collector.addRule(".monaco-editor .tokens-inspect-widget { background-color: " + background + "; }");
    }
    var foreground = theme.getColor(editorHoverForeground);
    if (foreground) {
        collector.addRule(".monaco-editor .tokens-inspect-widget { color: " + foreground + "; }");
    }
});
