/* GLIB - Library of useful routines for C programming
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Modified by the GLib Team and others 1997-2000.  See the AUTHORS
 * file for a list of people on the GLib Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GLib at ftp://ftp.gtk.org/pub/gtk/. 
 */

/* 
 * MT safe
 */

#include "config.h"
#include "glibconfig.h"

#include <stdlib.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#ifndef G_OS_WIN32
#include <sys/time.h>
#include <time.h>
#include <errno.h>
#endif /* G_OS_WIN32 */

#ifdef G_OS_WIN32
#include <windows.h>
#endif /* G_OS_WIN32 */

#include "glib.h"
#include "galias.h"

#define G_NSEC_PER_SEC 1000000000

struct _GTimer
{
#ifdef G_OS_WIN32
  guint64 start;
  guint64 end;
#elif HAVE_CLOCK_GETTIME 
  struct timespec start;
  struct timespec end;
  gint clock;
#else /* uses gettimeofday */
  struct timeval start;
  struct timeval end;
#endif 

  guint active : 1;
};

#ifdef G_OS_WIN32
#  define GETTIME(v)				\
     GetSystemTimeAsFileTime ((FILETIME *)&v)
#elif HAVE_CLOCK_GETTIME
#  define GETTIME(v)				\
     clock_gettime (posix_clock, &v)
#else
#  define GETTIME(v)				\
     gettimeofday (&v, NULL)
#endif

#ifdef HAVE_CLOCK_GETTIME
static gint posix_clock = 0;

static void
init_posix_clock (void)
{
  static gboolean initialized = FALSE;

  if (!initialized)
    {
      initialized = TRUE;
#if !defined(_POSIX_MONOTONIC_CLOCK) || _POSIX_MONOTONIC_CLOCK >= 0
      if (sysconf (_SC_MONOTONIC_CLOCK) >= 0)
	posix_clock = CLOCK_MONOTONIC;
      else
#endif
	posix_clock = CLOCK_REALTIME;
    }
}
#endif

GTimer*
g_timer_new (void)
{
  GTimer *timer;

  timer = g_new (GTimer, 1);
  timer->active = TRUE;

#ifdef HAVE_CLOCK_GETTIME
  init_posix_clock ();
#endif

  GETTIME (timer->start);

  return timer;
}

void
g_timer_destroy (GTimer *timer)
{
  g_return_if_fail (timer != NULL);

  g_free (timer);
}

void
g_timer_start (GTimer *timer)
{
  g_return_if_fail (timer != NULL);

  timer->active = TRUE;

  GETTIME (timer->start);
}

void
g_timer_stop (GTimer *timer)
{
  g_return_if_fail (timer != NULL);

  timer->active = FALSE;

  GETTIME(timer->end);
}

void
g_timer_reset (GTimer *timer)
{
  g_return_if_fail (timer != NULL);

  GETTIME (timer->start);
}

void
g_timer_continue (GTimer *timer)
{
#ifdef G_OS_WIN32
  guint64 elapsed;
#elif HAVE_CLOCK_GETTIME
  struct timespec elapsed;
#else
  struct timeval elapsed;
#endif

  g_return_if_fail (timer != NULL);
  g_return_if_fail (timer->active == FALSE);

  /* Get elapsed time and reset timer start time
   *  to the current time minus the previously
   *  elapsed interval.
   */

#ifdef G_OS_WIN32

  elapsed = timer->end - timer->start;

  GETTIME (timer->start);

  timer->start -= elapsed;

#elif HAVE_CLOCK_GETTIME

  if (timer->start.tv_nsec > timer->end.tv_nsec)
    {
      timer->end.tv_nsec += G_NSEC_PER_SEC;
      timer->end.tv_sec--;
    }

  elapsed.tv_nsec = timer->end.tv_nsec - timer->start.tv_nsec;
  elapsed.tv_sec = timer->end.tv_sec - timer->start.tv_sec;

  GETTIME (timer->start);

  if (timer->start.tv_nsec < elapsed.tv_nsec)
    {
      timer->start.tv_nsec += G_NSEC_PER_SEC;
      timer->start.tv_sec--;
    }

  timer->start.tv_nsec -= elapsed.tv_nsec;
  timer->start.tv_sec -= elapsed.tv_sec;

#else

  if (timer->start.tv_usec > timer->end.tv_usec)
    {
      timer->end.tv_usec += G_USEC_PER_SEC;
      timer->end.tv_sec--;
    }

  elapsed.tv_usec = timer->end.tv_usec - timer->start.tv_usec;
  elapsed.tv_sec = timer->end.tv_sec - timer->start.tv_sec;

  GETTIME (timer->start);

  if (timer->start.tv_usec < elapsed.tv_usec)
    {
      timer->start.tv_usec += G_USEC_PER_SEC;
      timer->start.tv_sec--;
    }

  timer->start.tv_usec -= elapsed.tv_usec;
  timer->start.tv_sec -= elapsed.tv_sec;

#endif /* !G_OS_WIN32 */

  timer->active = TRUE;
}

gdouble
g_timer_elapsed (GTimer *timer,
		 gulong *microseconds)
{
  gdouble total;
#ifdef G_OS_WIN32
  gint64 elapsed;
#elif HAVE_CLOCK_GETTIME
  struct timespec elapsed;
#else
  struct timeval elapsed;
#endif 

  g_return_val_if_fail (timer != NULL, 0);

#ifdef G_OS_WIN32
  if (timer->active)
    GETTIME (timer->end);

  elapsed = timer->end - timer->start;

  total = elapsed / 1e7;

  if (microseconds)
    *microseconds = (elapsed / 10) % 1000000;
#elif HAVE_CLOCK_GETTIME
  if (timer->active)
    GETTIME (timer->end);

  if (timer->start.tv_nsec > timer->end.tv_nsec)
    {
      timer->end.tv_nsec += G_NSEC_PER_SEC;
      timer->end.tv_sec--;
    }

  elapsed.tv_nsec = timer->end.tv_nsec - timer->start.tv_nsec;
  elapsed.tv_sec = timer->end.tv_sec - timer->start.tv_sec;

  total = elapsed.tv_sec + ((gdouble) elapsed.tv_nsec / (gdouble) G_NSEC_PER_SEC);
  if (total < 0)
    {
      total = 0;

      if (microseconds)
	*microseconds = 0;
    }
  else if (microseconds)
    *microseconds = elapsed.tv_nsec / 1000;

#else
  if (timer->active)
    GETTIME (timer->end);

  if (timer->start.tv_usec > timer->end.tv_usec)
    {
      timer->end.tv_usec += G_USEC_PER_SEC;
      timer->end.tv_sec--;
    }

  elapsed.tv_usec = timer->end.tv_usec - timer->start.tv_usec;
  elapsed.tv_sec = timer->end.tv_sec - timer->start.tv_sec;

  total = elapsed.tv_sec + ((gdouble) elapsed.tv_usec / (gdouble) G_USEC_PER_SEC);
  if (total < 0)
    {
      total = 0;

      if (microseconds)
	*microseconds = 0;
    }
  else if (microseconds)
    *microseconds = elapsed.tv_usec;

#endif

  return total;
}

void
g_usleep (gulong microseconds)
{
#ifdef G_OS_WIN32
  Sleep (microseconds / 1000);
#else /* !G_OS_WIN32 */
# ifdef HAVE_NANOSLEEP
  struct timespec request, remaining;
  request.tv_sec = microseconds / G_USEC_PER_SEC;
  request.tv_nsec = 1000 * (microseconds % G_USEC_PER_SEC);
  while (nanosleep (&request, &remaining) == -1 && errno == EINTR)
    request = remaining;
# else /* !HAVE_NANOSLEEP */
  if (g_thread_supported ())
    {
      static GStaticMutex mutex = G_STATIC_MUTEX_INIT;
      static GCond* cond = NULL;
      GTimeVal end_time;
      
      g_get_current_time (&end_time);
      if (microseconds > G_MAXLONG)
	{
	  microseconds -= G_MAXLONG;
	  g_time_val_add (&end_time, G_MAXLONG);
	}
      g_time_val_add (&end_time, microseconds);

      g_static_mutex_lock (&mutex);
      
      if (!cond)
	cond = g_cond_new ();
      
      while (g_cond_timed_wait (cond, g_static_mutex_get_mutex (&mutex), 
				&end_time))
	/* do nothing */;
      
      g_static_mutex_unlock (&mutex);
    }
  else
    {
      struct timeval tv;
      tv.tv_sec = microseconds / G_USEC_PER_SEC;
      tv.tv_usec = microseconds % G_USEC_PER_SEC;
      select(0, NULL, NULL, NULL, &tv);
    }
# endif /* !HAVE_NANOSLEEP */
#endif /* !G_OS_WIN32 */
}

/**
 * g_time_val_add:
 * @time_: a #GTimeVal
 * @microseconds: number of microseconds to add to @time
 *
 * Adds the given number of microseconds to @time_. @microseconds can
 * also be negative to decrease the value of @time_.
 **/
void 
g_time_val_add (GTimeVal *time_, glong microseconds)
{
  g_return_if_fail (time_->tv_usec >= 0 && time_->tv_usec < G_USEC_PER_SEC);

  if (microseconds >= 0)
    {
      time_->tv_usec += microseconds % G_USEC_PER_SEC;
      time_->tv_sec += microseconds / G_USEC_PER_SEC;
      if (time_->tv_usec >= G_USEC_PER_SEC)
       {
         time_->tv_usec -= G_USEC_PER_SEC;
         time_->tv_sec++;
       }
    }
  else
    {
      microseconds *= -1;
      time_->tv_usec -= microseconds % G_USEC_PER_SEC;
      time_->tv_sec -= microseconds / G_USEC_PER_SEC;
      if (time_->tv_usec < 0)
       {
         time_->tv_usec += G_USEC_PER_SEC;
         time_->tv_sec--;
       }      
    }
}

/* converts a broken down date representation, relative to UTC, to
 * a timestamp; it uses timegm() if it's available.
 */
static time_t
mktime_utc (struct tm *tm)
{
  time_t retval;
  
#ifndef HAVE_TIMEGM
  static const gint days_before[] =
  {
    0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334
  };
#endif

#ifndef HAVE_TIMEGM
  if (tm->tm_mon < 0 || tm->tm_mon > 11)
    return (time_t) -1;

  retval = (tm->tm_year - 70) * 365;
  retval += (tm->tm_year - 68) / 4;
  retval += days_before[tm->tm_mon] + tm->tm_mday - 1;
  
  if (tm->tm_year % 4 == 0 && tm->tm_mon < 2)
    retval -= 1;
  
  retval = ((((retval * 24) + tm->tm_hour) * 60) + tm->tm_min) * 60 + tm->tm_sec;
#else
  retval = timegm (tm);
#endif /* !HAVE_TIMEGM */
  
  return retval;
}

/**
 * g_time_val_from_iso8601:
 * @iso_date: a ISO 8601 encoded date string
 * @time_: a #GTimeVal
 *
 * Converts a string containing an ISO 8601 encoded date and time
 * to a #GTimeVal and puts it into @time_.
 *
 * Return value: %TRUE if the conversion was successful.
 *
 * Since: 2.12
 */
gboolean
g_time_val_from_iso8601 (const gchar *iso_date,
			 GTimeVal    *time_)
{
  struct tm tm;
  long val;

  g_return_val_if_fail (iso_date != NULL, FALSE);
  g_return_val_if_fail (time_ != NULL, FALSE);

  val = strtoul (iso_date, (char **)&iso_date, 10);
  if (*iso_date == '-')
    {
      /* YYYY-MM-DD */
      tm.tm_year = val - 1900;
      iso_date++;
      tm.tm_mon = strtoul (iso_date, (char **)&iso_date, 10) - 1;
      
      if (*iso_date++ != '-')
       	return FALSE;
      
      tm.tm_mday = strtoul (iso_date, (char **)&iso_date, 10);
    }
  else
    {
      /* YYYYMMDD */
      tm.tm_mday = val % 100;
      tm.tm_mon = (val % 10000) / 100 - 1;
      tm.tm_year = val / 10000 - 1900;
    }

  if (*iso_date++ != 'T')
    return FALSE;
  
  val = strtoul (iso_date, (char **)&iso_date, 10);
  if (*iso_date == ':')
    {
      /* hh:mm:ss */
      tm.tm_hour = val;
      iso_date++;
      tm.tm_min = strtoul (iso_date, (char **)&iso_date, 10);
      
      if (*iso_date++ != ':')
        return FALSE;
      
      tm.tm_sec = strtoul (iso_date, (char **)&iso_date, 10);
    }
  else
    {
      /* hhmmss */
      tm.tm_sec = val % 100;
      tm.tm_min = (val % 10000) / 100;
      tm.tm_hour = val / 10000;
    }

  time_->tv_sec = mktime_utc (&tm);
  time_->tv_usec = 1;
  
  if (*iso_date == '.')
    time_->tv_usec = strtoul (iso_date + 1, (char **)&iso_date, 10);
    
  if (*iso_date == '+' || *iso_date == '-')
    {
      gint sign = (*iso_date == '+') ? -1 : 1;
      
      val = 60 * strtoul (iso_date + 1, (char **)&iso_date, 10);
      
      if (*iso_date == ':')
	val = 60 * val + strtoul (iso_date + 1, NULL, 10);
      else
        val = 60 * (val / 100) + (val % 100);

      time_->tv_sec += (time_t) (val * sign);
    }

  return TRUE;
}

/**
 * g_time_val_to_iso8601:
 * @time_: a #GTimeVal
 * 
 * Converts @time_ into a ISO 8601 encoded string, relative to the
 * Coordinated Universal Time (UTC).
 *
 * Return value: a newly allocated string containing a ISO 8601 date
 *
 * Since: 2.12
 */
gchar *
g_time_val_to_iso8601 (GTimeVal *time_)
{
  gchar *retval;

  g_return_val_if_fail (time_->tv_usec >= 0 && time_->tv_usec < G_USEC_PER_SEC, NULL);

#define ISO_8601_LEN 	21
#define ISO_8601_FORMAT "%Y-%m-%dT%H:%M:%SZ"
  retval = g_new0 (gchar, ISO_8601_LEN + 1);
  
  strftime (retval, ISO_8601_LEN,
	    ISO_8601_FORMAT,
	    gmtime (&(time_->tv_sec)));
  
  return retval;
}

#define __G_TIMER_C__
#include "galiasdef.c"
