/* Copyright © Charliecloud contributors. */

#define _GNU_SOURCE
#include "config.h"

#include <dirent.h>
#include <fnmatch.h>
#include <libgen.h>
#include <string.h>
#include <sys/mount.h>
#include <sys/statvfs.h>
#include <unistd.h>

#include "all.h"


/** Function prototypes (private) **/

void mkdir_overmount(const char *path, const char *scratch);


/** Functions **/

/* Return a newly-allocated, null-terminated list of filenames in directory
   path that match fnmatch(3)-pattern glob, excluding “.” and “..”. For a list
   of everything, pass "*" for glob. Leading dots *do* match “*”.

   We use readdir(3) rather than scandir(3) because the latter allocates
   memory with malloc(3). */
char **dir_glob(const char *path, const char *glob)
{
   DIR *dp;
   char **entries = list_new(16, sizeof(char *));

   Tfe (dp = opendir(path), "can't open directory: %s", path);
   while (true) {
      struct dirent *entry;
      int matchp;
      errno = 0;
      entry = readdir(dp);
      if (entry == NULL) {
         Zfe (errno, "can’t read directory: %s", path);
         break;  // EOF
      }
      if (streq(entry->d_name, ".") || streq(entry->d_name, ".."))
         continue;
      matchp = fnmatch(glob, entry->d_name, FNM_EXTMATCH);
      if (matchp != 0)   // glob doesn’t match or error
         T__ (matchp == FNM_NOMATCH);
      else {             // does match
         char *name = strdup_ch(entry->d_name);  // entry gets overwritten
         list_append((void **)&entries, &name, sizeof(char *));
      }
   }
   Zfe (closedir(dp), "can't close directory: %s", path);

   return entries;
}

/* Return the number of matches for glob in path. */
int dir_glob_count(const char *path, const char *glob)
{
   return list_count(dir_glob(path, glob), sizeof(char *));
}

/** Get the parent directory of a path.

    @param path  Path to parse.

    @returns the directory portion of @p path in a newly allocated string.

    This has the same API as @c dirname(3) except that @p path is not modified
    and the result is a new buffer. */
#undef dirname
char *dirname_ch(const char *path)
{
   // Use dirname(3) to ensure it works the same. The first copy is because
   // dirname(3) might modify its argument; the second is because the returned
   // pointer might be to some static buffer we don’t own.
   return strdup_ch(dirname(strdup_ch(path)));
}
#define dirname FN_BLOCKED

/** Return path of current working directory as a newly allocated string.

    Eliminates the need for getcwd(3) callers to either provide or free a
    buffer to hold the path.

    On error, exit with error message. */
#undef getcwd
char *getcwd_ch(void)
{
   char *buf;
   size_t sz;

   // PATH_MAX is unreliable [1], so loop on buffer size until it’s large
   // enough. For starting size, I guessed at something that would usually be
   // sufficient, but too small often enough that multiple iterations is
   // exercised reasonably frequently.
   buf = NULL;
   sz = 64;
   while (true) {
      buf = realloc_ch(buf, sz, false);
      if (getcwd(buf, sz))
         return buf;
      else {
         Tfe (errno == ERANGE, "can't getcwd(3)");
         sz *= 2;
      }
   }

   return buf;
}
#define getcwd FN_BLOCKED

/* Return a new null-terminated string containing the next record from fp,
   where records are delimited by delim (e.g., pass '\n' to get the next
   line). If no more records available, return NULL. Exit on error.

   Unlike getdelim(3), the delimiter is *not* part of the returned string.

   Warnings:

     1. Records cannot contain the zero byte, and behavior is undefined if fp
        containes any zeros and delimiter is not '\0'.

     2. The returned buffer is likely larger than needed. We assume wasting
        this space is better than the overhead of realloc’ing down to a
        precise size. */
char *getdelim_ch(FILE *fp, char delim)
{
   size_t bytes_read = 0;
   size_t buf_len = 8;  // non-zero start avoids early frequent realloc
   char *buf = malloc_ch(buf_len, false);

   while (true) {
      int c = fgetc(fp);
      if (c == EOF)
         break;
      bytes_read++;
      if (bytes_read > buf_len) {      // room for terminator ensured later
         buf_len *= 2;
         buf = realloc_ch(buf, buf_len, false);
      }
      buf[bytes_read-1] = c;
      if (c == delim)
         break;
   }

   if (bytes_read >= 1 && buf[bytes_read-1] == delim)  // found delimiter
      buf[bytes_read-1] = '\0';  // replace delimiter with terminator
   else {
      Tfe (feof(fp), "error reading file");  // filename unknown here
      if (bytes_read == 0)  // no record left
         return NULL;
      else {                // record ends at EOF (no delimiter)
         if (bytes_read >= buf_len) {
            T__ (bytes_read == buf_len);
            buf = realloc_ch(buf, buf_len + 1, false);
         }
         buf[bytes_read] = '\0';  // append terminator
      }
   }

   return buf;
}

/* Create the directory at path, despite its parent not allowing write access,
   by overmounting a new, writeable directory atop it. We preserve the old
   contents by bind-mounting the old directory as a subdirectory, then setting
   up a symlink ranch.

   The new directory lives initially in scratch, which must not be used for
   any other purpose. No cleanup is done here, so a disposable tmpfs is best.
   If anything goes wrong, exit with an error message. */
void mkdir_overmount(const char *path, const char *scratch)
{
   char *parent, *over, *path_dst;
   char *orig_dir = ".orig";  // resisted calling this .weirdal
   int entry_ct;
   char **entries;

   parent = dirname_ch(path);
   VERBOSE("making writeable via symlink ranch: %s", parent);
   over = asprintf_ch("%s/%d", scratch, dir_glob_count(scratch, "*") + 1);
   path_dst = path_join(over, orig_dir);

   // bind-mounts
   Z_e (mkdir(over, 0755));
   Z_e (mkdir(path_dst, 0755));
   Zfe (mount(parent, path_dst, NULL, MS_BIND|MS_REC, NULL),
        "can't bind-mount: %s -> %s", parent, path_dst);
   Zfe (mount(over, parent, NULL, MS_BIND|MS_REC, NULL),
        "can't bind-mount: %s- > %s", over, parent);

   // symlink ranch
   entries = dir_glob(path_dst, "*");
   entry_ct = list_count(entries, sizeof(entries[0]));
   DEBUG("existing entries: %d", entry_ct);
   for (int i = 0; i < entry_ct; i++) {
      char * src = path_join(parent, entries[i]);
      char * dst = path_join(orig_dir, entries[i]);
      Zfe (symlink(dst, src), "can't symlink: %s -> %s", src, dst);
   }

   Zfe (mkdir(path, 0755), "can't mkdir even after overmount: %s", path);
}

/* Create directories in path under base. Exit with an error if anything goes
   wrong. For example, mkdirs("/foo", "/bar/baz") will create directories
   /foo/bar and /foo/bar/baz if they don't already exist, but /foo must exist
   already. Symlinks are followed. path must remain under base, i.e. you can't
   use symlinks or ".." to climb out. denylist is a null-terminated array of
   paths under which no directories may be created, or NULL if none.

   Can defeat an un-writeable directory by overmounting a new writeable
   directory atop it. To enable this behavior, pass the path to an appropriate
   scratch directory in scratch. */
void mkdirs(const char *base, const char *path, char **denylist,
            const char *scratch)
{
   char *basec, *component, *next, *nextc, *pathw, *saveptr;
   char *denylist_null[] = { NULL };
   struct stat sb;

   T__ (base[0] != 0   && path[0] != 0);      // no empty paths
   T__ (base[0] == '/' && path[0] == '/');    // absolute paths only
   if (denylist == NULL)
      denylist = denylist_null;  // literal here causes intermittent segfaults

   basec = realpath_ch(base);
   T__ (path_exists(basec, NULL, false));

   TRACE("mkdirs: base: %s", basec);
   TRACE("mkdirs: path: %s", path);
   for (int i = 0; denylist[i] != NULL; i++)
      TRACE("mkdirs: deny: %s", denylist[i]);

   pathw = strdup_ch(path);  // writeable copy
   saveptr = NULL;           // avoid warning (#1048; see also strtok_r(3))
   component = strtok_r(pathw, "/", &saveptr);
   nextc = basec;
   next = NULL;
   while (component != NULL) {
      next = path_join(nextc, component);  // canonical except for last
      TRACE("mkdirs: next: %s", next);
      component = strtok_r(NULL, "/", &saveptr);  // next NULL if current last
      if (path_exists(next, &sb, false)) {
         if (S_ISLNK(sb.st_mode)) {
            char buf;                             // we only care if absolute
            Tfe (1 == readlink(next, &buf, 1), "can't read symlink: %s", next);
            Tf_ (buf != '/', "can't mkdir: symlink not relative: %s", next);
            Tf_ (path_exists(next, &sb, true),    // resolve symlink
                 "can't mkdir: broken symlink: %s", next);
         }
         Tf_ (S_ISDIR(sb.st_mode) || !component,   // last component not dir OK
              "can't mkdir: exists but not a directory: %s", next);
         nextc = realpath_ch(next);
         Tf_ (path_exists(nextc, NULL, false), NULL);
         TRACE("mkdirs: exists, canonical: %s", nextc);
      } else {
         Tf_ (path_subdir_p(basec, next),
              "can't mkdir: %s not subdirectory of %s", next, basec);
         for (size_t i = 0; denylist[i] != NULL; i++)
            Zf_ (path_subdir_p(denylist[i], next),
                 "can't mkdir: %s under existing bind-mount %s",
                 next, denylist[i]);
         if (mkdir(next, 0755)) {
            if (scratch && (errno == EACCES || errno == EPERM))
               mkdir_overmount(next, scratch);
            else
               Tfe (0, "can't mkdir: %s", next);
         }
         nextc = next;  // canonical b/c we just created last component as dir
         TRACE("mkdirs: created: %s", nextc);
      }
   }
   TRACE("mkdirs: done");
}

/** Return @p path as an absolute path, in a new buffer.

    @param path          Relative path to convert.

    @param cwd           Path that @p path is relative to. This can itself be
                         relative; if so, it is relative to the actual CWD. To
                         obtain @p path relative to the CWD, pass @c "." here.

    @param cwd_parent_p  If true, use the *parent* of @p cwd rather than @p
                         cwd itself. This facilitates making siblings of @p
                         cwd, e.g. if it’s a file and you want to put stuff in
                         the same directory.

    @note
    This operation is purely textual, without reference to what’s actually on
    the filesystem, in contrast with e.g. @c realpath(3). */
char *path_absolve(const char *path, const char *cwd, bool cwd_parent_p)
{
   char *cwd_ = (char *)cwd;  // defeat const
   T__ (path != NULL);
   T__ (cwd_ != NULL);
   T__ (path[0] != '/');

   if (cwd_[0] != '/')
      cwd_ = path_join(getcwd_ch(), cwd_);

   if (cwd_parent_p)
      path_split(cwd_, &cwd_, NULL);

   return path_join(cwd_, path);
}

/* Return true if the given path exists, false otherwise. On error, exit. If
   statbuf is non-null, store the result of stat(2) there. If follow_symlink
   is true and the last component of path is a symlink, stat(2) the target of
   the symlink; otherwise, lstat(2) the link itself. */
bool path_exists(const char *path, struct stat *statbuf, bool follow_symlink)
{
   struct stat statbuf_;

   if (statbuf == NULL)
      statbuf = &statbuf_;

   if (follow_symlink) {
      if (stat(path, statbuf) == 0)
         return true;
   } else {
      if (lstat(path, statbuf) == 0)
         return true;
   }

   Tfe (errno == ENOENT, "can't stat: %s", path);
   return false;
}

/* Concatenate paths a and b, then return the result. */
char *path_join(const char *a, const char *b)
{
   T__ (a != NULL);
   T__ (b != NULL);
   T__ (strlen(a) > 0 || strlen(b) > 0);

   return path_tidy(cats(3, a, "/", b));
}

/* Return the mount flags of the file system containing path, suitable for
   passing to mount(2).

   This is messy because the flags we get from statvfs(3) are ST_* while the
   flags needed by mount(2) are MS_*. My glibc has a comment in bits/statvfs.h
   that the ST_* “should be kept in sync with” the MS_* flags, and the values
   do seem to match, but there are additional undocumented flags in there.
   Also, the kernel contains a test “unprivileged-remount-test.c” that
   manually translates the flags. Thus, I wasn’t comfortable simply passing
   the output of statvfs(3) to mount(2). */
unsigned long path_mount_flags(const char *path)
{
   struct statvfs sv;
   unsigned long known_flags =   ST_MANDLOCK   | ST_NOATIME  | ST_NODEV
                               | ST_NODIRATIME | ST_NOEXEC   | ST_NOSUID
                               | ST_RDONLY     | ST_RELATIME | ST_SYNCHRONOUS;

   Z__ (statvfs(path, &sv));

   // Flag 0x20 is ST_VALID according to the kernel [1], which clashes with
   // MS_REMOUNT, so inappropriate to pass through. Glibc unsets it from the
   // flag bits returned by statvfs(2) [2], but musl doesn’t [3], so unset it.
   //
   // [1]: https://github.com/torvalds/linux/blob/3644286f/include/linux/statfs.h#L27
   // [2]: https://sourceware.org/git?p=glibc.git;a=blob;f=sysdeps/unix/sysv/linux/internal_statvfs.c;h=b1b8dfefe6be909339520d120473bd67e4bece57
   // [3]: https://git.musl-libc.org/cgit/musl/tree/src/stat/statvfs.c?h=v1.2.2
   sv.f_flag &= ~0x20;

   Zf_ (sv.f_flag & ~known_flags, "unknown mount flags: 0x%lx %s",
        sv.f_flag & ~known_flags, path);

   return   (sv.f_flag & ST_MANDLOCK    ? MS_MANDLOCK    : 0)
          | (sv.f_flag & ST_NOATIME     ? MS_NOATIME     : 0)
          | (sv.f_flag & ST_NODEV       ? MS_NODEV       : 0)
          | (sv.f_flag & ST_NODIRATIME  ? MS_NODIRATIME  : 0)
          | (sv.f_flag & ST_NOEXEC      ? MS_NOEXEC      : 0)
          | (sv.f_flag & ST_NOSUID      ? MS_NOSUID      : 0)
          | (sv.f_flag & ST_RDONLY      ? MS_RDONLY      : 0)
          | (sv.f_flag & ST_RELATIME    ? MS_RELATIME    : 0)
          | (sv.f_flag & ST_SYNCHRONOUS ? MS_SYNCHRONOUS : 0);
}

/* Split path into dirname and basename. If dir and/or base is NULL, then skip
   that output. */
void path_split(const char *path, char **dir, char **base)
{
   if (dir != NULL)
      *dir = dirname_ch(path);
   if (base != NULL)
      *base = basename(strdup_ch(path));
}

/* Return true if path is a subdirectory of base, false otherwise. Acts on the
   paths as given, with no canonicalization or other reference to the
   filesystem. For example:

      path_subdir_p("/foo", "/foo/bar")   => true
      path_subdir_p("/foo", "/bar")       => false
      path_subdir_p("/foo/bar", "/foo/b") => false */
bool path_subdir_p(const char *base, const char *path)
{
   int base_len = strlen(base);
   int path_len = strlen(base);

   // remove trailing slashes
   while (base[base_len-1] == '/' && base_len >= 1)
      base_len--;
   while (path[path_len-1] == '/' && path_len >= 1)
      path_len--;

   if (base_len > path_len)
      return false;

   if (streq(base, "/"))  // below logic breaks if base is root
      return true;

   return (   streqn(base, path, base_len)
           && (path[base_len] == '/' || path[base_len] == 0));
}

/** Canonicalize @p path and return the result.

    1. Remove all trailing slashes.
    2. Replace repeated slashes with a single slash.
    3. Remove no-op components:
       a. All single dots (“.”).
       b. Double dots not preceded by the beginning of the path.

    @p path must contain at least one non-empty component. */
char *path_tidy(const char *path)
{
   char **cs;
   char *path_new;
   int nonempty_ct;
   bool absolute_p = (path[0] == '/');

   // Split path into components. This cleans up the slashes, i.e. 1 and 2,
   // but also deletes any leading slash. Therefore, paths consisting of
   // solely the root directory (“/”) will come back as an empty list.
   cs = list_new_strings('/', path);
   T__ (absolute_p || cs[0] != NULL);

   // Find all up-components that have a corresponding down-component and
   // replace both with dots.
   for (int i = 1; cs[i] != NULL; i++)
      if (streq(cs[i], ".."))
         for (int j = i-1; j >= 0; j--)
            if (!streq(cs[j], ".") && !streq(cs[j], "..")) {
               cs[i] = ".";  // up
               cs[j] = ".";  // down
               break;
            }

   // Re-join the components, skipping “.”, i.e. 3a.
   path_new = "";
   nonempty_ct = 0;
   for (int i = 0; cs[i] != NULL; i++)
      if (!streq(cs[i], ".")) {
         nonempty_ct++;
         path_new = cats(3, path_new, "/", cs[i]);  // path_join() calls us
      }
   T__ (absolute_p || nonempty_ct > 0);

   // Skip leading slash if relative.
   if (!absolute_p)
      path_new++;

   return path_new;
}

/** Canonicalize @p path to the extent possible and return the result.

    This function is similar @c realpath(3) but decays to text-only processing
    for the part of the path that doesn’t exist, can’t be accessed, or
    otherwise would cause @c realpath(3) to fail. Specifically:

    1. If relative, convert @p path to absolute (using @c $CWD).
    2. Tidy according to the rules in @c path_tidy().
    3. Resolve symlinks for the (leading) accessible components of @p path.

    @param path  Path to canonicalize. May be @c NULL; in this case, return @c
                 NULL. May not be the empty string.*/
#undef realpath
char *realpath_ch(const char *path)
{
   char *ok = (char *)path;  // accessible components of path
   char *not = "";           // non-accessible components

   if (!path)
      return NULL;
   T__ (path[0] != '\0');

   // Convert to absolute.
   if (ok[0] != '/')
      ok = path_absolve(ok, ".", false);

   // Use realpath(3) to tidy/resolve everything we can.
   while (true) {
      char *p = realpath(ok, NULL);
      if (p) {
         ok = p;
         break;
      } else {  // realpath(3) failed; go up one level and try again
         char *base;
         T__ (!streq(ok, "/"));  // hosed if realpath(3) fails on “/”
         path_split(ok, &ok, &base);
         not = path_join(base, not);
      }
   }

   return path_join(ok, not);  // tidies not
}
#define realpath FN_BLOCKED
