//////////////////////////////////////////////////////////////////////////////
//
// 		  Copyright (C) 1996,1997  Matthew Doar  doar@pobox.com
// 
// Permission to use, copy, and distribute this software and its documentation 
// for any purpose with or without fee is hereby granted, provided that the 
// above copyright notice appears with all copies and that both that copyright 
// notice and this permission notice appear in supporting documentation. 
// 
// Permission to modify the software is granted, but not the right to 
// distribute the modified code. Modifications are to be distributed as 
// patches to the released version. 
// 
// This software is provided "as is" without express or implied warranty. 
//
//////////////////////////////////////////////////////////////////////////////

// tiers.cc
#include <iostream.h>

#ifndef _TIERS_HH
#include "tiers.hh"
#endif



////////////////////////////////////////////////////////////////////////
// Model::CreateWanNodes
//
// Create the WAN and its node placings.
////////////////////////////////////////////////////////////////////////
bool Model::CreateWanNodes()
{
  bool ret = true;
  Node*	nodes = 0;
  Edge*	edges = 0;

  if (NW != 1)
	{
	  cerr << "CreateWanNodes:: Only 1 WAN currently supported in the model" 
		   << endl;
	  return false;
	}

  // Create an array of NW WANs - just one WAN currently suppported
  WanList = new NodesAndEdges[NW];
  if (!WanList) return false;

  for (unsigned long int i = 0; i < NW; i++)
	{
	  // Create a WAN 
	  // first get a typical value for SW from the parameter SW
	  // specificSW is always less than or equal to SW
	  const unsigned long int specificSW = SWdist(SW);	
	  WanList[i].nodes = new Node[specificSW];
	  
	  // edges is the NumNodes * NumNodes connectivity matrix
	  WanList[i].edges = new Edge[specificSW*specificSW];

	  // Could have up to specificSW-1 + NM edges at a single WAN node
	  // by the WAN being a star and all NM MANs having their primary 
	  // connection to the centre of the star. Note that RMW internetwork 
	  // redundancy never adds another edge to the primary connection point 
	  // in the WAN
	  WanList[i].MaxNumEdges(specificSW+NM);

	  nodes = WanList[i].nodes; // an alias
	  edges = WanList[i].edges; // an alias
	  if (!nodes || !edges) return false;

	  for (unsigned long int j = 0; j < specificSW; j++)
		{
		  for (unsigned long int k = 0; k < specificSW; k++)
			{
			  // zero the edge array
			  edges[j*specificSW+k].start = 0;	
			  edges[j*specificSW+k].end = 0;	
			  edges[j*specificSW+k].delay = 0;	
			  edges[j*specificSW+k].bw = 0;	
			  edges[j*specificSW+k].state = INACTIVE;	
			}
		  nodes[j].number = CurrNodeNum;
		  nodes[j].type = WAN;	// used by place
		  // place node j in the grid
		  if (!place(j, nodes, specificSW)) 
			{
			  cerr << "CreateWanNodes:: failed to place a node" << endl;
			  return false;
			}
		  CurrNodeNum++;
		}

	  WanList[i].NumNodes(specificSW);
	}

  return ret;
}



////////////////////////////////////////////////////////////////////////
// Model::CreateManNodes
//
// Create the list of MANs and their node placings.
// Separate function from CreateWanNodes for future flexibility
////////////////////////////////////////////////////////////////////////
bool Model::CreateManNodes()
{
  bool ret = true;
  Node*	nodes = 0;
  Edge*	edges = 0;

  // Create an array of NM MANs
  // get the real number of MANs. Currently just NM
  unsigned long int specificNM = NMdist(NM);	
  ManList = new NodesAndEdges[specificNM];
  if (!ManList) return false;

  for (unsigned long int i = 0; i < specificNM; i++)
	{
	  // Create a MAN with SM nodes
	  // first get a typical value for SM from the parameter SM
	  const unsigned long int specificSM = SMdist(SM);	
	  ManList[i].nodes = new Node[specificSM];

	  ManList[i].edges = new Edge[specificSM*specificSM];

	  // Could have up to specificSM-1 + NL + RMW edges at a single WAN node
	  // by the MAN being a star and all NL LANs having their primary 
	  // connection to the centre of the star. NL is the *maximum* number of 
	  // LANs per MAN. If there are fewer LANs in one MAN, they are not 
	  // attached to any other MAN.
	  //
	  // The extra RMW edges in this case could occur if all the 
	  // RMW redundancy were from the central star of the MAN.
	  ManList[i].MaxNumEdges(specificSM+RMW+NL);

	  nodes = ManList[i].nodes; // an alias
	  edges = ManList[i].edges; // an alias
	  if (!nodes || !edges) return false;

	  for (unsigned long int j = 0; j < specificSM; j++)
		{
		  for (unsigned long int k = 0; k < specificSM; k++)
			{
			  // zero the edge array
			  edges[j*specificSM+k].start = 0;	
			  edges[j*specificSM+k].end = 0;	
			  edges[j*specificSM+k].delay = 0;	
			  edges[j*specificSM+k].bw = 0;	
			  edges[j*specificSM+k].state = INACTIVE;	
			}
		  nodes[j].number = CurrNodeNum;
		  nodes[j].type = MAN;	// used by place
		  // place node j in the grid
		  if (!place(j, nodes, specificSM)) 
			{
			  cerr << "CreateManNodes:: failed to place a node" << endl;
			  return false;
			}
		  CurrNodeNum++;
		}

	  ManList[i].NumNodes(specificSM);
	}
  WanList[0].NumNetworks(specificNM);

  return ret;
}



////////////////////////////////////////////////////////////////////////
// Model::CreateLanNodes
//
// Create LANs and place their nodes
////////////////////////////////////////////////////////////////////////
bool Model::CreateLanNodes()
{
  bool ret = true;
  Node*	nodes = 0;
  Edge*	edges = 0;
  unsigned long int NumMans = WanList[0].NumNetworks();
  unsigned long int NumLans = 0;

  for (unsigned long int man = 0; man < NumMans; man++)
	{
	  // Decide how many LANs should be attached to this MAN
	  unsigned long int specificNL = NLdist(NL);	
	  ManList[man].NumNetworks(specificNL);
	  NumLans += specificNL;
	}

  // Create an array of up to NL LANs to be attached to up to NM MANs
  LanList = new NodesAndEdges[NumLans*NumMans];
  if (!LanList) return false;

  for (unsigned long int i = 0; i < NumLans; i++)
	{
	  // Create a LAN
	  // first get a typical value for SL from the parameter SL
	  const unsigned long int specificSL = SLdist(SL);	
	  LanList[i].nodes = new Node[specificSL];

	  // Since LANs are modelled as stars, no need to allocate
	  // specificSL*specificSL space for edges
	  // Need to allocate space for edges for internetwork redundancy
	  // These edges appear at the end of the edges array
	  LanList[i].edges = new Edge[specificSL+RLM]; // 
	  // and store the edge array dimension value
	  LanList[i].MaxNumEdges(specificSL+RLM);

	  nodes = LanList[i].nodes; // an alias
	  edges = LanList[i].edges; // an alias
	  if (!nodes || !edges) return false;

	  for (unsigned long int k = 0; k < (specificSL+RLM); k++)
		{
		  // zero the edge array
		  edges[k].start = 0;	
		  edges[k].end = 0;	
		  edges[k].delay = 0;	
		  edges[k].bw = 0;	
		  edges[k].state = INACTIVE;	
		}

	  for (unsigned long int j = 0; j < specificSL; j++)
		{
		  nodes[j].number = CurrNodeNum;
		  nodes[j].type = LAN;	// used by place
		  // place node j in the grid
		  if (!place(j, nodes, specificSL)) 
			{
			  cerr << "CreateLanNodes:: failed to place a node" << endl;
			  return false;
			}
		  CurrNodeNum++;
		}

	  LanList[i].NumNodes(specificSL);

	  // create and zero the edge list count for counting edges from node 0
	  if (!LanList[i].newEdgeList(1))
		{
		  cerr << "CreateLanNodes:: failed to allocate space for edge list" 
			   << endl;
		  return false;
		}
	  LanList[i].NumEdges(0,0);
	}

  return ret;
}



////////////////////////////////////////////////////////////////////////
// Model::CreateWanEdges
//
// For the WAN, create an edge array of all possible edges in it.
// Create a minimum spanning tree from those edges.
////////////////////////////////////////////////////////////////////////
bool Model::CreateWanEdges()
{
  bool ret = true;
  Node*	nodes = 0;
  Edge*	edges = 0;

  for (unsigned long int i = 0; i < 1 /* NW should equal 1 */; i++)
	{
	  const unsigned long int NumWanNodes = WanList[i].NumNodes();	

	  nodes = WanList[i].nodes; // an alias
	  edges = WanList[i].edges; // an alias

	  // Create edges for (j,k) and (k,j)
	  // The cost table makes these symmetrical if they are so.
	  for (unsigned long int j = 0; j < NumWanNodes; j++)
		{
		  for (unsigned long int k = 0; k < NumWanNodes; k++)
			{
			  // Create an edge from node j to node k and mark it inactive 
			  // The edge (j,j) is undefined
			  if (j!=k)
				{
				  // there are 3 network types
				  CostTableEntry anEntry = theCostTable[(WAN*3)+WAN];
				  unsigned long int index = j*NumWanNodes+k;

				  edges[index].start = nodes[j].number;
				  edges[index].end = nodes[k].number;
				  unsigned long int dummy = anEntry.d1 + 
					((anEntry.d2)*Euclid(nodes[j], nodes[k])*WAN_SCALE);
				  edges[index].delay = dummy;
				  edges[index].bw = anEntry.bw;
				  edges[index].state = Model::INACTIVE;
				}
			} // k
		} // j

	  // Calculate a minimum spanning tree and leave the result in newedges
	  if (!Prims(&(WanList[i]), max(RW,RMW-1))) 
		{
		  return false;
		}
	} // i

  return ret;
}



////////////////////////////////////////////////////////////////////////
// Model::CreateManEdges
//
// For every MAN, create an edge array of all possible edges in it.
// Create a minimum spanning tree from those edges.
////////////////////////////////////////////////////////////////////////
bool Model::CreateManEdges()
{
  bool ret = true;
  Node*	nodes = 0;
  Edge*	edges = 0;

  for (unsigned long int i = 0; i < WanList[0].NumNetworks(); i++)
	{
	  const unsigned long int NumManNodes = ManList[i].NumNodes();	

	  nodes = ManList[i].nodes; // an alias
	  edges = ManList[i].edges; // an alias

	  // Create edges for (j,k) and (k,j)
	  // The cost table makes these symmetrical if they are so.
	  for (unsigned long int j = 0; j < NumManNodes; j++)
		{
		  for (unsigned long int k = 0; k < NumManNodes; k++)
			{
			  // Create an edge from node j to node k and mark it inactive 
			  // The edge (j,j) is undefined
			  if (j!=k)
				{
				  // there are 3 network types
				  CostTableEntry anEntry = theCostTable[(MAN*3)+MAN];
				  unsigned long int index = j*NumManNodes+k;

				  edges[index].start = nodes[j].number;
				  edges[index].end = nodes[k].number;
				  edges[index].delay = anEntry.d1 + 
					((anEntry.d2)*Euclid(nodes[j], nodes[k])*(MAN_SCALE));
				  edges[index].bw = anEntry.bw;
				  edges[index].state = Model::INACTIVE;
				}
			} // k
		} // j

	  // Calculate a minimum spanning tree and leave the result in newedges 
	  if (!Prims(&(ManList[i]), max(RM,RMW-1))) 
		{
		  return false;
		}

	} // i

  return ret;
}


////////////////////////////////////////////////////////////////////////
// Model::CreateLanEdges
//
// For every LAN, create an edge list with node 0 as the centre of a star
// Do not calculate an MST or permit intranetwork redundancy (for simplicity)
////////////////////////////////////////////////////////////////////////
bool Model::CreateLanEdges()
{
  bool ret = true;
  Node*	nodes = 0;
  Edge*	edges = 0;
  unsigned long int NumMans = WanList[0].NumNetworks();

  // Count the total number of LANs which need to be created
  unsigned long int NumLans = 0;
  for (unsigned long int man = 0; man < NumMans; man++)
	{
	  NumLans += ManList[man].NumNetworks();
	}

  // The edge from node 0 to node j is in edges[j]
  for (unsigned long int i = 0; i < (NumMans*NumLans); i++)
	{
	  const unsigned long int NumLanNodes = LanList[i].NumNodes();	

	  nodes = LanList[i].nodes; // an alias
	  edges = LanList[i].edges; // an alias

	  // Create edges for (0,j) and (j,0)
	  // All edges in a LAN are assumed symmetrical
	  for (unsigned long int j = 1; j < NumLanNodes; j++)
		{
		  // Create an edge from node 0 to node j and mark it active 
		  // there are 3 network types
		  CostTableEntry anEntry = theCostTable[(LAN*3)+LAN];
		  unsigned long int index = j;

		  edges[index].start = nodes[0].number;
		  edges[index].end = nodes[j].number;
		  edges[index].delay = anEntry.d1 + 
			((anEntry.d2)*Euclid(nodes[0], nodes[j])*(LAN_SCALE));
		  edges[index].bw = anEntry.bw;
		  edges[index].state = ACTIVE;

		  LanList[i].IncNumEdges(0);
		} // j

	} // i

  return ret;
}

// end of file

