/*
 * Copyright (C) 2000-2002 Daniel Linder / UIUC DRES
 * Based on work by the Accessible Linux Group
 * Rishi Dubey, Aaron Klish, Kristian Rickert, Shane Smith
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 *   USA
 *
 */

#include "ax.h"


///////////////////////////////////////////////////////////////////////////////
//
// Function getValue
//
// Takes an input string and returns that string's integer conversion in
// "ret".  Return value is a normalized result of strltol().
//
///////////////////////////////////////////////////////////////////////////////


int getValue(char* instring, int* ret)
{
  // Check for an incoming NULL string
  if(!instring) {
    return 2;
  }

  char* outptr;

  *ret = (int)strtol(instring, &outptr, 10);

  if(outptr[0] == '\0') {     // Success!
    return 0;
  }
  else {                      // Failure.
    return 1;
  }
}


///////////////////////////////////////////////////////////////////////////////
//
// Function showHelp
//
// Displays command line info
//
///////////////////////////////////////////////////////////////////////////////


void showHelp(void) {

  cout << "Usage: accessx [options & option/value pairs]"
       << endl << endl;
  cout << "status:                  Show status of AccessX Features" << endl
       << "help:                    Show this help text" << endl
       << "loadconfig:              Load the ~/AccessX config file" << endl
       << endl
       << "+timeout:                Enable AccessX Timeout" << endl
       << "-timeout:                Disable AccessX Timeout" << endl
       << "timeout <value>:         Set AccessX Timeout" << endl
       << endl
       << "+mousekeys:              Enable MouseKeys" << endl
       << "-mousekeys:              Disable MouseKeys" << endl
       << "mousedelay <value>:      Set MouseKeys Delay" << endl
       << "mouseinterval <value>:   Set MouseKeys Interval" << endl
       << "mousetimetomax <value>:  Set MouseKeys time to max speed" << endl
       << "mousemaxspeed <value>:   Set MouseKeys max speed" << endl
       << "mousecurve <value>:      Set MouseKeys acceleration curve" << endl
       << endl
       << "+slowkeys:               Enable SlowKeys" << endl
       << "-slowkeys:               Disable SlowKeys" << endl
       << "slowdelay <value>:       Set SlowKeys Delay" << endl
       << endl
       << "+stickykeys:             Enable StickyKeys" << endl
       << "-stickykeys:             Disable StickyKeys" << endl
       << "+stickylatchtolock:      Enable S.K. Latch To Lock" << endl
       << "-stickylatchtolock:      Disbale S.K. Latch To Lock" << endl
       << "+stickytwokeydisable:    Enable S.K. Two Key Disable" << endl
       << "-stickytwokeydisable:    Disable S.K. Two Key Disable" << endl
       << endl
       << "+bouncekeys:             Enable BounceKeys" << endl
       << "-bouncekeys:             Disable BounceKeys" << endl
       << "bouncedelay <value>:     Set Debounce Delay" << endl
       << endl
       << "+repeatkeys:             Enable RepeatKeys" << endl
       << "-repeatkeys:             Disable RepeatKeys" << endl
       << "repeatdelay <value>:     Set RepeatKeys Delay" << endl
       << "repeatinterval <value>:  Set RepeatKeys Interval" << endl << endl
       << "With no options, loads config file." << endl
       << endl;
}


///////////////////////////////////////////////////////////////////////////////
//
// ShowRawStatus(Access* ax)
//
// Used to communicate with the "accessx" Tcl/Tk GUI
//
///////////////////////////////////////////////////////////////////////////////


void showRawStatus(Access* ax)
{
  cout << ax->GetAccessXTOState() << endl;
  cout << (int)ceil(ax->GetAccessXDelay() / 60.0) << endl;
  cout << ax->GetMouseKeysState() << endl;
  cout << (int)ceil(ax->GetMouseKeysTimeToMax() / 1000.0) << endl;
  cout << ax->GetMouseKeysMaxSpeed() << endl;
  cout << ax->GetSlowKeysState() << endl;
  cout << (float)(ax->GetSlowKeysDelay() / 1000.0) << endl;
  cout << ax->GetStickyKeysState() << endl;
  cout << ax->GetStickyKeysLatchToLockState() << endl;
  cout << ax->GetStickyKeysTwoKeyDisableState() << endl;
  cout << ax->GetBounceKeysState() << endl;
  cout << (float)(ax->GetBounceKeysDelay() / 1000.0) << endl;
  cout << ax->GetRepeatKeysState() << endl;
  cout << (float)(ax->GetRepeatKeysDelay() / 1000.0) << endl;
  cout << (float)(1000.0 / ax->GetRepeatKeysInterval()) << endl;
  cout << ToggleKeysToggleSet << endl;
  cout << SoundOnOffToggleSet << endl;
  cout << SlowKeysOnAcceptToggleSet << endl;
  cout << SlowKeysOnPressToggleSet << endl;
  cout << EnableAccessXToggleSet << endl;
}

///////////////////////////////////////////////////////////////////////////////
//
// Function parseCommandLine
//
// Parses and sets command line parameters
//
///////////////////////////////////////////////////////////////////////////////


void parseCommandLine(int argc, char** argv, Access* ax)
{
  // Begin Parsing Command Line ---------------------------------------------

  int done = 0;
  int index = 1;

  int rc, value;

  if(argc < 2) {
    parseConfigFile(ax, 0);
    done = 1;
  }
  else done = 0;

  while(!done) {
    if(!strcmp(argv[index], "status"))
      ax->ShowDiagnostics();
    else if(!strcmp(argv[index], "showraw"))
      showRawStatus(ax);
    else if(!strcmp(argv[index], "help"))
      showHelp();
    else if(!strcmp(argv[index], "loadconfig"))
      parseConfigFile(ax, 1);

    // AccessX Timeout Features ---------------------------------------------

    else if(!strcmp(argv[index], "+timeout")) {
      ax->SetAccessXTOState(1);
    }
    else if(!strcmp(argv[index], "-timeout")) {
      ax->SetAccessXTOState(0);
    }
    else if(!strcmp(argv[index], "timeoout")) {
      index++;
      rc = getValue(argv[index], &value);
      if(rc) {
	cout << "Invalid Option." << endl;
	exit(2);
      }
      ax->SetAccessXDelay(value);
    }

    // AccessX MouseKeys Features -------------------------------------------

    else if(!strcmp(argv[index], "+mousekeys")) {
      ax->SetMouseKeysState(1);
    }
    else if(!strcmp(argv[index], "-mousekeys")) {
      ax->SetMouseKeysState(0);
    }
    else if(!strcmp(argv[index], "mousedelay")) {
      index++;
      rc = getValue(argv[index], &value);
      if(rc) {
	cout << "Invalid Option." << endl;
	exit(2);
      }
      ax->SetMouseKeysDelay(value);
    }
    else if(!strcmp(argv[index], "mouseinterval")) {
      index++;
      rc = getValue(argv[index], &value);
      if(rc) {
	cout << "Invalid Option." << endl;
	exit(2);
      }
      ax->SetMouseKeysInterval(value);
    }
    else if(!strcmp(argv[index], "mousetimetomax")) {
      index++;
      rc = getValue(argv[index], &value);
      if(rc) {
	cout << "Invalid Option." << endl;
	exit(2);
      }
      ax->SetMouseKeysTimeToMax(value);
    }
    else if(!strcmp(argv[index], "mousemaxspeed")) {
      index++;
      rc = getValue(argv[index], &value);
      if(rc) {
	cout << "Invalid Option." << endl;
	exit(2);
      }
      ax->SetMouseKeysMaxSpeed(value);
    }
    else if(!strcmp(argv[index], "mousecurve")) {
      index++;
      rc = getValue(argv[index], &value);
      if(rc) {
	cout << "Invalid Option." << endl;
	exit(2);
      }
      ax->SetMouseKeysCurve(value);
    }

    // AccessX SlowKeys Features --------------------------------------------

    else if(!strcmp(argv[index], "+slowkeys")) {
      ax->SetSlowKeysState(1);
    }
    else if(!strcmp(argv[index], "-slowkeys")) {
      ax->SetSlowKeysState(0);
    }
    else if(!strcmp(argv[index], "slowdelay")) {
      index++;
      rc = getValue(argv[index], &value);
      if(rc) {
	cout << "Invalid Option." << endl;
	exit(2);
      }
      ax->SetSlowKeysDelay(value);
    }

    // AccessX StickyKeys Features ------------------------------------------

    else if(!strcmp(argv[index], "+stickykeys")) {
      ax->SetStickyKeysState(1);
    }
    else if(!strcmp(argv[index], "-stickykeys")) {
      ax->SetStickyKeysState(0);
    }
    else if(!strcmp(argv[index], "+stickylatchtolock")) {
      ax->SetStickyKeysLatchToLockState(1);
    }
    else if(!strcmp(argv[index], "-stickylatchtolock")) {
      ax->SetStickyKeysLatchToLockState(0);
    }
    else if(!strcmp(argv[index], "+stickytwokeydisable")) {
      ax->SetStickyKeysTwoKeyDisableState(1);
    }
    else if(!strcmp(argv[index], "-stickytwokeydisable")) {
      ax->SetStickyKeysTwoKeyDisableState(0);
    }

    // AccessX BounceKeys Features ------------------------------------------

    else if(!strcmp(argv[index], "+bouncekeys")) {
      ax->SetBounceKeysState(1);
    }
    else if(!strcmp(argv[index], "-bouncekeys")) {
      ax->SetBounceKeysState(0);
    }
    else if(!strcmp(argv[index], "bouncedelay")) {
      index++;
      rc = getValue(argv[index], &value);
      if(rc) {
	cout << "Invalid Option." << endl;
	exit(2);
      }
      ax->SetBounceKeysDelay(value);
    }

    // AccessX RepeatKeys Features ------------------------------------------

    else if(!strcmp(argv[index], "+repeatkeys")) {
      ax->SetRepeatKeysState(1);
    }
    else if(!strcmp(argv[index], "-repeatkeys")) {
      ax->SetRepeatKeysState(0);
    }
    else if(!strcmp(argv[index], "repeatdelay")) {
      index++;
      rc = getValue(argv[index], &value);
      if(rc) {
	cout << "Invalid Option." << endl;
	exit(2);
      }
      ax->SetRepeatKeysDelay(value);
    }
    else if(!strcmp(argv[index], "repeatinterval")) {
      index++;
      rc = getValue(argv[index], &value);
      if(rc) {
	cout << "Invalid Option." << endl;
	exit(2);
      }
      ax->SetRepeatKeysInterval(value);
    }

    // AccessX Unknown Features ---------------------------------------------

    else {
      cout << "Unknown Option!" << endl;
      exit(1);
    }

    // End of the Command Line Parser ---------------------------------------

    index++;
    if(index >= argc) done = 1;
  }
}


///////////////////////////////////////////////////////////////////////////////
//
// Function parseConfigFile
//
// Parses and sets config file parameters
//
///////////////////////////////////////////////////////////////////////////////


void parseConfigFile(Access* ax, int quiet)
{
  int KRGSlowKeysDelayScaleValue;
  int KRGSlowKeysDelayScaleDecimalPoints;

  int TimeOutScaleValue;
  int TimeOutScaleDecimalPoints;

  int MouseMaxSpeedScaleValue;
  int MouseMaxSpeedScaleDecimalPoints;
  int MouseTimeToMaxScaleValue;
  int MouseTimeToMaxScaleDecimalPoints;

  int KRGDebounceScaleValue;
  int KRGDebounceScaleDecimalPoints;

  int KRGRepeatDelayScaleValue;
  int KRGRepeatDelayScaleDecimalPoints;
  int KRGRepeatRateScaleValue;
  int KRGRepeatRateScaleDecimalPoints;

  // Come up with path for AccessX config file and open the file

  char* homePath;
  homePath = getenv("HOME");
  int pathLength = strlen(homePath);
  int newPathLength = pathLength + 10;
  char newHomePath[newPathLength];
  strncpy(newHomePath, homePath, pathLength+1);
  strcat(newHomePath, "/AccessX");

  FILE* infile;
  infile = fopen(newHomePath, "r");

  // If the file exists, open & parse it, otherwise don't.

  if(infile) {
    char param[256];   // Parameter to operate on
    char value[64];    // Value of Parameter
    int rc;            // General Return Code
    int error = 0;     // flag to raise if error occurs

    while( !(feof(infile) || error) ) {
      fscanf(infile, "%s\t%s", param, value);

      if(!strcmp(param, "*StickyTwoKeysToggle.set:"))
	if(!strcmp(value, "True"))
	  ax->SetStickyKeysTwoKeyDisableState(1);
	else if(!strcmp(value, "False"))
	  ax->SetStickyKeysTwoKeyDisableState(0);
	else exit(10);

      else if(!strcmp(param, "*ToggleKeysToggle.set:"))
	if(!strcmp(value, "True"))
	  ToggleKeysToggleSet = 1;
	else if(!strcmp(value, "False"))
	  ToggleKeysToggleSet = 0;
	else exit(10);

      else if(!strcmp(param, "*StickyModSoundToggle.set:"))
	if(!strcmp(value, "True"))
	  ax->SetStickyKeysLatchToLockState(1);
	else if(!strcmp(value, "False"))
	  ax->SetStickyKeysLatchToLockState(0);
	else exit(10);

      else if(!strcmp(param, "*SlowKeysToggle.set:"))
	if(!strcmp(value, "True"))
	  ax->SetSlowKeysState(1);
	else if(!strcmp(value, "False"))
	  ax->SetSlowKeysState(0);
	else exit(10);

      else if(!strcmp(param, "*SoundOnOffToggle.set:"))
	if(!strcmp(value, "True"))
	  SoundOnOffToggleSet = 1;
	else if(!strcmp(value, "False"))
	  SoundOnOffToggleSet = 0;
	else exit(10);

      else if(!strcmp(param, "*SlowKeysOnAcceptToggle.set:"))
	if(!strcmp(value, "True"))
	  SlowKeysOnAcceptToggleSet = 1;
	else if(!strcmp(value, "False"))
	  SlowKeysOnAcceptToggleSet = 0;
	else exit(10);

      else if(!strcmp(param, "*TimeOutToggle.set:"))
	if(!strcmp(value, "True"))
	  ax->SetAccessXTOState(1);
	else if(!strcmp(value, "False"))
	  ax->SetAccessXTOState(0);
	else exit(10);

      else if(!strcmp(param, "*SlowKeysOnPressToggle.set:"))
	if(!strcmp(value, "True"))
	  SlowKeysOnPressToggleSet = 1;
	else if(!strcmp(value, "False"))
	  SlowKeysOnPressToggleSet = 0;
	else exit(10);

      else if(!strcmp(param, "*EnableAccessXToggle.set:"))
	if(!strcmp(value, "True"))
	  EnableAccessXToggleSet = 1;
	else if(!strcmp(value, "False"))
	  EnableAccessXToggleSet = 0;
	else exit(10);

      else if(!strcmp(param, "*RepeatKeysToggle.set:"))
	if(!strcmp(value, "True"))
	  ax->SetRepeatKeysState(1);
	else if(!strcmp(value, "False"))
	  ax->SetRepeatKeysState(0);
	else exit(10);

      else if(!strcmp(param, "*BounceKeysToggle.set:"))
	if(!strcmp(value, "True"))
	  ax->SetBounceKeysState(1);
	else if(!strcmp(value, "False"))
	  ax->SetBounceKeysState(0);
	else exit(10);

      else if(!strcmp(param, "*StickyKeysToggle.set:"))
	if(!strcmp(value, "True"))
	  ax->SetStickyKeysState(1);
	else if(!strcmp(value, "False"))
	  ax->SetStickyKeysState(0);
	else exit(10);

      else if(!strcmp(param, "*MouseKeysToggle.set:"))
	if(!strcmp(value, "True"))
	  ax->SetMouseKeysState(1);
	else if(!strcmp(value, "False"))
	  ax->SetMouseKeysState(0);
	else exit(10);

      else if(!strcmp(param, "*TimeOutScale.value:")) {
	rc = getValue(value, &TimeOutScaleValue);
	if(rc) {
	  cout << "Invalid Config File Option." << endl;
	  exit(10);
	}
      }
      else if(!strcmp(param, "*TimeOutScale.decimalPoints:")) {
	rc = getValue(value, &TimeOutScaleDecimalPoints);
	if(rc) {
	  cout << "Invalid Config File Option." << endl;
	  exit(10);
	}
      }

      else if(!strcmp(param, "*KRGSlowKeysDelayScale.value:")) {
	rc = getValue(value, &KRGSlowKeysDelayScaleValue);
	if(rc) {
	  cout << "Invalid Config File Option." << endl;
	  exit(10);
	}
      }

      else if(!strcmp(param, "*KRGSlowKeysDelayScale.decimalPoints:")) {
	rc = getValue(value, &KRGSlowKeysDelayScaleDecimalPoints);
	if(rc) {
	  cout << "Invalid Config File Option." << endl;
	  exit(10);
	}
      }

      else if(!strcmp(param, "*MouseMaxSpeedScale.value:")) {
	rc = getValue(value, &MouseMaxSpeedScaleValue);
	if(rc) {
	  cout << "Invalid Config File Option." << endl;
	  exit(10);
	}
      }

      else if(!strcmp(param, "*MouseMaxSpeedScale.decimalPoints:")) {
	rc = getValue(value, &MouseMaxSpeedScaleDecimalPoints);
	if(rc) {
	  cout << "Invalid Config File Option." << endl;
	  exit(10);
	}
      }

      else if(!strcmp(param, "*KRGDebounceScale.value:")) {
	rc = getValue(value, &KRGDebounceScaleValue);
	if(rc) {
	  cout << "Invalid Config File Option." << endl;
	  exit(10);
	}
      }

      else if(!strcmp(param, "*KRGDebounceScale.decimalPoints:")) {
	rc = getValue(value, &KRGDebounceScaleDecimalPoints);
	if(rc) {
	  cout << "Invalid Config File Option." << endl;
	  exit(10);
	}
      }

      else if(!strcmp(param, "*KRGRepeatDelayScale.value:")) {
	rc = getValue(value, &KRGRepeatDelayScaleValue);
	if(rc) {
	  cout << "Invalid Config File Option." << endl;
	  exit(10);
	}
      }

      else if(!strcmp(param, "*KRGRepeatDelayScale.decimalPoints:")) {
	rc = getValue(value, &KRGRepeatDelayScaleDecimalPoints);
	if(rc) {
	  cout << "Invalid Config File Option." << endl;
	  exit(10);
	}
      }

      else if(!strcmp(param, "*KRGRepeatRateScale.value:")) {
	rc = getValue(value, &KRGRepeatRateScaleValue);
	if(rc) {
	  cout << "Invalid Config File Option." << endl;
	  exit(10);
	}
      }

      else if(!strcmp(param, "*KRGRepeatRateScale.decimalPoints:")) {
	rc = getValue(value, &KRGRepeatRateScaleDecimalPoints);
	if(rc) {
	  cout << "Invalid Config File Option." << endl;
	  exit(10);
	}
      }

      else if(!strcmp(param, "*MouseTimeToMaxScale.value:")) {
	rc = getValue(value, &MouseTimeToMaxScaleValue);
	if(rc) {
	  cout << "Invalid Config File Option." << endl;
	  exit(10);
	}
      }

      else if(!strcmp(param, "*MouseTimeToMaxScale.decimalPoints:")) {
	rc = getValue(value, &MouseTimeToMaxScaleDecimalPoints);
	if(rc) {
	  cout << "Invalid Config File Option." << endl;
	  exit(10);
	}
      }

      else {
	cout << "Invalid Config File Parameter." << endl;
	exit(10);
      }
    }
    fclose(infile);

    // The on/off toggles have all been set.  Now, adjust the numeric controls

    double SlowKeysDelayValue  = KRGSlowKeysDelayScaleValue;
    double TimeOutValue        = TimeOutScaleValue;
    double MouseMaxSpeedValue  = MouseMaxSpeedScaleValue;
    double MouseTimeToMaxValue = MouseTimeToMaxScaleValue;
    double DebounceValue       = KRGDebounceScaleValue;
    double RepeatDelayValue    = KRGRepeatDelayScaleValue;
    double RepeatRateValue     = KRGRepeatRateScaleValue;

    int i;

    for(i=0; i<KRGSlowKeysDelayScaleDecimalPoints; i++)
      SlowKeysDelayValue /= 10;

    if(SlowKeysDelayValue == 0) {
      cout << "Config File: Slow Keys Delay Cannot Be Zero." << endl;
      exit(10);
    }

    for(i=0; i<TimeOutScaleDecimalPoints; i++)
      TimeOutValue /= 10;

    if(TimeOutValue == 0) {
      cout << "Config File: Timeout Value Cannot Be Zero." << endl;
      exit(10);
    }

    for(i=0; i<MouseMaxSpeedScaleDecimalPoints; i++)
      MouseMaxSpeedValue /= 10;

    if(MouseMaxSpeedValue == 0) {
      cout << "Config File: Mouse Keys Max Speed Cannot Be Zero." << endl;
      exit(10);
    }

    for(i=0; i<MouseTimeToMaxScaleDecimalPoints; i++)
      MouseTimeToMaxValue /= 10;

    if(MouseTimeToMaxValue == 0) {
      cout << "Config File: Mouse Keys Time To Max Cannot Be Zero." << endl;
      exit(10);
    }

    for(i=0; i<KRGDebounceScaleDecimalPoints; i++)
      DebounceValue /= 10;

    if(DebounceValue == 0) {
      cout << "Config File: Debounce Delay Cannot Be Zero." << endl;
      exit(10);
    }

    for(i=0; i<KRGRepeatDelayScaleDecimalPoints; i++)
      RepeatDelayValue /= 10;

    if(RepeatDelayValue == 0) {
      cout << "Config File: Repeat Keys Delay Cannot Be Zero." << endl;
      exit(10);
    }

    for(i=0; i<KRGRepeatRateScaleDecimalPoints; i++)
      RepeatRateValue /= 10;

    if(RepeatRateValue == 0) {
      cout << "Config File: Repeat Keys Rate Cannot Be Zero." << endl;
      exit(10);
    }

    // min->sec
    ax->SetAccessXDelay((int)floor(TimeOutValue*60.0));

    // sec->msec
    ax->SetSlowKeysDelay((int)floor(SlowKeysDelayValue*1000.0));

    // pixels->pixels
    ax->SetMouseKeysMaxSpeed((int)floor(MouseMaxSpeedValue));

    // sec->msec
    ax->SetMouseKeysTimeToMax((int)floor(MouseTimeToMaxValue*1000.0));

    // sec->msec
    ax->SetBounceKeysDelay((int)floor(DebounceValue*1000.0));

    // sec->msec
    ax->SetRepeatKeysDelay((int)floor(RepeatDelayValue*1000.0));

    // cps->interval between keys
    ax->SetRepeatKeysInterval((int)floor(1000.0/RepeatRateValue));

    if(!quiet) {
      cout << "Config file ~/AccessX loaded." << endl
	   << "\"ax help\" for help." << endl;
    }
  }
  else {
    if(!quiet) {
      cout << "Config File ~/AccessX not found!" << endl
	   << "\"ax help\" for help." << endl;
    }
  }
}


///////////////////////////////////////////////////////////////////////////////
//
// Function Main
//
///////////////////////////////////////////////////////////////////////////////


int main(int argc, char** argv)
{
  Access ax;
  ax.QueryServer();

  parseCommandLine(argc, argv, &ax);

  ax.FlushNewSettings();

  exit(0);
}


