/* ------------------------------------------------------------------------
 *
 * client_tipc.c
 *
 * Short description: TIPC hello world demo (client side)
 * ------------------------------------------------------------------------
 *
 * Copyright (c) 2003, Ericsson Research Canada
 * Copyright (c) 2010 Wind River Systems
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * Neither the name of Ericsson Research Canada nor the names of its
 * contributors may be used to endorse or promote products derived from this
 * software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * ------------------------------------------------------------------------
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <netinet/in.h>
#include <linux/tipc.h>

#define SERVER_TYPE  18888
#define SERVER_INST  17
#define BUF_SIZE 40
#ifndef TIPC_SERVICE_ADDR
#define TIPC_SERVICE_ADDR  2
#endif

void wait_for_service(uint32_t type, uint32_t instance, int wait)
{
	struct tipc_event event;
	struct sockaddr_tipc topsrv = {
		.family = AF_TIPC,
		.addrtype = TIPC_SERVICE_ADDR,
		.addr.name.name.type = TIPC_TOP_SRV,
		.addr.name.name.instance = TIPC_TOP_SRV,
		.addr.name.domain = 0
	};
	struct tipc_subscr subscr = {
		.seq.type = type,
		.seq.lower = instance,
		.seq.upper = instance,
		.timeout = wait,
		.filter = TIPC_SUB_SERVICE
	};

	int sd = socket(AF_TIPC, SOCK_SEQPACKET, 0);

	/* Connect to topology server */
	if (0 > connect(sd, (struct sockaddr *)&topsrv, sizeof(topsrv))) {
		perror("Client: failed to connect to topology server");
		exit(1);
	}

	if (send(sd, &subscr, sizeof(subscr), 0) != sizeof(subscr)) {
		perror("Client: failed to send subscription");
		exit(1);
	}

	/* Now wait for the subscription to fire */
	if (recv(sd, &event, sizeof(event), 0) != sizeof(event)) {
		perror("Client: failed to receive event");
		exit(1);
	}
	if (event.event != TIPC_PUBLISHED) {
		printf("Client: server {%u,%u} not published within %u [s]\n",
		       type, instance, wait/1000);
		exit(1);
	}

	close(sd);
}


int main(int argc, char *argv[])
{
	char buf[BUF_SIZE] = {"Hello World!!!"};
	struct sockaddr_tipc server = {
		.family = AF_TIPC,
		.addrtype = TIPC_SERVICE_ADDR,
		.addr.name.name.type = SERVER_TYPE,
		.addr.name.name.instance = SERVER_INST,
		.addr.name.domain = 0
	};
	int sd;

	printf("****** TIPC hello world client started ******\n");

	wait_for_service(SERVER_TYPE, SERVER_INST, 10000);

	sd = socket(AF_TIPC, SOCK_RDM, 0);
	if (0 > sendto(sd, buf, strlen(buf) + 1, 0, (void*)&server, sizeof(server))) {
		perror("Client: failed to send");
		exit(1);
	}
	printf("Client: sent message: %s \n", buf);

	if (0 >= recv(sd, buf, sizeof(buf), 0)) {
		perror("Client: unexpected response");
		exit(1);
	}
	printf("Client: received response: %s \n", buf);

	printf("****** TIPC hello client finished ******\n");
	exit(0);
}
