//! Defines constants which identify keyboard keys and modifiers.
//!
//! Please refer to the Best Keyboard Practices document for details on what
//! this information means and how best to use it.
//!
//! <https://wiki.libsdl.org/SDL3/BestKeyboardPractices>

use super::stdinc::*;

use super::scancode::*;

/// The SDL virtual key representation.
///
/// Values of this type are used to represent keyboard keys using the current
/// layout of the keyboard. These values include Unicode values representing
/// the unmodified character that would be generated by pressing the key, or an
/// `SDLK_*` constant for those keys that do not generate characters.
///
/// A special exception is the number keys at the top of the keyboard which map
/// to SDLK_0...SDLK_9 on AZERTY layouts.
///
/// Keys with the `SDLK_EXTENDED_MASK` bit set do not map to a scancode or
/// unicode code point.
///
/// ### Availability
/// This datatype is available since SDL 3.2.0.
///
/// ### Known values (`sdl3-sys`)
/// | Constant | Description |
/// | -------- | ----------- |
/// | [`SDLK_EXTENDED_MASK`] | |
/// | [`SDLK_SCANCODE_MASK`] | |
/// | [`SDLK_UNKNOWN`] | 0 |
/// | [`SDLK_RETURN`] | '\r' |
/// | [`SDLK_ESCAPE`] | '\x1B' |
/// | [`SDLK_BACKSPACE`] | '\b' |
/// | [`SDLK_TAB`] | '\t' |
/// | [`SDLK_SPACE`] | ' ' |
/// | [`SDLK_EXCLAIM`] | '!' |
/// | [`SDLK_DBLAPOSTROPHE`] | '"' |
/// | [`SDLK_HASH`] | '#' |
/// | [`SDLK_DOLLAR`] | '$' |
/// | [`SDLK_PERCENT`] | '%' |
/// | [`SDLK_AMPERSAND`] | '&' |
/// | [`SDLK_APOSTROPHE`] | '\'' |
/// | [`SDLK_LEFTPAREN`] | '(' |
/// | [`SDLK_RIGHTPAREN`] | ')' |
/// | [`SDLK_ASTERISK`] | '*' |
/// | [`SDLK_PLUS`] | '+' |
/// | [`SDLK_COMMA`] | ',' |
/// | [`SDLK_MINUS`] | '-' |
/// | [`SDLK_PERIOD`] | '.' |
/// | [`SDLK_SLASH`] | '/' |
/// | [`SDLK_0`] | '0' |
/// | [`SDLK_1`] | '1' |
/// | [`SDLK_2`] | '2' |
/// | [`SDLK_3`] | '3' |
/// | [`SDLK_4`] | '4' |
/// | [`SDLK_5`] | '5' |
/// | [`SDLK_6`] | '6' |
/// | [`SDLK_7`] | '7' |
/// | [`SDLK_8`] | '8' |
/// | [`SDLK_9`] | '9' |
/// | [`SDLK_COLON`] | ':' |
/// | [`SDLK_SEMICOLON`] | ';' |
/// | [`SDLK_LESS`] | '<' |
/// | [`SDLK_EQUALS`] | '=' |
/// | [`SDLK_GREATER`] | '>' |
/// | [`SDLK_QUESTION`] | '?' |
/// | [`SDLK_AT`] | '@' |
/// | [`SDLK_LEFTBRACKET`] | '[' |
/// | [`SDLK_BACKSLASH`] | '\\' |
/// | [`SDLK_RIGHTBRACKET`] | ']' |
/// | [`SDLK_CARET`] | '^' |
/// | [`SDLK_UNDERSCORE`] | '_' |
/// | [`SDLK_GRAVE`] | '`' |
/// | [`SDLK_A`] | 'a' |
/// | [`SDLK_B`] | 'b' |
/// | [`SDLK_C`] | 'c' |
/// | [`SDLK_D`] | 'd' |
/// | [`SDLK_E`] | 'e' |
/// | [`SDLK_F`] | 'f' |
/// | [`SDLK_G`] | 'g' |
/// | [`SDLK_H`] | 'h' |
/// | [`SDLK_I`] | 'i' |
/// | [`SDLK_J`] | 'j' |
/// | [`SDLK_K`] | 'k' |
/// | [`SDLK_L`] | 'l' |
/// | [`SDLK_M`] | 'm' |
/// | [`SDLK_N`] | 'n' |
/// | [`SDLK_O`] | 'o' |
/// | [`SDLK_P`] | 'p' |
/// | [`SDLK_Q`] | 'q' |
/// | [`SDLK_R`] | 'r' |
/// | [`SDLK_S`] | 's' |
/// | [`SDLK_T`] | 't' |
/// | [`SDLK_U`] | 'u' |
/// | [`SDLK_V`] | 'v' |
/// | [`SDLK_W`] | 'w' |
/// | [`SDLK_X`] | 'x' |
/// | [`SDLK_Y`] | 'y' |
/// | [`SDLK_Z`] | 'z' |
/// | [`SDLK_LEFTBRACE`] | '{' |
/// | [`SDLK_PIPE`] | '|' |
/// | [`SDLK_RIGHTBRACE`] | '}' |
/// | [`SDLK_TILDE`] | '~' |
/// | [`SDLK_DELETE`] | '\x7F' |
/// | [`SDLK_PLUSMINUS`] | '\xB1' |
/// | [`SDLK_CAPSLOCK`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CAPSLOCK`]) |
/// | [`SDLK_F1`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F1`]) |
/// | [`SDLK_F2`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F2`]) |
/// | [`SDLK_F3`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F3`]) |
/// | [`SDLK_F4`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F4`]) |
/// | [`SDLK_F5`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F5`]) |
/// | [`SDLK_F6`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F6`]) |
/// | [`SDLK_F7`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F7`]) |
/// | [`SDLK_F8`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F8`]) |
/// | [`SDLK_F9`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F9`]) |
/// | [`SDLK_F10`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F10`]) |
/// | [`SDLK_F11`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F11`]) |
/// | [`SDLK_F12`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F12`]) |
/// | [`SDLK_PRINTSCREEN`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_PRINTSCREEN`]) |
/// | [`SDLK_SCROLLLOCK`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_SCROLLLOCK`]) |
/// | [`SDLK_PAUSE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_PAUSE`]) |
/// | [`SDLK_INSERT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_INSERT`]) |
/// | [`SDLK_HOME`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_HOME`]) |
/// | [`SDLK_PAGEUP`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_PAGEUP`]) |
/// | [`SDLK_END`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_END`]) |
/// | [`SDLK_PAGEDOWN`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_PAGEDOWN`]) |
/// | [`SDLK_RIGHT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_RIGHT`]) |
/// | [`SDLK_LEFT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_LEFT`]) |
/// | [`SDLK_DOWN`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_DOWN`]) |
/// | [`SDLK_UP`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_UP`]) |
/// | [`SDLK_NUMLOCKCLEAR`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_NUMLOCKCLEAR`]) |
/// | [`SDLK_KP_DIVIDE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_DIVIDE`]) |
/// | [`SDLK_KP_MULTIPLY`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MULTIPLY`]) |
/// | [`SDLK_KP_MINUS`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MINUS`]) |
/// | [`SDLK_KP_PLUS`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_PLUS`]) |
/// | [`SDLK_KP_ENTER`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_ENTER`]) |
/// | [`SDLK_KP_1`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_1`]) |
/// | [`SDLK_KP_2`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_2`]) |
/// | [`SDLK_KP_3`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_3`]) |
/// | [`SDLK_KP_4`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_4`]) |
/// | [`SDLK_KP_5`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_5`]) |
/// | [`SDLK_KP_6`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_6`]) |
/// | [`SDLK_KP_7`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_7`]) |
/// | [`SDLK_KP_8`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_8`]) |
/// | [`SDLK_KP_9`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_9`]) |
/// | [`SDLK_KP_0`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_0`]) |
/// | [`SDLK_KP_PERIOD`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_PERIOD`]) |
/// | [`SDLK_APPLICATION`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_APPLICATION`]) |
/// | [`SDLK_POWER`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_POWER`]) |
/// | [`SDLK_KP_EQUALS`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_EQUALS`]) |
/// | [`SDLK_F13`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F13`]) |
/// | [`SDLK_F14`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F14`]) |
/// | [`SDLK_F15`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F15`]) |
/// | [`SDLK_F16`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F16`]) |
/// | [`SDLK_F17`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F17`]) |
/// | [`SDLK_F18`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F18`]) |
/// | [`SDLK_F19`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F19`]) |
/// | [`SDLK_F20`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F20`]) |
/// | [`SDLK_F21`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F21`]) |
/// | [`SDLK_F22`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F22`]) |
/// | [`SDLK_F23`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F23`]) |
/// | [`SDLK_F24`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F24`]) |
/// | [`SDLK_EXECUTE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_EXECUTE`]) |
/// | [`SDLK_HELP`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_HELP`]) |
/// | [`SDLK_MENU`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MENU`]) |
/// | [`SDLK_SELECT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_SELECT`]) |
/// | [`SDLK_STOP`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_STOP`]) |
/// | [`SDLK_AGAIN`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AGAIN`]) |
/// | [`SDLK_UNDO`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_UNDO`]) |
/// | [`SDLK_CUT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CUT`]) |
/// | [`SDLK_COPY`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_COPY`]) |
/// | [`SDLK_PASTE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_PASTE`]) |
/// | [`SDLK_FIND`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_FIND`]) |
/// | [`SDLK_MUTE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MUTE`]) |
/// | [`SDLK_VOLUMEUP`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_VOLUMEUP`]) |
/// | [`SDLK_VOLUMEDOWN`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_VOLUMEDOWN`]) |
/// | [`SDLK_KP_COMMA`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_COMMA`]) |
/// | [`SDLK_KP_EQUALSAS400`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_EQUALSAS400`]) |
/// | [`SDLK_ALTERASE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_ALTERASE`]) |
/// | [`SDLK_SYSREQ`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_SYSREQ`]) |
/// | [`SDLK_CANCEL`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CANCEL`]) |
/// | [`SDLK_CLEAR`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CLEAR`]) |
/// | [`SDLK_PRIOR`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_PRIOR`]) |
/// | [`SDLK_RETURN2`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_RETURN2`]) |
/// | [`SDLK_SEPARATOR`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_SEPARATOR`]) |
/// | [`SDLK_OUT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_OUT`]) |
/// | [`SDLK_OPER`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_OPER`]) |
/// | [`SDLK_CLEARAGAIN`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CLEARAGAIN`]) |
/// | [`SDLK_CRSEL`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CRSEL`]) |
/// | [`SDLK_EXSEL`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_EXSEL`]) |
/// | [`SDLK_KP_00`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_00`]) |
/// | [`SDLK_KP_000`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_000`]) |
/// | [`SDLK_THOUSANDSSEPARATOR`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_THOUSANDSSEPARATOR`]) |
/// | [`SDLK_DECIMALSEPARATOR`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_DECIMALSEPARATOR`]) |
/// | [`SDLK_CURRENCYUNIT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CURRENCYUNIT`]) |
/// | [`SDLK_CURRENCYSUBUNIT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CURRENCYSUBUNIT`]) |
/// | [`SDLK_KP_LEFTPAREN`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_LEFTPAREN`]) |
/// | [`SDLK_KP_RIGHTPAREN`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_RIGHTPAREN`]) |
/// | [`SDLK_KP_LEFTBRACE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_LEFTBRACE`]) |
/// | [`SDLK_KP_RIGHTBRACE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_RIGHTBRACE`]) |
/// | [`SDLK_KP_TAB`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_TAB`]) |
/// | [`SDLK_KP_BACKSPACE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_BACKSPACE`]) |
/// | [`SDLK_KP_A`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_A`]) |
/// | [`SDLK_KP_B`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_B`]) |
/// | [`SDLK_KP_C`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_C`]) |
/// | [`SDLK_KP_D`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_D`]) |
/// | [`SDLK_KP_E`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_E`]) |
/// | [`SDLK_KP_F`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_F`]) |
/// | [`SDLK_KP_XOR`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_XOR`]) |
/// | [`SDLK_KP_POWER`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_POWER`]) |
/// | [`SDLK_KP_PERCENT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_PERCENT`]) |
/// | [`SDLK_KP_LESS`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_LESS`]) |
/// | [`SDLK_KP_GREATER`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_GREATER`]) |
/// | [`SDLK_KP_AMPERSAND`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_AMPERSAND`]) |
/// | [`SDLK_KP_DBLAMPERSAND`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_DBLAMPERSAND`]) |
/// | [`SDLK_KP_VERTICALBAR`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_VERTICALBAR`]) |
/// | [`SDLK_KP_DBLVERTICALBAR`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_DBLVERTICALBAR`]) |
/// | [`SDLK_KP_COLON`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_COLON`]) |
/// | [`SDLK_KP_HASH`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_HASH`]) |
/// | [`SDLK_KP_SPACE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_SPACE`]) |
/// | [`SDLK_KP_AT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_AT`]) |
/// | [`SDLK_KP_EXCLAM`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_EXCLAM`]) |
/// | [`SDLK_KP_MEMSTORE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MEMSTORE`]) |
/// | [`SDLK_KP_MEMRECALL`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MEMRECALL`]) |
/// | [`SDLK_KP_MEMCLEAR`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MEMCLEAR`]) |
/// | [`SDLK_KP_MEMADD`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MEMADD`]) |
/// | [`SDLK_KP_MEMSUBTRACT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MEMSUBTRACT`]) |
/// | [`SDLK_KP_MEMMULTIPLY`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MEMMULTIPLY`]) |
/// | [`SDLK_KP_MEMDIVIDE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MEMDIVIDE`]) |
/// | [`SDLK_KP_PLUSMINUS`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_PLUSMINUS`]) |
/// | [`SDLK_KP_CLEAR`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_CLEAR`]) |
/// | [`SDLK_KP_CLEARENTRY`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_CLEARENTRY`]) |
/// | [`SDLK_KP_BINARY`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_BINARY`]) |
/// | [`SDLK_KP_OCTAL`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_OCTAL`]) |
/// | [`SDLK_KP_DECIMAL`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_DECIMAL`]) |
/// | [`SDLK_KP_HEXADECIMAL`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_HEXADECIMAL`]) |
/// | [`SDLK_LCTRL`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_LCTRL`]) |
/// | [`SDLK_LSHIFT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_LSHIFT`]) |
/// | [`SDLK_LALT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_LALT`]) |
/// | [`SDLK_LGUI`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_LGUI`]) |
/// | [`SDLK_RCTRL`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_RCTRL`]) |
/// | [`SDLK_RSHIFT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_RSHIFT`]) |
/// | [`SDLK_RALT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_RALT`]) |
/// | [`SDLK_RGUI`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_RGUI`]) |
/// | [`SDLK_MODE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MODE`]) |
/// | [`SDLK_SLEEP`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_SLEEP`]) |
/// | [`SDLK_WAKE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_WAKE`]) |
/// | [`SDLK_CHANNEL_INCREMENT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CHANNEL_INCREMENT`]) |
/// | [`SDLK_CHANNEL_DECREMENT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CHANNEL_DECREMENT`]) |
/// | [`SDLK_MEDIA_PLAY`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_PLAY`]) |
/// | [`SDLK_MEDIA_PAUSE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_PAUSE`]) |
/// | [`SDLK_MEDIA_RECORD`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_RECORD`]) |
/// | [`SDLK_MEDIA_FAST_FORWARD`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_FAST_FORWARD`]) |
/// | [`SDLK_MEDIA_REWIND`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_REWIND`]) |
/// | [`SDLK_MEDIA_NEXT_TRACK`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_NEXT_TRACK`]) |
/// | [`SDLK_MEDIA_PREVIOUS_TRACK`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_PREVIOUS_TRACK`]) |
/// | [`SDLK_MEDIA_STOP`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_STOP`]) |
/// | [`SDLK_MEDIA_EJECT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_EJECT`]) |
/// | [`SDLK_MEDIA_PLAY_PAUSE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_PLAY_PAUSE`]) |
/// | [`SDLK_MEDIA_SELECT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_SELECT`]) |
/// | [`SDLK_AC_NEW`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_NEW`]) |
/// | [`SDLK_AC_OPEN`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_OPEN`]) |
/// | [`SDLK_AC_CLOSE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_CLOSE`]) |
/// | [`SDLK_AC_EXIT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_EXIT`]) |
/// | [`SDLK_AC_SAVE`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_SAVE`]) |
/// | [`SDLK_AC_PRINT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_PRINT`]) |
/// | [`SDLK_AC_PROPERTIES`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_PROPERTIES`]) |
/// | [`SDLK_AC_SEARCH`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_SEARCH`]) |
/// | [`SDLK_AC_HOME`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_HOME`]) |
/// | [`SDLK_AC_BACK`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_BACK`]) |
/// | [`SDLK_AC_FORWARD`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_FORWARD`]) |
/// | [`SDLK_AC_STOP`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_STOP`]) |
/// | [`SDLK_AC_REFRESH`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_REFRESH`]) |
/// | [`SDLK_AC_BOOKMARKS`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_BOOKMARKS`]) |
/// | [`SDLK_SOFTLEFT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_SOFTLEFT`]) |
/// | [`SDLK_SOFTRIGHT`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_SOFTRIGHT`]) |
/// | [`SDLK_CALL`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CALL`]) |
/// | [`SDLK_ENDCALL`] | SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_ENDCALL`]) |
/// | [`SDLK_LEFT_TAB`] | Extended key Left Tab |
/// | [`SDLK_LEVEL5_SHIFT`] | Extended key Level 5 Shift |
/// | [`SDLK_MULTI_KEY_COMPOSE`] | Extended key Multi-key Compose |
/// | [`SDLK_LMETA`] | Extended key Left Meta |
/// | [`SDLK_RMETA`] | Extended key Right Meta |
/// | [`SDLK_LHYPER`] | Extended key Left Hyper |
/// | [`SDLK_RHYPER`] | Extended key Right Hyper |
pub type SDL_Keycode = Uint32;

pub const SDLK_EXTENDED_MASK: SDL_Keycode = ((536870912_u32) as SDL_Keycode);

pub const SDLK_SCANCODE_MASK: SDL_Keycode = ((1073741824_u32) as SDL_Keycode);

#[inline(always)]
pub const fn SDL_SCANCODE_TO_KEYCODE(X: SDL_Scancode) -> SDL_Keycode {
    ((X.0 as SDL_Keycode) | SDLK_SCANCODE_MASK)
}

/// 0
pub const SDLK_UNKNOWN: SDL_Keycode = (0x00000000 as SDL_Keycode);

/// '\r'
pub const SDLK_RETURN: SDL_Keycode = (0x0000000d as SDL_Keycode);

/// '\x1B'
pub const SDLK_ESCAPE: SDL_Keycode = (0x0000001b as SDL_Keycode);

/// '\b'
pub const SDLK_BACKSPACE: SDL_Keycode = (0x00000008 as SDL_Keycode);

/// '\t'
pub const SDLK_TAB: SDL_Keycode = (0x00000009 as SDL_Keycode);

/// ' '
pub const SDLK_SPACE: SDL_Keycode = (0x00000020 as SDL_Keycode);

/// '!'
pub const SDLK_EXCLAIM: SDL_Keycode = (0x00000021 as SDL_Keycode);

/// '"'
pub const SDLK_DBLAPOSTROPHE: SDL_Keycode = (0x00000022 as SDL_Keycode);

/// '#'
pub const SDLK_HASH: SDL_Keycode = (0x00000023 as SDL_Keycode);

/// '$'
pub const SDLK_DOLLAR: SDL_Keycode = (0x00000024 as SDL_Keycode);

/// '%'
pub const SDLK_PERCENT: SDL_Keycode = (0x00000025 as SDL_Keycode);

/// '&'
pub const SDLK_AMPERSAND: SDL_Keycode = (0x00000026 as SDL_Keycode);

/// '\''
pub const SDLK_APOSTROPHE: SDL_Keycode = (0x00000027 as SDL_Keycode);

/// '('
pub const SDLK_LEFTPAREN: SDL_Keycode = (0x00000028 as SDL_Keycode);

/// ')'
pub const SDLK_RIGHTPAREN: SDL_Keycode = (0x00000029 as SDL_Keycode);

/// '*'
pub const SDLK_ASTERISK: SDL_Keycode = (0x0000002a as SDL_Keycode);

/// '+'
pub const SDLK_PLUS: SDL_Keycode = (0x0000002b as SDL_Keycode);

/// ','
pub const SDLK_COMMA: SDL_Keycode = (0x0000002c as SDL_Keycode);

/// '-'
pub const SDLK_MINUS: SDL_Keycode = (0x0000002d as SDL_Keycode);

/// '.'
pub const SDLK_PERIOD: SDL_Keycode = (0x0000002e as SDL_Keycode);

/// '/'
pub const SDLK_SLASH: SDL_Keycode = (0x0000002f as SDL_Keycode);

/// '0'
pub const SDLK_0: SDL_Keycode = (0x00000030 as SDL_Keycode);

/// '1'
pub const SDLK_1: SDL_Keycode = (0x00000031 as SDL_Keycode);

/// '2'
pub const SDLK_2: SDL_Keycode = (0x00000032 as SDL_Keycode);

/// '3'
pub const SDLK_3: SDL_Keycode = (0x00000033 as SDL_Keycode);

/// '4'
pub const SDLK_4: SDL_Keycode = (0x00000034 as SDL_Keycode);

/// '5'
pub const SDLK_5: SDL_Keycode = (0x00000035 as SDL_Keycode);

/// '6'
pub const SDLK_6: SDL_Keycode = (0x00000036 as SDL_Keycode);

/// '7'
pub const SDLK_7: SDL_Keycode = (0x00000037 as SDL_Keycode);

/// '8'
pub const SDLK_8: SDL_Keycode = (0x00000038 as SDL_Keycode);

/// '9'
pub const SDLK_9: SDL_Keycode = (0x00000039 as SDL_Keycode);

/// ':'
pub const SDLK_COLON: SDL_Keycode = (0x0000003a as SDL_Keycode);

/// ';'
pub const SDLK_SEMICOLON: SDL_Keycode = (0x0000003b as SDL_Keycode);

/// '<'
pub const SDLK_LESS: SDL_Keycode = (0x0000003c as SDL_Keycode);

/// '='
pub const SDLK_EQUALS: SDL_Keycode = (0x0000003d as SDL_Keycode);

/// '>'
pub const SDLK_GREATER: SDL_Keycode = (0x0000003e as SDL_Keycode);

/// '?'
pub const SDLK_QUESTION: SDL_Keycode = (0x0000003f as SDL_Keycode);

/// '@'
pub const SDLK_AT: SDL_Keycode = (0x00000040 as SDL_Keycode);

/// '['
pub const SDLK_LEFTBRACKET: SDL_Keycode = (0x0000005b as SDL_Keycode);

/// '\\'
pub const SDLK_BACKSLASH: SDL_Keycode = (0x0000005c as SDL_Keycode);

/// ']'
pub const SDLK_RIGHTBRACKET: SDL_Keycode = (0x0000005d as SDL_Keycode);

/// '^'
pub const SDLK_CARET: SDL_Keycode = (0x0000005e as SDL_Keycode);

/// '_'
pub const SDLK_UNDERSCORE: SDL_Keycode = (0x0000005f as SDL_Keycode);

/// '`'
pub const SDLK_GRAVE: SDL_Keycode = (0x00000060 as SDL_Keycode);

/// 'a'
pub const SDLK_A: SDL_Keycode = (0x00000061 as SDL_Keycode);

/// 'b'
pub const SDLK_B: SDL_Keycode = (0x00000062 as SDL_Keycode);

/// 'c'
pub const SDLK_C: SDL_Keycode = (0x00000063 as SDL_Keycode);

/// 'd'
pub const SDLK_D: SDL_Keycode = (0x00000064 as SDL_Keycode);

/// 'e'
pub const SDLK_E: SDL_Keycode = (0x00000065 as SDL_Keycode);

/// 'f'
pub const SDLK_F: SDL_Keycode = (0x00000066 as SDL_Keycode);

/// 'g'
pub const SDLK_G: SDL_Keycode = (0x00000067 as SDL_Keycode);

/// 'h'
pub const SDLK_H: SDL_Keycode = (0x00000068 as SDL_Keycode);

/// 'i'
pub const SDLK_I: SDL_Keycode = (0x00000069 as SDL_Keycode);

/// 'j'
pub const SDLK_J: SDL_Keycode = (0x0000006a as SDL_Keycode);

/// 'k'
pub const SDLK_K: SDL_Keycode = (0x0000006b as SDL_Keycode);

/// 'l'
pub const SDLK_L: SDL_Keycode = (0x0000006c as SDL_Keycode);

/// 'm'
pub const SDLK_M: SDL_Keycode = (0x0000006d as SDL_Keycode);

/// 'n'
pub const SDLK_N: SDL_Keycode = (0x0000006e as SDL_Keycode);

/// 'o'
pub const SDLK_O: SDL_Keycode = (0x0000006f as SDL_Keycode);

/// 'p'
pub const SDLK_P: SDL_Keycode = (0x00000070 as SDL_Keycode);

/// 'q'
pub const SDLK_Q: SDL_Keycode = (0x00000071 as SDL_Keycode);

/// 'r'
pub const SDLK_R: SDL_Keycode = (0x00000072 as SDL_Keycode);

/// 's'
pub const SDLK_S: SDL_Keycode = (0x00000073 as SDL_Keycode);

/// 't'
pub const SDLK_T: SDL_Keycode = (0x00000074 as SDL_Keycode);

/// 'u'
pub const SDLK_U: SDL_Keycode = (0x00000075 as SDL_Keycode);

/// 'v'
pub const SDLK_V: SDL_Keycode = (0x00000076 as SDL_Keycode);

/// 'w'
pub const SDLK_W: SDL_Keycode = (0x00000077 as SDL_Keycode);

/// 'x'
pub const SDLK_X: SDL_Keycode = (0x00000078 as SDL_Keycode);

/// 'y'
pub const SDLK_Y: SDL_Keycode = (0x00000079 as SDL_Keycode);

/// 'z'
pub const SDLK_Z: SDL_Keycode = (0x0000007a as SDL_Keycode);

/// '{'
pub const SDLK_LEFTBRACE: SDL_Keycode = (0x0000007b as SDL_Keycode);

/// '|'
pub const SDLK_PIPE: SDL_Keycode = (0x0000007c as SDL_Keycode);

/// '}'
pub const SDLK_RIGHTBRACE: SDL_Keycode = (0x0000007d as SDL_Keycode);

/// '~'
pub const SDLK_TILDE: SDL_Keycode = (0x0000007e as SDL_Keycode);

/// '\x7F'
pub const SDLK_DELETE: SDL_Keycode = (0x0000007f as SDL_Keycode);

/// '\xB1'
pub const SDLK_PLUSMINUS: SDL_Keycode = (0x000000b1 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CAPSLOCK`])
pub const SDLK_CAPSLOCK: SDL_Keycode = (0x40000039 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F1`])
pub const SDLK_F1: SDL_Keycode = (0x4000003a as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F2`])
pub const SDLK_F2: SDL_Keycode = (0x4000003b as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F3`])
pub const SDLK_F3: SDL_Keycode = (0x4000003c as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F4`])
pub const SDLK_F4: SDL_Keycode = (0x4000003d as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F5`])
pub const SDLK_F5: SDL_Keycode = (0x4000003e as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F6`])
pub const SDLK_F6: SDL_Keycode = (0x4000003f as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F7`])
pub const SDLK_F7: SDL_Keycode = (0x40000040 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F8`])
pub const SDLK_F8: SDL_Keycode = (0x40000041 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F9`])
pub const SDLK_F9: SDL_Keycode = (0x40000042 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F10`])
pub const SDLK_F10: SDL_Keycode = (0x40000043 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F11`])
pub const SDLK_F11: SDL_Keycode = (0x40000044 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F12`])
pub const SDLK_F12: SDL_Keycode = (0x40000045 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_PRINTSCREEN`])
pub const SDLK_PRINTSCREEN: SDL_Keycode = (0x40000046 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_SCROLLLOCK`])
pub const SDLK_SCROLLLOCK: SDL_Keycode = (0x40000047 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_PAUSE`])
pub const SDLK_PAUSE: SDL_Keycode = (0x40000048 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_INSERT`])
pub const SDLK_INSERT: SDL_Keycode = (0x40000049 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_HOME`])
pub const SDLK_HOME: SDL_Keycode = (0x4000004a as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_PAGEUP`])
pub const SDLK_PAGEUP: SDL_Keycode = (0x4000004b as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_END`])
pub const SDLK_END: SDL_Keycode = (0x4000004d as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_PAGEDOWN`])
pub const SDLK_PAGEDOWN: SDL_Keycode = (0x4000004e as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_RIGHT`])
pub const SDLK_RIGHT: SDL_Keycode = (0x4000004f as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_LEFT`])
pub const SDLK_LEFT: SDL_Keycode = (0x40000050 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_DOWN`])
pub const SDLK_DOWN: SDL_Keycode = (0x40000051 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_UP`])
pub const SDLK_UP: SDL_Keycode = (0x40000052 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_NUMLOCKCLEAR`])
pub const SDLK_NUMLOCKCLEAR: SDL_Keycode = (0x40000053 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_DIVIDE`])
pub const SDLK_KP_DIVIDE: SDL_Keycode = (0x40000054 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MULTIPLY`])
pub const SDLK_KP_MULTIPLY: SDL_Keycode = (0x40000055 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MINUS`])
pub const SDLK_KP_MINUS: SDL_Keycode = (0x40000056 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_PLUS`])
pub const SDLK_KP_PLUS: SDL_Keycode = (0x40000057 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_ENTER`])
pub const SDLK_KP_ENTER: SDL_Keycode = (0x40000058 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_1`])
pub const SDLK_KP_1: SDL_Keycode = (0x40000059 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_2`])
pub const SDLK_KP_2: SDL_Keycode = (0x4000005a as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_3`])
pub const SDLK_KP_3: SDL_Keycode = (0x4000005b as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_4`])
pub const SDLK_KP_4: SDL_Keycode = (0x4000005c as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_5`])
pub const SDLK_KP_5: SDL_Keycode = (0x4000005d as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_6`])
pub const SDLK_KP_6: SDL_Keycode = (0x4000005e as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_7`])
pub const SDLK_KP_7: SDL_Keycode = (0x4000005f as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_8`])
pub const SDLK_KP_8: SDL_Keycode = (0x40000060 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_9`])
pub const SDLK_KP_9: SDL_Keycode = (0x40000061 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_0`])
pub const SDLK_KP_0: SDL_Keycode = (0x40000062 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_PERIOD`])
pub const SDLK_KP_PERIOD: SDL_Keycode = (0x40000063 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_APPLICATION`])
pub const SDLK_APPLICATION: SDL_Keycode = (0x40000065 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_POWER`])
pub const SDLK_POWER: SDL_Keycode = (0x40000066 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_EQUALS`])
pub const SDLK_KP_EQUALS: SDL_Keycode = (0x40000067 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F13`])
pub const SDLK_F13: SDL_Keycode = (0x40000068 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F14`])
pub const SDLK_F14: SDL_Keycode = (0x40000069 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F15`])
pub const SDLK_F15: SDL_Keycode = (0x4000006a as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F16`])
pub const SDLK_F16: SDL_Keycode = (0x4000006b as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F17`])
pub const SDLK_F17: SDL_Keycode = (0x4000006c as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F18`])
pub const SDLK_F18: SDL_Keycode = (0x4000006d as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F19`])
pub const SDLK_F19: SDL_Keycode = (0x4000006e as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F20`])
pub const SDLK_F20: SDL_Keycode = (0x4000006f as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F21`])
pub const SDLK_F21: SDL_Keycode = (0x40000070 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F22`])
pub const SDLK_F22: SDL_Keycode = (0x40000071 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F23`])
pub const SDLK_F23: SDL_Keycode = (0x40000072 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_F24`])
pub const SDLK_F24: SDL_Keycode = (0x40000073 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_EXECUTE`])
pub const SDLK_EXECUTE: SDL_Keycode = (0x40000074 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_HELP`])
pub const SDLK_HELP: SDL_Keycode = (0x40000075 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MENU`])
pub const SDLK_MENU: SDL_Keycode = (0x40000076 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_SELECT`])
pub const SDLK_SELECT: SDL_Keycode = (0x40000077 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_STOP`])
pub const SDLK_STOP: SDL_Keycode = (0x40000078 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AGAIN`])
pub const SDLK_AGAIN: SDL_Keycode = (0x40000079 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_UNDO`])
pub const SDLK_UNDO: SDL_Keycode = (0x4000007a as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CUT`])
pub const SDLK_CUT: SDL_Keycode = (0x4000007b as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_COPY`])
pub const SDLK_COPY: SDL_Keycode = (0x4000007c as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_PASTE`])
pub const SDLK_PASTE: SDL_Keycode = (0x4000007d as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_FIND`])
pub const SDLK_FIND: SDL_Keycode = (0x4000007e as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MUTE`])
pub const SDLK_MUTE: SDL_Keycode = (0x4000007f as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_VOLUMEUP`])
pub const SDLK_VOLUMEUP: SDL_Keycode = (0x40000080 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_VOLUMEDOWN`])
pub const SDLK_VOLUMEDOWN: SDL_Keycode = (0x40000081 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_COMMA`])
pub const SDLK_KP_COMMA: SDL_Keycode = (0x40000085 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_EQUALSAS400`])
pub const SDLK_KP_EQUALSAS400: SDL_Keycode = (0x40000086 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_ALTERASE`])
pub const SDLK_ALTERASE: SDL_Keycode = (0x40000099 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_SYSREQ`])
pub const SDLK_SYSREQ: SDL_Keycode = (0x4000009a as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CANCEL`])
pub const SDLK_CANCEL: SDL_Keycode = (0x4000009b as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CLEAR`])
pub const SDLK_CLEAR: SDL_Keycode = (0x4000009c as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_PRIOR`])
pub const SDLK_PRIOR: SDL_Keycode = (0x4000009d as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_RETURN2`])
pub const SDLK_RETURN2: SDL_Keycode = (0x4000009e as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_SEPARATOR`])
pub const SDLK_SEPARATOR: SDL_Keycode = (0x4000009f as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_OUT`])
pub const SDLK_OUT: SDL_Keycode = (0x400000a0 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_OPER`])
pub const SDLK_OPER: SDL_Keycode = (0x400000a1 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CLEARAGAIN`])
pub const SDLK_CLEARAGAIN: SDL_Keycode = (0x400000a2 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CRSEL`])
pub const SDLK_CRSEL: SDL_Keycode = (0x400000a3 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_EXSEL`])
pub const SDLK_EXSEL: SDL_Keycode = (0x400000a4 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_00`])
pub const SDLK_KP_00: SDL_Keycode = (0x400000b0 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_000`])
pub const SDLK_KP_000: SDL_Keycode = (0x400000b1 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_THOUSANDSSEPARATOR`])
pub const SDLK_THOUSANDSSEPARATOR: SDL_Keycode = (0x400000b2 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_DECIMALSEPARATOR`])
pub const SDLK_DECIMALSEPARATOR: SDL_Keycode = (0x400000b3 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CURRENCYUNIT`])
pub const SDLK_CURRENCYUNIT: SDL_Keycode = (0x400000b4 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CURRENCYSUBUNIT`])
pub const SDLK_CURRENCYSUBUNIT: SDL_Keycode = (0x400000b5 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_LEFTPAREN`])
pub const SDLK_KP_LEFTPAREN: SDL_Keycode = (0x400000b6 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_RIGHTPAREN`])
pub const SDLK_KP_RIGHTPAREN: SDL_Keycode = (0x400000b7 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_LEFTBRACE`])
pub const SDLK_KP_LEFTBRACE: SDL_Keycode = (0x400000b8 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_RIGHTBRACE`])
pub const SDLK_KP_RIGHTBRACE: SDL_Keycode = (0x400000b9 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_TAB`])
pub const SDLK_KP_TAB: SDL_Keycode = (0x400000ba as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_BACKSPACE`])
pub const SDLK_KP_BACKSPACE: SDL_Keycode = (0x400000bb as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_A`])
pub const SDLK_KP_A: SDL_Keycode = (0x400000bc as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_B`])
pub const SDLK_KP_B: SDL_Keycode = (0x400000bd as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_C`])
pub const SDLK_KP_C: SDL_Keycode = (0x400000be as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_D`])
pub const SDLK_KP_D: SDL_Keycode = (0x400000bf as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_E`])
pub const SDLK_KP_E: SDL_Keycode = (0x400000c0 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_F`])
pub const SDLK_KP_F: SDL_Keycode = (0x400000c1 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_XOR`])
pub const SDLK_KP_XOR: SDL_Keycode = (0x400000c2 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_POWER`])
pub const SDLK_KP_POWER: SDL_Keycode = (0x400000c3 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_PERCENT`])
pub const SDLK_KP_PERCENT: SDL_Keycode = (0x400000c4 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_LESS`])
pub const SDLK_KP_LESS: SDL_Keycode = (0x400000c5 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_GREATER`])
pub const SDLK_KP_GREATER: SDL_Keycode = (0x400000c6 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_AMPERSAND`])
pub const SDLK_KP_AMPERSAND: SDL_Keycode = (0x400000c7 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_DBLAMPERSAND`])
pub const SDLK_KP_DBLAMPERSAND: SDL_Keycode = (0x400000c8 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_VERTICALBAR`])
pub const SDLK_KP_VERTICALBAR: SDL_Keycode = (0x400000c9 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_DBLVERTICALBAR`])
pub const SDLK_KP_DBLVERTICALBAR: SDL_Keycode = (0x400000ca as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_COLON`])
pub const SDLK_KP_COLON: SDL_Keycode = (0x400000cb as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_HASH`])
pub const SDLK_KP_HASH: SDL_Keycode = (0x400000cc as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_SPACE`])
pub const SDLK_KP_SPACE: SDL_Keycode = (0x400000cd as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_AT`])
pub const SDLK_KP_AT: SDL_Keycode = (0x400000ce as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_EXCLAM`])
pub const SDLK_KP_EXCLAM: SDL_Keycode = (0x400000cf as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MEMSTORE`])
pub const SDLK_KP_MEMSTORE: SDL_Keycode = (0x400000d0 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MEMRECALL`])
pub const SDLK_KP_MEMRECALL: SDL_Keycode = (0x400000d1 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MEMCLEAR`])
pub const SDLK_KP_MEMCLEAR: SDL_Keycode = (0x400000d2 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MEMADD`])
pub const SDLK_KP_MEMADD: SDL_Keycode = (0x400000d3 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MEMSUBTRACT`])
pub const SDLK_KP_MEMSUBTRACT: SDL_Keycode = (0x400000d4 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MEMMULTIPLY`])
pub const SDLK_KP_MEMMULTIPLY: SDL_Keycode = (0x400000d5 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_MEMDIVIDE`])
pub const SDLK_KP_MEMDIVIDE: SDL_Keycode = (0x400000d6 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_PLUSMINUS`])
pub const SDLK_KP_PLUSMINUS: SDL_Keycode = (0x400000d7 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_CLEAR`])
pub const SDLK_KP_CLEAR: SDL_Keycode = (0x400000d8 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_CLEARENTRY`])
pub const SDLK_KP_CLEARENTRY: SDL_Keycode = (0x400000d9 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_BINARY`])
pub const SDLK_KP_BINARY: SDL_Keycode = (0x400000da as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_OCTAL`])
pub const SDLK_KP_OCTAL: SDL_Keycode = (0x400000db as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_DECIMAL`])
pub const SDLK_KP_DECIMAL: SDL_Keycode = (0x400000dc as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_KP_HEXADECIMAL`])
pub const SDLK_KP_HEXADECIMAL: SDL_Keycode = (0x400000dd as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_LCTRL`])
pub const SDLK_LCTRL: SDL_Keycode = (0x400000e0 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_LSHIFT`])
pub const SDLK_LSHIFT: SDL_Keycode = (0x400000e1 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_LALT`])
pub const SDLK_LALT: SDL_Keycode = (0x400000e2 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_LGUI`])
pub const SDLK_LGUI: SDL_Keycode = (0x400000e3 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_RCTRL`])
pub const SDLK_RCTRL: SDL_Keycode = (0x400000e4 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_RSHIFT`])
pub const SDLK_RSHIFT: SDL_Keycode = (0x400000e5 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_RALT`])
pub const SDLK_RALT: SDL_Keycode = (0x400000e6 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_RGUI`])
pub const SDLK_RGUI: SDL_Keycode = (0x400000e7 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MODE`])
pub const SDLK_MODE: SDL_Keycode = (0x40000101 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_SLEEP`])
pub const SDLK_SLEEP: SDL_Keycode = (0x40000102 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_WAKE`])
pub const SDLK_WAKE: SDL_Keycode = (0x40000103 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CHANNEL_INCREMENT`])
pub const SDLK_CHANNEL_INCREMENT: SDL_Keycode = (0x40000104 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CHANNEL_DECREMENT`])
pub const SDLK_CHANNEL_DECREMENT: SDL_Keycode = (0x40000105 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_PLAY`])
pub const SDLK_MEDIA_PLAY: SDL_Keycode = (0x40000106 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_PAUSE`])
pub const SDLK_MEDIA_PAUSE: SDL_Keycode = (0x40000107 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_RECORD`])
pub const SDLK_MEDIA_RECORD: SDL_Keycode = (0x40000108 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_FAST_FORWARD`])
pub const SDLK_MEDIA_FAST_FORWARD: SDL_Keycode = (0x40000109 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_REWIND`])
pub const SDLK_MEDIA_REWIND: SDL_Keycode = (0x4000010a as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_NEXT_TRACK`])
pub const SDLK_MEDIA_NEXT_TRACK: SDL_Keycode = (0x4000010b as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_PREVIOUS_TRACK`])
pub const SDLK_MEDIA_PREVIOUS_TRACK: SDL_Keycode = (0x4000010c as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_STOP`])
pub const SDLK_MEDIA_STOP: SDL_Keycode = (0x4000010d as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_EJECT`])
pub const SDLK_MEDIA_EJECT: SDL_Keycode = (0x4000010e as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_PLAY_PAUSE`])
pub const SDLK_MEDIA_PLAY_PAUSE: SDL_Keycode = (0x4000010f as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_MEDIA_SELECT`])
pub const SDLK_MEDIA_SELECT: SDL_Keycode = (0x40000110 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_NEW`])
pub const SDLK_AC_NEW: SDL_Keycode = (0x40000111 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_OPEN`])
pub const SDLK_AC_OPEN: SDL_Keycode = (0x40000112 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_CLOSE`])
pub const SDLK_AC_CLOSE: SDL_Keycode = (0x40000113 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_EXIT`])
pub const SDLK_AC_EXIT: SDL_Keycode = (0x40000114 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_SAVE`])
pub const SDLK_AC_SAVE: SDL_Keycode = (0x40000115 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_PRINT`])
pub const SDLK_AC_PRINT: SDL_Keycode = (0x40000116 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_PROPERTIES`])
pub const SDLK_AC_PROPERTIES: SDL_Keycode = (0x40000117 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_SEARCH`])
pub const SDLK_AC_SEARCH: SDL_Keycode = (0x40000118 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_HOME`])
pub const SDLK_AC_HOME: SDL_Keycode = (0x40000119 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_BACK`])
pub const SDLK_AC_BACK: SDL_Keycode = (0x4000011a as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_FORWARD`])
pub const SDLK_AC_FORWARD: SDL_Keycode = (0x4000011b as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_STOP`])
pub const SDLK_AC_STOP: SDL_Keycode = (0x4000011c as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_REFRESH`])
pub const SDLK_AC_REFRESH: SDL_Keycode = (0x4000011d as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_AC_BOOKMARKS`])
pub const SDLK_AC_BOOKMARKS: SDL_Keycode = (0x4000011e as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_SOFTLEFT`])
pub const SDLK_SOFTLEFT: SDL_Keycode = (0x4000011f as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_SOFTRIGHT`])
pub const SDLK_SOFTRIGHT: SDL_Keycode = (0x40000120 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_CALL`])
pub const SDLK_CALL: SDL_Keycode = (0x40000121 as SDL_Keycode);

/// SDL_SCANCODE_TO_KEYCODE([`SDL_SCANCODE_ENDCALL`])
pub const SDLK_ENDCALL: SDL_Keycode = (0x40000122 as SDL_Keycode);

/// Extended key Left Tab
pub const SDLK_LEFT_TAB: SDL_Keycode = (0x20000001 as SDL_Keycode);

/// Extended key Level 5 Shift
pub const SDLK_LEVEL5_SHIFT: SDL_Keycode = (0x20000002 as SDL_Keycode);

/// Extended key Multi-key Compose
pub const SDLK_MULTI_KEY_COMPOSE: SDL_Keycode = (0x20000003 as SDL_Keycode);

/// Extended key Left Meta
pub const SDLK_LMETA: SDL_Keycode = (0x20000004 as SDL_Keycode);

/// Extended key Right Meta
pub const SDLK_RMETA: SDL_Keycode = (0x20000005 as SDL_Keycode);

/// Extended key Left Hyper
pub const SDLK_LHYPER: SDL_Keycode = (0x20000006 as SDL_Keycode);

/// Extended key Right Hyper
pub const SDLK_RHYPER: SDL_Keycode = (0x20000007 as SDL_Keycode);

/// Valid key modifiers (possibly OR'd together).
///
/// ### Availability
/// This datatype is available since SDL 3.2.0.
///
/// ### Known values (`sdl3-sys`)
/// | Constant | Description |
/// | -------- | ----------- |
/// | [`SDL_KMOD_NONE`] | no modifier is applicable. |
/// | [`SDL_KMOD_LSHIFT`] | the left Shift key is down. |
/// | [`SDL_KMOD_RSHIFT`] | the right Shift key is down. |
/// | [`SDL_KMOD_LEVEL5`] | the Level 5 Shift key is down. |
/// | [`SDL_KMOD_LCTRL`] | the left Ctrl (Control) key is down. |
/// | [`SDL_KMOD_RCTRL`] | the right Ctrl (Control) key is down. |
/// | [`SDL_KMOD_LALT`] | the left Alt key is down. |
/// | [`SDL_KMOD_RALT`] | the right Alt key is down. |
/// | [`SDL_KMOD_LGUI`] | the left GUI key (often the Windows key) is down. |
/// | [`SDL_KMOD_RGUI`] | the right GUI key (often the Windows key) is down. |
/// | [`SDL_KMOD_NUM`] | the Num Lock key (may be located on an extended keypad) is down. |
/// | [`SDL_KMOD_CAPS`] | the Caps Lock key is down. |
/// | [`SDL_KMOD_MODE`] | the !AltGr key is down. |
/// | [`SDL_KMOD_SCROLL`] | the Scroll Lock key is down. |
/// | [`SDL_KMOD_CTRL`] | Any Ctrl key is down. |
/// | [`SDL_KMOD_SHIFT`] | Any Shift key is down. |
/// | [`SDL_KMOD_ALT`] | Any Alt key is down. |
/// | [`SDL_KMOD_GUI`] | Any GUI key is down. |
pub type SDL_Keymod = Uint16;

/// no modifier is applicable.
pub const SDL_KMOD_NONE: SDL_Keymod = (0x0000 as SDL_Keymod);

/// the left Shift key is down.
pub const SDL_KMOD_LSHIFT: SDL_Keymod = (0x0001 as SDL_Keymod);

/// the right Shift key is down.
pub const SDL_KMOD_RSHIFT: SDL_Keymod = (0x0002 as SDL_Keymod);

/// the Level 5 Shift key is down.
pub const SDL_KMOD_LEVEL5: SDL_Keymod = (0x0004 as SDL_Keymod);

/// the left Ctrl (Control) key is down.
pub const SDL_KMOD_LCTRL: SDL_Keymod = (0x0040 as SDL_Keymod);

/// the right Ctrl (Control) key is down.
pub const SDL_KMOD_RCTRL: SDL_Keymod = (0x0080 as SDL_Keymod);

/// the left Alt key is down.
pub const SDL_KMOD_LALT: SDL_Keymod = (0x0100 as SDL_Keymod);

/// the right Alt key is down.
pub const SDL_KMOD_RALT: SDL_Keymod = (0x0200 as SDL_Keymod);

/// the left GUI key (often the Windows key) is down.
pub const SDL_KMOD_LGUI: SDL_Keymod = (0x0400 as SDL_Keymod);

/// the right GUI key (often the Windows key) is down.
pub const SDL_KMOD_RGUI: SDL_Keymod = (0x0800 as SDL_Keymod);

/// the Num Lock key (may be located on an extended keypad) is down.
pub const SDL_KMOD_NUM: SDL_Keymod = (0x1000 as SDL_Keymod);

/// the Caps Lock key is down.
pub const SDL_KMOD_CAPS: SDL_Keymod = (0x2000 as SDL_Keymod);

/// the !AltGr key is down.
pub const SDL_KMOD_MODE: SDL_Keymod = (0x4000 as SDL_Keymod);

/// the Scroll Lock key is down.
pub const SDL_KMOD_SCROLL: SDL_Keymod = (0x8000 as SDL_Keymod);

/// Any Ctrl key is down.
pub const SDL_KMOD_CTRL: SDL_Keymod = (SDL_KMOD_LCTRL | SDL_KMOD_RCTRL);

/// Any Shift key is down.
pub const SDL_KMOD_SHIFT: SDL_Keymod = (SDL_KMOD_LSHIFT | SDL_KMOD_RSHIFT);

/// Any Alt key is down.
pub const SDL_KMOD_ALT: SDL_Keymod = (SDL_KMOD_LALT | SDL_KMOD_RALT);

/// Any GUI key is down.
pub const SDL_KMOD_GUI: SDL_Keymod = (SDL_KMOD_LGUI | SDL_KMOD_RGUI);

#[cfg(doc)]
use crate::everything::*;
