#!/usr/bin/env python
#
# Author: Mike McKerns (mmckerns @caltech and @uqfoundation)
# Copyright (c) 2008-2016 California Institute of Technology.
# Copyright (c) 2016-2025 The Uncertainty Quantification Foundation.
# License: 3-clause BSD.  The full license text is available at:
#  - https://github.com/uqfoundation/dill/blob/master/LICENSE

import os
import sys
# drop support for older python
if sys.version_info < (3, 8):
    unsupported = 'Versions of Python before 3.8 are not supported'
    raise ValueError(unsupported)

# get distribution meta info
here = os.path.abspath(os.path.dirname(__file__))
sys.path.append(here)
from version import (__version__, __author__, __contact__ as AUTHOR_EMAIL,
                     get_license_text, get_readme_as_rst, write_info_file)
LICENSE = get_license_text(os.path.join(here, 'LICENSE'))
README = get_readme_as_rst(os.path.join(here, 'README.md'))
    
# write meta info file
write_info_file(here, 'dill', doc=README, license=LICENSE,
                version=__version__, author=__author__)
del here, get_license_text, get_readme_as_rst, write_info_file

# check if setuptools is available
try:
    from setuptools import setup
    from setuptools.dist import Distribution
    has_setuptools = True
except ImportError:
    from distutils.core import setup
    Distribution = object
    has_setuptools = False

# build the 'setup' call
setup_kwds = dict(
    name='dill',
    version=__version__,
    description='serialize all of Python',
    long_description = README.strip(),
    author = __author__,
    author_email = AUTHOR_EMAIL,
    maintainer = __author__,
    maintainer_email = AUTHOR_EMAIL,
    license = 'BSD-3-Clause',
    platforms = ['Linux', 'Windows', 'Mac'],
    url = 'https://github.com/uqfoundation/dill',
    download_url = 'https://pypi.org/project/dill/#files',
    project_urls = {
        'Documentation':'http://dill.rtfd.io',
        'Source Code':'https://github.com/uqfoundation/dill',
        'Bug Tracker':'https://github.com/uqfoundation/dill/issues',
    },
    python_requires = '>=3.8',
    classifiers = [
        'Development Status :: 5 - Production/Stable',
        'Intended Audience :: Developers',
        'Intended Audience :: Science/Research',
        'License :: OSI Approved :: BSD License',
        'Programming Language :: Python :: 3',
        'Programming Language :: Python :: 3.8',
        'Programming Language :: Python :: 3.9',
        'Programming Language :: Python :: 3.10',
        'Programming Language :: Python :: 3.11',
        'Programming Language :: Python :: 3.12',
        'Programming Language :: Python :: 3.13',
        'Programming Language :: Python :: Implementation :: CPython',
        'Programming Language :: Python :: Implementation :: PyPy',
        'Topic :: Scientific/Engineering',
        'Topic :: Software Development',
    ],
    packages = ['dill','dill.tests'],
    package_dir = {'dill':'dill', 'dill.tests':'dill/tests'},
    scripts=['scripts/undill','scripts/get_objgraph','scripts/get_gprof'],
)

# force python-, abi-, and platform-specific naming of bdist_wheel
class BinaryDistribution(Distribution):
    """Distribution which forces a binary package with platform name"""
    def has_ext_modules(foo):
        return True

# define dependencies
ctypes_version = 'ctypes>=1.0.1'
objgraph_version = 'objgraph>=1.7.2'
gprof2dot_version = 'gprof2dot>=2022.7.29'
pyreadline_version = 'pyreadline>=1.7.1'
# add dependencies
depend = [ctypes_version]
if sys.platform[:3] == 'win':
    extras = {'readline': [pyreadline_version], 'graph': [objgraph_version], 'profile': [gprof2dot_version]}
else:
    extras = {'readline': [], 'graph': [objgraph_version], 'profile': [gprof2dot_version]}
# update setup kwds
if has_setuptools:
    setup_kwds.update(
        zip_safe=False,
        # distclass=BinaryDistribution,
        # install_requires=depend,
        extras_require=extras,
    )

# call setup
setup(**setup_kwds)

# if dependencies are missing, print a warning
try:
    pass
    #import ctypes
    #import objgraph
    #import gprof2dot
    #import readline
except ImportError:
    print ("\n***********************************************************")
    print ("WARNING: One of the following dependencies is unresolved:")
#   print ("    %s" % ctypes_version)
    print ("    %s (optional)" % objgraph_version)
    print ("    %s (optional)" % gprof2dot_version)
    if sys.platform[:3] == 'win':
        print ("    %s (optional)" % pyreadline_version)
    print ("***********************************************************\n")
