/*
 * Endpoint - Linux SBP2 Disk Target
 *
 * Copyright (C) 2003 Oracle.  All rights reserved.
 *
 * Author: Manish Singh <manish.singh@oracle.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <glib.h>

#include "rbcdisk.h"
#include "rbcprivate.h"


static gsize process_filename (const gchar        *filename,
			       gint               *out_fd);

static gint  partition_order  (const RBCPartition *a,
			       const RBCPartition *b);


RBCDisk *
rbc_disk_new (RBCDiskType type)
{
  RBCDisk *disk;

  disk = g_new (RBCDisk, 1);

  disk->type = type;

  disk->filename = NULL;
  disk->partitions = NULL;

  disk->fd = -1;
  disk->capacity = 0;

  disk->active = FALSE;

  return disk;
}

void
rbc_disk_destroy (RBCDisk *disk)
{
  GList        *list;
  RBCPartition *partition;

  g_return_if_fail (disk != NULL);

  if (disk->fd != -1)
    close (disk->fd);

  if (disk->filename)
    g_free (disk->filename);

  if (disk->partitions)
    {
      list = disk->partitions;
      
      while (list)
	{
	  partition = list->data;

	  if (partition->fd != -1)
	    close (partition->fd);

	  if (partition->filename)
	    g_free (partition->filename);

	  list = list->next;
	}

      g_list_free (disk->partitions);
    }

  g_free (disk);
}

RBCDiskType
rbc_disk_get_type (RBCDisk *disk)
{
  g_return_val_if_fail (disk != NULL, RBC_DISK_PHYSICAL);

  return disk->type;
}

gboolean
rbc_device_is_equal (RBCDisk *a,
                     RBCDisk *b)
{
  g_return_val_if_fail (a != NULL, FALSE);
  g_return_val_if_fail (b != NULL, FALSE);

  /* TODO: implement */

  return a == b;
}

gboolean
rbc_disk_set_filename (RBCDisk     *disk,
                       const gchar *filename)
{
  gsize capacity;

  g_return_val_if_fail (disk != NULL, FALSE);
  g_return_val_if_fail (disk->type == RBC_DISK_PHYSICAL, FALSE);
  g_return_val_if_fail (disk->active == FALSE, FALSE);
  g_return_val_if_fail (filename != NULL, FALSE);
  g_return_val_if_fail (disk->filename == NULL, FALSE);

  capacity = process_filename (filename, NULL);

  if (capacity == 0)
    return FALSE;

  disk->filename = g_strdup (filename);
  disk->capacity = capacity;

  return TRUE;
}

gboolean
rbc_disk_add_partition (RBCDisk     *disk,
                        const gchar *filename,
			guint        order)
{
  gsize         capacity;
  RBCPartition *partition;

  g_return_val_if_fail (disk != NULL, FALSE);
  g_return_val_if_fail (disk->type == RBC_DISK_VIRTUAL, FALSE);
  g_return_val_if_fail (disk->active == FALSE, FALSE);
  g_return_val_if_fail (filename != NULL, FALSE);

  capacity = process_filename (filename, NULL);

  if (capacity == 0)
    return FALSE;

  partition = g_new (RBCPartition, 1);

  partition->filename = g_strdup (filename);
  partition->order = order;

  partition->fd = -1;
  partition->capacity = capacity;

  disk->partitions = g_list_insert_sorted (disk->partitions, partition,
                                           (GCompareFunc) partition_order);

  return TRUE;
}

gsize
rbc_disk_get_capacity (RBCDisk *disk)
{
  g_return_val_if_fail (disk != NULL, 0);

  return disk->capacity;
}

gsize
rbc_disk_get_block_size (RBCDisk *disk)
{
  g_return_val_if_fail (disk != NULL, 0);

  return RBC_BLOCK_SIZE;
}

gboolean
rbc_disk_activate (RBCDisk *disk)
{
  GList        *list;
  RBCPartition *partition;

  g_return_val_if_fail (disk != NULL, FALSE);

  switch (disk->type)
    {
    case RBC_DISK_PHYSICAL:
      g_return_val_if_fail (disk->filename != NULL, FALSE);

      disk->capacity = process_filename (disk->filename, &disk->fd);

      if (disk->capacity == 0)
	return FALSE;

      break;

    case RBC_DISK_VIRTUAL:
      g_return_val_if_fail (disk->partitions != NULL, FALSE);

      list = disk->partitions;

      while (list)
	{
	  partition = list->data;

	  partition->capacity = process_filename (partition->filename,
	                                          &partition->fd);

	  if (partition->capacity == 0)
	    goto cleanup;

	  list = list->next;
	}

      break;

    default:
      g_assert_not_reached ();
      break;
    }

  return TRUE;
  
cleanup:
  list = list->prev;

  while (list)
    {
      partition = list->data;

      close (partition->fd);

      list = list->prev;
    }

  return FALSE;
}

static gsize
process_filename (const gchar *filename,
                  gint        *out_fd)
{
  gint  fd;
  gsize capacity;

  g_return_val_if_fail (filename != NULL, 0);

  fd = open (filename, O_RDWR);

  if (fd == -1)
    return 0;

  capacity = lseek (fd, 0, SEEK_END) / RBC_BLOCK_SIZE;

  if (out_fd)
    *out_fd = fd;
  else
    close (fd);

  return capacity;
}

static gint
partition_order (const RBCPartition *a,
                 const RBCPartition *b)
{
  if (a->order < b->order)
    return -1;
  else if (a->order > b->order)
    return 1;

  return 0;
}
