/*
 * Endpoint - Linux SBP2 Disk Target
 *
 * Copyright (C) 2003 Oracle.  All rights reserved.
 *
 * Author: Manish Singh <manish.singh@oracle.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */

#include <sys/types.h>
#include <unistd.h>

#include <sys/wait.h>
#include <signal.h>

#include <glib.h>

#include "app.h"
#include "cleanup.h"
#include "process.h"
#include "util.h"
#include "wire.h"


#define REAP_DELAY       1000
#define REAP_RETRY_COUNT 5


static void     close_descriptors (gint          except_reader,
				   gint          except_writer);

static gboolean reap_process      (GIOChannel   *channel,
                                   GIOCondition  condition,
				   Process      *process);
static gboolean waitpid_timeout   (Process      *process);


static inline void
close_fd (gint fd)
{
  if (fd != -1)
    close (fd);
}

Process *
process_start (EndpointApp       *app,
               ProcessReapNotify  notify)
{
  gint        ipipe[2] = { -1, -1 };
  gint        opipe[2] = { -1, -1 };
  gint        my_reader, sub_reader;
  gint        my_writer, sub_writer;
  GIOChannel *reader;
  GIOChannel *writer;
  gint        pid;
  Process    *process;

  g_return_val_if_fail (app != NULL, NULL);

  if (pipe (ipipe) < 0 || pipe (opipe) < 0)
    goto fail;

  my_reader  = ipipe[0];
  my_writer  = opipe[1];
  sub_reader = opipe[0];
  sub_writer = ipipe[1];

  pid = fork ();

  if (pid < 0)
    goto fail;

  process = g_new (Process, 1);

  if (pid == 0)
    {
      //close_descriptors (sub_reader, sub_writer);

      cleanup_unregister (app);

      reader = util_io_channel_new (sub_reader);
      writer = util_io_channel_new (sub_writer);

      g_main_loop_quit (app->loop);

      app->context = g_main_context_new ();

      app->wire_funcs = NULL;
      app->user_data = NULL;
    }
  else
    {
      close (sub_reader);
      close (sub_writer);

      reader = util_io_channel_new (my_reader);
      writer = util_io_channel_new (my_writer);

      util_io_add_watch (app->context, writer,
			 G_IO_ERR | G_IO_HUP,
			 (GIOFunc) reap_process, process);
    }
  
  util_io_add_watch (app->context, reader,
                     G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
		     (GIOFunc) wire_message_process, app);

  process->app = app;

  process->pid = pid;

  process->reader = reader;
  process->writer = writer;

  process->notify = notify;
  process->count = REAP_RETRY_COUNT;

  return process;

fail:
  close_fd (ipipe[0]);
  close_fd (ipipe[1]);
  close_fd (opipe[0]);
  close_fd (opipe[1]);

  return NULL;
}

static void
close_descriptors (gint except_reader,
                   gint except_writer)
{
  glong open_max;
  gint  i;

  g_return_if_fail (except_reader > 3);
  g_return_if_fail (except_writer > 3);

  open_max = sysconf(_SC_OPEN_MAX);

  for (i = 3; i < open_max; i++)
    if (i != except_reader && i != except_writer)
      close (i);
}

static gboolean
reap_process (GIOChannel   *channel,
	      GIOCondition  condition,
	      Process      *process)
{
  g_return_val_if_fail (channel != NULL, FALSE);
  g_return_val_if_fail (process != NULL, FALSE);
  g_return_val_if_fail (process->app->context != NULL, FALSE);

  util_timeout_add (process->app->context, REAP_DELAY,
                    (GSourceFunc) waitpid_timeout, process);

  return FALSE;
}

static gboolean
waitpid_timeout (Process *process)
{
  gint pid;
  gint status;

  g_return_val_if_fail (process != NULL, TRUE);
  g_return_val_if_fail (process->pid > 0, TRUE);

  process->count--;

  pid = waitpid (process->pid, &status, WNOHANG);

  if (pid == -1)
    {
      /* ERROR MESSAGE */
    }
  else if (pid == 0)
    {
      if (process->count == 0)
	kill (process->pid, SIGTERM);
      else if (process->count < 0)
	kill (process->pid, SIGKILL);

      return TRUE;
    }

  process->notify (process, status);
  g_free (process);

  return FALSE;
}
