﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/budgets/BudgetsRequest.h>
#include <aws/budgets/Budgets_EXPORTS.h>
#include <aws/budgets/model/Notification.h>
#include <aws/budgets/model/Subscriber.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Budgets {
namespace Model {

/**
 * <p> Request of DeleteSubscriber </p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/budgets-2016-10-20/DeleteSubscriberRequest">AWS
 * API Reference</a></p>
 */
class DeleteSubscriberRequest : public BudgetsRequest {
 public:
  AWS_BUDGETS_API DeleteSubscriberRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteSubscriber"; }

  AWS_BUDGETS_API Aws::String SerializePayload() const override;

  AWS_BUDGETS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The <code>accountId</code> that is associated with the budget whose
   * subscriber you want to delete.</p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  DeleteSubscriberRequest& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the budget whose subscriber you want to delete.</p>
   */
  inline const Aws::String& GetBudgetName() const { return m_budgetName; }
  inline bool BudgetNameHasBeenSet() const { return m_budgetNameHasBeenSet; }
  template <typename BudgetNameT = Aws::String>
  void SetBudgetName(BudgetNameT&& value) {
    m_budgetNameHasBeenSet = true;
    m_budgetName = std::forward<BudgetNameT>(value);
  }
  template <typename BudgetNameT = Aws::String>
  DeleteSubscriberRequest& WithBudgetName(BudgetNameT&& value) {
    SetBudgetName(std::forward<BudgetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The notification whose subscriber you want to delete.</p>
   */
  inline const Notification& GetNotification() const { return m_notification; }
  inline bool NotificationHasBeenSet() const { return m_notificationHasBeenSet; }
  template <typename NotificationT = Notification>
  void SetNotification(NotificationT&& value) {
    m_notificationHasBeenSet = true;
    m_notification = std::forward<NotificationT>(value);
  }
  template <typename NotificationT = Notification>
  DeleteSubscriberRequest& WithNotification(NotificationT&& value) {
    SetNotification(std::forward<NotificationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The subscriber that you want to delete.</p>
   */
  inline const Subscriber& GetSubscriber() const { return m_subscriber; }
  inline bool SubscriberHasBeenSet() const { return m_subscriberHasBeenSet; }
  template <typename SubscriberT = Subscriber>
  void SetSubscriber(SubscriberT&& value) {
    m_subscriberHasBeenSet = true;
    m_subscriber = std::forward<SubscriberT>(value);
  }
  template <typename SubscriberT = Subscriber>
  DeleteSubscriberRequest& WithSubscriber(SubscriberT&& value) {
    SetSubscriber(std::forward<SubscriberT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_accountId;
  bool m_accountIdHasBeenSet = false;

  Aws::String m_budgetName;
  bool m_budgetNameHasBeenSet = false;

  Notification m_notification;
  bool m_notificationHasBeenSet = false;

  Subscriber m_subscriber;
  bool m_subscriberHasBeenSet = false;
};

}  // namespace Model
}  // namespace Budgets
}  // namespace Aws
