﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cloudsearch/CloudSearchRequest.h>
#include <aws/cloudsearch/CloudSearch_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CloudSearch {
namespace Model {

/**
 * <p>Container for the parameters to the
 * <code><a>DescribeDomainEndpointOptions</a></code> operation. Specify the name of
 * the domain you want to describe. To show the active configuration and exclude
 * any pending changes, set the Deployed option to <code>true</code>.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cloudsearch-2013-01-01/DescribeDomainEndpointOptionsRequest">AWS
 * API Reference</a></p>
 */
class DescribeDomainEndpointOptionsRequest : public CloudSearchRequest {
 public:
  AWS_CLOUDSEARCH_API DescribeDomainEndpointOptionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeDomainEndpointOptions"; }

  AWS_CLOUDSEARCH_API Aws::String SerializePayload() const override;

 protected:
  AWS_CLOUDSEARCH_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>A string that represents the name of a domain.</p>
   */
  inline const Aws::String& GetDomainName() const { return m_domainName; }
  inline bool DomainNameHasBeenSet() const { return m_domainNameHasBeenSet; }
  template <typename DomainNameT = Aws::String>
  void SetDomainName(DomainNameT&& value) {
    m_domainNameHasBeenSet = true;
    m_domainName = std::forward<DomainNameT>(value);
  }
  template <typename DomainNameT = Aws::String>
  DescribeDomainEndpointOptionsRequest& WithDomainName(DomainNameT&& value) {
    SetDomainName(std::forward<DomainNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether to retrieve the latest configuration (which might be in a Processing
   * state) or the current, active configuration. Defaults to <code>false</code>.</p>
   */
  inline bool GetDeployed() const { return m_deployed; }
  inline bool DeployedHasBeenSet() const { return m_deployedHasBeenSet; }
  inline void SetDeployed(bool value) {
    m_deployedHasBeenSet = true;
    m_deployed = value;
  }
  inline DescribeDomainEndpointOptionsRequest& WithDeployed(bool value) {
    SetDeployed(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_domainName;
  bool m_domainNameHasBeenSet = false;

  bool m_deployed{false};
  bool m_deployedHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudSearch
}  // namespace Aws
