package codec

import (
	"bytes"
	"encoding/hex"
	"reflect"
	"testing"
	"time"

	"github.com/hashicorp/go-msgpack/v2/codec/internal/testdata"
)

// Tests that msgpack correctly deserializes all legacy hashicorp/raft entries, byte-for-byte.
// These were generated by running all the Raft tests and capturing all msgpack encodings.

func decodeMsgPack(buf []byte, out interface{}) error {
	r := bytes.NewBuffer(buf)
	hd := MsgpackHandle{}
	dec := NewDecoder(r, &hd)
	return dec.Decode(out)
}

func serializeTests(t *testing.T, tests []testdata.SerializeTest) {
	t.Helper()
	for _, test := range tests {
		t.Run(test.Name, func(t *testing.T) {
			expected, err := hex.DecodeString(test.EncodedBytesHex)
			if err != nil {
				t.Fatal(err)
			}
			typ := reflect.TypeOf(test.ExpectedData)
			x := reflect.New(typ).Elem().Interface()

			err = decodeMsgPack(expected, &x)
			if err != nil {
				t.Fatal(err)
			}
			if !reflect.DeepEqual(x, test.ExpectedData) {
				// to help figure out what is different since the Logs are pointers
				if aer, ok := x.(testdata.AppendEntriesRequest); ok {
					for _, log := range aer.Entries {
						if log != nil {
							t.Errorf("log %##v", *log)
						}
					}
				}
				t.Errorf("expected %#v but got %#v", test.ExpectedData, x)
			}
		})
	}

}

func TestRaftBytes_v116(t *testing.T) {
	serializeTests(t, testdata.RaftV116)
}

func TestRaftBytes_v115(t *testing.T) {
	serializeTests(t, testdata.RaftV115)
}

func TestRaftBytes_v055(t *testing.T) {
	serializeTests(t, testdata.RaftV055)
}

func TestTimeBytes(t *testing.T) {
	testCases := []struct {
		name     string
		data     string
		expected string
	}{
		{
			"v1.1.6 non-UTC with nanos",
			"af010000000eda5a48a43b9ac9ff01a4",
			"2022-07-08T22:47:48.999999999+07:00",
		},
		{
			"v1.1.6",
			"af010000000eda5aab140e044990ffff",
			"2022-07-08T22:47:48.235162Z",
		},
		{
			"v1.1.5",
			"a83811264062c8b414",
			"2022-07-08T22:47:48.235162Z",
		},
	}

	for _, test := range testCases {
		var got time.Time
		data, err := hex.DecodeString(test.data)
		if err != nil {
			t.Fatal(err)
		}
		err = decodeMsgPack(data, &got)
		if err != nil {
			t.Fatal(err)
		}
		expected, err := time.Parse(time.RFC3339Nano, test.expected)
		if err != nil {
			t.Fatal(err)
		}
		if !got.Equal(expected) {
			t.Errorf("%s: Got %s expected %s", test.name, got.Format(time.RFC3339Nano), expected.Format(time.RFC3339Nano))
		}
	}
}
