/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
** documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
** rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
** permit persons to whom the Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in all copies or substantial portions
** of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
** WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
** COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
** TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
** SOFTWARE.
**/

package comms

import (
	"slices"
	"testing"

	"github.com/google/go-cmp/cmp"
)

func Test_NewTLSConfigSetting(t *testing.T) {
	t.Parallel()

	type args struct {
		input string
	}

	tests := []struct {
		name    string
		args    args
		want    ConfigSetting
		wantErr bool
	}{
		{
			name: "+tlsConnect",
			args: args{input: "TLSConnect"},
			want: TLSConnect,
		},
		{
			name: "+tlsCertFile",
			args: args{input: "TLSCertFile"},
			want: TLSCertFile,
		},
		{
			name:    "-invalidSetting",
			args:    args{input: "InvalidSetting"},
			wantErr: true,
		},
		{
			name:    "-emptyString",
			args:    args{input: ""},
			wantErr: true,
		},
		{
			name:    "-caseSensitive",
			args:    args{input: "tlsconnect"},
			wantErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got, err := NewTLSConfigSetting(tt.args.input)

			if (err != nil) != tt.wantErr {
				t.Fatalf("NewTLSConfigSetting() error = %v, wantErr %v", err, tt.wantErr)
			}

			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Errorf("NewTLSConfigSetting() mismatch (-want +got):\n%s", diff)
			}
		})
	}
}

func Test_AllTLSConfigSettings(t *testing.T) {
	t.Parallel()

	// Get the expected settings from the source set.
	expected := make([]ConfigSetting, 0, len(tlsConfigSettingsSet))
	for setting := range tlsConfigSettingsSet {
		expected = append(expected, setting)
	}
	// Sort for consistent comparison.
	slices.Sort(expected)

	// Get the actual settings from the function.
	got := AllTLSConfigSettings()
	// Sort for consistent comparison.
	slices.Sort(got)

	if diff := cmp.Diff(expected, got); diff != "" {
		t.Fatalf("AllTLSConfigSettings() mismatch (-want +got):\n%s", diff)
	}
}
