/***************************************************************************
 *	Copyright (C) 2005 by karye												*
 *	karye@users.sourceforge.net												*
 *																			*
 *	This program is free software; you can redistribute it and/or modify	*
 *	it under the terms of the GNU General Public License as published by	*
 *	the Free Software Foundation; either version 2 of the License, or		*
 *	(at your option) any later version.										*
 *																			*
 *	This program is distributed in the hope that it will be useful,			*
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of			*
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *	GNU General Public License for more details.							*
 *																			*
 *	You should have received a copy of the GNU General Public License		*
 *	along with this program; if not, write to the							*
 *	Free Software Foundation, Inc.,											*
 *	59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.				*
 ***************************************************************************/

#include <cstddef>            // for NULL
#include <kio/filecopyjob.h>   // for file_copy, FileCopyJob
#include <klocalizedstring.h>  // for i18n
#include <kmessagebox.h>       // for error
#include <qbytearray.h>        // for QByteArray
#include <qcheckbox.h>         // for QCheckBox
#include <qdatetime.h>         // for QDateTime
#include <qdebug.h>            // for QDebug
#include <qfile.h>             // for QFile
#include <qglobal.h>           // for QGlobalStatic, QGlobalStatic<>::Type
#include <qiodevice.h>         // for QIODevice, QIODevice::WriteOnly
#include <qobject.h>           // for QObject
#include <qobjectdefs.h>       // for emit
#include <qstring.h>           // for QString, operator+
#include <qtextbrowser.h>      // for QTextBrowser
#include <qtextstream.h>       // for QTextStream
#include <qurl.h>              // for QUrl

#include "common.h"            // for EMERGE, EMERGELOG, ERROR, KUROO, TOLOG
#include "global.h"            // for kurooDir
#include "log.h"               // for Log
#include "logs/log.h"

namespace KIO {
class Job;
}  // namespace KIO

uint Log::buffer_MaxLines = 2000;

/**
* @class Log
* @short Log output from all actions as emerge, scanning... to log window and to file.
*
* @todo: save all output to log file
*/
Log::Log( QObject* m_parent )
	: QObject( m_parent ), m_verboseLog( nullptr ), m_saveLog( nullptr ), m_logBrowser( nullptr ), numLines( 0 )
{}

Log::~Log()
{
	m_logFile.close();
}

/**
* Open persistent log.
* @return log file name
*/
auto Log::init( QWidget *parent ) -> const QString
{
	m_parent = parent;

	QString logName = kurooDir + QStringLiteral("kuroo.log");
	qDebug() << logName << "\n";
	m_logFile.setFileName( logName );
	if( !m_logFile.open( QIODevice::WriteOnly ) ) {
		qCritical() << "Writing: " << kurooDir << "kuroo.log";
		KMessageBox::error( parent, i18n( "Writing %1kuroo.log.", kurooDir ), i18n( "Saving" ) );
		return QString();
	}
	return logName;

}

/**
* Backup job of "kuroo.log".
* @return copy job
*/
auto Log::backupLog() -> KIO::Job*
{
	if ( m_saveLog && m_saveLog->isChecked() ) {
		QDateTime dt = QDateTime::currentDateTime();
		KIO::Job *cpjob = KIO::file_copy( QUrl::fromLocalFile( kurooDir + QStringLiteral("kuroo.log") ),
										QUrl::fromLocalFile( QStringLiteral("%1kuroo_%2.log").arg( kurooDir, dt.toString(QStringLiteral("yyyyMMdd_hhmm")) ) ));
		return cpjob;
	}
	return nullptr;
}

/**
* Connect log output browser in gui.
* @param logBrowserGui			pointer to text browser.
* @param checkBoxVerboseGui	pointer to verbose checkbox.
*/
void Log::setGui( QTextBrowser* logBrowserGui, QCheckBox* verboseLogGui, QCheckBox* saveLogGui )
{
	m_logBrowser = logBrowserGui;
	m_verboseLog = verboseLogGui;
	m_saveLog = saveLogGui;
}

/**
* Write log lines to text browser and log file.
* @param output		line of text.
* @param logType			type of log = EMERGE, KUROO, WARNING, TOLOG, EMERGELOG.
*/
void Log::writeLog( const QString& output, int logType )
{
	QString line(output);
	line.toUtf8().replace( '\'', "''" );

	switch ( logType ) {

		case EMERGE: {
			if ( m_verboseLog && m_verboseLog->isChecked() ) {
				//line = line.replace('>', "&gt;").replace('<', "&lt;");
				addText( line );
				Q_EMIT signalLogChanged();
			}
			break;
		}

		case KUROO: {
			addText( line );
			break;
		}

		case ERROR: {
			line = line.replace( u'>', QStringLiteral("&gt;") ).replace( u'<', QStringLiteral("&lt;") );
			addText( line );
			break;
		}

		case TOLOG: {
			QTextStream st( &m_logFile );
			st << line << "\n";
			m_logFile.flush();
			break;
		}

		case EMERGELOG: {
			line = line.replace( u'>', QStringLiteral("&gt;") ).replace( u'<', QStringLiteral("&lt;") );
			addText( line );
		}
	}
}

void Log::addText( const QString& str ) {
	//qDebug() << "Max Buffer Lines: " << buffer_MaxLines << "\n";
	if (numLines > buffer_MaxLines) {
		m_logBrowser->clear();
		numLines=0;
	}
	if (m_logBrowser) {
		m_logBrowser->append( str );
		numLines++;
	}
}


