" tar.vim: Handles browsing tarfiles
"            AUTOLOAD PORTION
" Date:			Sep 29, 2006
" Version:		11
" Maintainer:	Charles E Campbell, Jr <NdrOchip@ScampbellPfamily.AbizM-NOSPAM>
" License:		Vim License  (see vim's :help license)
"
"	Contains many ideas from Michael Toren's <tar.vim>
"
" Copyright:  * Copyright 2007 Jan Minar <rdancer@gmail.com> {{{1
"
"	      *	Based on tar.vim version 11, which carries the following
"		copyright notice:
"
"		Copyright (C) 2005 Charles E. Campbell, Jr.
"               Permission is hereby granted to use and distribute this code,
"               with or without modifications, provided that this copyright
"               notice is copied with it. Like anything else that's free,
"               tarPlugin.vim is provided *as is* and comes with no warranty
"               of any kind, either expressed or implied. By using this
"               plugin, you agree that in no event will the copyright
"               holder be liable for any damages resulting from the use
"               of this software.
"
"	      *	Based also on vim71/autoload/gzip.vim.

" ---------------------------------------------------------------------
" Load Once: {{{1
let s:keepcpo= &cpo
set cpo&vim
if &cp || exists("g:loaded_tar") || v:version < 700
 finish
endif
let g:loaded_tar= "v11"
"call Decho("loading autoload/tar.vim")

" ---------------------------------------------------------------------
"  Default Settings: {{{1
if !exists("g:tar_browseoptions")
 let g:tar_browseoptions= "Ptf"
endif
if !exists("g:tar_readoptions")
 let g:tar_readoptions= "OPxf"
endif
if !exists("g:tar_cmd")
 let g:tar_cmd= "tar"
endif
if !exists("g:tar_writeoptions")
 let g:tar_writeoptions= "uf"
endif
if !exists("g:tar_shq")
 if has("unix")
  let g:tar_shq= "'"
 else
  let g:tar_shq= '"'
 endif
endif

" ----------------
"  Functions: {{{1
" ----------------

" ---------------------------------------------------------------------
" tar#Browse: {{{2
fun! tar#Browse(tarfile)
"  call Dfunc("tar#Browse(tarfile<".a:tarfile.">)")
  let repkeep= &report
  set report=10

  " sanity checks
  if !executable(g:tar_cmd)
   redraw!
   echohl Error | echo '***error*** (tar#Browse) "'.g:tar_cmd.'" not available on your system'
"   call inputsave()|call input("Press <cr> to continue")|call inputrestore()
   let &report= repkeep
"   call Dret("tar#Browse")
   return
  endif
  if !filereadable(a:tarfile)
"   call Decho('a:tarfile<'.a:tarfile.'> not filereadable')
   if a:tarfile !~# '^\a\+://'
    " if its an url, don't complain, let url-handlers such as vim do its thing
    redraw!
    echohl Error | echo "***error*** (tar#Browse) File not readable<".a:tarfile.">" | echohl None
"    call inputsave()|call input("Press <cr> to continue")|call inputrestore()
   endif
   let &report= repkeep
"   call Dret("tar#Browse : file<".a:tarfile."> not readable")
   return
  endif
  if &ma != 1
   set ma
  endif
  let w:tarfile= a:tarfile

  setlocal noswapfile
  setlocal buftype=nofile
  setlocal bufhidden=hide
  setlocal nobuflisted
  setlocal nowrap
  set ft=tar

  " give header
"  call Decho("printing header")
  exe "$put ='".'\"'." tar.vim version ".g:loaded_tar."'"
  exe "$put ='".'\"'." Browsing tarfile '"
  let w:fsav = @f
   let @f = w:tarfile
   exe "normal A\<c-r>\<c-r>f"
  let @f = w:fsav
  unlet w:fsav
  exe "$put ='".'\"'." Select a file with cursor and press ENTER"."'"
  0d
  $

  " Escape a:tarfile, poorly {{{3
  let tarfile= a:tarfile
  if has("win32") && executable("cygpath")
   " assuming cygwin
   " XXX Whatever this does  -- rdancer 2007-07-13
   let tarfile=substitute(system("cygpath -u ".tarfile),'\n$','','e')
  endif
  let curlast= line("$")
  " XXX Don't let %-expansion happen in the first place
  " XXX The list of expandable chars is incomplete
  let tarfile_s = fnameescape(tarfile)
  " Unroll the tarball, according to compression type {{{3
  if tarfile =~# '\.\(gz\|tgz\)$'
"   call Decho("silent r! gzip -d -c -- ".tarfile_s."| ".g:tar_cmd." -".g:tar_browseoptions." - ")
   exe "silent r! gzip -d -c -- ".tarfile_s."| ".g:tar_cmd." -".g:tar_browseoptions." - "
  elseif tarfile =~# '\.bz2$'
"   call Decho("silent r! bzip2 -d -c -- ".tarfile_s."| ".g:tar_cmd." -".g:tar_browseoptions." - ")
   exe "silent r! bzip2 -d -c -- ".tarfile_s."| ".g:tar_cmd." -".g:tar_browseoptions." - "
  else
"   call Decho("silent r! ".g:tar_cmd." -".g:tar_browseoptions." ".tarfile_s)
   exe "silent r! ".g:tar_cmd." -".g:tar_browseoptions." ".tarfile_s
  endif
  " Errors and housekeeping {{{3
  if v:shell_error != 0
   redraw!
   echohl WarningMsg | echo "***warning*** (tar#Browse) please check your g:tar_browseoptions<".g:tar_browseoptions.">"
"   call inputsave()|call input("Press <cr> to continue")|call inputrestore()
"   call Dret("tar#Browse : a:tarfile<".a:tarfile.">")
   return
  endif
  if line("$") == curlast || ( line("$") == (curlast + 1) && getline("$") =~ '\c\%(warning\|error\|inappropriate\|unrecognized\)')
   redraw!
   echohl WarningMsg | echo "***warning*** (tar#Browse) ".a:tarfile." doesn't appear to be a tar file" | echohl None
"   call inputsave()|call input("Press <cr> to continue")|call inputrestore()
   silent %d
   let eikeep= &ei
   set ei=BufReadCmd,FileReadCmd
   exe "r ".fnameescape(a:tarfile)
   let &ei= eikeep
   1d
"   call Dret("tar#Browse : a:tarfile<".a:tarfile.">")
   return
  endif

  setlocal noma nomod ro
  noremap <silent> <buffer> <cr> :call <SID>TarBrowseSelect()<cr>

  let &report= repkeep
"  call Dret("tar#Browse : w:tarfile<".w:tarfile.">")
endfun

" ---------------------------------------------------------------------
" TarBrowseSelect: {{{2
fun! s:TarBrowseSelect()
"  call Dfunc("TarBrowseSelect() w:tarfile<".w:tarfile."> curfile<".expand("%").">")
  let repkeep= &report
  set report=10
  let fname= getline(".")
"  call Decho("fname<".fname.">")

  " sanity check
  if fname =~ '^"'
   let &report= repkeep
"   call Dret("TarBrowseSelect")
   return
  endif

  " about to make a new window, need to use w:tarfile
  let tarfile= w:tarfile
  let curfile= expand("%")
  if has("win32") && executable("cygpath")
   " assuming cygwin
   let tarfile=substitute(system("cygpath -u ".tarfile),'\n$','','e')
  endif

  new
  wincmd _
  let s:tblfile_{winnr()}= curfile
  call tar#Read("tarfile:".tarfile.':'.fname,1)
  filetype detect

  let &report= repkeep
"  call Dret("TarBrowseSelect : s:tblfile_".winnr()."<".s:tblfile_{winnr()}.">")
endfun

" ---------------------------------------------------------------------
" tar#Read: {{{2
fun! tar#Read(fname,mode)
"  call Dfunc("tar#Read(fname<".a:fname.">,mode=".a:mode.")")
  let repkeep= &report
  set report=10
  " Sanitize tarfile, fname, poorly
  " XXX
  let tarfile = substitute(a:fname,'tarfile:\(.\{-}\):.*$','\1','')
  " XXX
  let fname   = substitute(a:fname,'tarfile:.\{-}:\(.*\)$','\1','')
  if has("win32") && executable("cygpath")
   " assuming cygwin
   " XXX
   let tarfile=substitute(system("cygpath -u ".tarfile),'\n$','','e')
  endif
"  call Decho("tarfile<".tarfile.">")
"  call Decho("fname<".fname.">")

  let tarfile_s = s:escape(tarfile)
  let fname_s = s:escape(fname)
  if tarfile =~# '\.\(gz\|tgz\)$'
"   call Decho("silent r! gzip -d -c -- ".tarfile_s."| ".g:tar_cmd." -".g:tar_readoptions." - -- ".fname_s)
   exe "silent r! gzip -d -c -- ".tarfile_s."| ".g:tar_cmd." -".g:tar_readoptions." - -- ".fname_s
  elseif tarfile =~# '\.bz2$'
"   call Decho("silent r! bzip2 -d -c -- ".tarfile_s."| ".g:tar_cmd." -".g:tar_readoptions." - -- ".fname_s)
   exe "silent r! bzip2 -d -c -- ".tarfile_s."| ".g:tar_cmd." -".g:tar_readoptions." - -- ".fname_s
  else
"   call Decho("silent r! ".g:tar_cmd." -".g:tar_readoptions." ".tarfile_s." -- ".fname_s)
   exe "silent r! ".g:tar_cmd." -".g:tar_readoptions." ".tarfile_s." -- ".fname_s
  endif
  let w:tarfile= a:fname
  " XXX Command substitution (:file `date`)
  exe "file tarfile:".fname

  " cleanup
  0d
  set nomod

  let &report= repkeep
"  call Dret("tar#Read : w:tarfile<".w:tarfile.">")
endfun

" ---------------------------------------------------------------------
" tar#Write: {{{2
fun! tar#Write(fname)
"  call Dfunc("tar#Write(fname<".a:fname.">) w:tarfile<".w:tarfile."> tblfile_".winnr()."<".s:tblfile_{winnr()}.">")
  let repkeep= &report
  set report=10

  " sanity checks
  if !executable(g:tar_cmd)
   redraw!
   echohl Error | echo '***error*** (tar#Browse) "'.g:tar_cmd.'" not available on your system'
"   call inputsave()|call input("Press <cr> to continue")|call inputrestore()
   let &report= repkeep
"   call Dret("tar#Write")
   return
  endif
  if !exists("*mkdir")
   redraw!
   echohl Error | echo "***error*** (tar#Write) sorry, mkdir() doesn't work on your system" | echohl None
"   call inputsave()|call input("Press <cr> to continue")|call inputrestore()
   let &report= repkeep
"   call Dret("tar#Write")
   return
  endif

  let curdir= getcwd()
  let tmpdir= tempname()
"  call Decho("orig tempname<".tmpdir.">")
  if tmpdir =~ '\.'
   " XXX
   let tmpdir= substitute(tmpdir,'\.[^.]*$','','e')
  endif
"  call Decho("tmpdir<".tmpdir.">")
  call mkdir(tmpdir,"p")

  " attempt to change to the indicated directory
  try
   " XXX
   exe "cd ".escape(tmpdir,' \')
  catch /^Vim\%((\a\+)\)\=:E344/
   redraw!
   echohl Error | echo "***error*** (tar#Write) cannot cd to temporary directory" | Echohl None
"   call inputsave()|call input("Press <cr> to continue")|call inputrestore()
   let &report= repkeep
"   call Dret("tar#Write")
   return
  endtry
"  call Decho("current directory now: ".getcwd())

  " place temporary files under .../_ZIPVIM_/
  if isdirectory("_ZIPVIM_")
   call s:Rmdir("_ZIPVIM_")
  endif
  call mkdir("_ZIPVIM_")
  cd _ZIPVIM_
"  call Decho("current directory now: ".getcwd())

  " XXX
  let tarfile = substitute(w:tarfile,'tarfile:\(.\{-}\):.*$','\1','')
  " XXX
  let fname   = substitute(w:tarfile,'tarfile:.\{-}:\(.*\)$','\1','')

  " handle compressed archives
  if tarfile =~# '\.gz'
   call system("gzip -d -- ".shellescape(tarfile))
   let tarfile = substitute(tarfile,'\.gz','','e')
   let compress= "gzip -- ".shellescape(tarfile)
  elseif tarfile =~# '\.tgz'
   call system("gzip -d -- "".shellescape(tarfile))
   let tarfile = substitute(tarfile,'\.tgz','.tar','e')
   let compress= "gzip -- ".shellescape(tarfile)
   let tgz     = 1
  elseif tarfile =~# '\.bz2'
   call system("bzip2 -d -- ".shellescape(tarfile))
   let tarfile = substitute(tarfile,'\.bz2','','e')
   let compress= "bzip2 -- ".shellescape(tarfile)
  endif

  if v:shell_error != 0
   redraw!
   echohl Error | echo "***error*** (tar#Write) sorry, unable to update ".tarfile." with ".fname | echohl None
"   call inputsave()|call input("Press <cr> to continue")|call inputrestore()
  else

"   call Decho("tarfile<".tarfile."> fname<".fname.">")
 
   if fname =~ '/'
    let dirpath = substitute(fname,'/[^/]\+$','','e')
    if executable("cygpath")
     let dirpath = substitute(system("cygpath ".dirpath),'\n','','e')
    endif
    call mkdir(dirpath,"p")
   endif
   if tarfile !~ '/'
    let tarfile= curdir.'/'.tarfile.
   endif
"   call Decho("tarfile<".tarfile."> fname<".fname.">")
 
   "XXX This is not adequate
"   call Decho ("w! ".escape(fname, '%#<>|\'))
   exe "w! ".escape(fname, '%#<>|\')
   if executable("cygpath")
    let tarfile = substitute(system("cygpath ".tarfile),'\n','','e')
   endif
 
   " delete old file from tarfile
"   call Decho("tar --delete -f ".shellescape(tarfile)." -- ".shellescape(fname))
   call system("tar --delete -f ".shellescape(tarfile)." -- ".shellescape(fname))
   if v:shell_error != 0
    redraw!
    echohl Error | echo "***error*** (tar#Write) sorry, unable to update ".tarfile." with ".fname | echohl None
"    call inputsave()|call input("Press <cr> to continue")|call inputrestore()
   else
 
    " update tarfile with new file 
"    call Decho("tar -".g:tar_writeoptions." ".shellescape(tarfile)." -- ".shellescape(fname))
    call system("tar -".g:tar_writeoptions." ".shellescape(tarfile)." -- ".shellescape(fname))
    if v:shell_error != 0
     redraw!
     echohl Error | echo "***error*** (tar#Write) sorry, unable to update ".tarfile." with ".fname | echohl None
"     call inputsave()|call input("Press <cr> to continue")|call inputrestore()
    elseif exists("compress")
"     call Decho("call system(".compress.")")
     call system(compress)
     if exists("tgz")
"      call Decho("rename(".tarfile.".gz,".substitute(tarfile,'\.tar$','.tgz','e').")")
      call rename(tarfile.".gz",substitute(tarfile,'\.tar$','.tgz','e'))
     endif
    endif
   endif

   " support writing tarfiles across a network
   if s:tblfile_{winnr()} =~ '^\a\+://'
"    call Decho("handle writing <".tarfile."> across network to <".s:tblfile_{winnr()}.">")
    let tblfile= s:tblfile_{winnr()}
    1split|enew
    let binkeep= &binary
    let eikeep = &ei
    set binary ei=all
    exe "e! ".tarfile
    call netrw#NetWrite(tblfile)
    let &ei     = eikeep
    let &binary = binkeep
    q!
    unlet s:tblfile_{winnr()}
   endif
  endif
  
  " cleanup and restore current directory
  cd ..
  call s:Rmdir("_ZIPVIM_")
  exe "cd ".escape(curdir,' \')
  setlocal nomod

  let &report= repkeep
"  call Dret("tar#Write")
endfun

" ---------------------------------------------------------------------
" Rmdir: {{{2
fun! s:Rmdir(fname)
"  call Dfunc("Rmdir(fname<".a:fname.">)")
  if has("unix")
   call system("/bin/rm -rf -- ".shellescape(a:fname))
  elseif has("win32") || has("win95") || has("win64") || has("win16")
   if &shell =~? "sh$"
   call system("/bin/rm -rf -- ".shellescape(a:fname))
   else
    call system("del /S ".shellescape(a:fname))
   endif
  endif
"  call Dret("Rmdir")
endfun

" ---------------------------------------------------------------------
" s:escape(name): {{{1
" Copied from vim71/autoload/gzip.vim
fun! s:escape(name)
  " shellescape() was added by patch 7.0.111
  if exists("*shellescape")
  " XXX Don't let %-expansion happen in the first place
  " XXX The list of expandable chars is incomplete
    return escape(shellescape(a:name), '%#<>')
  endif
  echoerr("Can't reliaby escape shell commands.  Quitting.")
  quit
endfun

" ------------------------------------------------------------------------
" Modelines And Restoration: {{{1
let &cpo= s:keepcpo
unlet s:keepcpo
"  vim:ts=8 fdm=marker
