/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file misc.c
 * \brief Misc functions
 */

#include <ffgtk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include "callbycall.h"

/* set it here as it is not defined at gdk-pixbuf.h */
gboolean gdk_pixbuf_set_option( GdkPixbuf *pixbuf, const gchar *key, const gchar *value);

/* local definitions */

/* current numberFormat */
enum {
	unknown,
	localNumber,
	nationalNumber,
	internationalNumber
};

enum {
	requestLocalFormat = 0,
	requestNationalFormat = 1,
	requestInternationalFormat = 2,
	requestInternationalPlusFormat = 3
};

/** translations from event to text for file monitor events */
struct sEventTranslate {
	GFileMonitorEvent nEvent;
	char *pnText;
};

/** text values of file monitor events */
struct sEventTranslate sEventTranslationTable[] = {
	{ G_FILE_MONITOR_EVENT_CHANGED , "file changed" },
	{ G_FILE_MONITOR_EVENT_CHANGES_DONE_HINT, "file changes finished" },
	{ G_FILE_MONITOR_EVENT_DELETED,"file deleted" },
	{ G_FILE_MONITOR_EVENT_CREATED, "file created" },
	{ G_FILE_MONITOR_EVENT_ATTRIBUTE_CHANGED, "file attrbutes changed" },
	{ G_FILE_MONITOR_EVENT_PRE_UNMOUNT, "file system about to be unmounted" },
	{ G_FILE_MONITOR_EVENT_UNMOUNTED, "file system unmounted" },
	{ -1, "" }
};

/** users ffgtk directory */
static const gchar *pnUserDir = NULL;
/** users home directory */
static const gchar *pnHomeDir = NULL;

/** call in icon */
static GdkPixbuf *psCallIn = NULL;
/** call out icon */
static GdkPixbuf *psCallOut = NULL;
/** call in failed icon */
static GdkPixbuf *psCallInFailed = NULL;
/** voicebox icon */
static GdkPixbuf *psVoiceBox = NULL;
/** faxbox icon */
static GdkPixbuf *psFaxBox = NULL;

/**
 * \brief Get users home directory
 * \return home dir
 */
const gchar *getHomeDir( void ) {
	if ( pnHomeDir == NULL ) {
		pnHomeDir = g_get_home_dir();
	}

	return pnHomeDir;
}

/**
 * \brief Get users ffgtk directory
 * \return ffgtk directory
 */
const gchar *getUserDir( void ) {
	if ( pnUserDir == NULL ) {
		pnUserDir = g_build_filename( getHomeDir(), FFGTK_USER_DIR, NULL );
	}

	return pnUserDir;
}

/**
 * \brief Show error message
 * \param bThread enter threads flag
 * \param pnErrorMessage error message
 * \param pnText secondary text
 */
void ShowError( gboolean bThread, const gchar *pnErrorMessage, const gchar *pnText ) {
	GtkWidget *psErrorDialog;

	if ( bThread == TRUE ) {
		ffgtkLock();
	}

	psErrorDialog = gtk_message_dialog_new_with_markup( NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, "<span weight=\"bold\" size=\"larger\">%s</span>", pnErrorMessage );
	gtk_message_dialog_format_secondary_text( GTK_MESSAGE_DIALOG( psErrorDialog ), "%s", pnText );
	g_signal_connect( G_OBJECT( psErrorDialog ), "response", G_CALLBACK( gtk_widget_destroy ), psErrorDialog );
	gtk_widget_show_all( psErrorDialog );

	if ( bThread == TRUE ) {
		ffgtkUnlock();
	}
}

gchar *getDirectory( char *pnType ) {
#ifdef G_OS_WIN32
	GFile *psDirectory;
	GFile *psChild;
	char *pnTmp = NULL;

	pnTmp = g_win32_get_package_installation_directory_of_module( NULL );

	psDirectory = g_file_new_for_path( pnTmp );
	g_free( pnTmp );

	psChild = g_file_get_child( psDirectory, pnType );
	g_object_unref( psDirectory );

	psDirectory = psChild;

	pnTmp = g_file_get_path( psDirectory );

	return pnTmp;
#else
	return g_strdup( pnType );
#endif
}

gchar *getUiFile( const gchar *pnFile ) {
	return g_build_filename( getDirectory( PKGDATADIR ), pnFile, NULL );
}

/**
 * \brief Debug function, save data stream to file
 * \param pnName file name
 * \param pnData data pointer
 * \param nLen length of data or -1 for complete string
 */
void saveData( char *pnName, char *pnData, gint nLen ) {
	int nHandle, nResult;

	Debug( KERN_DEBUG, "Saving to file %s\n", pnName );

	if ( nLen == -1 ) {
		nLen = strlen( pnData );
	}

	nHandle = open( pnName, O_RDWR | O_CREAT | O_TRUNC, 0660 );
	if ( nHandle != -1 ) {
		nResult = write( nHandle, pnData, nLen );
		if ( nResult != strlen( pnData ) ) {
			Debug( KERN_WARNING, "Could not save file %s!\n", pnName );
		}
		close( nHandle );
	} else {
		Debug( KERN_DEBUG, "Could not open file %s\n", pnName );
	}
}

/**
 * \brief Save debug data
 * \param pnName file name
 * \param pnData data pointer
 * \param nLen length of data
 */
void saveDebugData( char *pnName, char *pnData, gint nLen ) {
	int nHandle, nResult;
	gchar *pnFile = NULL;

	if ( debugGetLevel( getActiveProfile() ) != KERN_DEBUG ) {
		return;
	}

	if ( pnData == NULL ) {
		return;
	}

	if ( nLen == -1 ) {
		nLen = strlen( pnData );
	}

	pnFile = g_strdup_printf( "%s/%s", g_get_tmp_dir(), pnName );

	Debug( KERN_DEBUG, "Saving to file %s\n", pnFile );

	nHandle = open( pnFile, O_RDWR | O_CREAT | O_TRUNC, 0660 );
	if ( nHandle != -1 ) {
		nResult = write( nHandle, pnData, nLen );
		if ( nResult != nLen ) {
			Debug( KERN_WARNING, "Could not save file %s!\n", pnFile );
		}
		close( nHandle );
	} else {
		Debug( KERN_DEBUG, "Could not open file %s\n", pnFile );
	}

	g_free( pnFile );
}

/**
 * \brief Debug function, add data stream to file
 * \param pnName file name
 * \param pnData data pointer
 */
void addData( char *pnName, char *pnData ) {
	int nHandle, nResult;

	nHandle = open( pnName, O_WRONLY | O_APPEND );
	if ( nHandle != -1 ) {
		nResult = write( nHandle, pnData, strlen( pnData ) );
		if ( nResult != strlen( pnData ) ) {
			Debug( KERN_WARNING, "Could not add data to file %s!\n", pnName );
		}
		close( nHandle );
	} else {
		Debug( KERN_DEBUG, "Could not open file %s\n", pnName );
	}
}

/**
 * \brief Debug function, load data from file
 * \param pnName file name
 * \param pnSize optional size storage
 * \return data pointer
 */
gchar *loadData( char *pnName, gint *pnSize ) {
	int nFile = -1;
	int nSize;
	int nRet;
	gchar *pnFileData = NULL;

	nFile = open( pnName, O_RDONLY );
	if ( nFile == -1 ) {
		return NULL;
	}

	nSize = lseek( nFile, 0, SEEK_END );

	lseek( nFile, 0, SEEK_SET );
	Debug( KERN_DEBUG, "file %s, %d bytes\n", pnName, nSize );

	pnFileData = g_malloc0( nSize + 1 );
	if ( pnFileData == NULL ) {
		close( nFile );
		return NULL;
	}

	nRet = read( nFile, pnFileData, nSize );
	if ( nRet == -1 ) {
		g_free( pnFileData );
		close( nFile );
		return NULL;
	}

	if ( pnSize != NULL ) {
		*pnSize = nRet;
	}

	close( nFile );

	return pnFileData;
}

/**
 * \brief Strip html entities from word
 * \param pnWord html string
 * \return stripped string
 */
static gchar *stripHtml( gchar *pnWord ) {
	GRegex *psTags = g_regex_new( "<(.|\n)*?>", G_REGEX_DOTALL | G_REGEX_OPTIMIZE, 0, NULL );
	GRegex *psSpace = g_regex_new( "&nbsp;", G_REGEX_DOTALL | G_REGEX_OPTIMIZE, 0, NULL );
	GRegex *psMisc = g_regex_new( "Â", G_REGEX_DOTALL | G_REGEX_OPTIMIZE, 0, NULL );
	gchar *pnStripped = g_regex_replace_literal( psTags, pnWord, -1, 0, g_strdup( "" ), 0, NULL );
	gchar *pnSpaced = g_regex_replace_literal( psSpace, pnStripped, -1, 0, g_strdup( " " ), 0, NULL );
	gchar *pnMisc = g_regex_replace_literal( psMisc, pnSpaced, -1, 0, g_strdup( "" ), 0, NULL );
	gchar *pnReturn = convertEntities( pnMisc );
	g_free( pnMisc );
	g_free( pnSpaced );
	g_free( pnStripped );
	return pnReturn;
}

/**
 * \brief Process result string
 * \param pnWord html string
 * \return stripped and trimmed string
 */
static gchar *processResult( gchar *pnWord ) {
	gchar *pnResult = stripHtml( pnWord );
	g_strstrip ( pnResult );
	return pnResult;
}

/**
 * \brief Extract information from lines via named regex
 * \param pnData data in one big string
 * \param ppnFirstName result first name
 * \param ppnLastName result last name
 * \param ppnStreet result street
 * \param ppnZipCode result zip code
 * \param ppnCity result city
 * \param pnRegex regular expression
 * \return TRUE when successful, otherwise FALSE
 */
gboolean extractInformationNamed( gchar *pnData, gchar **ppnFirstName, gchar **ppnLastName, gchar **ppnStreet, gchar **ppnZipCode, gchar **ppnCity, gchar *pnRegex ) {
	GRegex *psReg;
	GMatchInfo *psInfo;
	gboolean bRes;
	GError *psError = NULL;;
	gchar *pnFirstName;
	gchar *pnLastName;
	gchar *pnStreet;
	gchar *pnZipCode;
	gchar *pnCity;

	if ( pnData == NULL ) {
		Debug( KERN_WARNING, "ppnData is NULL!!!\n" );
		return FALSE;
	}

	psReg = g_regex_new( pnRegex, G_REGEX_MULTILINE, 0, &psError );
	if ( psReg == NULL ) {
		Debug( KERN_DEBUG, "Error in regex call (named):%s!\n", psError -> message );
		return FALSE;
	}

	if ( !g_regex_match( psReg, pnData, 0, &psInfo ) ) {
		Debug( KERN_DEBUG, "Regexp does not match!\n" );
		g_match_info_free( psInfo );
		g_regex_unref( psReg );
		return FALSE;
	}

	bRes = g_match_info_matches(psInfo);
	if ( !bRes ) {
		Debug( KERN_DEBUG, "Can not parse result from lookup\n" );
		g_match_info_free( psInfo );
		g_regex_unref( psReg );
		return FALSE;
	}
	if ( ppnFirstName  != NULL) {
		if ( ( pnFirstName = g_match_info_fetch_named( psInfo, "firstname" ) ) != NULL ) {
			*ppnFirstName = processResult( pnFirstName );
			g_free( pnFirstName );
		} else {
			*ppnFirstName = g_strdup( "" );
		}
	}

	if ( ppnLastName != NULL ) {
		if ( ( pnLastName = g_match_info_fetch_named( psInfo, "lastname" ) ) != NULL ) {
			*ppnLastName = processResult( pnLastName );
			g_free( pnLastName );
		} else {
			*ppnLastName = g_strdup( "" );
		}
	}

	if ( ppnStreet != NULL ) {
		if  ( ( pnStreet = g_match_info_fetch_named( psInfo, "street" ) ) != NULL ) {
			*ppnStreet = processResult( pnStreet );
			g_free( pnStreet );
		} else {
			*ppnStreet = g_strdup( "" );
		}
	}

	if ( ppnZipCode != NULL ) {
		if ( ( pnZipCode = g_match_info_fetch_named( psInfo, "zipcode" ) ) != NULL ) {
			*ppnZipCode = processResult( pnZipCode );
			g_free( pnZipCode );
		} else {
			*ppnZipCode = g_strdup( "" );
		}
	}


	if ( ppnCity != NULL ) {
		if (( pnCity = g_match_info_fetch_named( psInfo, "city" ) ) != NULL ) {
			*ppnCity = processResult( pnCity );
			g_free( pnCity );
		} else {
			*ppnCity = g_strdup( "" );
		}
	}

	g_match_info_free( psInfo );

	g_regex_unref( psReg );

	/* We want at least at least a last name */
	return ( strlen( *ppnLastName ) != 0 );
}


/**
 * \brief Extract information from lines via regex
 * \param ppnLines data in lines
 * \param pnRegex regular expression
 * \return extracted information
 */
gchar *extractInformation( gchar **ppnLines, gchar *pnRegex ) {
	GRegex *psReg;
	GMatchInfo *psInfo;
	int nIndex = 0;
	gchar *pnFirst = NULL;
	GError *psError = NULL;

	if ( ppnLines == NULL ) {
		Debug( KERN_WARNING, "ppnLines is NULL!!!\n" );
		return NULL;
	}

	psReg = g_regex_new( pnRegex, 0, 0, &psError );

	if ( psReg == NULL ) {
		Debug( KERN_DEBUG, "Error in regex call (per line):%s!\n", psError -> message );
		return NULL;
	}

	for ( nIndex = 0; ppnLines[ nIndex ] != NULL; nIndex++ ) {
		g_regex_match( psReg, ppnLines[ nIndex ], 0, &psInfo );

		while ( psInfo != NULL && g_match_info_matches( psInfo ) ) {
			gchar *pnWord = g_match_info_fetch( psInfo, 0 );
			gchar *pnResult = NULL;

			if ( pnWord == NULL ) {
				break;
			}

			pnResult = stripHtml( pnWord );
			if ( pnFirst == NULL ) {
				pnFirst = g_strdup( g_strstrip ( pnResult ) );
			}
			g_free( pnWord );
			if ( g_match_info_next( psInfo, NULL ) == FALSE ) {
				break;
			}
		}

		g_match_info_free( psInfo );
	}
	
	g_regex_unref( psReg );

	return pnFirst;
}

/** iso8859_1 entities */
static gchar *entities_iso8859_1[] = {
	"iexcl", "cent", "pound", "curren", "yen", "brvbar", "sect", "uml", "copy", "ordf",
		"laquo", "not", "shy", "reg", "macr", "deg", "plusmn", "sup2", "sup3", "acute", "micro",
		"para", "middot", "cedil", "sup1", "ordm", "raquo", "frac14", "frac12", "frac34", "iquest",
		"Agrave", "Aacute", "Acirc", "Atilde", "Auml", "Aring", "AElig", "Ccedil", "Egrave",
		"Eacute", "Ecirc", "Euml", "Igrave", "Iacute", "Icirc", "Iuml", "ETH", "Ntilde", "Ograve",
		"Oacute", "Ocirc", "Otilde", "Ouml", "times", "Oslash", "Ugrave", "Uacute", "Ucirc", "Uuml",
		"Yacute", "THORN", "szlig", "agrave", "aacute", "acirc", "atilde", "auml", "aring", "aelig",
		"ccedil", "egrave", "eacute", "ecirc", "euml", "igrave", "iacute", "icirc", "iuml", "eth",
		"ntilde", "ograve", "oacute", "ocirc", "otilde", "ouml", "divide", "oslash", "ugrave",
		"uacute", "ucirc", "uuml", "yacute", "thorn", "yuml", NULL
};

/** entities symbols */
static gchar *entities_symbols[] = {
	"fnof", "Alpha", "Beta", "Gamma", "Delta", "Epsilon", "Zeta", "Eta", "Theta", "Iota", "Kappa",
		"Lambda", "Mu", "Nu", "Xi", "Omicron", "Pi", "Rho", "Sigma", "Tau", "Uplsilon", "Phi",
		"Chi", "Psi", "Omega", "alpha", "beta", "gamma", "delta", "epsilon", "zeta", "eta", "theta",
		"iota", "kappa", "lambda", "mu", "nu", "xi", "omicron", "pi", "rho", "sigmaf", "sigma",
		"tau", "upsilon", "phi", "chi", "psi", "omega", "thetasym", "upsih", "piv", "bull",
		"hellip", "prime", "Prime", "oline", "frasl", "weierp", "image", "real", "trade", "alefsym",
		"larr", "uarr", "rarr", "darr", "harr", "crarr", "lArr", "uArr", "rArr", "dArr", "hArr",
		"forall", "part", "exist", "empty", "nabla", "isin" "notin", "ni", "prod", "sum", "minus",
		"lowast", "radic", "prop", "infin", "ang", "and", "or", "cap", "cup", "int", "there4",
		"sim", "cong", "asymp", "ne", "equiv", "le", "ge", "sub", "sup", "nsub", "sube", "supe",
		"oplus", "otimes", "perp", "sdot", "lceil", "rceil", "lfloor", "rfloor", "lang", "rang",
		"loz", "spades", "clubs", "hearts", "diams", NULL
};

/** special entities */
static gchar *entities_special[] = {
	"OElig", "oelig", "Scaron", "scaron", "Yuml", "circ", "tilde",
		"ensp", "emsp", "thinsp", "zwnj", "zwj", "lrm", "rlm", "ndash", "mdash", "lsquo", "rsquo",
		"sbquo", "ldquo", "rdquo", "bdquo", "dagger", "Dagger", "permil", "lsaquo", "rsaquo",
		"euro", NULL
};

/** xml entities */
static gchar *entities_xml[] = {
	"nbsp", "quot", "amp", "lt", "gt", NULL
};

/** the unicode characters for iso8859_1 are 161 + the index in the array */
static guint entity_unicode_symbols[] = {
	402, 913, 914, 915, 916, 917, 918, 919, 920, 921, 922, 923, 924, 925, 926, 927, 928, 929, 931,
		932, 933, 934, 935, 936, 937, 945, 946, 947, 948, 949, 950, 951, 952, 953, 954, 955, 956,
		957, 958, 959, 960, 961, 962, 963, 964, 965, 966, 967, 968, 977, 978, 982, 8226, 8230, 8242,
		8443, 8254, 8260, 8472, 8465, 8476, 8482, 8501, 8592, 8593, 8594, 8595, 8596, 8629, 8656,
		8657, 8658, 8659, 8660, 8704, 8706, 8707, 8709, 8711, 8712, 8713, 8715, 8719, 8721, 8722,
		8727, 8730, 8733, 8734, 8736, 8743, 8744, 8745, 8746, 8747, 8756, 8764, 8773, 8776, 8800,
		8801, 8804, 8805, 8834, 8835, 8836, 8838, 8839, 8853, 8855, 8869, 8901, 8968, 8969, 8970,
		8971, 9001, 9002, 9674, 9824, 9827, 9829, 9830, -1
};

/* special unicode entity */
static guint entity_unicode_special[] = {
	338, 339, 352, 353, 376, 710, 732, 8194, 8195, 8201, 8204, 8205, 8206, 8207,
		8211, 8212, 8216, 8217, 8218, 8220, 8221, 8222, 8224, 8225, 8240, 8249, 8250, 8364, -1
};

/** unicode xml entity */
static guint entity_unicode_xml[] = {
	160, 34, 38, 60, 62, -1
};

/**
 * \brief Returns index in array
 * \param arr array
 * \param string string to search in array
 * \return index or -1 on error
 */
static gint index_in_array(gchar **arr,gchar *string) {
	gint i;
	for (i=0;arr[i]!=NULL;i++) {
		if (strcmp(arr[i],string)==0) {
			return i;
		}
	}
	return -1;
}

/**
 * \brief Get unichar for entity
 * \param entity entity
 * \param numerical numerical format?
 * \param iso8859_1 test iso8859_1
 * \param symbols test symbols
 * \param specials test specials
 * \param xml test xml
 * \return unicode char
 */
static gunichar unichar_for_entity(gchar *entity,gboolean numerical, gboolean iso8859_1, gboolean symbols, gboolean specials, gboolean xml) {
	gint indx;
	if (!entity) return -1;
	if (entity[0] == '#') {
		if (numerical) {
			/* convert the remaining characters to a number */
			if (entity[1] == 'x') {
				/* from hexadecimal */
				return g_ascii_strtoull(&entity[2],NULL,16);
			} else {
				/* from decimal */
				return g_ascii_strtoull(&entity[1],NULL,10);
			}
		}
		return -1;
	}
	if (iso8859_1) {
		indx = index_in_array(entities_iso8859_1, entity);
		if (indx != -1) {
			return indx + 161;
		}
	}
	if (symbols) {
		indx = index_in_array(entities_symbols, entity);
		if (indx != -1) {
			return entity_unicode_symbols[indx];
		}
	}
	if (specials) {
		indx = index_in_array(entities_special, entity);
		if (indx != -1) {
			return entity_unicode_special[indx];
		}
	}
	if (xml) {
		indx = index_in_array(entities_xml, entity);
		if (indx != -1) {
			return entity_unicode_xml[indx];
		}
	}
	return -1;
}

/**
 * \brief Convert entities in string
 * \param inbuf input buffer
 * \return converted string
 */
gchar *convertEntities(const gchar *inbuf) {
	const gchar *found, *prevfound;
	gchar *outbuf, *outbufloc;

	outbuf = g_malloc0(strlen(inbuf)+1);
	prevfound = inbuf;
	outbufloc = outbuf;
	found = g_utf8_strchr(inbuf,-1, '&');
	while (found) {
		gchar *endfound;
		
		endfound = g_utf8_strchr(found,-1, ';');
		if (endfound && endfound - found <= 7) {
			gchar *entity, tmp[7];
			gint len;
			gunichar unic;
			
			entity = g_strndup(found+1, (endfound-found)-1);
			len = (found - prevfound);
			memcpy(outbufloc,prevfound,len);
			outbufloc += len;
			unic = unichar_for_entity(entity, TRUE,TRUE,TRUE,TRUE,TRUE);
			if (unic != -1) {
				memset(tmp, 0, 7);
				len = g_unichar_to_utf8(unic, tmp);
				len = strlen(tmp);
				memcpy(outbufloc, tmp, len);
				outbufloc += len;
			} else {
				len = endfound - found+1;
				memcpy(outbufloc, found, len);
				outbufloc += len;
			}
			g_free(entity);
			
			prevfound = g_utf8_next_char(endfound);
			found = g_utf8_strchr(prevfound,-1, '&');
		} else {
			found = g_utf8_strchr(g_utf8_next_char(found),-1, '&');
		}
	}
	memcpy(outbufloc,prevfound,strlen(prevfound)+1);
	return outbuf;
}

/**
 * \brief get call-by-call prefix length 
 * \param pnNumber input number string
 * \return length of call-by-call prefix
 */

gint callBycallPrefixLength( const char *pnNumber) {
	const gchar *myCountryCode = routerGetCountryCode( getActiveProfile() );
	callByCallEnt *entry;

	if ( ( myCountryCode == NULL) || (strlen( myCountryCode ) == 0)) {
		return 0;
	}

	for ( entry = callByCallTable; strlen( entry -> countryCode ) != 0; entry++) {
		if ( ( strcmp( myCountryCode, entry -> countryCode) == 0) &&
		     ( strncmp( pnNumber, entry -> prefix, strlen( entry -> prefix ) ) == 0) ) {
				return entry -> prefixLength;
			}
	}
	return 0;
}


/**
 * \brief Extract the real number, exclude '(' ')' ' ' ...
 * \param pnNumber input number
 * \return real number
 */
gchar *stripNumber( const gchar *pnNumber ) {
	GString *psNewNumber;
	const gchar * internationalPrefix = routerGetInternationalPrefix( getActiveProfile() );

	if ( pnNumber == NULL ) {
		return NULL;
	}

	psNewNumber = g_string_new( "" );

	while ( *pnNumber != 0 ) {
		if ( isdigit( *pnNumber ) || *pnNumber == '*' || *pnNumber == '#' ) {
			g_string_append_c( psNewNumber, *pnNumber );
		} else if ( *pnNumber == '+' ) {
			g_string_append( psNewNumber, internationalPrefix );
		}
		pnNumber++;
	}
	return g_string_free( psNewNumber, FALSE );
}


/**
 * \brief format number, exclude '(' ')' ' ' ...
 * \param pnNumber input number
 * \param outputFormat selected output format
 * \return real number
 */
gchar *formatNumber( const gchar *pnNumber, const gint outputFormat ) {
	gchar *pnTmp;
	gchar *pnStripped;
	const gchar * internationalPrefix = routerGetInternationalPrefix( getActiveProfile() );
	const gchar * nationalPrefix = routerGetNationalPrefix( getActiveProfile() );
	const gchar * myCountryCode = routerGetCountryCode( getActiveProfile() );
	const gchar * myAreaCode = routerGetAreaCode( getActiveProfile() );

	gint numberFormat = unknown;
	const gchar *myPrefix;
	gchar *pnResult = NULL;


	if ( pnNumber == NULL ) {
		return NULL;
	}


	if ( strchr( pnNumber, '@' ) != NULL ) {
		return g_strdup ( pnNumber );
	}

	pnStripped = pnTmp = stripNumber( pnNumber );

	/* we only need to check for international prefix, as stripNumber() already removes the + */
	if ( strncmp( pnTmp, internationalPrefix, strlen( internationalPrefix ) ) == 0 ) {

		/* international format number */
		pnTmp = pnTmp + strlen( internationalPrefix );
		numberFormat = internationalNumber;
	} 

	if ( (numberFormat == internationalNumber) &&
	     (strncmp ( pnTmp, myCountryCode, strlen ( myCountryCode )) == 0 ) )  {

		/* national number */
		pnTmp = pnTmp + strlen ( myCountryCode );
		numberFormat = nationalNumber;
	}

	if ( numberFormat == unknown ) {
		
		/* not an international format, test for national or local format */
		if ( strlen( nationalPrefix ) > 0 && strlen( myAreaCode ) > 0 && strncmp( pnTmp, nationalPrefix, strlen( nationalPrefix )) == 0 ) {
			pnTmp = pnTmp + strlen( nationalPrefix );
			numberFormat = nationalNumber;
		} else {
			numberFormat = localNumber;
		}
	}

	if ( ( numberFormat == nationalNumber ) && (strncmp ( pnTmp, myAreaCode, strlen ( myAreaCode ) ) == 0 ) ) {
		/* local number */
		pnTmp = pnTmp + strlen( myAreaCode );
		numberFormat = localNumber;
	} 

	/* number format must be set when we get here */

	g_assert( numberFormat != unknown );	
	g_assert( ( outputFormat >= 0 ) && ( outputFormat < 4 ) );

	switch( outputFormat ) {
	
	case requestLocalFormat:		/* local number format */
	case requestNationalFormat:		/* national number format */
		switch ( numberFormat ) {

		case localNumber:
			if ( outputFormat == requestLocalFormat ) {
				pnResult =  g_strdup( pnTmp );
			} else {
				pnResult = g_strdup_printf ( "%s%s%s", nationalPrefix, myAreaCode, pnTmp );
			}
			break;

		case nationalNumber:
			pnResult = g_strdup_printf ( "%s%s", nationalPrefix, pnTmp );
			break;

		case internationalNumber:
			pnResult = g_strdup_printf ( "%s%s", internationalPrefix, pnTmp );
			break;
		}
		break;

	case requestInternationalFormat:	/* international prefix + international format */
	case requestInternationalPlusFormat:	/* international format  prefixed by a + */

		myPrefix = (outputFormat == requestInternationalPlusFormat ) ? "+" : internationalPrefix;

		switch ( numberFormat ) {
		
		case localNumber:
			pnResult = g_strdup_printf ( "%s%s%s%s", myPrefix, myCountryCode, myAreaCode, pnTmp );
			break;

		case nationalNumber:
			pnResult = g_strdup_printf ( "%s%s%s", myPrefix, myCountryCode, pnTmp );
			break;


		case internationalNumber:
			pnResult = g_strdup_printf ( "%s%s", myPrefix, pnTmp );
			break;
		}
		break;
	default:
		g_assert(outputFormat);
	}
	g_free( pnStripped);
	g_assert(pnResult != NULL);
	return pnResult;
}

/**
 * \brief Normalize number to be stored in addressbook, , exclude '(' ')' ' ' ...
 * \param pnNumber input number
 * \return real number
 */
gchar *normalizeNumber( const gchar *pnNumber ) {
	const gint outputFormat = addressBookGetNumberFormat( getActiveProfile() );
	return formatNumber( pnNumber, outputFormat );
}

/**
 * \brief Convert number to a full number
 * \param pnNumber telephone number
 * \param bCountryCodePrefix include country code prefix?
 * \return complete number
 */
gchar *fullNumber( const char *pnNumber, gboolean bCountryCodePrefix ) {

	if ( pnNumber == NULL || strlen( pnNumber ) <= 0 ) {
		return NULL;
	}
	/* remove call-by-call (carrier preselect) prefix */

	pnNumber = pnNumber + callBycallPrefixLength( pnNumber );

	/* return requested format number */

	return formatNumber( pnNumber, bCountryCodePrefix ? requestInternationalFormat: requestNationalFormat );
}

/**
 * \brief Extract the real (dialable) number, exclude '(' ')' ' ' ...
 * \param pnNumber input number
 * \return real number
 */
gchar *fixNumber( const gchar *pnNumber ) {

	if ( pnNumber == NULL || strlen( pnNumber ) <= 0 ) {
		return NULL;
	}

	if ( strchr( pnNumber, '@' ) != NULL ) {
		return g_strdup( pnNumber );
	}

	/* return local format number */

	return formatNumber( pnNumber, requestLocalFormat );
}

/**
 * \brief Internal debug function
 * \param nLevel debug level of message
 * \param pnName function name
 * \param pnFormat debug message format
 */
void Debug2( int nLevel, const char *pnName, const char *pnFormat, ... ) {
	static FILE *psOut = NULL;
	FILE *psOutDebug = stdout;

	if ( nLevel <= debugGetLevel( getActiveProfile() ) ) {
		va_list pArgs;
		time_t sTime = time( NULL );
		struct tm *psNow = localtime( &sTime );

		if ( psOut == NULL ) {
			gchar *pnFileName = g_strdup_printf( "%s/ffgtk.log", getUserDir() );
			psOut = fopen( pnFileName, "w+" );
			g_free( pnFileName );
		}

		if ( psOut == NULL ) {
			return;
		}

		fprintf( psOut, "%02d:%02d:%02d %s(): ", psNow -> tm_hour, psNow -> tm_min, psNow -> tm_sec, pnName );
		va_start( pArgs, pnFormat );
		vfprintf( psOut, pnFormat, pArgs );
		va_end( pArgs );
		fflush( psOut );

		fprintf( psOutDebug, "%02d:%02d:%02d %s(): ", psNow -> tm_hour, psNow -> tm_min, psNow -> tm_sec, pnName );
		va_start( pArgs, pnFormat );
		vfprintf( psOutDebug, pnFormat, pArgs );
		va_end( pArgs );
		fflush( psOutDebug );
	}
}

/**
 * \brief Scale person image to requested size
 * \param psPerson person structure
 * \param fScale scale factor
 * \return scaled image or NULL
 */
GdkPixbuf *getScaledImage( struct sPerson *psPerson, float fScale ) {
	GdkPixbuf *psScaled = NULL;

	if ( psPerson != NULL && psPerson -> psImage != NULL ) {
		gint nWidth, nHeight;
		gint nOrigWidth, nOrigHeight;
		gfloat fFactor;
		gtk_icon_size_lookup( GTK_ICON_SIZE_DIALOG, &nOrigWidth, &nOrigHeight );

		nOrigWidth *= fScale;
		nOrigHeight *= fScale;

		nWidth = gdk_pixbuf_get_width( psPerson -> psImage );
		nHeight = gdk_pixbuf_get_height( psPerson -> psImage );
		if ( nWidth > nHeight ) {
			fFactor = ( float ) nWidth / ( float ) nHeight;
			nWidth = nOrigWidth;
			nHeight = nOrigHeight / fFactor;
		} else {
			fFactor = ( float ) nHeight / ( float ) nWidth;
			nHeight = nOrigHeight;
			nWidth = nOrigWidth / ( float ) fFactor;
		}

		psScaled = gdk_pixbuf_scale_simple( psPerson -> psImage, nWidth, nHeight, GDK_INTERP_BILINEAR );
	}

	return psScaled;
}

/**
 * \brief Get type icon
 * \param nType icon type
 * \return pixbuf of type
 */
GdkPixbuf *getTypeIcon( int nType ) {
	if ( psCallIn == NULL ) {
		/* create call in icon */
		gchar *pnCallIn = g_build_filename( getDirectory( PKGDATADIR ), "callin.png", NULL );
		psCallIn = gdk_pixbuf_new_from_file( pnCallIn, NULL );
		g_free( pnCallIn );

		if ( psCallIn != NULL ) {
			gdk_pixbuf_set_option( psCallIn, "title", "callin" );
		}
	}
	if ( psCallOut == NULL ) {
		/* create call out icon */
		gchar *pnCallOut = g_build_filename( getDirectory( PKGDATADIR ), "callout.png", NULL );
		psCallOut = gdk_pixbuf_new_from_file( pnCallOut, NULL );
		g_free( pnCallOut );
		if ( psCallOut != NULL ) {
			gdk_pixbuf_set_option( psCallOut, "title", "callout" );
		}
	}
	if ( psCallInFailed == NULL ) {
		/* create call in failed icon */
		gchar *pnCallInFailed = g_build_filename( getDirectory( PKGDATADIR ), "callinfailed.png", NULL );
		psCallInFailed = gdk_pixbuf_new_from_file( pnCallInFailed, NULL );
		g_free( pnCallInFailed );
		if ( psCallInFailed != NULL ) {
			gdk_pixbuf_set_option( psCallInFailed, "title", "callinfailed" );
		}
	}
	if ( psVoiceBox == NULL ) {
		/* create voicebox icon */
		GtkWidget *psImage = gtk_image_new();
#if GTK_MAJOR_VERSION < 3
		psVoiceBox = gtk_widget_render_icon( psImage, GTK_STOCK_MEDIA_RECORD, GTK_ICON_SIZE_SMALL_TOOLBAR, NULL );
#else
		psVoiceBox = gtk_widget_render_icon_pixbuf( psImage, GTK_STOCK_MEDIA_RECORD, GTK_ICON_SIZE_SMALL_TOOLBAR );
#endif
		gdk_pixbuf_set_option( psVoiceBox, "title", "voicebox" );
	}
	if ( psFaxBox == NULL ) {
		/* create faxbox icon */
		GtkWidget *psImage = gtk_image_new();
#if GTK_MAJOR_VERSION < 3
		psFaxBox = gtk_widget_render_icon( psImage, GTK_STOCK_PRINT_REPORT, GTK_ICON_SIZE_SMALL_TOOLBAR, NULL );
#else
		psFaxBox = gtk_widget_render_icon_pixbuf( psImage, GTK_STOCK_PRINT_REPORT, GTK_ICON_SIZE_SMALL_TOOLBAR );
#endif
		gdk_pixbuf_set_option( psFaxBox, "title", "faxbox" );
	}

	/* return requested icon */
	switch ( nType ) {
		case 1:
			return psCallIn;
		case 2:
			return psCallInFailed;
		case 3:
			return psCallOut;
		case 4:
			return psVoiceBox;
		case 5:
			return psFaxBox;
		default:
			break;
	}

	return NULL;
}

#if 0
/**
 * \brief Free type icons
 */
void freeTypeIcons( void ) {
	/* Free call in icon */
	if ( psCallIn != NULL ) {
		g_object_unref( G_OBJECT( psCallIn ) );
	}

	/* Free call in failed icon */
	if ( psCallInFailed != NULL ) {
		g_object_unref( G_OBJECT( psCallInFailed ) );
	}

	/* Free call out icon */
	if ( psCallOut != NULL ) {
		g_object_unref( G_OBJECT( psCallOut ) );
	}

	/* Free voicebox icon */
	if ( psVoiceBox != NULL ) {
		g_object_unref( G_OBJECT( psVoiceBox ) );
	}

	/* Free faxbox icon */
	if ( psFaxBox != NULL ) {
		g_object_unref( G_OBJECT( psFaxBox ) );
	}
}
#endif

/**
 * \brief Convert file monitor event to text
 * \param nEvent event type
 * \return event text
 */
const char *GFileMonitorEvent2Text( GFileMonitorEvent nEvent ) {
	int nIndex;

	if ( nEvent == -1 ) {
		return "empty";
	}
	for ( nIndex = 0; sEventTranslationTable[ nIndex ].nEvent != -1; nIndex++ ) {
		if ( sEventTranslationTable[ nIndex ].nEvent == nEvent ) {
			return sEventTranslationTable[ nIndex ].pnText;
		}
	}

	return "Undefined event";
	
}

//static int nLocked = 0;
static GMutex *psLocked = NULL;

void ffgtkLock( void ) {

	//g_mutex_lock( psLocked );

	//if ( nLocked == 0 ) {
		gdk_threads_enter();
	//} else {
	//	Debug( KERN_DEBUG, "Locked again!!\n" );
	//}

	//nLocked++;
}

void ffgtkUnlock( void ) {
	//nLocked--;

	//if ( nLocked == 0 ) {
		gdk_threads_leave();
	//}
	//g_mutex_unlock( psLocked );
}

void ffgtkInitLock( void ) {
    psLocked = g_mutex_new();
}

/**
 * \brief Search for case-sensitive needle in haystack
 * \param haystack haystack
 * \param needle needle
 * \return pointer to position or NULL
 */
char *g_strcasestr( const char *haystack, const char *needle ) {
	size_t n = strlen( needle );

	for ( ; *haystack; haystack++ ) {
		if ( g_ascii_strncasecmp( haystack, needle, n ) == 0 ) {
			return ( char * ) haystack;
		}
	}

	return NULL;
}

#if GTK_MAJOR_VERSION > 2
/** The following functions are deprecated within GTK3 so i readded them
 * in order to compile ffgtk with GTK2 and GTK3
 */

/**
 * \brief Append text to combo box
 * \param psComboBox combo box widget
 * \param pnText text to add
 */
void gtk_combo_box_append_text( GtkComboBox *psComboBox, const gchar *pnText ) {
	GtkTreeIter sIter;
	GtkListStore *psStore;

	psStore = GTK_LIST_STORE( gtk_combo_box_get_model( psComboBox ) );
	gtk_list_store_append( GTK_LIST_STORE( psStore ), &sIter );
	gtk_list_store_set( psStore, &sIter, 0, pnText, -1 );
}

/**
 * \brief Get active selected combo box text
 * \param psComboBox combo box widget
 * \return active text or NULL on error
 */
gchar *gtk_combo_box_get_active_text( GtkComboBox *psComboBox ) {
	GtkTreeIter sIter;
	gchar *pnText = NULL;
	GtkTreeModel *psModel = gtk_combo_box_get_model( psComboBox );

	if ( gtk_combo_box_get_active_iter( psComboBox, &sIter ) ) {
		gtk_tree_model_get( psModel, &sIter, 0, &pnText, -1 );
	}

	return pnText;
}

/**
 * \brief Create new combo box text
 * \return combo box widget
 */
GtkWidget *gtk_combo_box_new_text(void) {
	return gtk_combo_box_text_new();
}

#endif
