/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file ab_fritzfon_v1.c
 * \brief fritzfonV1 addressbook plugin
 */

#include <ffgtk.h>
#include <sys/stat.h>

/** Maximal fritz fon chunk size */
static int nfritzfonV1MaxSize = 2048;

/** Swap first/last name */
static gboolean fritzfonV1GetSwap( void ) {
	return prefsGetBool( getActiveProfile(), "/plugins/fritzfonV1/swap" );
}

/** Swap first/last name */
static void fritzfonV1SetSwap( gboolean bSwap ) {
	prefsSetBool( getActiveProfile(), "/plugins/fritzfonV1/swap", bSwap );
}

/**
 * \brief Get name from fritzbox structure
 * \param pnData data pointer
 * \param nPos starting position
 * \param ppnName where to store name to
 * \return error code, 0=success else fail
 */
static int fritzfonV1GetName( gchar *pnData, gint nPos, gchar **ppnName ) {
	int nNameStart;
	int nNameStop;

	nPos = findString( pnData, nPos, "TrFonName(" );
	if ( nPos == -1 ) {
		//Debug( KERN_DEBUG, "'TrFonName(' Not found\n" );
		return -1;
	}

	nNameStart = findString( pnData, nPos, "," );

	if ( nNameStart == -1 ) {
		Debug( KERN_DEBUG, "Error in nNameStart\n" );
		return -1;
	}
	nNameStart += 3;

	nNameStop = findString( pnData, nNameStart, "\"" );
	if ( nNameStop == -1 ) {
		Debug( KERN_DEBUG, "Error in nNameStop\n" );
		return -1;
	}
	nNameStop -= 1;

	*ppnName = getSubString( pnData, nNameStart, nNameStop - nNameStart + 1 );
	if ( *ppnName == NULL ) {
		Debug( KERN_DEBUG, "Error in pnName\n" );
		return -1;
	}

	return nNameStop;
}

/**
 * \brief Retrieve type and number from fritzbox structure
 * \param pnData data pointer
 * \param nPos starting position
 * \param ppnType type of number
 * \param ppnNumber number storage
 * \return error code, 0=success else fail
 */
static int fritzfonV1GetNumber( gchar *pnData, gint nPos, gchar **ppnType, gchar **ppnNumber ) {
	int nNextNamePos = findString( pnData, nPos, "document.write" );
	int nPtr, nTypeStart, nTypeStop, nNumberStart, nNumberStop;

	*ppnType = NULL;
	*ppnNumber = NULL;

	nPtr = findString( pnData, nPos, "TrFonNr(" );
	if ( nPtr == -1 || nPtr >= nNextNamePos ) {
		Debug( KERN_DEBUG, "Error in nPtr\n" );
		return -1;
	}
	nTypeStart = nPtr + 9;
	nNumberStart = findString( pnData, nTypeStart, "," );
	if ( nNumberStart == -1 || nNumberStart >= nNextNamePos ) {
		Debug( KERN_DEBUG, "Error in nNumberStart\n" );
		return -1;
	}
	nNumberStart += 3;
	nTypeStop = nNumberStart - 5;
	nNumberStop = findString( pnData, nNumberStart, "\"" );
	if ( nNumberStop == -1 || nNumberStop >= nNextNamePos ) {
		Debug( KERN_DEBUG, "Error in nNumberStop\n" );
		return -1;
	}
	nNumberStop -= 1;
	*ppnNumber = getSubString( pnData, nNumberStart, nNumberStop - nNumberStart + 1 );
	if ( *ppnNumber == NULL ) {
		Debug( KERN_DEBUG, "Error in pnNumber\n" );
		return -1;
	}

	*ppnType = getSubString( pnData, nTypeStart, nTypeStop - nTypeStart + 1 );
	if ( *ppnType == NULL ) {
		Debug( KERN_DEBUG, "Error in pnType\n" );
		g_free( *ppnNumber );
		return -1;
	}
					 
	return nNumberStop;
}

/**
 * \brief Read ebook list
 * \return error code
 */
static int fritzfonV1ReadBook( void ) {
	struct sUrlHandler *psHandler;
	gint nPos, nCount = 0, nLoop;
	gchar anBuffer[ 1024 ];
	gchar *pnName;
	gchar *pnConvName = NULL;
	gchar *pnType;
	gchar *pnNumber;
	gchar *pnHome = NULL;
	gchar *pnMobile = NULL;
	gchar *pnWork = NULL;
	gchar *pnFirstName = NULL;
	gchar *pnLastName = NULL;
	gchar *pnTmp;
	gint nError = -1;

	if ( routerLogin( getActiveProfile() ) == -1 ) {
		return -1;
	}

	snprintf( anBuffer, sizeof( anBuffer ), "%s/cgi-bin/webcm", routerGetHost( getActiveProfile() ) );
	psHandler = urlHandler( anBuffer, 80 );
	setPostData( psHandler, "&getpage=../html/de/menus/menu2.html&var:lang=de&var:pagename=fonbuch&var:menu=fon&sid=%s", getActiveProfile() -> sRouterInfo.pnSessionId );
	nError = readUrl( psHandler, getActiveProfile() );
	routerLogout( getActiveProfile() );
	if ( nError != 0 ) {
		printf( "Error: No data\n" );
		return -1;
	}

	psHandler -> pnData = loadData( "/home/buzz/tb.html", NULL );

	nPos = 0;
	while ( ( nPos = fritzfonV1GetName( psHandler -> pnData, nPos, &pnName ) ) != -1 ) {
		int nAct;

		pnHome = NULL;
		pnWork = NULL;
		pnMobile = NULL;

		nAct = nPos;
		for ( nLoop = 0; nLoop < 3; nLoop++ ) {
			nAct = fritzfonV1GetNumber( psHandler -> pnData, nAct, &pnType, &pnNumber );
			if ( nAct == -1 ) {
				break;
			}

			if ( strcmp( pnType, "mobile" ) == 0 ) {
				pnMobile = pnNumber;
			} else if ( strcmp( pnType, "home" ) == 0 ) {
				pnHome = pnNumber;
			} else if ( strcmp( pnType, "work" ) == 0 ) {
				pnWork = pnNumber;
			} else {
				Debug( KERN_WARNING, "Unknown type '%s'\n", pnType );
			}
			g_free( pnType );
			pnType = NULL;

			if ( nAct != -1 ) {
				nPos = nAct;
			}
		}

		pnConvName = convertEntities( pnName );
		//Debug( KERN_DEBUG, "pnConvName: '%s' (%d)\n", pnConvName, strlen( pnConvName ) );
		pnTmp = strchr( pnConvName, ' ' );
		if ( pnTmp != NULL ) {
			gint nLen = 0;
			pnLastName = strrchr( pnConvName, ' ' ) + 1;
			//Debug( KERN_DEBUG, "pnLastName: '%s', (%d)\n", pnLastName, strlen( pnLastName ) );
			nLen = strlen( pnConvName ) - strlen( pnLastName ) - 1;
			pnFirstName = g_malloc0( nLen + 1 );
			strncpy( pnFirstName, pnConvName, nLen );
			pnFirstName[ nLen ] = '\0';
			//Debug( KERN_DEBUG, "pnFirstName %s, (%d)\n", pnFirstName, nLen );
		} else {
			pnFirstName = NULL;
			pnLastName = pnConvName;
		}
	
		GHashTable *psTable = g_hash_table_new( NULL, NULL );
		gchar *pnId = g_strdup_printf( "%d", nCount++ );
		AddInfo( psTable, PERSON_ID, pnId );
		if ( fritzfonV1GetSwap() == FALSE ) {
			AddInfo( psTable, PERSON_FIRST_NAME, pnFirstName );
			AddInfo( psTable, PERSON_LAST_NAME, pnLastName );
		} else {
			AddInfo( psTable, PERSON_FIRST_NAME, pnLastName );
			AddInfo( psTable, PERSON_LAST_NAME, pnFirstName );
		}
		AddInfo( psTable, PERSON_DISPLAY_NAME, pnConvName );
		AddInfo( psTable, PERSON_BUSINESS_PHONE, pnWork );
		AddInfo( psTable, PERSON_PRIVATE_PHONE, pnHome );
		AddInfo( psTable, PERSON_PRIVATE_MOBILE, pnMobile );

		AddPerson( psTable, FALSE );
		g_free( pnId );

		g_hash_table_destroy( psTable );

		if ( pnFirstName != NULL ) {
			g_free( pnFirstName );
		}
		g_free( pnConvName );
		
		if ( pnHome != NULL ) {
			g_free( pnHome );
			pnHome = NULL;
		}
		if ( pnWork != NULL ) {
			g_free( pnWork );
			pnWork = NULL;
		}
		if ( pnMobile != NULL ) {
			g_free( pnMobile );
			pnMobile = NULL;
		}
		if ( pnName != NULL ) {
			g_free( pnName );
		}
	}

	freeHandler( psHandler );

	return 0;
}

/**
 * \brief Convert & to %26
 * \param pnInput input string
 * \return output string
 */
static gchar *fritzfonV1Convert( gchar *pnInput ) {
	GRegex *psLocal = g_regex_new( "&", G_REGEX_DOTALL | G_REGEX_OPTIMIZE, 0, NULL );
	gchar *pnTmp = g_regex_replace_literal( psLocal, pnInput, -1, 0, "%26", 0, NULL );

	return pnTmp;
}

/**
 * \brief Write ebook list
 * \return error code
 */
static int fritzfonV1WriteBook( void ) {
	struct sUrlHandler *psHandler;
	GString *psDelete = NULL;
	gchar anBuffer[ 1024 ];
	gint nError = -1, nIndex = -1;
	GList *psList = psPersonsList;
	GString *psEntry = NULL;

	if ( routerLogin( getActiveProfile() ) == -1 ) {
		return -1;
	}

	if ( nfritzfonV1MaxSize > CURL_MAX_WRITE_SIZE ) {
		nfritzfonV1MaxSize = CURL_MAX_WRITE_SIZE;
	}

	/* Delete list */
	snprintf( anBuffer, sizeof( anBuffer ), "%s/cgi-bin/webcm", routerGetHost( getActiveProfile() ) );
	psHandler = urlHandler( anBuffer, 80 );
	psDelete = g_string_new( "" );
	//g_string_printf( psDelete, "&getpage=../html/de/menus/menu2.html&var:lang=de&var:pagename=fonbuch&var:menu=fon&sid=%s", getActiveProfile() -> sRouterInfo.pnSessionId );

	nIndex = g_list_length( psList );
	nIndex = 100;
	Debug( KERN_DEBUG, "Length of list: %d\n", nIndex );
	while ( nIndex != 0 ) {
		g_string_append_printf( psDelete, "&telcfg:command/Phonebook/Entry%d=delete", --nIndex );
	}

	Debug( KERN_DEBUG, "Send str: %d [%s]\n", strlen( psDelete -> str ), psDelete -> str );
	g_string_append_printf( psDelete, "&sid=%s", getActiveProfile() -> sRouterInfo.pnSessionId );
	setPostData( psHandler, psDelete -> str );

	nError = readUrl( psHandler, getActiveProfile() );
	g_string_free( psDelete, TRUE );
	freeHandler( psHandler );

	/* Add list */
	snprintf( anBuffer, sizeof( anBuffer ), "%s/cgi-bin/webcm", routerGetHost( getActiveProfile() ) );
	psHandler = urlHandler( anBuffer, 80 );
	psDelete = g_string_new( "" );
	//g_string_printf( psDelete, "&getpage=../html/de/menus/menu2.html&var:lang=de&var:pagename=fonbuch2&var:menu=fon&sid=%s", getActiveProfile() -> sRouterInfo.pnSessionId );
	
	for ( nIndex = 0, psList = psPersonsList; psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		struct sPerson *psPerson = psList -> data;
		gint nNumber = 0;
		gchar anCode[ 3 ];

		if ( psPerson -> nFlags & PERSON_FLAG_DELETED ) {
			psPerson -> nFlags = PERSON_FLAG_UNCHANGED;
			continue;
		}

		snprintf( anCode, sizeof( anCode ), "%.2d", nIndex + 1 );

		if ( psPerson -> pnId != NULL ) {
			g_free( psPerson -> pnId );
		}
		psPerson -> pnId = g_strdup_printf( "%d", nIndex );

		psEntry = g_string_new( "" );

		/* Set general name entry */
		g_string_append_printf( psEntry, "&telcfg:settings/Phonebook/Entry%d/Name=", nIndex );

		if ( fritzfonV1GetSwap() == TRUE ) {
			/* Add last name if present */
			if ( psPerson -> pnLastName != NULL && strlen( psPerson -> pnLastName ) > 0 ) {
				gchar *pnConv = fritzfonV1Convert( psPerson -> pnLastName );
				g_string_append_printf( psEntry, "%s", pnConv );
			}

			/* If needed, add space between first and last name */
			if ( psPerson -> pnFirstName != NULL && strlen( psPerson -> pnFirstName ) > 0 ) {
				g_string_append_printf( psEntry, " " );
			}
		}

		/* Add firstname if present */
		if ( psPerson -> pnFirstName != NULL && strlen( psPerson -> pnFirstName ) > 0 ) {
			gchar *pnConv = fritzfonV1Convert( psPerson -> pnFirstName );
			g_string_append_printf( psEntry, "%s", pnConv );

			/* If needed, add space between first and last name */
			if ( psPerson -> pnLastName != NULL && strlen( psPerson -> pnLastName ) > 0 ) {
				g_string_append_printf( psEntry, " " );
			}
		}

		if ( fritzfonV1GetSwap() == FALSE ) {
			/* Add last name if present */
			if ( psPerson -> pnLastName != NULL && strlen( psPerson -> pnLastName ) > 0 ) {
				gchar *pnConv = fritzfonV1Convert( psPerson -> pnLastName );
				g_string_append_printf( psEntry, "%s", pnConv );
			}
		}

		/* Set category 0 */
		g_string_append_printf( psEntry, "&telcfg:settings/Phonebook/Entry%d/Category=%d", nIndex, 0 );

		if ( psPerson -> pnPrivatePhone != NULL && strlen( psPerson -> pnPrivatePhone ) > 0 ) {
			g_string_append_printf( psEntry,
					               "&telcfg:settings/Phonebook/Entry%d/Number%d/Type=%s&telcfg:settings/Phonebook/Entry%d/Number/Number%d=%s"
					               "&telcfg:settings/Phonebook/Entry%d/Number%d/Code=%s&telcfg:settings/Phonebook/Entry%d/Number%d/Vanity=",
					               nIndex, nNumber, "home", nIndex, nNumber, psPerson -> pnPrivatePhone,
								   nIndex, nNumber, ""/*nNumber == 0 ? anCode : ""*/, nIndex, nNumber );
			nNumber++;
		}

		if ( psPerson -> pnPrivateMobile != NULL && strlen( psPerson -> pnPrivateMobile ) > 0 ) {
			g_string_append_printf( psEntry,
					               "&telcfg:settings/Phonebook/Entry%d/Number%d/Type=%s&telcfg:settings/Phonebook/Entry%d/Number%d/Number=%s"
					               "&telcfg:settings/Phonebook/Entry%d/Number%d/Code=%s&telcfg:settings/Phonebook/Entry%d/Number%d/Vanity=",
					               nIndex, nNumber, "mobile", nIndex, nNumber, psPerson -> pnPrivateMobile, nIndex, nNumber, ""/*nNumber == 0 ? anCode : ""*/, nIndex, nNumber );
			nNumber++;
		}

		if ( psPerson -> pnBusinessPhone != NULL && strlen( psPerson -> pnBusinessPhone ) > 0 ) {
			g_string_append_printf( psEntry,
					               "&telcfg:settings/Phonebook/Entry%d/Number%d/Type=%s&telcfg:settings/Phonebook/Entry%d/Number%d/Number=%s"
					               "&telcfg:settings/Phonebook/Entry%d/Number%d/Code=%s&telcfg:settings/Phonebook/Entry%d/Number%d/Vanity=",
					               nIndex, nNumber, "work", nIndex, nNumber, psPerson -> pnBusinessPhone, nIndex, nNumber, ""/*nNumber == 0 ? anCode : ""*/, nIndex, nNumber );
		}
		
		if ( psDelete -> len + psEntry -> len + 22 >= nfritzfonV1MaxSize ) {
			g_string_append_printf( psDelete, "&sid=%s", getActiveProfile() -> sRouterInfo.pnSessionId );
			setPostData( psHandler, psDelete -> str );
			Debug( KERN_DEBUG, "write: %d/%d '%s'\n", psDelete -> len, CURL_MAX_WRITE_SIZE, psDelete -> str );
			nError = readUrl( psHandler, getActiveProfile() );
			g_string_free( psDelete, TRUE );

			psDelete = g_string_new( "" );
		}

		g_string_append_printf( psDelete, "%s", psEntry -> str );
		g_string_free( psEntry, TRUE );

		nIndex++;
		psPerson -> nFlags = PERSON_FLAG_UNCHANGED;
	}

	if ( psDelete -> len > 100 ) {
		Debug( KERN_DEBUG, "write: %d/%d '%s'\n", psDelete -> len, CURL_MAX_WRITE_SIZE, psDelete -> str );
		//Debug( KERN_DEBUG, "write: %d/%d\n", psDelete -> len, CURL_MAX_WRITE_SIZE );
		g_string_append_printf( psDelete, "&sid=%s", getActiveProfile() -> sRouterInfo.pnSessionId );
		setPostData( psHandler, psDelete -> str );

		nError = readUrl( psHandler, getActiveProfile() );
	}
	freeHandler( psHandler );
	g_string_free( psDelete, TRUE );

	routerLogout( getActiveProfile() );

	return nError;
}

void prefsAddNone( struct sProfile *psProfile, const char *pnName );

/**
 * \brief Display fritzfonV1 preferences window
 */
static void fritzfonV1Preferences( void ) {
	GtkWidget *psDialog = gtk_dialog_new_with_buttons( _( "FritzFon V1 Preferences" ), NULL, 0, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL );
	GtkWidget *psCheckBox = gtk_check_button_new_with_label( _( "Swap first/lastname" ) );
	GtkWidget *psBox;

	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( psCheckBox ), fritzfonV1GetSwap() );
	psBox = gtk_dialog_get_content_area( GTK_DIALOG( psDialog ) );
	gtk_box_pack_start( GTK_BOX( psBox ), psCheckBox, FALSE, TRUE, 15 );
	gtk_widget_set_size_request( psDialog, 300, 80 );
	gtk_widget_show( GTK_WIDGET( psCheckBox ) );
	gtk_dialog_run( GTK_DIALOG( psDialog ) );

	prefsAddNone( getActiveProfile(), "/plugins/fritzfonV1" );

	fritzfonV1SetSwap( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psCheckBox ) ) );
	gtk_widget_destroy( psDialog );
	SavePreferences( getActiveProfile() );

	freePersons();
	fritzfonV1ReadBook();
}

/** book definition */
static struct sAddressBook sFritzFonV1 = {
	fritzfonV1ReadBook,
	fritzfonV1WriteBook,
	PERSON_FIRST_NAME | PERSON_LAST_NAME | PERSON_DISPLAY_NAME |
	PERSON_PRIVATE_PHONE | PERSON_PRIVATE_MOBILE |
	PERSON_BUSINESS_PHONE,
	PERSON_FIRST_NAME | PERSON_LAST_NAME | PERSON_DISPLAY_NAME |
	PERSON_PRIVATE_PHONE | PERSON_PRIVATE_MOBILE |
	PERSON_BUSINESS_PHONE,
	fritzfonV1Preferences
};

MODULE_INIT( PLUGIN_TYPE_BOOK, _( "fritzfon V1 Addressbook" ), &sFritzFonV1, NULL, NULL );
