/*
 * Copyright (C) 2007 Felipe Weckx <felipe.weckx@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gbemol-utils.h"
#include "gbemol-mpd.h"
#include "gbemol-net.h"
#include "gbemol-amazon.h"

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdarg.h>

#ifdef ENABLE_ID3_APIC
/* ID3 Picture Tag */
#include <id3.h>
#endif

#include <gtk/gtk.h>
#include <glib/gstdio.h>

/* Got from brasero (http://perso.wanadoo.fr/bonfire) */
GtkWidget *
gbemol_utils_make_button (char *text, char *stock)
{
	GtkWidget *button;
	GtkWidget *box, *lbl;

	box = gtk_hbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (box),
			    gtk_image_new_from_stock (stock, GTK_ICON_SIZE_BUTTON),
			    FALSE,
			    FALSE,
			    0);

	button = gtk_button_new ();
	if (text)
	{
		lbl = gtk_label_new_with_mnemonic (text);
		gtk_box_pack_start (GTK_BOX (box),
				    lbl,
				    FALSE,
				    FALSE,
				    0);
		gtk_label_set_mnemonic_widget (GTK_LABEL (lbl), button);
	}

	gtk_container_add (GTK_CONTAINER (button), box);
	gtk_button_set_alignment (GTK_BUTTON (button), 0.5, 0.5);

	return button;
}

/* Same as above, but returns the label as well */
GtkWidget *
gbemol_utils_make_button_and_label (char *text, char *stock, GtkWidget** lbl)
{
	GtkWidget *button;
	GtkWidget *box;

	box = gtk_hbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (box),
			    gtk_image_new_from_stock (stock, GTK_ICON_SIZE_BUTTON),
			    FALSE,
			    FALSE,
			    0);

	button = gtk_button_new ();
	if (text)
	{
		*lbl = gtk_label_new_with_mnemonic (text);
		gtk_box_pack_start (GTK_BOX (box),
				    *lbl,
				    FALSE,
				    FALSE,
				    0);
		gtk_label_set_mnemonic_widget (GTK_LABEL (*lbl), button);
	}

	gtk_container_add (GTK_CONTAINER (button), box);
	gtk_button_set_alignment (GTK_BUTTON (button), 0.5, 0.5);

	return button;
}

void gbemol_utils_char_list_free (GList *l)
{
	if (l)
	{
		g_list_foreach (l, (GFunc) g_free, NULL);
		g_list_free (l);
	}
}

void gbemol_utils_song_list_free (GList *l)
{
	if (l)
	{
		g_list_foreach (l, (GFunc) gbemol_mpd_free_song, NULL);
		g_list_free (l);
	}
}

GList* 
gbemol_utils_get_selected (GtkTreeView* tvw)
{
	GtkTreeModel *model;
	GtkTreeSelection *select;
	GList *rows = NULL;

	select = gtk_tree_view_get_selection (tvw);
	if (gtk_tree_selection_count_selected_rows (select) > 0)
	{
		rows = gtk_tree_selection_get_selected_rows (select, &model);
		return rows;
	} 
	else
		return NULL;
}

void 
gbemol_utils_selected_list_free (GList* l)
{
        if (l)
        {
                g_list_foreach (l, (GFunc) gtk_tree_path_free, NULL);
                g_list_free (l);
        }
}

/*
 * Config file functions
 */
gboolean gbemol_cfg_create_file ()
{
	gchar* path;
	gint fd;

	path = g_strdup_printf ("%s/.gbemol.cfg", g_get_home_dir());

	if ((fd = open (path, O_CREAT)) != -1)
	{
		chmod (path, 0644);
		close (fd);
		/* Initial configs */
		gbemol_cfg_set_int ("win_main", "visible", 1);
		return TRUE;
	}

	return FALSE;
}

/*
 * Opens and reads the key file
 */
GKeyFile* gbemol_cfg_load_file ()
{
	gchar* path, *path2;
	GKeyFile* file;
	GError* error = NULL;

	path = g_strdup_printf ("%s/gbemol/gbemol.cfg", g_get_user_config_dir());
	file = g_key_file_new ();

	if (g_file_test (path, G_FILE_TEST_EXISTS))
		/* File exists, load it */
		g_key_file_load_from_file (file, path, G_KEY_FILE_NONE, &error);
	else
	{
		/* File doesn't exist, check if the directory exists */
		path2 = g_strdup_printf ("%s/gbemol", g_get_user_config_dir());		
		if (!(g_file_test (path2, G_FILE_TEST_EXISTS) && g_file_test (path2, G_FILE_TEST_IS_DIR)))
			g_mkdir (path2, 0755);
		g_free (path2);
	}

	g_free (path);
	
	return file;
}

gboolean gbemol_cfg_save_file (GKeyFile* file)
{
	gchar* path, *data;
	gsize len;
	FILE* cfg;
	
	path = g_strdup_printf ("%s/gbemol/gbemol.cfg", g_get_user_config_dir ());
	data = g_key_file_to_data (file, &len, NULL);

	cfg = fopen (path, "w");

	if (cfg)
	{
		fprintf (cfg, data);
		fclose (cfg);
		g_free (path);
		g_free (data);
		return TRUE;
	}

	g_free (path);
	g_free (data);
	g_key_file_free (file);

	return FALSE;
}
	
void gbemol_cfg_set_string (gchar* group, gchar* key, const gchar* value)
{
	GKeyFile *file;

	file = gbemol_cfg_load_file ();
	g_key_file_set_string (file, group, key, value);
	gbemol_cfg_save_file (file);
}

gchar* gbemol_cfg_get_string (gchar* group, gchar* key)
{
	GKeyFile *file;
	gchar* str;

	file = gbemol_cfg_load_file ();
	str = g_key_file_get_string (file, group, key, NULL);
	g_key_file_free (file);

	return str;
}

void gbemol_cfg_set_int (gchar* group, gchar* key, gint value)
{
	GKeyFile *file;

	file = gbemol_cfg_load_file ();
	g_key_file_set_integer (file, group, key, value);
	gbemol_cfg_save_file (file);
	g_key_file_free (file);

}

gint gbemol_cfg_get_int (gchar* group, gchar* key)
{
	GKeyFile *file;
	gint value;

	file = gbemol_cfg_load_file ();
	value = g_key_file_get_integer (file, group, key, NULL);
	g_key_file_free (file);

	return value;
}

void gbemol_cfg_set_bool(gchar* group, gchar* key, gboolean value)
{
	GKeyFile *file;

	file = gbemol_cfg_load_file ();
	g_key_file_set_boolean(file, group, key, value);
	gbemol_cfg_save_file (file);
	g_key_file_free (file);

}

gboolean gbemol_cfg_get_bool(gchar* group, gchar* key)
{
	GKeyFile *file;
	gboolean value;

	file = gbemol_cfg_load_file ();
	value = g_key_file_get_boolean(file, group, key, NULL);
	g_key_file_free (file);

	return value;
}

gboolean gbemol_cfg_get_bool_with_default(gchar* group, gchar* key, gboolean def)
{
	GKeyFile *file;
	gboolean value;
	GError* e = NULL;

	file = gbemol_cfg_load_file();
	value = g_key_file_get_boolean(file, group, key, &e);
	if ((e != NULL) && (e->code == G_KEY_FILE_ERROR_KEY_NOT_FOUND))
		value = def;
	g_key_file_free (file);
	
	return value;
}

void gbemol_cfg_set_double (gchar* group, gchar* key, gdouble value)
{
	GKeyFile *file;

	file = gbemol_cfg_load_file ();
	g_key_file_set_double (file, group, key, value);
	gbemol_cfg_save_file (file);

}

gdouble gbemol_cfg_get_double (gchar* group, gchar* key)
{
	GKeyFile *file;
	gdouble value;

	file = gbemol_cfg_load_file ();
	value = g_key_file_get_double (file, group, key, NULL);
	g_key_file_free (file);

	return value;
}

#ifdef ENABLE_ID3_APIC
/*
 * Got from EasyTag (http://easytag.sf.net)
 */
ID3_C_EXPORT size_t ID3Tag_Link_1 (ID3Tag *id3tag, const char *filename)
{
    size_t offset;

#   if (0) // Link the file with the both tags may cause damage to unicode strings
//#   if ( (ID3LIB_MAJOR >= 3) && (ID3LIB_MINOR >= 8) && (ID3LIB_PATCH >= 1) ) // Same test used in Id3tag_Read_File_Tag to use ID3Tag_HasTagType
        /* No problem of priority, so we link the file with the both tags
         * to manage => ID3Tag_HasTagType works correctly */
        offset = ID3Tag_LinkWithFlags(id3tag,filename,ID3TT_ID3V1 | ID3TT_ID3V2);
#   elif ( (ID3LIB_MAJOR >= 3) && (ID3LIB_MINOR >= 8) )
        /* Version 3.8.0pre2 gives priority to tag id3v1 instead of id3v2, so we
         * try to fix it by linking the file with the id3v2 tag first. This bug 
         * was fixed in the final version of 3.8.0 but we can't know it... */
        /* First, try to get the ID3v2 tags */
        offset = ID3Tag_LinkWithFlags(id3tag,filename,ID3TT_ID3V2);
        if (offset == 0)
        {
            /* No ID3v2 tags available => try to get the ID3v1 tags */
            offset = ID3Tag_LinkWithFlags(id3tag,filename,ID3TT_ID3V1);
        }
#   else
        /* Function 'ID3Tag_LinkWithFlags' is not defined up to id3lib-.3.7.13 */
        offset = ID3Tag_Link(id3tag,filename);
#   endif
    //g_print("ID3 TAG SIZE: %d\t%s\n",offset,g_path_get_basename(filename));
    return offset;
}

#endif

GdkPixbuf* gbemol_get_song_image_from_id3 (GbemolMpdSong* song)
{
	GdkPixbuf *pix = NULL;
#ifdef ENABLE_ID3_APIC	
	size_t offset;
	ID3Tag* id3_tag = NULL;
	ID3Frame* id3_frame;
	ID3Field* id3_field;
	gchar *path, *songs_path;
	GdkPixbufLoader *load;
	gulong size = 0;
	guchar  *data = NULL;

	if (!(songs_path = gbemol_cfg_get_string ("MPD", "music_dir")))
		return NULL;

	path = g_strdup_printf ("%s/%s", songs_path, song->file);

	id3_tag = ID3Tag_New();
	offset = ID3Tag_LinkWithFlags (id3_tag, path, ID3TT_ID3V1 | ID3TT_ID3V2);

	g_free (path);
	g_free (songs_path);

	if (!offset)
		return NULL;

	id3_frame = ID3Tag_FindFrameWithID(id3_tag, ID3FID_PICTURE);
	if (!id3_frame)
		return NULL;

	if ( (id3_field = ID3Frame_GetField(id3_frame, ID3FN_DATA)) )
        {
            size = ID3Field_Size(id3_field);
            data = g_malloc(size);
            ID3Field_GetBINARY(id3_field, data, size);
        }

	load = gdk_pixbuf_loader_new ();
	if (load)
        	if (gdk_pixbuf_loader_write (load, data, size, 0))
			pix = gdk_pixbuf_loader_get_pixbuf (load);

	g_free (data);
	g_free (id3_tag);
	g_free (id3_frame);
	g_free (id3_field);
#endif
	return pix;
}


/* Returns the cover filename and creates ~/.cover if it doesn't exists */
gchar* gbemol_utils_generate_cover_filename(GbemolMpdSong* song)
{
	gchar* str_dir, *str;
	GDir* dir_covers;
	
	str_dir = g_strdup_printf ("%s/.covers/", g_get_home_dir());	
	dir_covers = g_dir_open(str_dir, 0, NULL);
	if (dir_covers == NULL) {
		g_mkdir(str_dir, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH);
	} else {
		g_dir_close(dir_covers);
	}
	g_free(str_dir);
	str = g_strdup_printf ("%s/.covers/%s-%s.jpg", g_get_home_dir(), song->artist, song->album);
	return str;
}

/**
 * Try to load image from file in ~/.covers
 */
GdkPixbuf* gbemol_utils_get_cover_image_from_file(GbemolMpdSong* song)
{
	gchar *str;
	GdkPixbuf* img = NULL;
	
	str = gbemol_utils_generate_cover_filename(song);
	
	if (g_file_test (str, G_FILE_TEST_EXISTS))
	{
		GError *e = NULL;
		img = gdk_pixbuf_new_from_file (str, &e);
		if (e) 
		{
			g_message (e->message);		
			img = NULL;
		}
	}
	g_free(str);
	
	return img;
}

/**
 * Try to fetch image from Amazon.com
 */
GdkPixbuf* gbemol_utils_get_cover_image_from_amazon(GbemolMpdSong* song)
{
	GdkPixbuf* img = NULL;;	
	gchar *result = NULL;	
	GSList* results;
	
	results = gbemol_amazon_search_cover(song);
	if (results == NULL)
	{
		return NULL;
	}
	else
	{
		GError* err = NULL;
		gchar* str = gbemol_utils_generate_cover_filename(song);
		/* There's a cover, save it */
		result = results->data;		
		gbemol_net_download_file(result, str, &err);
		if (err != NULL)
		{
			g_message(err->message);
		}
		else
		{
			img = gdk_pixbuf_new_from_file (str, NULL);
		}
		g_free (str);		
	}
	
	/* Free everything */
	g_slist_foreach(results, (GFunc) g_free, NULL);

	return img;
}

GdkPixbuf* gbemol_utils_get_song_image (GbemolMpdSong* song)
{
	GdkPixbuf *img = NULL;
	
	if (gbemol_cfg_get_bool("cover_art", "file"))
		img = gbemol_utils_get_cover_image_from_file(song);
	
	/* No picture in  ~/.covers, try amazon */
	if (!img && gbemol_cfg_get_bool("cover_art", "amazon"))
	{
		img = gbemol_utils_get_cover_image_from_amazon(song);
	}

	return img;
}

GtkWidget* gbemol_utils_label_new_with_markup (gchar* text)
{
	GtkWidget* lbl;

	lbl = gtk_label_new (text);
	gtk_label_set_use_markup (GTK_LABEL (lbl), TRUE);
	gtk_misc_set_alignment (GTK_MISC (lbl), 0.0, 0.5);

	return lbl;
}

void gbemol_debug_msg (char *msg, ...)
{
#ifdef DEBUG
	va_list args;
	char *str;

	va_start (args);

	str = g_strdup_vprintf ("DEBUG: %s\n", args);
	g_print (str);
	g_free (str);
#endif
}

