/**
 * \file pappsomspp/msrun/private/timsmsrunreaderbase.cpp
 * \date 15/07/2024
 * \author Olivier Langella
 * \brief Base class for all tims ms run reader objects
 */

/*******************************************************************************
 * Copyright (c) 2024 Olivier Langella
 *<Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

#include "timsmsrunreaderbase.h"
#include "../../exception/exceptionnotimplemented.h"
#include "../../exception/exceptioninterrupted.h"
#include <QDebug>

namespace pappso
{
TimsMsRunReaderBase::TimsMsRunReaderBase(MsRunIdCstSPtr &msrun_id_csp)
  : MsRunReader(msrun_id_csp)
{
  initialize();
}

TimsMsRunReaderBase::TimsMsRunReaderBase(
  const TimsMsRunReaderBase &msrun_reader_base)
  : MsRunReader(msrun_reader_base.getMsRunId()),
    msp_timsData(msrun_reader_base.msp_timsData)
{
  initialize();
}


TimsMsRunReaderBase::~TimsMsRunReaderBase()
{

  msp_timsData = nullptr;
}

void
TimsMsRunReaderBase::initialize()
{
  if(msp_timsData == nullptr)
    msp_timsData =
      std::make_shared<TimsData>(mcsp_msRunId.get()->getFileName());

  if(msp_timsData.get() == nullptr)
    {
      throw PappsoException(
        QObject::tr("ERROR in TimsMsRunReaderBase::initialize "
                    "msp_timsData is null for MsRunId %1")
          .arg(mcsp_msRunId.get()->toString()));
    }
}


bool
TimsMsRunReaderBase::accept(const QString &file_name) const
{
  qDebug() << file_name;
  return true;
}

bool
TimsMsRunReaderBase::hasScanNumbers() const
{
  return false;
}


bool
TimsMsRunReaderBase::releaseDevice()
{
  msp_timsData = nullptr;
  return true;
}

bool
TimsMsRunReaderBase::acquireDevice()
{
  if(msp_timsData == nullptr)
    {
      initialize();
    }
  return true;
}

TimsDataSp
TimsMsRunReaderBase::getTimsDataSPtr()
{
  acquireDevice();
  return msp_timsData;
}


Trace
TimsMsRunReaderBase::getTicChromatogram()
{
  // Use the Sqlite database to fetch the total ion current chromatogram (TIC
  // chromatogram).

  acquireDevice();

  // The time unit here is seconds, not minutes!!!
  return msp_timsData->getTicChromatogram();
}


std::vector<double>
pappso::TimsMsRunReaderBase::getRetentionTimeLine()
{
  return msp_timsData.get()->getRetentionTimeLineInSeconds();
}


XicCoordSPtr
TimsMsRunReaderBase::newXicCoordSPtrFromSpectrumIndex(
  std::size_t spectrum_index [[maybe_unused]],
  pappso::PrecisionPtr precision [[maybe_unused]]) const
{
  throw ExceptionNotImplemented(QObject::tr("Not implemented %1 %2 %3")
                                  .arg(__FILE__)
                                  .arg(__FUNCTION__)
                                  .arg(__LINE__));
}

XicCoordSPtr
TimsMsRunReaderBase::newXicCoordSPtrFromQualifiedMassSpectrum(
  const pappso::QualifiedMassSpectrum &mass_spectrum [[maybe_unused]],
  pappso::PrecisionPtr precision [[maybe_unused]]) const
{
  throw ExceptionNotImplemented(QObject::tr("Not implemented %1 %2 %3")
                                  .arg(__FILE__)
                                  .arg(__FUNCTION__)
                                  .arg(__LINE__));
}


} // namespace pappso
