///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file BondsDataChannel.h
 * \brief Contains the definition of the AtomViz::BondsDataChannel class.
 */

#ifndef __BONDS_DATA_CHANNEL_H
#define __BONDS_DATA_CHANNEL_H

#include <atomviz/AtomViz.h>
#include "DataChannel.h"

#include <core/scene/animation/controller/Controller.h>

namespace AtomViz {

/**
 * \brief This data channel stores the bonds of atoms to their neighbors.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT BondsDataChannel : public DataChannel
{
	Q_OBJECT

public:

	/// \brief Serialization constructor that creates an empty data channel.
	///
	/// \note This constructor is only used when a data channel is loaded from a scene file.
	///       It must not be used to create a new data channel.
	explicit BondsDataChannel(bool isLoading = false);

	/// \brief Constructor that creates a standard data channel.
	/// \param which Specifies which standard data channel should be created.
	///              This should be DataChannelIdentifier::BondsChannel.
	/// \param maxBonds The maximum number of bonds per atom.
	///
	/// Data type, component count and name are automatically chosen by this constructor.
	BondsDataChannel(DataChannelIdentifier which, size_t maxBonds = 16);

	/// \brief Renders the channel' contents in a viewport.
	/// \param time The current animation time.
	/// \param vp The viewport into which the channel should be rendered.
	/// \param atoms The AtomsObject to which this DataChannel belongs to.
	/// \param contextNode The scene object the AtomsObject belongs to.
	virtual void render(TimeTicks time, Viewport* vp, AtomsObject* atoms, ObjectNode* contextNode);

	/// \brief Renders the channel's contents in high-quality mode to an offscreen buffer.
	/// \param time The current animation time.
	/// \param atoms The AtomsObject to which this DataChannel belongs to.
	/// \param view Describes the projection parameters.
	/// \param contextNode The scene object the AtomsObject belongs to.
	/// \param imageWidth The width of the image buffer in pixels.
	/// \param imageHeight The height of the image buffer in pixels.
	/// \param glcontext The OpenGL rendering context.
	virtual void renderHQ(TimeTicks time, AtomsObject* atoms, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext);

	/// \brief Lets the channel clear all its internal caches.
	///
	/// This method is automatically invoked for each DataChannel by the AtomsObject::invalidate()
	/// method. It informs the data channel that the AtomsObject it belongs to has
	/// changed in some way.
	virtual void clearCaches();

	/// \brief Returns the bond display width.
	FloatType bondWidth() const { return _bondWidth ? _bondWidth->getCurrentValue() : (FloatType)0.3; }

	/// Returns whether bonds are rendered in flat shading mode.
	bool flatBonds() const { return _flatBonds; }

public:

	/// \brief Removes all bonds stored in this channel.
	void clearBonds();

	/// \brief Adds a half bond.
	bool setHalfBond(int fromAtom, int bondIndex, int toAtom) {
		if(bondIndex >= (int)componentCount()) return false;
		setIntComponent(fromAtom, bondIndex, toAtom);
		return true;
	}

private:

	/// Copies the contents from the given source channel into this channel.
	/// Atoms for which the bit in the given mask is set are skipped.
	virtual void filterCopy(DataChannel* source, const dynamic_bitset<>& mask);

protected:

	/// Initializes the object. This is called from the constructors.
	void init(bool isLoading);

	/// Renders the atomic bonds as thin lines.
	void renderBondsLines(TimeTicks time, AtomsObject* atoms, bool isPerspective, Vector3 viewDir, const Point3& viewPoint, Viewport* vp);

	/// Renders the atomic bonds as shaded cylinders.
	void renderBondsShaded(TimeTicks time, AtomsObject* atoms, bool isPerspective, Vector3 viewDir, const Point3& viewPoint, Viewport* vp);

	/// Renders the atomic bonds as flat rectangles.
	void renderBondsFlat(TimeTicks time, AtomsObject* atoms, bool isPerspective, Vector3 viewDir, const Point3& viewPoint, Viewport* vp);

	/// Controls the display width of the bonds in world units.
	ReferenceField<FloatController> _bondWidth;

	/// Controls whether solid or flat bonds are being rendered.
	PropertyField<bool> _flatBonds;

	DECLARE_SERIALIZABLE_PLUGIN_CLASS(BondsDataChannel)
	DECLARE_REFERENCE_FIELD(_bondWidth)
	DECLARE_PROPERTY_FIELD(_flatBonds)
};

/**
 * \brief A properties editor for the BondsDataChannel class.
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT BondsDataChannelEditor : public PropertiesEditor
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(BondsDataChannelEditor)
};

};	// End of namespace AtomViz

#endif // __BONDS_DATA_CHANNEL_H
