///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportPanel.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <core/gui/ApplicationManager.h>

namespace Core {

/// The singleton instance of this class.
ViewportManager* ViewportManager::_singletonInstance = NULL;

/******************************************************************************
* Initializes the viewport manager.
******************************************************************************/
ViewportManager::ViewportManager() :
	_activeViewport(NULL), _maximizedViewport(NULL), forceCompleteUpdate(false),
	viewportSuspendCount(1), viewportsNeedUpdate(false)
{
}

/******************************************************************************
* Sets the active viewport.
******************************************************************************/
void ViewportManager::setActiveViewport(Viewport* vp)
{
	if(vp == _activeViewport) return;
	_activeViewport = vp;
	updateViewports(true);
	if(APPLICATION_MANAGER.guiMode()) {
		MAIN_FRAME->viewportPanel()->update();
	}
}

/******************************************************************************
* Sets the maximized viewport.
******************************************************************************/
void ViewportManager::setMaximizedViewport(Viewport* vp)
{
	if(vp == _maximizedViewport) return;
	_maximizedViewport = vp;

    Q_FOREACH(Viewport* viewport, viewports()) {
		if(_maximizedViewport == NULL || viewport == _maximizedViewport)
			viewport->setVisible(true);
		else
			viewport->setVisible(false);
	}

	// Also make it the active viewport.
	if(vp) setActiveViewport(vp);

	if(APPLICATION_MANAGER.guiMode()) {
		ViewportPanel* vpPanel = MAIN_FRAME->viewportPanel();
		CHECK_POINTER(vpPanel);
		updateViewports(true);
	}
}

/******************************************************************************
* This flags all viewports for redrawing.
******************************************************************************/
void ViewportManager::updateViewports(bool forceCompleteUpdate)
{
	// Ignore update request that are made during an update.
	if(forceCompleteUpdate == false && isRendering())
		return;

	this->forceCompleteUpdate = this->forceCompleteUpdate || forceCompleteUpdate;
	if(viewportSuspendCount > 0) {
		viewportsNeedUpdate = true;
		return;
	}
	viewportsNeedUpdate = false;

	if(APPLICATION_MANAGER.guiMode()) {

		MAIN_FRAME->viewportPanel()->layoutViewports();

		Q_FOREACH(Viewport* vp, viewports()) {
			vp->updateViewport(this->forceCompleteUpdate);
		}
	}
	this->forceCompleteUpdate = false;
}

/******************************************************************************
* This immediately redraws the viewports reflecting all
* changes made to the scene.
******************************************************************************/
void ViewportManager::processViewportUpdates()
{
	if(viewportSuspendCount > 0) {
		return;
	}

	if(APPLICATION_MANAGER.guiMode()) {
		Window3D::processWindowUpdates();
	}
}

/******************************************************************************
* Return true if there is currently a rendering operation going on.
* No new windows or dialogs should be shown during this phase
* to prevent an infinite update loop.
******************************************************************************/
bool ViewportManager::isRendering() const
{
	if(!APPLICATION_MANAGER.guiMode()) return false;

	ViewportPanel* vpPanel = MAIN_FRAME->viewportPanel();
	return vpPanel->isRendering();
}

/******************************************************************************
* This will resume redrawing of the viewports after a call to suspendViewportUpdates().
******************************************************************************/
void ViewportManager::resumeViewportUpdates()
{
	OVITO_ASSERT(viewportSuspendCount > 0);
	viewportSuspendCount--;
	if(viewportSuspendCount == 0 && viewportsNeedUpdate) {
		updateViewports();
	}
}

/******************************************************************************
* This gets a viewport configuration that should be
* used as template for new scene files.
******************************************************************************/
ViewportConfiguration::SmartPtr ViewportManager::defaultConfiguration()
{
	// Make sure the default configuration is initialized.
	if(!defaultConfig) {
		defaultConfig = new ViewportConfiguration();
		defaultConfig->addViewport(new ViewportRecord(Viewport::VIEW_TOP, Viewport::SHADING_WIREFRAME, false, DEFAULT_ORTHOGONAL_FIELD_OF_VIEW, IDENTITY, false));
		defaultConfig->addViewport(new ViewportRecord(Viewport::VIEW_FRONT, Viewport::SHADING_WIREFRAME, false, DEFAULT_ORTHOGONAL_FIELD_OF_VIEW, AffineTransformation::rotationX(-FLOATTYPE_PI/2.0), false));
		defaultConfig->addViewport(new ViewportRecord(Viewport::VIEW_LEFT, Viewport::SHADING_WIREFRAME, false, DEFAULT_ORTHOGONAL_FIELD_OF_VIEW, AffineTransformation::rotationZ(FLOATTYPE_PI/2.0)*AffineTransformation::rotationY(FLOATTYPE_PI/2.0), false));
		defaultConfig->addViewport(new ViewportRecord(Viewport::VIEW_PERSPECTIVE, Viewport::SHADING_SHADED, false, DEFAULT_PERSPECTIVE_FIELD_OF_VIEW, AffineTransformation::lookAt(Point3(70,-100,80), ORIGIN, Vector3(0,0,1)), false));
		defaultConfig->setActiveViewport(0);
		defaultConfig->setMaximizedViewport(-1);
	}
	return defaultConfig;
}

/******************************************************************************
* Returns all viewports of the main viewport panel.
******************************************************************************/
const QVector<Viewport*>& ViewportManager::viewports() const
{
	static QVector<Viewport*> emptyList;
	if(APPLICATION_MANAGER.guiMode())
		return MAIN_FRAME->viewportPanel()->viewports();
	else
		return emptyList;
}

};
