/*
 *   Copyright 2009 - 2010 by Giulio Camuffo <giuliocamuffo@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as
 *   published by the Free Software Foundation; either version 2, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "wolflabel.h"

#include <QtGui/QStyleFactory>
#include <QtCore/QTimeLine>
#include <QtGui/QGraphicsItemAnimation>
#include <QtGui/QGraphicsSceneResizeEvent>
#include <QtGui/QLabel>

#include <KIO/Job>
#include <KIO/NetAccess>
#include <KTemporaryFile>
#include <KStandardDirs>

#include <Plasma/Label>

#include "scrollanimation.h"
#include <QPainter>
#include <qstyleoption.h>

static const int SPACING = 10;

WolfLabel::WolfLabel(QGraphicsWidget *parent)
    : QGraphicsWidget(parent),
      m_animation(0),
      m_font(QFont()),
      m_image(QPixmap()),
      m_resizeFont(true),
      m_text(QString()),
      m_textDirty(true),
      m_textPos(QPointF()),
      m_xOffset(0)
{
    setFlag(QGraphicsItem::ItemClipsToShape, true);
}

WolfLabel::~WolfLabel()
{
}

void WolfLabel::setText(const QString &text)
{
    m_text = text;
    m_textDirty = true;
    m_image = QPixmap();

    update();
}

void WolfLabel::setImage(const QString &path)
{
    m_text = QString();
    m_textDirty = true;

    KUrl url(path);
    QString nocover = KGlobal::dirs()->findResource("data", QString("amarok/images/nocover.png"));
    KTemporaryFile tmpFile;
    if (tmpFile.open()) {
        KIO::Job* copyJob = KIO::file_copy( url, KUrl( tmpFile.fileName() ), -1, KIO::Overwrite | KIO::HideProgressInfo );
            if (KIO::NetAccess::synchronousRun(copyJob, 0)) {
                m_image.load( tmpFile.fileName() );
            } else {
                m_image.load( nocover );
            }
    } else {
        m_image.load( nocover );
    }

    update();
}

void WolfLabel::clear()
{
    m_text = QString();
    m_image = QPixmap();

    update();
}

void WolfLabel::resizeEvent(QGraphicsSceneResizeEvent *event)
{
    Q_UNUSED(event)

    if (!m_text.isEmpty()) {
        m_textDirty = true;
    }
}

bool WolfLabel::isResizeFont() const
{
    return m_resizeFont;
}

void WolfLabel::setResizeFont(bool resize, int min)
{
    m_resizeFont = resize;
    m_minFont = min;
    m_textDirty = true;

    update();
}

int WolfLabel::minimumFont() const
{
    return m_minFont;
}

void WolfLabel::setFont(const QFont &customFont)
{
    m_font = customFont;
}

QFont WolfLabel::font() const
{
    return m_font;
}

bool WolfLabel::contains(const QPointF &point) const
{
    QRectF rect = geometry();

    return ((point.x() >= rect.x()) && (point.x() <= (rect.x() + rect.width())) &&
            (point.y() >= rect.y()) && (point.y() <= (rect.y() + rect.height())));
}

void WolfLabel::setXOffset(qreal offset)
{
    m_xOffset = offset;

    update();
}

qreal WolfLabel::xOffset() const
{
    return m_xOffset;
}

void WolfLabel::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_UNUSED(option)
    Q_UNUSED(widget)

    if (!m_text.isEmpty()) {
        if (m_textDirty) {
            m_textDirty = false;

            m_xOffset = 0;
            m_domesticatedFont = m_font;

            QFontMetricsF fm = QFontMetricsF(m_domesticatedFont);
            QSizeF stringSize(fm.width(m_text), fm.height());
            qreal y = (size().height() - stringSize.height()) / 2.0;
            while ((y < 0) && (m_domesticatedFont.pointSize() > 3)) {
                m_domesticatedFont.setPointSize(m_domesticatedFont.pointSize() - 1);
                fm = QFontMetricsF(m_domesticatedFont);
                stringSize.setHeight(fm.height());
                stringSize.setWidth(fm.width(m_text));
                y = (size().height() - stringSize.height()) / 2.0;
            }

            qreal x = boundingRect().size().width() - stringSize.width();
            if (m_resizeFont) {
                while ((x < 0) && (m_domesticatedFont.pointSize() > m_minFont)) {
                    m_domesticatedFont.setPointSize(m_domesticatedFont.pointSize() - 1);
                    fm = QFontMetricsF(m_domesticatedFont);
                    stringSize.setWidth(fm.width(m_text));
                    stringSize.setHeight(fm.height());
                    x = boundingRect().size().width() - stringSize.width();
                }
            }

            m_textPos = QPointF(0, size().height() - y);

            if (x < 0) {
                if (!m_animation) {
                    m_animation = new ScrollAnimation(this);
                    m_animation->setPropertyName("xOffset");
                    m_animation->setTargetObject(this);
                }
                m_animation->setDuration((2 * abs(x) + SPACING) * 150);
                m_animation->setNewStartValue(x  - SPACING);
                m_animation->setNewEndValue(SPACING);
                if (m_animation->state() != QAbstractAnimation::Running) {
                    m_animation->start();
                }
            } else {
                if (m_animation && (m_animation->state() != QAbstractAnimation::Stopped)) {
                    m_animation->stop();
                }
                m_xOffset = x / 2.0;
            }
        }

        painter->setFont(m_domesticatedFont);
        painter->drawText(m_textPos + QPointF(m_xOffset, 0), m_text);
    } else if (!m_image.isNull()) {
        QPixmap scaledPixmap(m_image.scaled(size().width(), size().height(), Qt::KeepAspectRatio, Qt::SmoothTransformation));
        QSize pixmapSize(scaledPixmap.size());
        QPoint pos((size().width() - pixmapSize.width()) / 2.0, (size().height() - pixmapSize.height()) / 2.0);

        painter->drawPixmap(QRect(pos, scaledPixmap.size()), scaledPixmap);
    }
}

#include "wolflabel.moc"
