# -*- coding: utf-8 -*-


# PyMeeus: Python module implementing astronomical algorithms.
# Copyright (C) 2018  Dagoberto Salazar
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.


from math import sin, cos, tan, acos, atan2, sqrt, radians, log10

from pymeeus.Angle import Angle
from pymeeus.Epoch import Epoch, JDE2000
from pymeeus.Interpolation import Interpolation
from pymeeus.Coordinates import (
    geometric_vsop_pos, apparent_vsop_pos, orbital_elements,
    nutation_longitude, true_obliquity, ecliptical2equatorial,
    passage_nodes_elliptic
)
from pymeeus.Earth import Earth
from pymeeus.Sun import Sun


"""
.. module:: Saturn
   :synopsis: Class to model Saturn planet
   :license: GNU Lesser General Public License v3 (LGPLv3)

.. moduleauthor:: Dagoberto Salazar
"""


VSOP87_L = [
    # L0
    [
        [87401354.029, 0.00000000000, 0.00000000000],
        [11107659.780, 3.96205090194, 213.29909543800],
        [1414150.958, 4.58581515873, 7.11354700080],
        [398379.386, 0.52112025957, 206.18554843720],
        [350769.223, 3.30329903015, 426.59819087600],
        [206816.296, 0.24658366938, 103.09277421860],
        [79271.288, 3.84007078530, 220.41264243880],
        [23990.338, 4.66976934860, 110.20632121940],
        [16573.583, 0.43719123541, 419.48464387520],
        [14906.995, 5.76903283845, 316.39186965660],
        [15820.300, 0.93808953760, 632.78373931320],
        [14609.562, 1.56518573691, 3.93215326310],
        [13160.308, 4.44891180176, 14.22709400160],
        [15053.509, 2.71670027883, 639.89728631400],
        [13005.305, 5.98119067061, 11.04570026390],
        [10725.066, 3.12939596466, 202.25339517410],
        [5863.207, 0.23657028777, 529.69096509460],
        [5227.771, 4.20783162380, 3.18139373770],
        [6126.308, 1.76328499656, 277.03499374140],
        [5019.658, 3.17787919533, 433.71173787680],
        [4592.541, 0.61976424374, 199.07200143640],
        [4005.862, 2.24479893937, 63.73589830340],
        [2953.815, 0.98280385206, 95.97922721780],
        [3873.696, 3.22282692566, 138.51749687070],
        [2461.172, 2.03163631205, 735.87651353180],
        [3269.490, 0.77491895787, 949.17560896980],
        [1758.143, 3.26580514774, 522.57741809380],
        [1640.183, 5.50504966218, 846.08283475120],
        [1391.336, 4.02331978116, 323.50541665740],
        [1580.641, 4.37266314120, 309.27832265580],
        [1123.515, 2.83726793572, 415.55249061210],
        [1017.258, 3.71698151814, 227.52618943960],
        [848.643, 3.19149825839, 209.36694217490],
        [1087.237, 4.18343232481, 2.44768055480],
        [956.752, 0.50740889886, 1265.56747862640],
        [789.205, 5.00745123149, 0.96320784650],
        [686.965, 1.74714407827, 1052.26838318840],
        [654.470, 1.59889331515, 0.04818410980],
        [748.811, 2.14398149298, 853.19638175200],
        [633.980, 2.29889903023, 412.37109687440],
        [743.584, 5.25276954625, 224.34479570190],
        [852.677, 3.42141350697, 175.16605980020],
        [579.857, 3.09259007048, 74.78159856730],
        [624.904, 0.97046831256, 210.11770170030],
        [529.861, 4.44938897119, 117.31986822020],
        [542.643, 1.51824320514, 9.56122755560],
        [474.279, 5.47527185987, 742.99006053260],
        [448.542, 1.28990416161, 127.47179660680],
        [546.358, 2.12678554211, 350.33211960040],
        [478.054, 2.96488054338, 137.03302416240],
        [354.944, 3.01286483030, 838.96928775040],
        [451.827, 1.04436664241, 490.33408917940],
        [347.413, 1.53928227764, 340.77089204480],
        [343.475, 0.24604039134, 0.52126486180],
        [309.001, 3.49486734909, 216.48048917570],
        [322.185, 0.96137456104, 203.73786788240],
        [372.308, 2.27819108625, 217.23124870110],
        [321.543, 2.57182354537, 647.01083331480],
        [330.196, 0.24715617844, 1581.95934828300],
        [249.116, 1.47010534421, 1368.66025284500],
        [286.688, 2.37043745859, 351.81659230870],
        [220.225, 4.20422424873, 200.76892246580],
        [277.775, 0.40020408926, 211.81462272970],
        [204.500, 6.01082206600, 265.98929347750],
        [207.663, 0.48349820488, 1162.47470440780],
        [208.655, 1.34516255304, 625.67019231240],
        [182.454, 5.49122292426, 2.92076130680],
        [226.609, 4.91003163138, 12.53017297220],
        [207.659, 1.28302218900, 39.35687591520],
        [173.914, 1.86305806814, 0.75075952540],
        [184.690, 3.50344404958, 149.56319713460],
        [183.511, 0.97254952728, 4.19278569400],
        [146.068, 6.23102544071, 195.13984817330],
        [164.541, 0.44005517520, 5.41662597140],
        [147.526, 1.53529320509, 5.62907429250],
        [139.666, 4.29450260069, 21.34064100240],
        [131.283, 4.06828961903, 10.29494073850],
        [117.283, 2.67920400584, 1155.36115740700],
        [149.299, 5.73594349789, 52.69019803950],
        [122.373, 1.97588777199, 4.66586644600],
        [113.747, 5.59427544714, 1059.38193018920],
        [102.702, 1.19748124058, 1685.05212250160],
        [118.156, 5.34072933900, 554.06998748280],
        [109.275, 3.43812715686, 536.80451209540],
        [110.399, 0.16604024090, 1.48447270830],
        [124.969, 6.27737805832, 1898.35121793960],
        [89.949, 5.80392934702, 114.13847448250],
        [103.956, 2.19210363069, 88.86568021700],
        [112.437, 1.10502663534, 191.20769491020],
        [106.570, 4.01156608514, 956.28915597060],
        [91.430, 1.87521577510, 38.13303563780],
        [83.791, 5.48810655641, 0.11187458460],
        [83.461, 2.28972767279, 628.85158605010],
        [96.987, 4.53666595763, 302.16477565500],
        [100.631, 4.96513666539, 269.92144674060],
        [75.491, 2.18045274099, 728.76296653100],
        [96.330, 2.83319189210, 275.55052103310],
        [82.363, 3.05469876064, 440.82528487760],
        [73.888, 5.08914205084, 1375.77379984580],
        [71.633, 5.10940743430, 65.22037101170],
        [70.409, 4.86846451411, 0.21244832110],
        [69.760, 3.71029022489, 14.97785352700],
        [88.772, 3.86334563977, 278.51946644970],
        [68.090, 0.73415460990, 1478.86657406440],
        [66.501, 0.02677580336, 70.84944530420],
        [65.682, 2.02165559602, 142.44965013380],
        [75.765, 1.61410487792, 284.14854074220],
        [63.153, 3.49493353034, 479.28838891550],
        [62.539, 2.58713611532, 422.66603761290],
        [69.313, 3.43979731402, 515.46387109300],
        [79.021, 4.45154941586, 35.42472265210],
        [63.664, 3.31749528708, 62.25142559510],
        [52.939, 5.51392725227, 0.26063243090],
        [53.011, 3.18480701697, 8.07675484730],
        [54.492, 2.45674090515, 22.09140052780],
        [50.514, 4.26749346978, 99.16062095550],
        [55.170, 0.96797446150, 942.06206196900],
        [49.288, 2.38641424063, 1471.75302706360],
        [47.199, 2.02515248245, 312.19908396260],
        [61.080, 1.50295092063, 210.85141488320],
        [45.126, 0.93109376473, 2001.44399215820],
        [60.556, 2.68715551585, 388.46515523820],
        [43.452, 2.52602011714, 288.08069400530],
        [42.544, 3.81793980322, 330.61896365820],
        [39.915, 5.71378652900, 408.43894361130],
        [50.145, 6.03164759907, 2214.74308759620],
        [45.860, 0.54229721801, 212.33588759150],
        [54.165, 0.78154835399, 191.95845443560],
        [47.016, 4.59934671151, 437.64389113990],
        [42.362, 1.90070070955, 430.53034413910],
        [39.722, 1.63259419913, 1066.49547719000],
        [36.345, 0.84756992711, 213.34727954780],
        [35.468, 4.18603772925, 215.74677599280],
        [36.344, 3.93295730315, 213.25091132820],
        [38.005, 0.31313803095, 423.41679713830],
        [44.746, 1.12488341174, 6.15033915430],
        [37.902, 1.19795851115, 2.70831298570],
        [43.402, 1.37363944007, 563.63121503840],
        [43.764, 3.93043802956, 525.49817940060],
        [34.825, 1.01566605408, 203.00415469950],
        [31.755, 1.69273634405, 0.16005869440],
        [30.880, 6.13525703832, 417.03696332040],
        [36.388, 6.00586032647, 18.15924726470],
        [29.032, 1.19660544505, 404.50679034820],
        [32.812, 0.53649479713, 107.02492748170],
        [30.433, 0.72335287989, 222.86032299360],
        [32.644, 0.81204701486, 1795.25844372100],
        [37.769, 3.69666903716, 1272.68102562720],
        [27.679, 1.45663979401, 7.16173111060],
        [27.187, 1.89731951902, 1045.15483618760],
        [37.699, 4.51997049537, 24.37902238820],
        [34.885, 4.46095761791, 214.26230328450],
        [32.650, 0.66372395761, 692.58748435350],
        [30.324, 5.30369950147, 33.94024994380],
        [27.480, 6.22702216249, 1.27202438720],
        [26.657, 4.56713198392, 7.06536289100],
        [31.745, 5.49798599565, 56.62235130260],
        [28.050, 5.64447420566, 128.95626931510],
        [24.277, 3.93966553574, 414.06801790380],
        [32.017, 5.22260660455, 92.04707395470],
        [26.976, 0.06705123981, 205.22234059070],
        [22.974, 3.65817751770, 207.67002114550],
        [31.775, 5.59198119173, 6069.77675455340],
        [23.153, 2.10054506119, 1788.14489672020],
        [31.025, 0.37190053329, 703.63318461740],
        [29.376, 0.14742155778, 131.40394986990],
        [22.562, 5.24009182383, 212.77783057620],
        [26.185, 5.41311252822, 140.00196957900],
        [25.673, 4.36038885283, 32.24332891440],
        [20.392, 2.82413909260, 429.77958461370],
        [20.659, 0.67091805084, 2317.83586181480],
        [24.397, 3.08740396398, 145.63104387150],
        [23.735, 2.54365387567, 76.26607127560],
        [20.157, 5.06708675157, 617.80588578620],
        [23.307, 3.97357729211, 483.22054217860],
        [22.878, 6.10452832642, 177.87437278590],
        [22.978, 3.20140795404, 208.63322899200],
        [20.638, 5.22128727027, 6.59228213900],
        [21.446, 0.72034565528, 1258.45393162560],
        [18.034, 6.11382719947, 210.37833413120],
        [22.380, 5.92299908546, 173.94221952280],
        [19.128, 5.77772013766, 213.82036029980],
        [20.871, 5.79126331864, 2531.13495725280],
        [19.327, 1.64147367403, 565.11568774670],
        [16.806, 3.27953583323, 98.89998852460],
        [20.833, 2.01655935909, 860.30992875280],
        [17.939, 3.14329498012, 831.85574074960],
        [15.653, 3.10137669623, 106.27416795630],
        [18.235, 5.22595172482, 73.29712585900],
        [19.302, 5.93947114050, 425.11371816770],
        [14.514, 2.75049388379, 1.22384027740],
        [14.562, 5.18795088579, 305.34616939270],
        [14.254, 3.88079504939, 54.17467074780],
        [14.594, 3.25016810034, 78.71375183040],
        [13.637, 2.55486219141, 405.25754987360],
        [13.914, 1.72356993808, 69.15252427480],
        [13.689, 2.37430586272, 125.98732389850],
        [13.496, 0.82683590985, 99.91138048090],
        [18.483, 0.73171264866, 9999.98645077300],
        [13.542, 3.58584380924, 234.63973644040],
        [13.741, 6.18458356845, 245.54242435240],
        [16.944, 0.72200792996, 2111.65031337760],
        [17.441, 0.23803796878, 134.58534360760],
        [14.181, 4.51963935804, 59.80374504030],
        [13.598, 2.53776983965, 1.69692102940],
        [12.240, 2.11973445754, 28.31117565130],
        [11.988, 1.62114832786, 1361.54670584420],
        [11.974, 4.07378735120, 280.96714700450],
        [12.758, 5.31146919749, 344.70304530790],
        [16.051, 3.97093160336, 355.74874557180],
        [11.427, 5.51123470805, 192.69216761850],
        [13.133, 4.69168003518, 767.36908292080],
        [14.746, 3.28998910617, 1589.07289528380],
        [11.417, 1.81615681635, 2104.53676637680],
        [11.626, 2.79410384978, 362.86229257260],
        [13.234, 4.16642914717, 225.82926841020],
        [10.599, 5.50554288376, 199.28444975750],
        [10.558, 3.57501718639, 1.43628859850],
        [10.485, 2.84462532686, 85.82729883120],
        [10.296, 0.22225264071, 198.32124191100],
        [10.552, 0.18716643576, 217.49188113200],
        [11.853, 0.11584857323, 7.63481186260],
        [10.248, 0.21904154170, 144.14657116320],
        [10.403, 1.68776321208, 31.01948863700],
        [10.313, 4.72132701805, 216.21985674480],
        [10.719, 2.60869377832, 339.28641933650],
        [9.636, 3.66746262954, 212.54833591260],
        [9.631, 3.34275630477, 223.59403617650],
        [9.684, 0.41556436593, 2634.22773147140],
        [9.885, 4.01798130416, 207.14875628370],
        [13.212, 6.00683506785, 214.78356814630],
        [11.346, 2.61898383052, 7.86430652620],
        [9.158, 5.39855118256, 342.25536475310],
        [11.882, 4.00188476744, 267.47376618580],
        [12.054, 3.59904816676, 124.43341522100],
        [8.921, 4.22716773496, 6.36278747540],
        [10.142, 3.60807025662, 14.01464568050],
        [9.350, 0.72255756005, 347.88443904560],
        [10.529, 2.36779614951, 831.10498122420],
        [8.587, 4.48439552745, 1692.16566950240],
        [10.142, 3.93620624488, 207.88246946660],
        [9.147, 4.28032835242, 312.45971639350],
        [8.088, 0.81225752596, 264.50482076920],
        [9.241, 4.26402650779, 20.60692781950],
        [9.614, 0.64291347187, 218.92816973050],
        [8.537, 0.48756672382, 1574.84580128220],
        [7.986, 4.71088791079, 333.65734504400],
        [8.951, 0.90641577433, 497.44763618020],
        [7.959, 2.73277594136, 4.14460158420],
        [9.133, 5.08250578843, 241.61027108930],
        [9.669, 1.60623316904, 0.89377187730],
        [8.883, 5.55491009279, 2847.52682690940],
        [8.926, 5.80857835271, 329.72519178090],
        [7.226, 0.60164771281, 206.23373254700],
        [7.655, 5.53676341721, 116.42609634290],
        [7.118, 0.18747501525, 209.10630974400],
        [7.507, 5.43555636173, 621.73803904930],
        [8.885, 5.36210591059, 343.21857259960],
        [7.056, 0.41911130648, 756.32338265690],
        [8.124, 4.05571025939, 237.67811782620],
        [8.964, 1.65023927130, 210.33015002140],
        [6.961, 3.17855200943, 543.02428721890],
        [8.916, 0.56503620503, 2428.04218303420],
        [6.926, 3.66869171435, 247.23934538180],
        [8.982, 4.25046722481, 46.47042291600],
        [7.089, 5.14399672225, 231.45834270270],
        [7.381, 1.25092810119, 217.96496188400],
        [7.134, 2.83090354854, 1148.24761040620],
        [6.353, 0.82582711056, 31.49256938900],
        [7.558, 5.62617378543, 518.64526483070],
        [6.383, 3.54809945181, 244.31858407500],
        [6.914, 3.70012837706, 206.13736432740],
        [6.286, 5.79144749096, 179.35884549420],
        [6.639, 4.55197585824, 120.35824960600],
        [5.823, 1.40737990571, 214.04985496340],
        [5.850, 4.86725483749, 320.32402291970],
        [6.213, 1.07959478499, 251.43213107580],
        [7.730, 3.82244175824, 380.12776796000],
        [5.716, 1.34909972549, 1677.93857550080],
        [6.469, 1.34776801494, 188.92007304980],
        [5.668, 2.28643368177, 20.44686912510],
        [6.092, 3.62275289839, 1169.58825140860],
        [5.711, 0.51687421521, 148.07872442630],
        [5.804, 1.54831552984, 2420.92863603340],
        [5.703, 5.05993483230, 2.96894541660],
        [5.913, 1.66225477547, 842.15068148810],
        [7.449, 1.36195943673, 166.82867252200],
        [6.482, 1.94032041024, 357.44566660120],
        [6.368, 2.44556930837, 654.12438031560],
        [6.327, 0.40654591365, 168.05251279940],
        [5.573, 2.69383455663, 750.10360753340],
        [7.216, 2.22547711392, 488.84961647110],
        [6.701, 6.03737590382, 160.60889739850],
        [6.938, 5.78362034410, 700.66423920080],
        [6.701, 3.14738404371, 491.81856188770],
        [5.684, 2.59531540359, 1.64453140270],
        [4.900, 2.03902856851, 0.80314915210],
        [5.147, 4.10182033298, 196.62432088160],
        [4.985, 2.96765983996, 258.87574647670],
        [5.911, 1.81507526918, 252.65597135320],
        [6.056, 3.33431010543, 182.27960680100],
        [6.195, 5.01900871714, 273.10284047830],
        [6.316, 5.49053160191, 206.70681329900],
        [5.529, 3.31498938717, 1905.46476494040],
        [5.102, 3.98171453610, 254.94359321360],
        [4.762, 2.24463685255, 635.96513305090],
        [5.213, 0.53609344278, 135.54855145410],
        [4.639, 0.04466373027, 213.18722085340],
        [5.951, 0.54565487490, 51.20572533120],
        [4.535, 0.16088614438, 2950.61960112800],
        [4.639, 4.73769153591, 213.41097002260],
        [4.716, 3.13636467789, 28.57180808220],
        [4.748, 1.12156952989, 6.21977512350],
        [5.735, 0.04425142145, 348.84764689210],
        [4.334, 2.68814219154, 81.75213321620],
        [4.538, 3.83676888638, 487.36514376280],
        [5.582, 3.63486861028, 248.72381809010],
        [4.106, 3.39164360376, 50.40257617910],
        [5.657, 3.59967787362, 282.45161971280],
        [5.145, 1.33329458239, 173.68158709190],
        [3.898, 4.11804949361, 213.51154375910],
        [3.898, 0.66430577257, 213.08664711690],
        [4.418, 0.10784811796, 905.88657979150],
        [4.935, 2.19060382431, 189.72322220190],
        [3.799, 2.60752583205, 546.95644048200],
        [3.960, 1.60339889010, 218.71572140940],
        [3.740, 3.30724497407, 274.06604832480],
        [3.778, 0.26606330942, 636.71589257630],
        [4.657, 0.37532078548, 2744.43405269080],
        [3.682, 5.11587898667, 458.84151979040],
        [4.230, 5.18313062329, 27.08733537390],
        [5.181, 3.75590784411, 3127.31333126180],
        [3.904, 2.21738744557, 358.93013930950],
        [4.784, 4.60666675927, 72.07328558160],
        [3.552, 3.23789349146, 543.91805909620],
        [3.502, 3.68869576093, 41.64449777560],
        [4.803, 4.73553427126, 240.38643081190],
        [3.768, 3.86077796242, 2008.55753915900],
        [3.680, 5.36657425183, 10.03430830760],
        [4.298, 3.15595944154, 738.79727483860],
        [3.388, 0.73176365772, 11.30633269480],
        [3.507, 2.62508475661, 13.33332212430],
        [3.552, 0.28967392251, 1891.23767093880],
        [3.604, 4.69324090480, 295.05122865420],
        [3.621, 6.25264336426, 129.91947716160],
        [3.334, 5.04221806054, 153.49535039770],
        [3.837, 5.31732096284, 3163.91869656600],
        [3.281, 5.59031570352, 2.00573757010],
        [4.042, 2.37081308090, 176.65053250850],
        [3.500, 2.54744268360, 1464.63948006280],
        [4.144, 5.46982520458, 6.90109867970],
        [3.691, 4.07518441665, 969.62247809490],
        [3.947, 4.27108449197, 181.80652604900],
        [3.867, 5.48643386310, 37.87240320690],
        [3.339, 6.05372370584, 9.40116886120],
        [3.484, 5.81097824751, 13.49338081870],
        [3.033, 2.38897886651, 221.37585028530],
        [2.990, 4.13995939326, 66.70484372000],
        [3.746, 5.29902286106, 561.18353448360],
        [3.233, 4.27743802321, 593.42686339800],
        [3.170, 1.75400477770, 235.39049596580],
        [4.114, 2.01006788412, 601.76425067620],
        [2.937, 4.76351448561, 213.55972786890],
        [2.932, 1.83671373509, 501.37978944330],
        [2.937, 0.01884528825, 213.03846300710],
        [3.268, 4.44653949711, 60.76695288680],
        [3.608, 0.14307251176, 552.58551477450],
        [2.947, 0.74753671556, 17.52610781830],
        [3.979, 0.76931722276, 424.15051032120],
        [2.803, 1.07518176128, 1994.33044515740],
        [2.905, 1.27201007426, 2737.32050569000],
        [3.610, 0.22394084000, 121.25202148330],
        [2.846, 5.11748545179, 205.66428357540],
        [3.156, 2.74955723696, 494.26624244250],
        [3.576, 4.49826302447, 167.08930495290],
        [2.746, 0.66908290712, 7.00167241620],
        [2.780, 2.10066625279, 894.84087952760],
        [2.875, 2.39009721774, 151.04766984290],
        [3.020, 0.25475826890, 40.84134862350],
        [2.731, 3.74814908509, 429.04587143080],
        [2.793, 4.17938837230, 292.01284726840],
        [2.706, 5.34438894925, 327.43756992050],
        [2.965, 0.61653881148, 643.82943957710],
        [2.616, 4.81901387560, 681.54178408960],
        [2.548, 3.78162580820, 1485.98012106520],
        [3.483, 5.76091147029, 141.22580985640],
        [3.257, 0.75722680616, 555.55446019110],
        [2.887, 6.15899159727, 425.63498302950],
        [2.450, 1.29619859767, 193.65537546500],
        [3.401, 2.48137843009, 43.28902917830],
        [3.208, 0.66002842340, 776.93031047640],
        [2.435, 4.58097103726, 477.80391620720],
        [2.577, 1.41538858001, 100.64509366380],
        [2.600, 3.73139519973, 17.40848773930],
        [2.428, 1.04400815278, 1279.79457262800],
        [2.569, 5.36004101928, 7.22542158540],
        [2.844, 2.47228767650, 280.00393915800],
        [2.847, 1.52706408796, 17.26547538740],
        [2.461, 2.73899140465, 172.24529849340],
        [3.228, 4.10258705369, 618.55664531160],
        [2.288, 0.18365494079, 426.64637498580],
        [2.952, 3.97748947007, 650.94298657790],
        [2.653, 0.14255829255, 162.89651925890],
        [2.291, 3.26940117011, 426.55000676620],
        [3.118, 2.80941831445, 2221.85663459700],
        [2.343, 4.24349768377, 113.38771495710],
        [2.780, 4.36271946528, 130.44074202340],
        [2.539, 5.58396427573, 381.35160823740],
        [2.673, 2.74210116623, 45.57665103870],
        [3.017, 3.72208070740, 228.27694896500],
        [2.781, 0.36312756349, 8.59801970910],
        [2.377, 4.49193242045, 25.12978191360],
        [2.140, 5.43424670725, 630.33605875840],
        [2.456, 1.71617205116, 313.68355667090],
        [2.071, 2.40453395841, 16.46232623530],
        [2.050, 6.19704773331, 3267.01147078460],
        [2.764, 0.40107063007, 219.44943459230],
        [2.307, 2.61462153778, 26.82670294300],
        [2.650, 0.05892373791, 5856.47765911540],
        [1.974, 2.15890150781, 746.92221379570],
        [1.949, 3.13157993205, 226.63241756230],
        [2.063, 0.75916097286, 472.17484191470],
        [2.172, 1.41622302638, 23.57587323610],
        [2.378, 3.45446288811, 241.87090352020],
        [2.314, 2.92766120608, 170.76082578510],
        [2.409, 1.55291842382, 112.65400177420],
        [2.092, 4.33481587531, 210.59078245230],
        [1.883, 4.75777119721, 115.62294719080],
        [1.963, 5.63940648232, 454.90936652730],
        [1.871, 2.14579836453, 135.33610313300],
        [2.304, 0.11816226543, 3060.82592234740],
        [2.221, 4.34506511014, 556.51766803760],
        [1.867, 5.70943358261, 19.12245511120],
        [2.269, 3.36100653157, 696.51963761660],
        [2.127, 0.44754929310, 216.00740842370],
        [1.807, 6.15427316170, 5.84152261360],
        [2.213, 3.42223891884, 533.62311835770],
        [1.866, 3.90535444843, 220.36445832900],
        [1.767, 0.94232357739, 213.45915413240],
        [1.767, 3.84003619647, 213.13903674360],
        [1.910, 3.72504487558, 104.05598206510],
        [1.750, 0.82378244287, 220.46082654860],
        [1.838, 0.06310147657, 436.15941843160],
        [2.146, 4.41415180481, 184.09414790940],
        [1.730, 2.21039276178, 416.30325013750],
        [1.715, 0.26601715797, 103.14095832840],
        [1.710, 0.63515407580, 181.05576652360],
        [2.307, 3.29544714308, 569.04784100980],
        [1.906, 5.30639447218, 405.99126305650],
        [1.863, 4.68613642432, 286.59622129700],
        [1.873, 2.26516020863, 1781.03134971940],
        [2.035, 3.85188859267, 672.14061522840],
        [2.236, 3.01959133214, 105.54045477340],
        [1.767, 1.45800271562, 16.67477455640],
        [1.633, 0.16030477876, 18.91000679010],
        [2.116, 2.90186702031, 486.40193591630],
        [2.202, 3.88125957017, 427.56139872250],
        [1.706, 3.35213628354, 103.04459010880],
        [1.604, 2.48973273967, 55.65914345610],
        [1.744, 1.83791106739, 1044.40407666220],
        [1.569, 6.10089581118, 106.01353552540],
        [2.081, 6.03810192844, 916.93228005540],
        [1.799, 5.01592570405, 731.94436026870],
        [1.737, 1.49651330833, 25.86349509650],
        [1.695, 3.53314158403, 627.36711334180],
        [1.543, 0.81384993001, 2310.72231481400],
        [1.896, 3.38169845451, 2324.94940881560],
        [1.926, 4.66519027283, 353.30106501700],
        [1.765, 5.14740716994, 107.28555991260],
        [1.556, 1.12431826916, 230.82520325630],
        [1.843, 0.02435960281, 102.12956637210],
        [1.501, 4.18415120927, 194.17664032680],
        [1.528, 1.00328674046, 3053.71237534660],
        [1.529, 5.58893570479, 212.02707105080],
        [1.684, 5.08547245125, 3480.31056622260],
        [1.461, 2.31020597821, 721.64941953020],
        [1.480, 5.34331643017, 418.52143602870],
        [1.601, 5.53623000915, 391.17346822390],
        [1.893, 3.62340803433, 204.70107572890],
        [1.529, 6.06535432009, 77.96299230500],
        [1.529, 5.47660937625, 214.57111982520],
        [1.552, 2.06693539836, 36.64856292950],
        [1.453, 6.04709831442, 165.60483224460],
        [1.393, 2.28253369060, 403.02231763990],
        [1.444, 2.90650214018, 447.93883187840],
        [1.924, 1.37028714759, 468.24268865160],
        [1.426, 0.13255011458, 2207.62954059540],
        [1.389, 2.21739183113, 643.07868005170],
        [1.365, 1.63853880518, 629.60234557550],
        [1.362, 3.35131049142, 93.53154666300],
        [1.376, 5.36989538450, 180.16199464630],
        [1.584, 0.85642767335, 271.40591944890],
        [1.405, 5.69231057947, 25.27279426550],
        [1.681, 5.30308110734, 835.03713448730],
        [1.598, 3.04233449432, 42.53826965290],
        [1.759, 3.59043066940, 508.35032409220],
        [1.394, 4.55070863290, 426.07692601420],
        [1.314, 1.81147178081, 1382.88734684660],
        [1.281, 4.26508388040, 123.53964334370],
        [1.742, 5.71133189432, 22.89454967990],
        [1.483, 1.84687831602, 289.56516671360],
        [1.257, 3.01131200921, 409.92341631960],
        [1.285, 4.41168551011, 558.00214074590],
        [1.355, 3.87115897452, 1802.37199072180],
        [1.333, 0.08474224795, 411.62033734900],
        [1.235, 4.08060394635, 28.45418800320],
        [1.373, 5.06955106471, 427.11945573780],
        [1.565, 2.32953532704, 41.05379694460],
        [1.656, 6.06169130804, 268.43697403230],
        [1.212, 3.05966957556, 420.96911658350],
        [1.238, 5.25936700679, 412.58354519550],
        [1.220, 3.92987038126, 2.66012887590],
        [1.552, 1.48184004773, 9786.68735533500],
        [1.240, 1.46716327302, 291.26208774300],
        [1.133, 5.39046583617, 423.67742956920],
        [1.319, 5.79905891015, 1108.13997496560],
        [1.329, 0.92291650117, 778.41478318470],
        [1.399, 2.55906860098, 421.93232443000],
        [1.120, 3.86777259232, 1033.35837639830],
        [1.164, 4.10048660918, 685.47393735270],
        [1.321, 1.45843550806, 1073.60902419080],
        [1.313, 0.11761534168, 71.81265315070],
        [1.438, 2.57741975416, 100.38446123290],
        [1.190, 5.63379509659, 5.10780943070],
        [1.289, 5.20604565993, 278.25883401880],
        [1.157, 5.00101860101, 230.56457082540],
        [1.233, 2.70207317014, 282.66406803390],
        [1.209, 4.02230498135, 980.66817835880],
        [1.070, 5.17569455055, 313.21047591890],
        [1.292, 4.30946655209, 219.89137757700],
        [1.399, 2.58476795858, 2538.24850425360],
        [1.038, 0.14212199680, 820.05928096030],
        [1.245, 4.08278897130, 525.75881183150],
        [1.254, 2.46275017735, 457.61767951300],
        [1.021, 1.11239421009, 69.36497259590],
        [1.009, 1.01709171385, 143.93412284210],
        [1.075, 2.39196853318, 48.75804477640],
        [1.180, 6.18938910429, 3377.21779200400],
        [0.989, 5.94928603657, 3583.40334044120],
        [0.972, 4.25434114756, 397.39324334740],
        [0.983, 0.04442608551, 140.96517742550],
        [0.972, 5.67683107883, 422.40540518200],
        [1.298, 1.34524469231, 875.83029900100],
        [1.190, 0.67933974618, 699.70103135430],
        [0.950, 2.66964340700, 92.30770638560],
        [0.933, 0.63000656580, 406.95447090300],
        [0.959, 1.77556884452, 67.66805156650],
        [1.185, 3.70140604185, 285.63301345050],
        [0.956, 5.18928530992, 319.31263096340],
        [1.014, 1.97449310063, 2097.42321937600],
        [1.048, 3.69659410655, 117.91056905120],
        [1.153, 2.53320305623, 104.57724692690],
        [1.258, 2.51536062507, 694.07195706180],
        [0.971, 5.19147635849, 240.12579838100],
        [0.940, 3.94701776697, 35.21227433100],
        [1.047, 6.12360979460, 238.90195810360],
        [1.185, 5.28289734361, 638.41281360570],
        [0.893, 0.95364488395, 14.81779483260],
        [1.094, 3.93009679240, 945.24345570670],
        [0.949, 3.46451925897, 443.86366626340],
        [1.002, 3.18639902867, 337.73251065900],
        [1.017, 2.87111101661, 211.60217440860],
        [0.875, 0.58638080067, 2.28762186040],
        [0.925, 1.54981519784, 19.64371997300],
        [1.152, 1.68528608590, 691.10301164520],
        [0.832, 2.64637256467, 436.89313161450],
        [0.834, 3.88913521570, 331.20966448920],
        [0.825, 2.15437872210, 739.80866679490],
        [0.848, 3.18263239100, 196.03362005060],
        [1.044, 3.87842686803, 532.61172640140],
        [0.846, 5.38853773752, 97.67614824720],
        [1.021, 2.93075488512, 184.98791978670],
        [0.843, 6.12012061227, 616.32141307790],
        [0.820, 1.01380400969, 480.77286162380],
        [0.842, 3.56523575381, 421.18156490460],
        [1.076, 3.28234305253, 5.67725840230],
        [0.808, 2.78227865672, 212.07525516060],
        [0.812, 0.94281737163, 108.72184851110],
        [0.808, 2.21202653278, 610.69233878540],
        [0.808, 2.00008111713, 214.52293571540],
        [0.875, 2.14897461363, 114.39910691340],
        [0.791, 2.35474255596, 1.37259812370],
        [0.960, 0.25496742364, 710.74673161820],
        [1.001, 2.34471240227, 16.04163511000],
        [0.994, 0.63700664871, 84.93352695390],
        [0.985, 2.63664920104, 395.57870223900],
        [0.874, 2.59112594967, 418.00017116690],
        [0.758, 0.58117487362, 2627.11418447060],
        [0.894, 1.48976897396, 760.25553592000],
        [0.768, 5.25095392845, 305.08553696180],
        [1.020, 2.73153988233, 268.95823889410],
        [0.903, 0.13224671457, 238.42887735160],
        [0.750, 0.76128043194, 724.83081326790],
        [0.903, 3.37105323370, 526.50957135690],
        [0.930, 2.83622594110, 2641.34127847220],
        [0.808, 5.23759255053, 216.26804085460],
        [0.864, 1.22059443823, 570.74476203920],
        [0.798, 3.72388187653, 124.50285119020],
        [0.753, 0.72747041757, 3370.10424500320],
        [0.844, 2.03251767810, 511.53171782990],
        [0.827, 4.49936223096, 444.75743814070],
        [0.848, 3.74330244183, 2118.76386037840],
        [0.795, 0.27939057139, 101.86893394120],
        [0.754, 5.68583497533, 662.53120356300],
        [0.750, 4.78778128003, 102.57150935680],
        [0.709, 1.64518562815, 159.12442469020],
        [0.770, 4.02404991950, 909.81873305460],
        [0.765, 1.62693133597, 465.95506679120],
        [0.911, 5.04635658282, 913.96333463880],
        [0.861, 2.78971410809, 495.75071515080],
        [0.688, 1.10207467005, 1.53686233500],
        [0.803, 4.33043919090, 453.42489381900],
        [0.673, 0.03439333853, 2524.02141025200],
        [0.856, 3.50639182375, 439.12836384820],
        [0.684, 3.93906807606, 337.80194662820],
        [0.716, 6.18909854987, 310.71461125430],
        [0.922, 1.70634200320, 125.18417474640],
        [0.885, 1.69955870744, 6283.07584999140],
        [0.656, 1.28102954508, 432.01481684740],
        [0.808, 1.64410808383, 299.12639426920],
        [0.656, 1.49449164620, 849.26422848890],
        [0.679, 4.02962984490, 429.51895218280],
        [0.854, 3.04068731741, 298.23262239190],
        [0.676, 2.17631477883, 576.16138801060],
        [0.881, 5.47733557925, 220.93390730060],
        [0.739, 5.85330901725, 938.12990870590],
        [0.637, 2.96294462433, 425.84743135060],
        [0.665, 1.62998758015, 221.16340196420],
        [0.693, 3.53871697600, 1182.92157353290],
        [0.659, 1.85319023888, 72.33391801250],
        [0.631, 2.01234919422, 58.10682401090],
        [0.806, 5.21763933753, 428.08266358430],
        [0.804, 5.94281804567, 26.02355379090],
        [0.679, 2.11784460940, 256.42806592190],
        [0.692, 1.88791537515, 214.99601646740],
        [0.731, 1.95762888351, 19.01058052660],
        [0.786, 0.91252523635, 518.38463239980],
        [0.684, 4.89288171806, 3796.70243587920],
        [0.612, 2.08511492036, 1038.04128918680],
        [0.598, 3.48814927085, 219.66188291340],
        [0.828, 0.31689472200, 25.60286266560],
        [0.773, 4.57524006328, 624.91943278700],
        [0.775, 6.12920077021, 432.22726516850],
        [0.820, 4.11320326130, 141.48644228730],
        [0.588, 1.95775535412, 211.86280683950],
        [0.661, 5.30100397707, 103.61403908040],
        [0.588, 2.82460441973, 214.73538403650],
        [0.651, 5.56732715834, 393.46109008430],
        [0.564, 4.01666572198, 850.01498801430],
        [0.657, 2.58166087726, 526.98265210890],
        [0.722, 0.68494219480, 953.10776223290],
        [0.592, 2.37190662490, 205.43478891180],
        [0.738, 1.07981019512, 239.16259053450],
        [0.590, 6.03587790219, 188.02630117250],
        [0.559, 5.76010635813, 430.79097657000],
        [0.535, 5.80499883199, 100.17201291180],
        [0.611, 5.95439360100, 3693.60966166060],
        [0.591, 4.22379888536, 505.31194270640],
        [0.691, 2.96568305933, 606.76018552230],
        [0.648, 2.33387623043, 30.75885620610],
        [0.544, 2.21686115865, 92.79783348010],
        [0.517, 2.68282421083, 262.05714021440],
        [0.563, 1.69735688719, 2413.81508903260],
        [0.569, 2.85075508949, 227.31374111850],
        [0.531, 2.17713708433, 263.02034806090],
        [0.504, 4.44947885193, 343.73983746140],
        [0.578, 3.31462999801, 33.72780162270],
        [0.619, 1.83409636127, 867.42347575360],
        [0.546, 4.82201187196, 1048.33622992530],
        [0.517, 1.42016110098, 1246.65747183630],
        [0.681, 1.94124532036, 25874.60404613620],
        [0.551, 5.70617358907, 1119.18567522950],
        [0.523, 5.78878978812, 366.79444583570],
        [0.486, 1.90063955671, 1063.31408345230],
        [0.552, 3.64325031166, 256.58812461630],
        [0.612, 2.39349965241, 2854.64037391020],
        [0.495, 3.46833581240, 597.35901666110],
        [0.622, 1.86539391351, 524.01370669230],
        [0.480, 5.33557742428, 29.20494752860],
        [0.492, 4.64109549618, 384.05992122310],
        [0.520, 2.32171681836, 2957.73314812880],
        [0.545, 0.53274778710, 431.26405732200],
        [0.479, 2.13325177240, 319.57326339430],
        [0.526, 4.17771910249, 136.06981631590],
        [0.612, 4.56148986681, 774.48262992160],
        [0.642, 3.25195912708, 67.88049988760],
        [0.527, 4.00299045889, 2435.15573003500],
        [0.524, 4.69817741494, 336.83873878170],
        [0.495, 5.95703962647, 765.88461021250],
        [0.463, 6.09780322600, 54.33472944220],
        [0.538, 0.22116216124, 450.97721326420],
        [0.465, 1.87487942848, 958.57677783100],
        [0.500, 1.54084756342, 572.22923474750],
        [0.528, 3.54764543325, 233.90602325750],
        [0.451, 5.72664397006, 3899.79521009780],
        [0.514, 5.00509609437, 273.85360000370],
        [0.471, 2.96871670899, 306.83064210100],
        [0.447, 1.87279400375, 62.03897727400],
        [0.456, 5.45521998520, 1171.87587326900],
        [0.499, 1.92821778530, 217.44369702220],
        [0.548, 3.21873307934, 824.74219374880],
        [0.579, 2.29424247924, 810.65811209910],
        [0.550, 0.67875196590, 315.16802937920],
        [0.556, 1.30277646899, 133.10087089930],
        [0.449, 6.05954557724, 141.69889060840],
        [0.447, 5.83737433218, 823.99143422340],
        [0.517, 3.62832879827, 934.94851496820],
        [0.482, 1.04855231956, 1055.44977692610],
        [0.482, 4.07207792722, 195.89060769870],
        [0.428, 0.65142455407, 427.34895040140],
        [0.585, 5.48406138684, 376.19561469690],
        [0.469, 3.87344294455, 320.58465535060],
        [0.488, 2.83523964260, 460.53844081980],
        [0.450, 4.30419652064, 88.11492069160],
        [0.537, 0.85582040238, 214.19286731530],
        [0.438, 1.32216133929, 963.40270297140],
        [0.560, 2.50374228728, 952.09637027660],
        [0.442, 2.80002649649, 209.15449385380],
        [0.443, 4.31062007978, 9992.87290377220],
        [0.469, 0.45235276525, 464.73122651380],
        [0.488, 0.35817443686, 36.90919536040],
        [0.418, 4.81080887468, 775.23338944700],
        [0.417, 4.93943593579, 306.09692891810],
        [0.483, 3.92847922420, 39.61750834610],
        [0.467, 1.89153069484, 30.05628079050],
        [0.410, 5.52148731635, 118.07062774560],
        [0.406, 1.35991757653, 945.99421523210],
        [0.446, 4.06656112480, 380.38840039090],
        [0.466, 3.65954736664, 988.53248488500],
        [0.418, 1.40185532806, 313.94418910180],
        [0.481, 1.80873987903, 43.12897048390],
        [0.437, 0.86746182155, 170.97327410620],
        [0.483, 4.49894122772, 46.20979048510],
        [0.398, 2.90977731924, 131.54696222180],
        [0.529, 3.74604329884, 699.17976649250],
        [0.396, 0.34033987778, 2943.50605412720],
        [0.545, 2.97400965609, 305.60680182360],
        [0.412, 3.81935995126, 84.34282612290],
        [0.425, 2.60672101181, 121.84272231430],
        [0.474, 2.41769418569, 838.21852822500],
        [0.457, 1.27246488727, 107.75864066460],
        [0.519, 3.12247974037, 10213.28554621100],
        [0.495, 4.63705386984, 301.41401612960],
        [0.537, 3.92653937147, 212.40532356070],
        [0.385, 3.33476325050, 806.72595883600],
        [0.477, 1.66001855277, 175.42669223110],
        [0.378, 0.47722247042, 200.55647414470],
        [0.459, 5.14821844812, 960.22130923370],
        [0.401, 4.36420932544, 739.05790726950],
        [0.467, 2.96423984091, 170.01006625970],
        [0.457, 4.45829983338, 33.13710079170],
        [0.476, 3.63257697985, 20.49505323490],
        [0.462, 3.57494442856, 71.60020482960],
        [0.376, 2.94770389653, 6062.66320755260],
        [0.473, 5.32759318114, 373.90799283650],
        [0.383, 1.84111991862, 346.39996633730],
        [0.366, 2.05039240297, 87.31177153950],
        [0.383, 2.00608370504, 3274.12501778540],
        [0.374, 5.65444305031, 540.73666535850],
        [0.410, 5.62629715297, 58.31927233200],
        [0.441, 6.26992749567, 378.90392768260],
        [0.451, 3.05920369737, 898.77303279070],
        [0.371, 5.65787287655, 89.75945209430],
        [0.367, 5.71998148487, 96.87299909510],
        [0.410, 1.06290837961, 1257.70317210020],
        [0.418, 2.03053179312, 146.59425171800],
        [0.492, 0.89529424356, 423.62924545940],
        [0.474, 2.43080822444, 705.11765732570],
        [0.457, 3.42347802916, 829.62050851590],
        [0.419, 0.32183412086, 90.56260124640],
        [0.347, 3.60116807440, 449.23210812500],
        [0.397, 2.29159647723, 782.34693644780],
        [0.433, 3.40938468811, 32.45577723550],
        [0.343, 4.30952656038, 401.32539661050],
        [0.351, 0.42160026295, 3686.49611465980],
        [0.360, 5.83204569879, 491.55792945680],
        [0.389, 2.73429108352, 36.17548217750],
        [0.434, 0.33664392528, 55.13787859430],
        [0.346, 5.09191837323, 392.65794093220],
        [0.336, 2.38696934868, 295.19424100610],
        [0.336, 1.56832822829, 233.74596456310],
        [0.400, 3.08774286568, 745.91082183940],
        [0.468, 4.21298903633, 832.58945393250],
        [0.400, 1.32415028336, 551.10104206620],
        [0.344, 1.52193307438, 754.83890994860],
        [0.433, 3.06696455642, 885.43971066640],
        [0.346, 4.76971433870, 4113.09430553580],
        [0.329, 3.31034285904, 952.35700270750],
        [0.425, 2.90590905341, 462.02291352810],
        [0.342, 5.87738962422, 561.93429400900],
        [0.439, 0.20791179302, 768.85355562910],
        [0.330, 2.08599193524, 614.62449204850],
        [0.394, 3.59805431851, 1261.63532536330],
        [0.369, 6.03075127328, 199.80571461930],
        [0.375, 2.56075851331, 732.69511979410],
        [0.408, 0.69233617038, 328.24071907260],
        [0.331, 4.28333309370, 541.53981451060],
        [0.323, 0.06694179304, 433.75992198660],
        [0.431, 4.50603340920, 2914.01423582380],
        [0.343, 2.18898146246, 80.19822453870],
        [0.437, 5.34124436008, 387.24131496080],
        [0.371, 4.82569229712, 103.35340664950],
        [0.318, 3.75720880396, 749.20983565610],
        [0.344, 0.70749566894, 229.97386999440],
        [0.312, 6.02741985422, 361.37781986430],
        [0.340, 0.64358874512, 303.86169668440],
        [0.394, 0.49801994278, 248.46318565920],
        [0.309, 5.83535487659, 236.19364511790],
        [0.325, 2.65292455786, 757.21715453420],
        [0.365, 0.56788400592, 402.21916848780],
        [0.342, 3.83450185886, 519.39602435610],
        [0.306, 0.35126074897, 354.99798604640],
        [0.314, 5.42086935152, 1151.42900414390],
        [0.384, 0.09367760642, 201.51968199120],
        [0.307, 5.62165090342, 426.48631629140],
        [0.301, 1.78550205039, 1354.43315884340],
        [0.300, 5.23426324539, 190.40454575810],
        [0.296, 3.15801280224, 192.85222631290],
        [0.381, 1.98600461808, 109.94568878850],
        [0.370, 5.57659190517, 562.14674233010],
        [0.305, 2.73187068494, 840.66620877980],
        [0.374, 3.73375594662, 420.44785172170],
        [0.307, 4.03149340189, 426.71006546060],
        [0.320, 1.42665180100, 2730.20695868920],
        [0.392, 0.14343294900, 206.39799675830],
        [0.288, 2.44787565138, 623.22251175760],
        [0.344, 1.57825843162, 6290.18939699220],
        [0.317, 1.84490263693, 214.94362684070],
        [0.346, 2.16849893508, 3171.03224356680],
        [0.286, 1.01623455966, 315.64111013120],
        [0.361, 3.44807605339, 259.76951835400],
        [0.280, 4.64272946177, 254.14044406150],
        [0.285, 2.09430258387, 335.14181775230],
        [0.281, 4.72136141979, 317.14262918200],
        [0.289, 4.70265740009, 29.74746424980],
        [0.290, 2.74360609522, 551.03160609700],
        [0.300, 5.23308974173, 1321.43907040360],
        [0.283, 1.21193797828, 1699.27921650320],
        [0.280, 0.45877292284, 38.60611638980],
        [0.273, 1.81952809851, 1056.20053645150],
        [0.336, 5.63115066542, 95.22846769240],
        [0.309, 3.33676739908, 1193.96727379680],
        [0.288, 2.57603349418, 1166.40685767090],
        [0.277, 3.35359995343, 532.87235883230],
        [0.287, 0.37229115993, 114.94162363460],
        [0.274, 4.21164027953, 90.82323367730],
        [0.267, 5.61367132586, 870.46185713940],
        [0.376, 6.14391456675, 913.00012679230],
        [0.296, 5.75705165982, 4010.00153131720],
        [0.304, 1.97982468634, 495.96316347190],
        [0.308, 5.08911712262, 481.73606947030],
        [0.265, 0.02893016561, 172.45774681450],
        [0.291, 2.10106037916, 619.29035849450],
        [0.273, 4.78343050572, 771.30123618390],
        [0.358, 0.04093867464, 637.44960575920],
        [0.272, 5.86505586319, 332.17287233570],
        [0.266, 3.28626000731, 560.71045373160],
        [0.312, 3.91213951915, 1226.21060271120],
        [0.258, 3.41185135958, 426.81063919710],
        [0.257, 1.06772454181, 714.67888488130],
        [0.258, 6.24129294573, 426.38574255490],
        [0.256, 0.75289096697, 103.84353374400],
        [0.254, 3.21116652124, 102.34201469320],
        [0.254, 6.14632777985, 620.25356634100],
        [0.266, 2.54280196709, 132.88842257820],
        [0.314, 0.31303204249, 991.71387862270],
        [0.317, 2.93589163442, 357.23321828010],
        [0.266, 6.12280636670, 57.51612317990],
        [0.254, 2.99730079627, 642.34496686880],
        [0.267, 5.54663413439, 628.59095361920],
        [0.348, 1.33319249154, 815.06334611420],
        [0.278, 5.59573741920, 334.55111692130],
        [0.303, 3.22789148979, 409.18970313670],
        [0.246, 3.90430259983, 441.57604440300],
        [0.260, 3.86355293530, 639.37602145220],
        [0.250, 0.07635434166, 2840.41327990860],
        [0.246, 5.71009371698, 476.31944349890],
        [0.301, 6.15272106984, 559.69906177530],
        [0.268, 3.73809606804, 658.05653357870],
        [0.316, 4.63035287047, 745.27768239300],
        [0.311, 3.51827727424, 2751.54759969160],
        [0.239, 0.51133317457, 1041.22268292450],
        [0.238, 5.46347279420, 4216.18707975440],
        [0.262, 3.20254407166, 1251.34038462480],
        [0.238, 1.02679111743, 1262.38608488870],
        [0.277, 3.32996713394, 545.47196777370],
        [0.271, 0.49229846069, 419.53282798500],
        [0.303, 3.77087511317, 285.37238101960],
        [0.234, 3.64328189368, 407.47573576480],
        [0.270, 3.93080966194, 313.47110834980],
        [0.302, 3.15201610429, 915.23535902600],
        [0.264, 0.46127639727, 720.89866000480],
        [0.235, 1.74238582338, 369.08206769610],
        [0.284, 5.20345848078, 395.10562148700],
        [0.295, 5.03016292492, 594.65070367540],
        [0.290, 1.94941528794, 907.37105249980],
        [0.229, 0.11049722694, 3259.89792378380],
        [0.304, 1.81516929363, 49.72125262290],
        [0.268, 5.54944666204, 12352.85260454480],
        [0.248, 4.39993603541, 385.54439393140],
        [0.234, 1.63365119249, 3590.51688744200],
        [0.317, 4.74907646809, 420.00590873700],
        [0.228, 4.89094697746, 1181.43710082460],
        [0.258, 2.41510535278, 550.13783421970],
        [0.236, 4.10002766188, 6467.92575796160],
        [0.249, 1.97767956861, 589.49471013490],
        [0.226, 1.60308230318, 316.27999507200],
        [0.286, 6.10513234736, 484.70501488690],
        [0.299, 3.71741328977, 1123.11782849260],
        [0.263, 1.85502475341, 608.87779767700],
        [0.220, 0.94624116595, 316.50374424120],
        [0.292, 3.12099513976, 47.69426319340],
        [0.217, 4.93010394323, 281.17959532560],
        [0.295, 3.18346450625, 1050.99635880120],
        [0.248, 5.48523310249, 638.93407846750],
        [0.276, 1.09325899753, 544.50875992720],
        [0.279, 2.65446123759, 134.11226285560],
        [0.247, 4.17284927352, 950.13881681630],
        [0.212, 2.45582771491, 1164.76232626820],
        [0.266, 2.30028827109, 314.90739694830],
        [0.215, 3.55148746518, 1097.09427470170],
        [0.223, 0.36862624591, 81.89514556810],
        [0.283, 0.35013012349, 1269.49963188950],
        [0.263, 0.58255768951, 386.98068252990],
        [0.208, 4.61303945066, 668.20846196530],
        [0.222, 4.51639912193, 304.12232911530],
        [0.274, 0.63572336701, 679.25416222920],
        [0.215, 0.74685271552, 1008.97935401010],
        [0.268, 3.43326489941, 598.84348936940],
        [0.271, 3.98364990268, 453.68552624990],
        [0.215, 2.24139383226, 661.23792731640],
        [0.264, 2.58516335032, 2527.20280398970],
        [0.205, 1.94472028389, 650.19222705250],
        [0.258, 1.50724315598, 1759.83372106890],
        [0.272, 6.27135287672, 990.22940591440],
        [0.201, 1.11891338895, 97.41551581630],
        [0.236, 4.13547185065, 348.63519857100],
        [0.201, 2.67485193508, 1546.53462563090],
        [0.250, 5.44391862722, 1254.52177836250],
        [0.203, 5.48660442144, 557.03893289940],
        [0.250, 0.32142312427, 25448.00585526019],
        [0.198, 0.91019549387, 1310.39337013970],
        [0.200, 0.90438804926, 47.06112374700],
        [0.194, 4.05730813129, 426.85882330690],
        [0.197, 0.58292199999, 156.67674413540],
        [0.197, 2.59384188520, 639.84910220420],
        [0.201, 1.49980256114, 827.92358748650],
        [0.197, 5.79126360809, 639.94547042380],
        [0.266, 1.45514683392, 109.24311337290],
        [0.239, 4.63640382863, 868.71675200020],
        [0.194, 5.59582424111, 426.33755844510],
        [0.247, 2.91348766248, 689.61853893690],
        [0.205, 1.04616771894, 448.68959140380],
        [0.232, 1.76503818816, 354.26427286350],
        [0.224, 3.45391027762, 1190.03512053370],
        [0.221, 6.27951545913, 1596.18644228460],
        [0.237, 1.24961141369, 882.94384600180],
        [0.206, 5.17010664247, 253.45912050530],
        [0.203, 0.25518217825, 4002.88798431640],
        [0.245, 3.94368773869, 769.81676347560],
        [0.255, 6.11790751550, 763.43692965770],
        [0.194, 2.83197251801, 263.70167161710],
        [0.237, 2.14007373880, 2700.71514038580],
        [0.203, 3.00078001542, 1385.17496870700],
        [0.203, 3.88000057282, 419.43645976540],
        [0.186, 4.79530535895, 843.63515419640],
        [0.203, 0.16707173895, 535.91074021810],
        [0.245, 0.76762638475, 5643.17856367740],
        [0.183, 2.20151176434, 35.16409022120],
        [0.242, 3.41613986919, 864.24208201590],
        [0.249, 3.47840802421, 1045.88854937050],
        [0.199, 4.70077509959, 1276.61317889030],
        [0.189, 1.84247610390, 434.67494572330],
        [0.192, 3.83148077309, 666.72398925700],
        [0.201, 1.34290804373, 1012.91150727320],
        [0.210, 5.12097279511, 3494.53766022420],
        [0.192, 1.36630036429, 904.40210708320],
        [0.195, 1.62629576534, 364.34676528090],
        [0.179, 1.66502999050, 244.79166482700],
        [0.186, 4.55663748319, 347.41135829360],
        [0.201, 0.50219680740, 36.38793049860],
        [0.179, 4.55153097409, 97.46369992610],
        [0.200, 0.68398141887, 2015.67108615980],
        [0.195, 2.20599546209, 66.18357885820],
        [0.186, 4.48925201018, 611.44309831080],
        [0.186, 3.13663606153, 433.66355376700],
        [0.177, 3.62811326217, 326.68681039510],
        [0.206, 0.38552475035, 857.12853501510],
        [0.229, 1.22330638216, 2906.90068882300],
        [0.213, 6.08436923028, 271.61836777000],
        [0.174, 6.18833529233, 3576.28979344040],
        [0.220, 2.89866380776, 322.61164478010],
        [0.202, 2.43755817264, 812.14258480740],
        [0.211, 4.27999470527, 1127.26243007680],
        [0.188, 0.97173474238, 1080.72257119160],
        [0.168, 0.65008064023, 1493.09366806600],
        [0.208, 4.68942071563, 5429.87946823940],
        [0.182, 0.76058628096, 504.56118318100],
        [0.167, 6.22608046965, 108.50940019000],
        [0.176, 2.08816807487, 670.91677495100],
        [0.170, 1.55680702386, 1670.82502850000],
        [0.187, 4.12687876136, 9985.75935677140],
        [0.166, 1.58596354370, 1379.70595310890],
        [0.170, 1.30946662341, 837.69726336320],
        [0.166, 0.02413278965, 224.60542813280],
        [0.204, 6.12072939658, 9360.08916445900],
        [0.189, 0.28191962964, 1175.80802653210],
        [0.184, 2.87461093640, 398.14400287280],
        [0.172, 5.41410693088, 2306.79016155090],
        [0.174, 0.94052578814, 632.03297978780],
        [0.181, 2.20017945285, 1049.08698945070],
        [0.176, 1.09857632660, 531.97858695500],
        [0.201, 2.68411933635, 795.68025857210],
        [0.197, 1.43444932583, 347.36317418380],
        [0.200, 4.33954193534, 1364.72809958190],
        [0.166, 2.83936872840, 3553.91152213780],
        [0.160, 1.28778451384, 962.50893109410],
        [0.206, 3.02518737424, 1141.13406340540],
        [0.200, 5.54644533685, 308.31511480930],
        [0.158, 4.75462437610, 1534.73816584160],
        [0.156, 3.42359004405, 241.75328344120],
        [0.182, 5.26192506263, 968.13800538660],
        [0.168, 0.37055108740, 10007.09999777380],
        [0.174, 4.61279044571, 223.33340374560],
        [0.158, 4.23116519562, 821.70381236300],
        [0.212, 5.98406058023, 432.74853003030],
        [0.160, 4.08093018212, 632.73555520340],
        [0.179, 5.74376301842, 924.04582705620],
        [0.185, 3.01715709315, 55.87159177720],
        [0.185, 0.75718598244, 1286.90811962880],
        [0.170, 3.46903106526, 1304.92435454160],
        [0.167, 3.19767851189, 635.23141986800],
        [0.153, 3.83006208210, 318.67949151700],
        [0.156, 1.61558494091, 110.25450532920],
        [0.206, 1.62702696825, 389.94962794650],
        [0.184, 3.31730160238, 1578.02719501990],
        [0.184, 6.19913208096, 731.68372783780],
        [0.158, 5.58628906712, 42.32582133180],
        [0.159, 2.00727485442, 702.14871190910],
        [0.155, 0.02889842368, 1357.61455258110],
        [0.151, 5.80616069064, 680.05731138130],
        [0.157, 5.67235483479, 77837.11123384659],
        [0.156, 2.78001360754, 1567.73225428140],
        [0.203, 5.91820261629, 971.10695080320],
        [0.156, 5.68482948401, 649.45851386960],
        [0.186, 5.72093247348, 664.27630870220],
        [0.184, 4.05878767739, 976.73602509570],
        [0.159, 3.10978439502, 230.70758317730],
        [0.154, 4.54479813612, 1239.54392483550],
        [0.177, 4.68148789757, 2921.12778282460],
        [0.158, 4.30854435851, 633.74694715970],
        [0.195, 5.33376523453, 1130.23137549340],
        [0.188, 2.04023570497, 1127.04998175570],
        [0.174, 2.88817750136, 25668.41849769900],
        [0.161, 2.96300008890, 152.74459087230],
        [0.170, 1.70779560770, 493.30303459600],
        [0.156, 3.05331862578, 913.75088631770],
        [0.169, 3.18325334507, 757.80785536520],
        [0.145, 0.89672198690, 632.83192342300],
        [0.149, 2.69519361988, 203.26478713040],
        [0.167, 2.96547549589, 1201.83158032300],
        [0.187, 3.17592113403, 842.90144101350],
        [0.173, 1.91528836350, 3487.42411322340],
        [0.154, 0.37169915171, 285.11174858870],
        [0.198, 3.14620903981, 640.86049416050],
        [0.143, 3.42760427364, 520.12973753900],
        [0.157, 2.34913118792, 5959.57043333400],
        [0.178, 5.34566184657, 272.58157561650],
        [0.171, 3.34185265551, 3067.93946934820],
        [0.171, 4.87848878292, 354.52490529440],
        [0.189, 0.07069084691, 1585.89150154610],
        [0.137, 1.28925665885, 214.10224459010],
        [0.137, 3.49438482900, 212.49594628590],
        [0.174, 3.07450555579, 64.95973858080],
        [0.181, 0.78652249647, 657.16276170140],
        [0.158, 2.93809799392, 211.65456403530],
        [0.133, 5.70724173596, 469.72716135990],
        [0.140, 0.35530248121, 219.51887056150],
        [0.132, 1.60614108449, 1372.59240610810],
        [0.164, 1.68236348359, 707.56533788050],
        [0.128, 1.49742950959, 45.24658263860],
        [0.132, 4.98075995751, 238.57188970350],
        [0.140, 3.62845435388, 423.88987789030],
        [0.134, 1.41837795083, 3906.90875709860],
        [0.127, 1.48966646069, 856.37777548970],
        [0.158, 2.57990141197, 369.97583957340],
        [0.132, 2.39662455993, 184.72728735580],
        [0.140, 4.36785639279, 207.07932031450],
        [0.157, 4.56655940808, 251.17149864490],
        [0.125, 4.60433583965, 6076.89030155420],
        [0.121, 1.59513269211, 184.84490743480],
        [0.145, 5.85121885906, 221.89711514710],
        [0.137, 0.76499603885, 476.10699517780],
        [0.139, 6.02467582226, 429.30650386170],
        [0.117, 0.23611722307, 426.75824957040],
        [0.117, 3.13382984215, 426.43813218160],
        [0.116, 4.34801448720, 418.96337901340],
        [0.135, 2.69789181467, 455.16999895820],
        [0.134, 1.19058772772, 502.86426215160],
        [0.123, 3.97560160218, 499.89531673500],
        [0.110, 2.37032413718, 439.93151300030],
        [0.109, 6.20303291896, 220.30076785420],
        [0.109, 1.38979633175, 325.95309721220],
        [0.115, 0.05845336809, 631.82053146670],
        [0.139, 2.21086387259, 9573.38825989700],
        [0.142, 6.14666228712, 3340.61242669980],
        [0.122, 4.77182119272, 604.47256366190],
        [0.119, 3.03461367061, 528.20649238630],
        [0.127, 2.97263950543, 498.93210888850],
        [0.111, 5.01192320232, 220.20019411770],
        [0.126, 3.95743516129, 566.60016045500],
        [0.098, 2.36210526249, 634.26821202150],
        [0.108, 2.46872857126, 83.37961827640],
        [0.101, 4.88259474274, 425.32616648880],
        [0.127, 4.89006771254, 162.09337010680],
        [0.099, 0.90073463816, 586.31331639720],
        [0.099, 0.16310526294, 394.35486196160],
        [0.115, 0.49160291262, 517.16079212240],
        [0.101, 3.86043866859, 198.10879358990],
        [0.105, 3.48222097813, 5863.59120611620],
        [0.104, 4.61148354671, 220.52451702340],
        [0.101, 4.77041950285, 427.87021526320],
        [0.098, 2.90784721214, 199.96577331370],
        [0.127, 0.14136936897, 2332.06295581640],
        [0.091, 6.22628300117, 211.29335786790],
        [0.100, 5.14847283436, 226.79247625670],
        [0.091, 4.84054379386, 215.30483300810],
        [0.099, 4.37583492400, 640.41855117580],
        [0.093, 5.30395179617, 222.70026429920],
        [0.086, 4.57481701854, 636.97652500720],
        [0.111, 0.61102669309, 1089.12939443900],
        [0.086, 4.50969293872, 625.88264063350],
        [0.088, 0.36828594935, 444.12429869430],
        [0.093, 5.76287499885, 203.89792657680],
        [0.082, 2.85558603378, 318.83955021140],
        [0.082, 5.19618475111, 1467.82087380050],
        [0.086, 0.97633784601, 200.03520928290],
        [0.082, 4.78557953017, 195.77298761970],
        [0.106, 2.29520624233, 799.61241183520],
        [0.081, 3.57727166766, 205.97310011610],
        [0.078, 5.50343512580, 262.80789973980],
        [0.087, 0.76830756075, 201.99276274320],
        [0.102, 2.11516755277, 206.93630796260],
        [0.081, 5.29639775054, 111.16952906590],
        [0.075, 2.77117107886, 255.83736509090],
        [0.074, 5.81587984729, 316.44005376640],
        [0.076, 1.78743197928, 171.65459766240],
        [0.094, 4.99996904753, 378.64329525170],
        [0.089, 5.85818860151, 807.94979911340],
        [0.072, 0.99858616883, 280.21638747910],
        [0.099, 0.15018241445, 186.21176006410],
        [0.072, 5.15715918322, 110.15813710960],
    ],
    # L1
    [
        [21354295595.986, 0.00000000000, 0.00000000000],
        [1296855.005, 1.82820544701, 213.29909543800],
        [564347.566, 2.88500136429, 7.11354700080],
        [98323.030, 1.08070061328, 426.59819087600],
        [107678.770, 2.27769911872, 206.18554843720],
        [40254.586, 2.04128257090, 220.41264243880],
        [19941.734, 1.27954662736, 103.09277421860],
        [10511.706, 2.74880392800, 14.22709400160],
        [6939.233, 0.40493079985, 639.89728631400],
        [4803.325, 2.44194097666, 419.48464387520],
        [4056.325, 2.92166618776, 110.20632121940],
        [3768.630, 3.64965631460, 3.93215326310],
        [3384.684, 2.41694251653, 3.18139373770],
        [3302.200, 1.26256486715, 433.71173787680],
        [3071.382, 2.32739317750, 199.07200143640],
        [1953.036, 3.56394683300, 11.04570026390],
        [1249.348, 2.62803737519, 95.97922721780],
        [921.683, 1.96089834250, 227.52618943960],
        [705.587, 4.41689249330, 529.69096509460],
        [649.654, 6.17418093659, 202.25339517410],
        [627.603, 6.11088227167, 309.27832265580],
        [486.843, 6.03998200305, 853.19638175200],
        [468.377, 4.61707843907, 63.73589830340],
        [478.501, 4.98776987984, 522.57741809380],
        [417.010, 2.11708169277, 323.50541665740],
        [407.630, 1.29949556676, 209.36694217490],
        [343.826, 3.95854178574, 412.37109687440],
        [339.724, 3.63396398752, 316.39186965660],
        [335.936, 3.77173072712, 735.87651353180],
        [331.933, 2.86077699882, 210.11770170030],
        [352.489, 2.31707079463, 632.78373931320],
        [289.429, 2.73263080235, 117.31986822020],
        [265.801, 0.54344631312, 647.01083331480],
        [230.493, 1.64428879621, 216.48048917570],
        [280.911, 5.74398845416, 2.44768055480],
        [191.667, 2.96512946582, 224.34479570190],
        [172.891, 4.07695221044, 846.08283475120],
        [167.131, 2.59745202658, 21.34064100240],
        [136.328, 2.28580246629, 10.29494073850],
        [131.364, 3.44108355646, 742.99006053260],
        [127.838, 4.09533471247, 217.23124870110],
        [108.862, 6.16141072262, 415.55249061210],
        [93.909, 3.48397279899, 1052.26838318840],
        [92.482, 3.94755499926, 88.86568021700],
        [97.584, 4.72845436677, 838.96928775040],
        [86.600, 1.21951325061, 440.82528487760],
        [83.463, 3.11269504725, 625.67019231240],
        [77.588, 6.24408938835, 302.16477565500],
        [61.557, 1.82789612597, 195.13984817330],
        [61.900, 4.29344363385, 127.47179660680],
        [67.106, 0.28961738595, 4.66586644600],
        [56.919, 5.01889578112, 137.03302416240],
        [54.160, 5.12628572382, 490.33408917940],
        [54.585, 0.28356341456, 74.78159856730],
        [51.425, 1.45766406064, 536.80451209540],
        [65.843, 5.64757042732, 9.56122755560],
        [57.780, 2.47630552035, 191.95845443560],
        [44.444, 2.70873627665, 5.41662597140],
        [46.799, 1.17721211050, 149.56319713460],
        [40.380, 3.88870105683, 728.76296653100],
        [37.768, 2.53379013859, 12.53017297220],
        [46.649, 5.14818326902, 515.46387109300],
        [45.891, 2.23198878761, 956.28915597060],
        [40.400, 0.41281520440, 269.92144674060],
        [37.191, 3.78239026411, 2.92076130680],
        [33.778, 3.21070688046, 1368.66025284500],
        [37.969, 0.64665967180, 422.66603761290],
        [32.857, 0.30063884563, 351.81659230870],
        [33.050, 5.43038091186, 1066.49547719000],
        [30.276, 2.84067004928, 203.00415469950],
        [35.116, 6.08421794089, 5.62907429250],
        [29.667, 3.39052569135, 1059.38193018920],
        [33.217, 4.64063092111, 277.03499374140],
        [31.876, 4.38622923770, 1155.36115740700],
        [28.913, 2.02614760507, 330.61896365820],
        [28.264, 2.74178953996, 265.98929347750],
        [30.089, 6.18684614308, 284.14854074220],
        [31.329, 2.43455855525, 52.69019803950],
        [26.493, 4.51214170121, 340.77089204480],
        [21.983, 5.14437352579, 4.19278569400],
        [22.230, 1.96481952451, 203.73786788240],
        [20.824, 6.16048095923, 860.30992875280],
        [21.690, 2.67578768862, 942.06206196900],
        [22.552, 5.88579123000, 210.85141488320],
        [19.807, 2.31345263487, 437.64389113990],
        [19.447, 4.76573277668, 70.84944530420],
        [19.310, 4.10209060369, 18.15924726470],
        [22.662, 4.13732273379, 191.20769491020],
        [18.209, 0.90310796389, 429.77958461370],
        [17.667, 1.84954766042, 234.63973644040],
        [17.547, 2.44735118493, 423.41679713830],
        [15.428, 4.23790088205, 1162.47470440780],
        [14.608, 3.59713247857, 1045.15483618760],
        [14.111, 2.94262468353, 1685.05212250160],
        [16.328, 4.05665272725, 949.17560896980],
        [13.348, 6.24509592240, 38.13303563780],
        [15.918, 1.06434204938, 56.62235130260],
        [14.059, 1.43503954068, 408.43894361130],
        [13.093, 5.75815864257, 138.51749687070],
        [15.772, 5.59350835225, 6.15033915430],
        [14.962, 5.77192239389, 22.09140052780],
        [16.024, 1.93900586533, 1272.68102562720],
        [16.751, 5.96673627422, 628.85158605010],
        [12.843, 4.24658666814, 405.25754987360],
        [13.628, 4.09892958087, 1471.75302706360],
        [15.067, 0.74142807591, 200.76892246580],
        [10.961, 1.55022573283, 223.59403617650],
        [11.695, 1.81237511034, 124.43341522100],
        [10.346, 3.46814088412, 1375.77379984580],
        [12.056, 1.85655834555, 131.40394986990],
        [10.123, 2.38221133049, 107.02492748170],
        [9.855, 3.95166998848, 430.53034413910],
        [9.803, 2.55389483994, 99.91138048090],
        [10.614, 5.36692189034, 215.74677599280],
        [12.080, 4.84549317054, 831.85574074960],
        [10.210, 6.07692961370, 32.24332891440],
        [9.245, 3.65417467270, 142.44965013380],
        [8.984, 1.23808405498, 106.27416795630],
        [9.336, 5.81062768434, 7.16173111060],
        [9.717, 1.38703872827, 145.63104387150],
        [8.394, 4.42341211111, 703.63318461740],
        [8.370, 5.64015188458, 62.25142559510],
        [8.244, 2.42225929772, 1258.45393162560],
        [7.784, 0.52562994711, 654.12438031560],
        [7.626, 3.75258725596, 312.19908396260],
        [7.222, 0.28429555677, 0.75075952540],
        [8.236, 6.22250515902, 14.97785352700],
        [7.054, 0.53177810740, 388.46515523820],
        [6.567, 3.48657341701, 35.42472265210],
        [9.011, 4.94919626910, 208.63322899200],
        [8.980, 0.08138173719, 288.08069400530],
        [6.421, 3.32905264657, 1361.54670584420],
        [6.489, 2.89389587598, 114.13847448250],
        [6.244, 0.54973852782, 65.22037101170],
        [6.154, 2.67885860584, 2001.44399215820],
        [6.742, 0.23586769279, 8.07675484730],
        [7.297, 4.85321224483, 222.86032299360],
        [6.302, 3.80651124694, 1788.14489672020],
        [5.824, 4.39327457448, 81.75213321620],
        [6.102, 0.88585782895, 92.04707395470],
        [6.914, 2.04631426723, 99.16062095550],
        [5.363, 5.47995103139, 563.63121503840],
        [5.172, 2.11968421583, 214.26230328450],
        [5.117, 5.76987684107, 565.11568774670],
        [6.197, 1.62553688800, 1589.07289528380],
        [4.970, 0.41949366126, 76.26607127560],
        [6.640, 5.82582210639, 483.22054217860],
        [5.277, 4.57975789757, 134.58534360760],
        [4.974, 4.20243895902, 404.50679034820],
        [5.150, 4.67582673243, 212.33588759150],
        [4.764, 4.59303997414, 554.06998748280],
        [4.573, 3.24875415786, 231.45834270270],
        [4.811, 0.46206327592, 362.86229257260],
        [5.148, 3.33570646174, 1.48447270830],
        [4.654, 5.80233066659, 217.96496188400],
        [4.509, 5.37581684215, 497.44763618020],
        [4.443, 0.11349392292, 295.05122865420],
        [4.943, 3.78020789259, 1265.56747862640],
        [4.211, 4.88306021960, 98.89998852460],
        [4.252, 5.00120115113, 213.34727954780],
        [4.774, 4.53259894142, 1148.24761040620],
        [3.911, 0.58582192963, 750.10360753340],
        [5.069, 2.20305668335, 207.88246946660],
        [3.553, 0.35374030841, 333.65734504400],
        [3.771, 0.98542435766, 24.37902238820],
        [3.458, 1.84990273999, 225.82926841020],
        [3.401, 5.31342401626, 347.88443904560],
        [3.347, 0.21414641376, 635.96513305090],
        [3.637, 1.61315058382, 245.54242435240],
        [3.416, 2.19551489078, 1574.84580128220],
        [3.655, 0.80544245690, 343.21857259960],
        [4.260, 1.80258750109, 213.25091132820],
        [3.110, 3.03815175282, 1677.93857550080],
        [3.052, 1.33858964447, 543.91805909620],
        [3.694, 0.81606028298, 344.70304530790],
        [3.016, 3.36219319026, 7.86430652620],
        [2.937, 4.86927342776, 144.14657116320],
        [2.768, 2.42707131609, 2317.83586181480],
        [3.059, 4.30820099442, 6062.66320755260],
        [3.650, 5.12802531219, 218.92816973050],
        [2.963, 3.53480751374, 2104.53676637680],
        [3.230, 2.88057019783, 216.21985674480],
        [2.984, 2.52971310583, 1692.16566950240],
        [2.897, 5.73256482240, 9992.87290377220],
        [2.591, 3.79880285744, 17.26547538740],
        [3.495, 5.29902525443, 350.33211960040],
        [2.859, 3.72804950659, 6076.89030155420],
        [2.775, 0.23549396237, 357.44566660120],
        [2.976, 2.48769315964, 46.47042291600],
        [2.487, 4.37868078530, 217.49188113200],
        [2.711, 5.15376840150, 10007.09999777380],
        [3.127, 1.92343235583, 17.40848773930],
        [3.181, 1.72419900322, 1169.58825140860],
        [2.348, 0.77373103004, 414.06801790380],
        [2.606, 3.42836913440, 31.01948863700],
        [2.556, 0.91735028377, 479.28838891550],
        [2.399, 4.82440545738, 1279.79457262800],
        [2.245, 3.76323995584, 425.11371816770],
        [3.020, 0.25310250109, 120.35824960600],
        [2.503, 2.10679832121, 168.05251279940],
        [2.564, 1.63158205055, 182.27960680100],
        [2.221, 3.15472373256, 212.77783057620],
        [2.357, 2.33145013714, 218.71572140940],
        [2.510, 4.51903989011, 198.32124191100],
        [2.715, 5.76330259543, 618.55664531160],
        [2.204, 3.35952557362, 160.60889739850],
        [2.648, 0.71962005233, 85.82729883120],
        [2.029, 5.28642331696, 125.98732389850],
        [2.497, 1.36671447252, 1905.46476494040],
        [2.017, 1.11498225426, 447.93883187840],
        [2.052, 1.27587874735, 14.01464568050],
        [2.254, 3.22447674190, 273.10284047830],
        [2.014, 0.39787014152, 358.93013930950],
        [1.981, 2.33696859021, 28.45418800320],
        [2.197, 5.93386789705, 13.33332212430],
        [2.237, 3.64433751164, 213.82036029980],
        [1.930, 1.85671740340, 1.27202438720],
        [2.037, 5.05300562628, 424.15051032120],
        [1.994, 1.35690802366, 20.60692781950],
        [1.911, 3.44106886717, 69.15252427480],
        [1.925, 3.75243031545, 28.31117565130],
        [2.297, 4.24557050896, 1464.63948006280],
        [2.117, 2.25897766314, 116.42609634290],
        [1.847, 5.40631472802, 31.49256938900],
        [1.841, 1.56916484272, 650.94298657790],
        [1.884, 6.27233535258, 25.12978191360],
        [1.960, 4.89484014840, 275.55052103310],
        [2.016, 5.45791785675, 842.15068148810],
        [2.282, 4.96276947440, 258.87574647670],
        [1.709, 3.99098237135, 416.30325013750],
        [2.176, 0.00746756006, 0.89377187730],
        [1.634, 5.30978165487, 251.43213107580],
        [1.687, 0.41586020065, 54.17467074780],
        [1.910, 2.59825755790, 329.72519178090],
        [2.113, 2.56582292726, 59.80374504030],
        [1.921, 2.42279051938, 113.38771495710],
        [1.658, 5.47323651540, 1073.60902419080],
        [1.590, 2.77545297350, 1994.33044515740],
        [1.936, 3.47558926847, 1581.95934828300],
        [1.649, 1.82779010589, 128.95626931510],
        [1.598, 1.71806465300, 129.91947716160],
        [1.967, 1.25160413795, 621.73803904930],
        [1.702, 1.91076102800, 278.51946644970],
        [1.569, 0.16491194947, 643.07868005170],
        [1.989, 5.28799230992, 508.35032409220],
        [1.520, 0.56950979689, 320.32402291970],
        [1.501, 1.99815894193, 1891.23767093880],
        [1.532, 3.27362317849, 2420.92863603340],
        [1.701, 2.72041261115, 767.36908292080],
        [1.561, 6.09424459628, 280.96714700450],
        [1.331, 4.20944443790, 546.95644048200],
        [1.381, 2.06768100830, 192.69216761850],
        [1.368, 6.28049502257, 1795.25844372100],
        [1.519, 2.20299556153, 2008.55753915900],
        [1.356, 4.01521042413, 721.64941953020],
        [1.296, 4.84815978742, 45.57665103870],
        [1.267, 5.28146654999, 173.94221952280],
        [1.402, 6.12951551550, 39.35687591520],
        [1.252, 2.19169926554, 2634.22773147140],
        [1.466, 4.16354845643, 26.82670294300],
        [1.285, 3.76170874847, 2.28762186040],
        [1.500, 5.41022492529, 214.04985496340],
        [1.396, 4.78595583428, 219.44943459230],
        [1.430, 0.70934745161, 254.94359321360],
        [1.195, 3.71281085322, 264.50482076920],
        [1.181, 0.42635230882, 41.64449777560],
        [1.190, 2.02079286787, 1485.98012106520],
        [1.160, 5.23649231796, 181.05576652360],
        [1.535, 3.62746990294, 561.18353448360],
        [1.120, 1.09127922130, 6.59228213900],
        [1.100, 0.27844612141, 184.09414790940],
        [1.227, 1.39969681270, 209.10630974400],
        [1.353, 6.12903657666, 207.67002114550],
        [1.124, 6.05105541765, 291.26208774300],
        [1.194, 4.79565407023, 1478.86657406440],
        [1.082, 4.73602931755, 78.71375183040],
        [1.202, 3.47301104146, 51.20572533120],
        [1.298, 2.34761557822, 210.37833413120],
        [1.166, 4.20037524355, 417.03696332040],
        [1.228, 3.94985981275, 1781.03134971940],
        [1.401, 2.41318931513, 636.71589257630],
        [1.009, 6.17414889934, 2111.65031337760],
        [1.084, 3.68958647346, 274.06604832480],
        [1.068, 0.80258823981, 436.89313161450],
        [1.007, 3.42792508860, 629.60234557550],
        [0.998, 5.57130056835, 205.22234059070],
        [1.058, 1.05742945779, 237.67811782620],
        [1.020, 3.33667290300, 166.82867252200],
        [0.965, 6.08359503243, 601.76425067620],
        [1.005, 3.56310748091, 643.82943957710],
        [0.987, 0.97129012811, 305.34616939270],
        [0.927, 3.87717400791, 135.33610313300],
        [1.129, 5.94840103961, 196.62432088160],
        [1.118, 5.25415059584, 189.72322220190],
        [1.200, 1.16671933467, 2221.85663459700],
        [0.909, 2.14001565047, 617.80588578620],
        [0.899, 2.31811625712, 312.45971639350],
        [1.081, 0.91006048421, 313.21047591890],
        [0.891, 3.74923531791, 916.93228005540],
        [0.886, 4.76066858907, 776.93031047640],
        [0.912, 0.99592540858, 491.81856188770],
        [0.880, 3.67349449376, 25.27279426550],
        [1.203, 1.39749267410, 337.73251065900],
        [0.867, 0.11684071625, 267.47376618580],
        [0.879, 6.12222682852, 867.42347575360],
        [1.080, 0.15038819285, 175.16605980020],
        [0.988, 3.12456192471, 214.78356814630],
        [0.889, 4.70508769146, 148.07872442630],
        [0.827, 6.08977582217, 488.84961647110],
        [0.889, 5.05124166027, 220.46082654860],
        [0.828, 6.27262544155, 1382.88734684660],
        [1.040, 5.76735098196, 501.37978944330],
        [1.103, 0.48706477230, 692.58748435350],
        [0.810, 2.50362385080, 2310.72231481400],
        [0.850, 4.55410385197, 77.96299230500],
        [1.108, 5.31792012163, 235.39049596580],
        [0.790, 0.89213206336, 342.25536475310],
        [0.775, 2.85873930879, 211.81462272970],
        [0.842, 2.99884993009, 2737.32050569000],
        [0.784, 0.05748459240, 543.02428721890],
        [0.754, 5.18317747668, 244.31858407500],
        [0.969, 1.31760425414, 486.40193591630],
        [0.943, 5.48641674428, 339.28641933650],
        [0.759, 6.25347177163, 151.04766984290],
        [0.710, 2.41619968810, 247.23934538180],
        [0.794, 2.59522645936, 1.64453140270],
        [0.857, 1.99318788624, 248.72381809010],
        [0.717, 4.56798357445, 121.25202148330],
        [0.671, 2.50955477476, 444.75743814070],
        [0.683, 5.51033310275, 487.36514376280],
        [0.684, 0.01892628603, 228.27694896500],
        [0.665, 1.47172657769, 427.56139872250],
        [0.761, 4.61079808671, 23.57587323610],
        [0.807, 3.21513718120, 1898.35121793960],
        [0.645, 1.92436523628, 2950.61960112800],
        [0.624, 6.05830190539, 241.61027108930],
        [0.699, 4.02804515616, 425.63498302950],
        [0.624, 5.85966148394, 696.51963761660],
        [0.620, 1.86426453489, 2207.62954059540],
        [0.641, 5.69868017561, 319.57326339430],
        [0.646, 3.78920578728, 1038.04128918680],
        [0.672, 2.54160055954, 271.40591944890],
        [0.768, 1.80484245332, 2324.94940881560],
        [0.737, 1.50539891226, 268.43697403230],
        [0.836, 1.26583811010, 212.54833591260],
        [0.753, 5.27536166240, 204.70107572890],
        [0.633, 2.19920009577, 1802.37199072180],
        [0.720, 2.58587107868, 472.17484191470],
        [0.683, 3.83223866420, 43.28902917830],
        [0.740, 6.21601938401, 556.51766803760],
        [0.795, 1.14460330178, 381.35160823740],
        [0.678, 3.65930963429, 2097.42321937600],
        [0.568, 5.92158661090, 2428.04218303420],
        [0.570, 1.18024241664, 131.54696222180],
        [0.566, 4.74157739398, 380.12776796000],
        [0.586, 5.71168743146, 570.74476203920],
        [0.550, 4.92413290959, 188.92007304980],
        [0.712, 2.69456114358, 16.67477455640],
        [0.545, 5.38725529600, 206.23373254700],
        [0.572, 5.79167804981, 195.89060769870],
        [0.602, 5.81756794592, 963.40270297140],
        [0.588, 4.25026865253, 426.64637498580],
        [0.563, 3.28295055824, 193.65537546500],
        [0.583, 5.44099997963, 526.50957135690],
        [0.679, 4.45748326743, 105.54045477340],
        [0.516, 5.99843937287, 289.56516671360],
        [0.520, 2.19322568805, 180.16199464630],
        [0.543, 4.19333695628, 213.18722085340],
        [0.586, 3.03470168346, 6275.96230299060],
        [0.572, 3.96788877624, 140.00196957900],
        [0.611, 4.15392239870, 436.15941843160],
        [0.505, 2.95739392583, 135.54855145410],
        [0.587, 4.55320395537, 5863.59120611620],
        [0.492, 2.71595874382, 84.93352695390],
        [0.576, 5.98300938454, 9793.80090233580],
        [0.489, 5.68450383182, 533.62311835770],
        [0.519, 3.09688510923, 327.43756992050],
        [0.486, 5.24220804875, 5849.36411211460],
        [0.475, 4.51295931678, 411.62033734900],
        [0.540, 4.44843952768, 10206.17199921020],
        [0.479, 0.87707794164, 207.14875628370],
        [0.468, 0.46572028197, 306.09692891810],
        [0.586, 0.86387928244, 2538.24850425360],
        [0.475, 6.19152982788, 397.39324334740],
        [0.541, 1.47958133221, 42.53826965290],
        [0.496, 6.07879620658, 576.16138801060],
        [0.447, 2.59259132013, 7.22542158540],
        [0.445, 5.06827300470, 778.41478318470],
        [0.560, 0.00461017471, 221.37585028530],
        [0.456, 4.60143715337, 710.74673161820],
        [0.449, 5.79223649465, 685.47393735270],
        [0.501, 1.91370965325, 831.10498122420],
        [0.595, 4.90329839607, 824.74219374880],
        [0.447, 4.88662794571, 429.04587143080],
        [0.445, 1.74764943142, 525.75881183150],
        [0.457, 0.80892712530, 458.84151979040],
        [0.543, 2.60317945475, 213.41097002260],
        [0.493, 0.61947189193, 41.05379694460],
        [0.455, 2.69847252264, 3053.71237534660],
        [0.429, 3.89071982978, 92.79783348010],
        [0.411, 1.34981168865, 27.08733537390],
        [0.448, 1.84775051361, 980.66817835880],
        [0.445, 4.21745990439, 905.88657979150],
        [0.403, 2.33067250642, 2627.11418447060],
        [0.404, 5.00179215709, 431.26405732200],
        [0.384, 1.65634584042, 241.75328344120],
        [0.410, 0.76907037678, 395.57870223900],
        [0.456, 1.98353741244, 213.51154375910],
        [0.459, 2.04878772547, 285.63301345050],
        [0.396, 5.04141834913, 298.23262239190],
        [0.377, 5.68073822097, 2744.43405269080],
        [0.415, 4.41600504868, 179.35884549420],
        [0.396, 4.29872851950, 206.70681329900],
        [0.389, 5.69091953122, 849.26422848890],
        [0.369, 1.36192003466, 835.03713448730],
        [0.374, 0.41402282126, 9779.57380833420],
        [0.379, 1.72255764532, 184.98791978670],
        [0.365, 5.88205574821, 19.64371997300],
        [0.456, 4.81297899859, 213.08664711690],
        [0.359, 1.06819138836, 206.13736432740],
        [0.367, 1.14184327929, 569.04784100980],
        [0.352, 3.04388401587, 638.41281360570],
        [0.463, 1.55834877017, 421.18156490460],
        [0.459, 5.34648461645, 699.70103135430],
        [0.383, 4.05921035379, 739.80866679490],
        [0.354, 1.09760553168, 738.79727483860],
        [0.382, 0.05348541587, 252.65597135320],
        [0.344, 1.18536656224, 439.12836384820],
        [0.382, 2.10483762147, 532.61172640140],
        [0.361, 0.50215018154, 50.40257617910],
        [0.351, 3.49546336297, 1354.43315884340],
        [0.395, 4.26278871560, 432.22726516850],
        [0.345, 2.38455893509, 426.07692601420],
        [0.350, 1.51541607946, 259.76951835400],
        [0.426, 5.29998227949, 934.94851496820],
        [0.339, 5.59774645356, 519.39602435610],
        [0.388, 3.40083809779, 2413.81508903260],
        [0.324, 3.68352014131, 72.07328558160],
        [0.323, 1.79597508586, 405.99126305650],
        [0.366, 3.56764349139, 1119.18567522950],
        [0.358, 4.11241839677, 37.87240320690],
        [0.423, 1.45116702108, 2641.34127847220],
        [0.314, 0.68465789313, 757.21715453420],
        [0.320, 3.12697568936, 945.99421523210],
        [0.338, 4.89782013581, 898.77303279070],
        [0.319, 5.76881401291, 69.36497259590],
        [0.310, 5.35598720822, 815.06334611420],
        [0.369, 4.46143610142, 421.93232443000],
        [0.311, 2.19275640712, 5856.47765911540],
        [0.306, 2.99917010799, 1130.23137549340],
        [0.330, 0.64102961163, 558.00214074590],
        [0.305, 0.40963115602, 661.23792731640],
        [0.320, 3.29267319940, 760.25553592000],
        [0.298, 5.48693246086, 702.14871190910],
        [0.352, 2.18179692198, 2118.76386037840],
        [0.299, 5.94980651345, 572.22923474750],
        [0.343, 2.62900083650, 213.55972786890],
        [0.296, 4.12563821701, 73.29712585900],
        [0.360, 2.94387423457, 2214.74308759620],
        [0.293, 5.71837797264, 60.76695288680],
        [0.326, 1.93806509331, 480.77286162380],
        [0.335, 2.60120542851, 518.38463239980],
        [0.322, 2.89685459163, 427.11945573780],
        [0.367, 2.20489848330, 518.64526483070],
        [0.361, 3.31464351282, 630.33605875840],
        [0.288, 0.87760478150, 887.72733252680],
        [0.290, 0.24071300709, 705.11765732570],
        [0.332, 5.96464701829, 100.64509366380],
        [0.284, 1.58760551116, 681.54178408960],
        [0.281, 1.68339116394, 3267.01147078460],
        [0.287, 3.54730637851, 756.32338265690],
        [0.331, 2.74250642576, 22.89454967990],
        [0.281, 4.79802388453, 409.92341631960],
        [0.372, 1.08754087151, 426.55000676620],
        [0.340, 0.59629116557, 627.36711334180],
        [0.325, 4.07319450014, 511.53171782990],
        [0.273, 0.71334827688, 305.08553696180],
        [0.272, 1.76124839309, 945.24345570670],
        [0.295, 4.00327005783, 432.74853003030],
        [0.271, 5.28903262032, 1080.72257119160],
        [0.276, 3.89192411657, 610.69233878540],
        [0.294, 2.80121651058, 724.83081326790],
        [0.319, 5.24824059915, 229.97386999440],
        [0.264, 2.36406383589, 731.94436026870],
        [0.288, 4.67818844930, 170.76082578510],
        [0.326, 3.81328980623, 525.49817940060],
        [0.283, 3.52027709716, 319.31263096340],
        [0.264, 0.25871603855, 494.26624244250],
        [0.261, 4.08135671345, 25.86349509650],
        [0.296, 4.49129913731, 693.55069220000],
        [0.292, 0.65370180027, 25867.49049913539],
        [0.292, 0.12510953311, 25881.71759313700],
        [0.254, 4.03912322565, 990.22940591440],
        [0.288, 3.98604904657, 707.77778620160],
        [0.285, 1.92328297431, 3134.42687826260],
        [0.284, 2.45411523294, 3120.19978426100],
        [0.256, 3.63282757780, 430.79097657000],
        [0.283, 2.51091647682, 286.59622129700],
        [0.325, 4.33261281211, 732.69511979410],
        [0.264, 0.05450228136, 650.19222705250],
        [0.273, 4.90735780421, 409.18970313670],
        [0.304, 4.61759348542, 468.24268865160],
        [0.285, 5.72467903890, 33.94024994380],
        [0.242, 5.28336514054, 403.02231763990],
        [0.270, 0.51583145648, 263.70167161710],
        [0.263, 4.81670787366, 1055.44977692610],
        [0.237, 2.92617048443, 913.96333463880],
        [0.246, 2.19675150666, 2943.50605412720],
        [0.278, 4.58404840578, 398.14400287280],
        [0.234, 2.64374114605, 739.05790726950],
        [0.229, 3.80445074468, 58.10682401090],
        [0.300, 2.06111081979, 429.51895218280],
        [0.223, 3.39888651505, 188.02630117250],
        [0.301, 2.96411385108, 624.91943278700],
        [0.221, 1.79137414078, 2524.02141025200],
        [0.220, 0.95686592581, 1894.41906467650],
        [0.225, 4.30669421945, 637.44960575920],
        [0.214, 1.70442143644, 658.05653357870],
        [0.227, 3.22613053351, 638.93407846750],
        [0.220, 2.66798936385, 953.10776223290],
        [0.253, 3.09377787768, 29.20494752860],
        [0.244, 3.15828383212, 7.00167241620],
        [0.295, 4.95843934543, 714.67888488130],
        [0.209, 0.94525938634, 864.24208201590],
        [0.216, 0.12221236180, 28.57180808220],
        [0.214, 2.80190604605, 373.90799283650],
        [0.212, 2.07343849515, 1357.61455258110],
        [0.216, 1.25531205533, 477.80391620720],
        [0.206, 5.35971491902, 3060.82592234740],
        [0.204, 3.08579410460, 67.66805156650],
        [0.210, 1.91489853604, 938.12990870590],
        [0.209, 1.46554109301, 952.35700270750],
        [0.202, 3.57670882297, 334.55111692130],
        [0.228, 5.66209641464, 1699.27921650320],
        [0.197, 4.61055255182, 464.73122651380],
        [0.193, 4.24606721746, 141.69889060840],
        [0.266, 0.69665031373, 2854.64037391020],
        [0.227, 1.31845358943, 230.70758317730],
        [0.192, 5.26739976413, 504.56118318100],
        [0.187, 0.85537192230, 273.85360000370],
        [0.199, 3.91291687807, 418.52143602870],
        [0.192, 6.15674105214, 611.44309831080],
        [0.210, 1.47873602747, 205.43478891180],
        [0.194, 2.37167703302, 3370.10424500320],
        [0.228, 2.15266015145, 55.13787859430],
        [0.201, 2.71380671608, 586.31331639720],
        [0.194, 3.29560033731, 1670.82502850000],
        [0.201, 4.23447633663, 1493.09366806600],
        [0.181, 3.61567262848, 9786.68735533500],
        [0.181, 2.83211558346, 1262.38608488870],
        [0.242, 4.69869158516, 1141.13406340540],
        [0.184, 4.66807336402, 1251.34038462480],
        [0.221, 2.25887876254, 355.74874557180],
        [0.200, 1.17340443616, 4952.06359328620],
        [0.222, 2.23360866067, 2435.15573003500],
        [0.175, 0.04701598422, 107.75864066460],
        [0.171, 5.02500742690, 93.53154666300],
        [0.184, 5.19723697138, 835.78789401270],
        [0.221, 4.49141283681, 913.00012679230],
        [0.195, 0.92088046109, 551.03160609700],
        [0.166, 5.01778115937, 354.99798604640],
        [0.165, 2.26267552932, 406.95447090300],
        [0.189, 0.31221126958, 420.96911658350],
        [0.196, 2.70333585839, 774.48262992160],
        [0.176, 6.12029409039, 181.80652604900],
        [0.172, 1.94132177757, 3259.89792378380],
        [0.160, 0.55319954265, 5429.87946823940],
        [0.161, 2.88623631474, 184.84490743480],
        [0.192, 0.26639534884, 295.19424100610],
        [0.167, 3.71345214172, 1056.20053645150],
        [0.195, 4.83926717598, 1596.18644228460],
        [0.156, 2.81916058733, 428.08266358430],
        [0.215, 1.88276472005, 220.36445832900],
        [0.167, 2.68872854428, 423.67742956920],
        [0.154, 1.66553954375, 115.62294719080],
        [0.175, 0.20216461467, 384.05992122310],
        [0.201, 4.38095931887, 418.00017116690],
        [0.167, 1.86485857353, 393.46109008430],
        [0.155, 0.92480392431, 282.66406803390],
        [0.146, 1.97663966745, 9360.08916445900],
        [0.160, 2.62483919699, 353.30106501700],
        [0.186, 1.37307151419, 292.01284726840],
        [0.198, 1.15631374887, 2957.73314812880],
        [0.144, 4.82956915076, 453.42489381900],
        [0.149, 3.60682821788, 205.66428357540],
        [0.147, 4.48377791879, 81.89514556810],
        [0.147, 5.74795037748, 856.37777548970],
        [0.142, 3.53823120158, 212.02707105080],
        [0.140, 0.70476909062, 640.86049416050],
        [0.139, 1.39047667205, 1261.63532536330],
        [0.153, 3.29559426243, 391.17346822390],
        [0.158, 1.79872341304, 326.68681039510],
        [0.174, 3.98677435872, 1049.08698945070],
        [0.171, 4.16825100469, 213.03846300710],
        [0.133, 4.74095454922, 0.04818410980],
        [0.155, 5.32313618730, 2015.67108615980],
        [0.158, 2.67557086253, 2531.13495725280],
        [0.158, 4.64622526567, 427.34895040140],
        [0.123, 2.20103444636, 210.59078245230],
        [0.160, 1.85888551524, 201.51968199120],
        [0.119, 3.12572799769, 238.57188970350],
        [0.120, 4.62897224203, 203.26478713040],
        [0.129, 4.92592016162, 1286.90811962880],
        [0.132, 3.44682160054, 156.67674413540],
        [0.143, 0.67951827513, 425.84743135060],
        [0.114, 5.46519773276, 552.58551477450],
        [0.132, 1.76335093671, 432.01481684740],
        [0.113, 0.68933513038, 450.97721326420],
        [0.128, 2.13986068877, 2751.54759969160],
        [0.123, 4.59695145319, 216.00740842370],
        [0.119, 1.04688666457, 462.02291352810],
        [0.108, 5.36873170289, 3377.21779200400],
        [0.142, 6.24626256472, 299.12639426920],
        [0.118, 0.63448253510, 369.97583957340],
        [0.105, 2.31570619675, 200.55647414470],
        [0.124, 1.87110815140, 850.01498801430],
        [0.106, 0.55623662570, 114.39910691340],
        [0.102, 3.95315219638, 361.37781986430],
        [0.095, 4.10658529323, 10213.28554621100],
        [0.097, 1.13534710734, 387.24131496080],
        [0.096, 4.46689094543, 401.32539661050],
        [0.119, 2.33636675091, 318.83955021140],
        [0.115, 3.37508073115, 313.94418910180],
        [0.106, 3.73586211650, 220.93390730060],
        [0.090, 0.59788492023, 227.31374111850],
        [0.103, 5.09172929383, 213.45915413240],
        [0.097, 5.95268532215, 1044.40407666220],
        [0.103, 1.70625660572, 213.13903674360],
        [0.080, 0.86872596168, 233.90602325750],
        [0.089, 5.35990932230, 214.19286731530],
        [0.080, 2.69565238975, 540.73666535850],
        [0.095, 1.19504849611, 460.53844081980],
        [0.105, 0.58624363205, 481.73606947030],
        [0.099, 2.68841109007, 219.89137757700],
        [0.098, 1.59923557478, 484.70501488690],
        [0.081, 1.12279793521, 420.44785172170],
        [0.075, 4.58892231446, 394.35486196160],
        [0.099, 4.68895851750, 448.68959140380],
        [0.076, 1.66929798365, 196.03362005060],
        [0.087, 3.12477195090, 857.12853501510],
        [0.078, 5.59819387460, 364.34676528090],
        [0.079, 3.53267171729, 969.62247809490],
    ],
    # L2
    [
        [116441.181, 1.17987850633, 7.11354700080],
        [91920.844, 0.07425261094, 213.29909543800],
        [90592.251, 0.00000000000, 0.00000000000],
        [15276.909, 4.06492007503, 206.18554843720],
        [10631.396, 0.25778277414, 220.41264243880],
        [10604.979, 5.40963595885, 426.59819087600],
        [4265.368, 1.04595556630, 14.22709400160],
        [1215.527, 2.91860042123, 103.09277421860],
        [1164.684, 4.60942128971, 639.89728631400],
        [1081.967, 5.69130351670, 433.71173787680],
        [1020.079, 0.63369182642, 3.18139373770],
        [1044.754, 4.04206453611, 199.07200143640],
        [633.582, 4.38825410036, 419.48464387520],
        [549.329, 5.57303134242, 3.93215326310],
        [456.914, 1.26840971349, 110.20632121940],
        [425.100, 0.20935499279, 227.52618943960],
        [273.739, 4.28841011784, 95.97922721780],
        [161.571, 1.38139149420, 11.04570026390],
        [129.494, 1.56586884170, 309.27832265580],
        [117.008, 3.88120915956, 853.19638175200],
        [105.415, 4.90003203599, 647.01083331480],
        [100.967, 0.89270493100, 21.34064100240],
        [95.227, 5.62561150598, 412.37109687440],
        [81.948, 1.02477558315, 117.31986822020],
        [74.857, 4.76178468163, 210.11770170030],
        [82.727, 6.05030934786, 216.48048917570],
        [95.659, 2.91093561539, 316.39186965660],
        [63.696, 0.35179804917, 323.50541665740],
        [84.860, 5.73472777961, 209.36694217490],
        [60.647, 4.87517850190, 632.78373931320],
        [66.459, 0.48297940601, 10.29494073850],
        [67.184, 0.45648612616, 522.57741809380],
        [53.281, 2.74730541387, 529.69096509460],
        [45.827, 5.69296621745, 440.82528487760],
        [45.293, 1.66856699796, 202.25339517410],
        [42.330, 5.70768187703, 88.86568021700],
        [32.140, 0.07050050346, 63.73589830340],
        [31.573, 1.67190022213, 302.16477565500],
        [31.150, 4.16379537691, 191.95845443560],
        [24.631, 5.65564728570, 735.87651353180],
        [26.558, 0.83256214407, 224.34479570190],
        [20.108, 5.94364609981, 217.23124870110],
        [17.511, 4.90014736798, 625.67019231240],
        [17.130, 1.62593421274, 742.99006053260],
        [13.744, 3.76497167300, 195.13984817330],
        [12.236, 4.71789723976, 203.00415469950],
        [11.940, 0.12620714199, 234.63973644040],
        [16.040, 0.57886320845, 515.46387109300],
        [11.154, 5.92216844780, 536.80451209540],
        [14.068, 0.20675293700, 838.96928775040],
        [11.013, 5.60207982774, 728.76296653100],
        [11.718, 3.12098483554, 846.08283475120],
        [9.962, 4.15472049127, 860.30992875280],
        [10.601, 3.20327613035, 1066.49547719000],
        [10.072, 0.25709351996, 330.61896365820],
        [9.490, 0.46379969328, 956.28915597060],
        [10.240, 4.98736656070, 422.66603761290],
        [8.287, 2.13990364272, 269.92144674060],
        [7.238, 5.39724715258, 1052.26838318840],
        [7.730, 5.24602742309, 429.77958461370],
        [6.353, 4.46211130731, 284.14854074220],
        [5.935, 5.40967847103, 149.56319713460],
        [7.550, 4.03401153929, 9.56122755560],
        [5.779, 4.29380891110, 415.55249061210],
        [6.082, 5.93416924841, 405.25754987360],
        [5.711, 0.01824076994, 124.43341522100],
        [5.676, 6.02235682150, 223.59403617650],
        [4.757, 4.92804854717, 654.12438031560],
        [4.727, 2.27461984667, 18.15924726470],
        [4.509, 4.40688707557, 942.06206196900],
        [5.621, 0.29694719379, 127.47179660680],
        [5.453, 5.53868222772, 949.17560896980],
        [4.130, 4.68673560379, 74.78159856730],
        [4.098, 5.30851262200, 1045.15483618760],
        [4.223, 2.89014939299, 56.62235130260],
        [4.887, 3.20022991216, 277.03499374140],
        [3.905, 3.30270187305, 490.33408917940],
        [3.923, 6.09732996823, 81.75213321620],
        [3.755, 4.93065184796, 52.69019803950],
        [4.602, 6.13908576681, 1155.36115740700],
        [3.714, 0.40648076787, 137.03302416240],
        [3.407, 4.28514461015, 99.91138048090],
        [3.579, 0.20402442077, 1272.68102562720],
        [3.946, 0.36500928968, 12.53017297220],
        [3.246, 1.56761884227, 1059.38193018920],
        [4.063, 0.29084229143, 831.85574074960],
        [3.688, 0.15467406177, 437.64389113990],
        [2.895, 3.13473183482, 70.84944530420],
        [2.800, 0.32727938074, 191.20769491020],
        [2.672, 1.87612402267, 295.05122865420],
        [3.454, 4.77197610696, 423.41679713830],
        [2.623, 5.15237415384, 1368.66025284500],
        [2.457, 3.89612890177, 210.85141488320],
        [2.461, 1.58522876760, 32.24332891440],
        [2.595, 3.59007068361, 131.40394986990],
        [2.289, 4.76825865118, 351.81659230870],
        [2.357, 5.83099000562, 106.27416795630],
        [2.221, 5.98277491515, 6062.66320755260],
        [2.221, 2.05930402282, 6076.89030155420],
        [2.183, 5.94985336393, 145.63104387150],
        [2.718, 3.37801252354, 408.43894361130],
        [2.288, 3.14000619320, 22.09140052780],
        [2.090, 1.12304173562, 9992.87290377220],
        [2.089, 3.48276230686, 10007.09999777380],
        [2.570, 5.12167203704, 265.98929347750],
        [1.835, 4.15379879659, 1258.45393162560],
        [1.820, 5.05340615445, 1361.54670584420],
        [1.760, 4.13532689228, 107.02492748170],
        [1.921, 4.51790997496, 138.51749687070],
        [1.707, 1.35864593280, 231.45834270270],
        [1.956, 5.87006093798, 1471.75302706360],
        [2.133, 5.23409848720, 1265.56747862640],
        [1.595, 5.61962698786, 447.93883187840],
        [1.609, 3.74893709671, 628.85158605010],
        [1.490, 0.48352404940, 340.77089204480],
        [1.560, 5.97095003614, 430.53034413910],
        [1.352, 0.71405348653, 28.45418800320],
        [1.355, 2.91219493604, 215.74677599280],
        [1.298, 5.84254169775, 543.91805909620],
        [1.664, 6.23834873469, 1148.24761040620],
        [1.205, 2.83373725021, 200.76892246580],
        [1.192, 3.52219428945, 497.44763618020],
        [1.122, 2.60571030270, 1279.79457262800],
        [1.217, 6.23528359211, 1589.07289528380],
        [1.420, 0.85079202155, 6069.77675455340],
        [1.120, 4.95656566453, 1685.05212250160],
        [1.010, 3.39689646619, 1073.60902419080],
        [1.352, 2.27575429523, 9999.98645077300],
        [0.979, 1.58571463442, 1375.77379984580],
        [1.159, 0.71823181781, 508.35032409220],
        [1.014, 2.40759054741, 703.63318461740],
        [0.956, 2.66256831556, 134.58534360760],
        [1.110, 1.19713920197, 618.55664531160],
        [0.945, 4.68155456977, 362.86229257260],
        [0.953, 4.20749172571, 288.08069400530],
        [1.033, 1.08781255146, 184.84490743480],
        [0.942, 2.43465223460, 222.86032299360],
        [0.909, 4.51769385360, 38.13303563780],
        [1.002, 1.38543153271, 483.22054217860],
        [1.082, 4.52832816548, 635.96513305090],
        [1.008, 4.91325851448, 750.10360753340],
        [0.862, 4.79998518474, 1677.93857550080],
        [0.828, 2.21940849017, 333.65734504400],
        [0.745, 3.97279299984, 1574.84580128220],
        [0.903, 5.58963782799, 1788.14489672020],
        [0.735, 2.28191723259, 1162.47470440780],
        [0.773, 5.82270096882, 416.30325013750],
        [0.734, 2.35356586018, 120.35824960600],
        [0.745, 4.84266000843, 76.26607127560],
        [0.765, 2.50840146722, 343.21857259960],
        [0.908, 5.01046293458, 1581.95934828300],
        [0.707, 3.66631544506, 347.88443904560],
        [0.870, 0.77106152694, 113.38771495710],
        [0.686, 2.88543836068, 92.04707395470],
        [0.673, 3.75650667651, 203.73786788240],
        [0.656, 3.77718582702, 217.96496188400],
        [0.675, 5.62875135263, 17.26547538740],
        [0.691, 0.21330089609, 99.16062095550],
        [0.786, 4.49318079175, 643.07868005170],
        [0.641, 0.67588390141, 46.47042291600],
        [0.663, 5.74837848383, 721.64941953020],
        [0.809, 5.94893988352, 1464.63948006280],
        [0.638, 4.86195439622, 357.44566660120],
        [0.740, 6.00053422445, 337.73251065900],
        [0.555, 4.95858934298, 358.93013930950],
        [0.581, 3.87669679805, 565.11568774670],
        [0.541, 1.22296838713, 62.25142559510],
        [0.697, 0.00715950269, 1169.58825140860],
        [0.524, 1.53830423608, 195.89060769870],
        [0.518, 5.41992758537, 312.19908396260],
        [0.626, 5.26580317026, 436.89313161450],
        [0.537, 6.17031657600, 182.27960680100],
        [0.574, 5.98607898826, 1905.46476494040],
        [0.541, 0.30589337713, 98.89998852460],
        [0.603, 3.26888470585, 208.63322899200],
        [0.504, 3.80930996688, 168.05251279940],
        [0.477, 3.56642391994, 563.63121503840],
        [0.511, 4.70719837179, 2001.44399215820],
        [0.475, 1.06025557585, 5856.47765911540],
        [0.540, 0.87230551412, 1692.16566950240],
        [0.454, 2.48128029368, 9786.68735533500],
        [0.456, 3.18303484133, 218.92816973050],
        [0.462, 0.71358186864, 258.87574647670],
        [0.424, 4.89778948357, 636.71589257630],
        [0.537, 2.59376221736, 313.21047591890],
        [0.410, 4.22147787617, 867.42347575360],
        [0.408, 3.06057772788, 424.15051032120],
        [0.407, 3.79376013938, 24.37902238820],
        [0.569, 3.68547825941, 350.33211960040],
        [0.404, 0.91401255827, 114.13847448250],
        [0.395, 3.50478374207, 129.91947716160],
        [0.395, 2.86309689622, 212.33588759150],
        [0.386, 5.00762729432, 388.46515523820],
        [0.393, 6.26835522096, 241.75328344120],
        [0.401, 4.60258908692, 1994.33044515740],
        [0.385, 0.91582119643, 160.60889739850],
        [0.467, 0.54876489832, 404.50679034820],
        [0.368, 0.35674031808, 214.26230328450],
        [0.471, 0.67360047481, 207.88246946660],
        [0.379, 0.92901327825, 767.36908292080],
        [0.420, 5.69797398044, 225.82926841020],
        [0.356, 3.10092792842, 842.15068148810],
        [0.428, 5.35375368944, 2104.53676637680],
        [0.422, 2.67975581832, 77.96299230500],
        [0.370, 5.46144813372, 1038.04128918680],
        [0.379, 5.56429091578, 131.54696222180],
        [0.441, 5.68196668399, 1781.03134971940],
        [0.361, 5.20616019966, 629.60234557550],
        [0.341, 5.92928351979, 26.82670294300],
        [0.419, 5.26851686707, 85.82729883120],
        [0.322, 0.80223983857, 6283.07584999140],
        [0.323, 3.86700993914, 576.16138801060],
        [0.321, 2.17186032970, 10213.28554621100],
        [0.355, 2.80560859177, 344.70304530790],
        [0.311, 3.77477255556, 1891.23767093880],
        [0.318, 5.22020784209, 142.44965013380],
        [0.315, 0.52272202855, 5849.36411211460],
        [0.428, 4.63722058283, 1898.35121793960],
        [0.337, 0.68198429948, 45.57665103870],
        [0.316, 0.54074780109, 444.75743814070],
        [0.310, 1.41032075652, 273.10284047830],
        [0.311, 3.53744556230, 251.43213107580],
        [0.295, 1.93253015677, 436.15941843160],
        [0.296, 1.97705648834, 9779.57380833420],
        [0.326, 3.67854047003, 963.40270297140],
        [0.389, 5.76841276132, 39.35687591520],
        [0.277, 5.73995694175, 92.79783348010],
        [0.315, 4.96371610197, 757.21715453420],
        [0.295, 1.81638833900, 1493.09366806600],
        [0.287, 0.97698377929, 685.47393735270],
        [0.281, 2.66463042095, 1286.90811962880],
        [0.330, 5.79776922760, 650.94298657790],
        [0.292, 3.97858181479, 472.17484191470],
        [0.266, 4.13716111320, 601.76425067620],
        [0.262, 0.91887592474, 245.54242435240],
        [0.278, 3.08964256591, 778.41478318470],
        [0.277, 3.08750916880, 621.73803904930],
        [0.255, 3.93981592051, 181.05576652360],
        [0.333, 2.04835822938, 561.18353448360],
        [0.247, 2.92754257675, 219.44943459230],
        [0.306, 0.36127922606, 824.74219374880],
        [0.253, 1.80130756458, 5643.17856367740],
        [0.337, 4.97764462199, 175.16605980020],
        [0.273, 0.66599369335, 2008.55753915900],
        [0.227, 4.87285356383, 661.23792731640],
        [0.249, 3.14202895058, 144.14657116320],
        [0.220, 3.93526603081, 319.57326339430],
        [0.212, 5.85248164087, 546.95644048200],
        [0.234, 1.65314711167, 554.06998748280],
        [0.204, 0.88373842674, 31.49256938900],
        [0.205, 2.93169866171, 1596.18644228460],
        [0.201, 3.36504567824, 1080.72257119160],
        [0.224, 4.34612745705, 1382.88734684660],
        [0.192, 5.13697232918, 329.72519178090],
        [0.208, 3.08549771485, 41.64449777560],
        [0.236, 0.07998742860, 1141.13406340540],
        [0.203, 4.13011580915, 2627.11418447060],
        [0.203, 0.13969067385, 1485.98012106520],
        [0.204, 3.38137545713, 699.70103135430],
        [0.212, 4.52370676085, 2310.72231481400],
        [0.218, 5.79277335862, 2221.85663459700],
        [0.213, 0.50441377637, 934.94851496820],
        [0.210, 5.04017633795, 2420.92863603340],
        [0.214, 4.64286758581, 2317.83586181480],
        [0.178, 0.84588580004, 128.36556848410],
        [0.170, 2.75006619605, 710.74673161820],
        [0.171, 4.32615182967, 291.26208774300],
        [0.172, 3.46971306920, 501.37978944330],
        [0.170, 1.05408992106, 526.50957135690],
        [0.162, 1.15683042950, 519.39602435610],
        [0.180, 4.96266204107, 1670.82502850000],
        [0.172, 1.65385549578, 916.93228005540],
        [0.170, 2.30821101766, 429.04587143080],
        [0.170, 5.98716489326, 643.82943957710],
        [0.173, 5.19933564968, 1354.43315884340],
        [0.195, 4.50165508529, 2214.74308759620],
        [0.156, 4.16290662749, 572.22923474750],
        [0.153, 1.23776248578, 2413.81508903260],
        [0.150, 0.63076983213, 1478.86657406440],
        [0.169, 4.28090123029, 305.34616939270],
        [0.174, 6.23077892653, 3384.33133900480],
        [0.149, 3.13274908516, 9573.38825989700],
        [0.162, 6.25601818345, 213.25091132820],
        [0.149, 4.81749019484, 945.99421523210],
        [0.162, 0.88610129190, 216.21985674480],
        [0.133, 2.31915371262, 156.67674413540],
        [0.165, 6.06456216591, 732.69511979410],
        [0.141, 6.14293754333, 1795.25844372100],
        [0.133, 0.06530337135, 218.71572140940],
        [0.162, 3.17058130506, 213.34727954780],
        [0.125, 2.07143636845, 425.11371816770],
        [0.146, 1.88627500632, 211.81462272970],
        [0.113, 2.79541965778, 235.39049596580],
        [0.117, 0.76464798684, 479.28838891550],
        [0.108, 3.95650672786, 570.74476203920],
        [0.106, 0.12820734703, 188.02630117250],
        [0.134, 3.58244908862, 849.26422848890],
        [0.114, 0.25990388555, 398.14400287280],
        [0.112, 2.39181495831, 217.49188113200],
        [0.091, 2.50716605179, 121.25202148330],
        [0.091, 1.75367948574, 213.82036029980],
        [0.088, 5.26121947108, 395.57870223900],
        [0.096, 3.98832609364, 289.56516671360],
        [0.091, 0.35318362186, 312.45971639350],
        [0.112, 1.14387590923, 1802.37199072180],
        [0.082, 3.73605217214, 207.67002114550],
        [0.082, 6.06283262812, 210.37833413120],
        [0.084, 3.34470673492, 67.66805156650],
        [0.086, 2.73917300180, 5863.59120611620],
        [0.083, 2.81499116485, 776.93031047640],
        [0.091, 1.26160093170, 212.77783057620],
        [0.090, 2.08722491981, 2111.65031337760],
        [0.080, 2.13136842916, 421.93232443000],
        [0.082, 4.16358350281, 9793.80090233580],
        [0.077, 2.96973341607, 431.26405732200],
        [0.079, 3.42790361067, 417.03696332040],
        [0.079, 3.18693585419, 320.32402291970],
        [0.080, 0.78975763683, 204.70107572890],
        [0.077, 1.89354243952, 556.51766803760],
        [0.073, 4.85923277221, 2118.76386037840],
        [0.071, 3.64551577433, 198.32124191100],
    ],
    # L3
    [
        [16038.734, 5.73945377424, 7.11354700080],
        [4249.793, 4.58539675603, 213.29909543800],
        [1906.524, 4.76082050205, 220.41264243880],
        [1465.687, 5.91326678323, 206.18554843720],
        [1162.041, 5.61973132428, 14.22709400160],
        [1066.581, 3.60816533142, 426.59819087600],
        [239.377, 3.86088273439, 433.71173787680],
        [236.975, 5.76826451465, 199.07200143640],
        [165.641, 5.11641150216, 3.18139373770],
        [131.409, 4.74327544615, 227.52618943960],
        [151.352, 2.73594641861, 639.89728631400],
        [61.630, 4.74287052463, 103.09277421860],
        [63.365, 0.22850089497, 419.48464387520],
        [40.437, 5.47298059144, 21.34064100240],
        [40.205, 5.96420266720, 95.97922721780],
        [38.746, 5.83386199529, 110.20632121940],
        [28.025, 3.01235311514, 647.01083331480],
        [25.029, 0.98808170740, 3.93215326310],
        [18.101, 1.02506397063, 412.37109687440],
        [17.879, 3.31913418974, 309.27832265580],
        [16.208, 3.89825272754, 440.82528487760],
        [15.763, 5.61667809625, 117.31986822020],
        [19.014, 1.91614237463, 853.19638175200],
        [18.262, 4.96738415934, 10.29494073850],
        [12.947, 1.18068953942, 88.86568021700],
        [17.919, 4.20376505349, 216.48048917570],
        [11.453, 5.57520615096, 11.04570026390],
        [10.548, 5.92906266269, 191.95845443560],
        [10.389, 3.94838736947, 209.36694217490],
        [8.650, 3.39335369698, 302.16477565500],
        [7.580, 4.87736913157, 323.50541665740],
        [6.697, 0.38198725552, 632.78373931320],
        [5.864, 1.05621157685, 210.11770170030],
        [5.449, 4.64268475485, 234.63973644040],
        [6.327, 2.25492722762, 522.57741809380],
        [3.602, 2.30677010956, 515.46387109300],
        [3.229, 2.20309400066, 860.30992875280],
        [3.701, 3.14159265359, 0.00000000000],
        [2.583, 4.93447677059, 224.34479570190],
        [2.543, 0.42393884183, 625.67019231240],
        [2.213, 3.19814958289, 202.25339517410],
        [2.421, 4.76621391814, 330.61896365820],
        [2.850, 0.58604395010, 529.69096509460],
        [1.965, 4.39525359412, 124.43341522100],
        [2.154, 1.35488209144, 405.25754987360],
        [2.296, 3.34809165905, 429.77958461370],
        [2.018, 3.06693569701, 654.12438031560],
        [1.979, 1.02981005658, 728.76296653100],
        [1.868, 3.09383546177, 422.66603761290],
        [1.846, 4.15225985450, 536.80451209540],
        [2.194, 1.18918501013, 1066.49547719000],
        [2.090, 4.15631351317, 223.59403617650],
        [1.481, 0.37916705169, 316.39186965660],
        [1.720, 5.82865773356, 195.13984817330],
        [1.460, 1.57663426355, 81.75213321620],
        [1.623, 6.03706764648, 742.99006053260],
        [1.286, 1.66154726117, 63.73589830340],
        [1.304, 5.02409881054, 956.28915597060],
        [1.446, 2.10575519127, 838.96928775040],
        [1.245, 3.88109752770, 269.92144674060],
        [1.018, 3.72599601656, 295.05122865420],
        [1.323, 1.38492882986, 735.87651353180],
        [1.318, 2.33460998999, 217.23124870110],
        [0.943, 2.75813531246, 284.14854074220],
        [0.906, 0.71155526266, 846.08283475120],
        [0.886, 3.83754799777, 447.93883187840],
        [0.943, 3.31480217015, 18.15924726470],
        [0.800, 4.71386673963, 56.62235130260],
        [0.908, 2.02119147951, 831.85574074960],
        [0.787, 0.80410269937, 1045.15483618760],
        [0.709, 4.27064410504, 437.64389113990],
        [0.651, 6.17565900032, 942.06206196900],
        [0.785, 2.40767785311, 203.00415469950],
        [0.702, 1.64585301418, 423.41679713830],
        [0.543, 2.86326941725, 184.84490743480],
        [0.532, 6.25762144463, 1059.38193018920],
        [0.521, 3.43013038466, 149.56319713460],
        [0.484, 4.88366060720, 1272.68102562720],
        [0.437, 5.40220619672, 408.43894361130],
        [0.388, 2.57589594168, 508.35032409220],
        [0.421, 4.05836524024, 543.91805909620],
        [0.375, 1.22747948298, 2324.94940881560],
        [0.347, 0.59237194522, 22.09140052780],
        [0.433, 1.69090148012, 1155.36115740700],
        [0.389, 1.46170367972, 1073.60902419080],
        [0.307, 1.82185086955, 628.85158605010],
        [0.409, 1.21858750514, 1052.26838318840],
        [0.309, 0.33610530663, 6076.89030155420],
        [0.309, 1.42279282226, 6062.66320755260],
        [0.340, 1.83325770310, 1141.13406340540],
        [0.303, 2.41584747330, 127.47179660680],
        [0.305, 5.34154702988, 131.40394986990],
        [0.298, 2.28594631393, 635.96513305090],
        [0.372, 1.03723911390, 313.21047591890],
        [0.338, 0.69100012338, 1361.54670584420],
        [0.325, 1.78816356937, 1148.24761040620],
        [0.322, 1.18628805010, 721.64941953020],
        [0.271, 2.45663156460, 415.55249061210],
        [0.251, 3.12046701975, 1382.88734684660],
        [0.254, 3.00353256829, 618.55664531160],
        [0.295, 0.35280179538, 2730.20695868920],
        [0.242, 1.52154324392, 70.84944530420],
        [0.296, 0.89576757167, 2104.53676637680],
        [0.264, 3.00987438634, 661.23792731640],
        [0.267, 0.31623829657, 1677.93857550080],
        [0.270, 2.56774718753, 643.07868005170],
        [0.261, 1.55058302472, 1457.52593306200],
        [0.246, 2.29214585472, 867.42347575360],
        [0.269, 3.18157515051, 750.10360753340],
        [0.272, 1.12208982319, 1788.14489672020],
        [0.256, 0.37673546414, 1279.79457262800],
        [0.206, 1.81129778306, 497.44763618020],
        [0.251, 0.61933213502, 2413.81508903260],
        [0.237, 3.35941544147, 436.89313161450],
        [0.247, 0.10102936687, 99.91138048090],
        [0.247, 0.93125798111, 52.69019803950],
        [0.221, 2.07880035795, 824.74219374880],
        [0.197, 6.16682223437, 1258.45393162560],
        [0.229, 5.57917534840, 2943.50605412720],
        [0.227, 0.43324651601, 2737.32050569000],
        [0.203, 4.12623986247, 337.73251065900],
        [0.214, 3.57607524509, 934.94851496820],
        [0.212, 1.25688162158, 1773.91780271860],
        [0.215, 0.88867647880, 1038.04128918680],
        [0.244, 5.51572084570, 231.45834270270],
        [0.181, 2.13821830481, 416.30325013750],
        [0.210, 4.19139167658, 2221.85663459700],
        [0.178, 2.91685344537, 74.78159856730],
        [0.201, 0.46214583002, 2854.64037391020],
        [0.236, 4.64388694899, 1905.46476494040],
        [0.199, 1.54991619669, 1471.75302706360],
        [0.199, 0.70725247497, 2420.92863603340],
        [0.162, 2.51488345020, 430.53034413910],
        [0.160, 1.23508694599, 1596.18644228460],
        [0.175, 4.14605894816, 2090.30967237520],
        [0.152, 0.05796022559, 32.24332891440],
        [0.176, 1.29002070623, 490.33408917940],
        [0.154, 3.60622857548, 650.94298657790],
        [0.185, 4.74969742128, 319.57326339430],
        [0.154, 1.54587199996, 1464.63948006280],
        [0.108, 4.25125786500, 145.63104387150],
        [0.106, 1.04047809351, 1162.47470440780],
        [0.114, 2.64055737100, 362.86229257260],
        [0.093, 3.36746275886, 483.22054217860],
        [0.091, 2.05796248692, 210.85141488320],
        [0.091, 4.53336314765, 241.75328344120],
        [0.072, 3.74361312157, 1485.98012106520],
        [0.076, 3.33892447677, 195.89060769870],
    ],
    # L4
    [
        [1661.894, 3.99826248978, 7.11354700080],
        [257.107, 2.98436499013, 220.41264243880],
        [236.344, 3.90241428075, 14.22709400160],
        [149.418, 2.74110824208, 213.29909543800],
        [109.598, 1.51515739251, 206.18554843720],
        [113.953, 3.14159265359, 0.00000000000],
        [68.390, 1.72120953337, 426.59819087600],
        [37.699, 1.23795458356, 199.07200143640],
        [40.060, 2.04644897412, 433.71173787680],
        [31.219, 3.01094184090, 227.52618943960],
        [15.111, 0.82897064529, 639.89728631400],
        [9.444, 3.71485300868, 21.34064100240],
        [5.690, 2.41995290633, 419.48464387520],
        [4.470, 1.45120818748, 95.97922721780],
        [5.608, 1.15607095740, 647.01083331480],
        [4.463, 2.11783225176, 440.82528487760],
        [3.229, 4.09278077834, 110.20632121940],
        [2.871, 2.77203153866, 412.37109687440],
        [2.796, 3.00730249564, 88.86568021700],
        [2.638, 0.00255721254, 853.19638175200],
        [2.574, 0.39246854091, 103.09277421860],
        [1.862, 5.07955457727, 309.27832265580],
        [2.225, 3.77689198137, 117.31986822020],
        [1.769, 5.19176876406, 302.16477565500],
        [1.921, 2.82884328662, 234.63973644040],
        [1.805, 2.23816036743, 216.48048917570],
        [1.211, 1.54685246534, 191.95845443560],
        [0.765, 3.44501766503, 323.50541665740],
        [0.763, 4.83197222448, 210.11770170030],
        [0.613, 4.19052656353, 515.46387109300],
        [0.648, 2.28591710303, 209.36694217490],
        [0.616, 4.03194472161, 522.57741809380],
        [0.630, 2.37952532019, 632.78373931320],
        [0.639, 0.29772678242, 860.30992875280],
        [0.559, 2.17110060530, 124.43341522100],
        [0.442, 2.23500083592, 447.93883187840],
        [0.407, 5.44515970990, 1066.49547719000],
        [0.469, 1.26889429317, 654.12438031560],
        [0.488, 3.20329778617, 405.25754987360],
        [0.415, 3.12435410343, 330.61896365820],
        [0.442, 3.38933498625, 81.75213321620],
        [0.332, 4.12464206608, 838.96928775040],
        [0.320, 3.18332026736, 529.69096509460],
        [0.312, 1.40962796637, 429.77958461370],
        [0.291, 3.18885372262, 1464.63948006280],
        [0.333, 2.94355912397, 728.76296653100],
        [0.235, 3.67049647573, 1148.24761040620],
        [0.286, 2.57895004576, 1045.15483618760],
        [0.223, 3.57980034401, 1155.36115740700],
        [0.261, 2.04564143519, 1677.93857550080],
        [0.218, 2.61967125327, 536.80451209540],
        [0.262, 2.48322150677, 625.67019231240],
        [0.191, 4.39064160974, 1574.84580128220],
        [0.176, 1.26161895188, 422.66603761290],
        [0.190, 2.32693171200, 223.59403617650],
        [0.185, 1.08713469614, 742.99006053260],
        [0.168, 0.69946458053, 824.74219374880],
        [0.177, 5.02663339078, 203.00415469950],
        [0.218, 0.40426546037, 867.42347575360],
        [0.178, 3.67593243311, 831.85574074960],
        [0.175, 5.75326979098, 1073.60902419080],
        [0.156, 3.02120117572, 1781.03134971940],
        [0.148, 2.28313808274, 295.05122865420],
        [0.150, 3.48436135302, 956.28915597060],
        [0.152, 1.91404443241, 942.06206196900],
        [0.146, 6.16519696640, 316.39186965660],
        [0.096, 2.93247663741, 224.34479570190],
        [0.088, 4.48383632427, 423.41679713830],
    ],
    # L5
    [
        [123.615, 2.25923345732, 7.11354700080],
        [34.190, 2.16250652689, 14.22709400160],
        [27.546, 1.19868150215, 220.41264243880],
        [5.818, 1.21584270184, 227.52618943960],
        [5.318, 0.23550400093, 433.71173787680],
        [3.677, 6.22669694355, 426.59819087600],
        [3.057, 2.97372046322, 199.07200143640],
        [2.861, 4.28710932685, 206.18554843720],
        [1.617, 6.25265362286, 213.29909543800],
        [1.279, 5.27612561266, 639.89728631400],
        [0.932, 5.56741549127, 647.01083331480],
        [0.756, 6.17716234487, 191.95845443560],
        [0.760, 0.69475544472, 302.16477565500],
        [1.038, 0.23516951637, 440.82528487760],
        [1.007, 3.14159265359, 0.00000000000],
        [0.549, 4.87733288264, 88.86568021700],
        [0.504, 4.77955496203, 419.48464387520],
        [0.346, 4.31847547394, 853.19638175200],
        [0.392, 5.69922389094, 654.12438031560],
        [0.242, 2.05052677361, 323.50541665740],
        [0.266, 1.11384528244, 234.63973644040],
        [0.199, 0.88505901097, 309.27832265580],
        [0.258, 5.10074489186, 95.97922721780],
        [0.166, 2.40063312194, 515.46387109300],
        [0.155, 4.70433216164, 860.30992875280],
        [0.089, 1.36371070380, 412.37109687440],
        [0.102, 0.49450039082, 117.31986822020],
    ],
]
"""This table contains Saturn's periodic terms (all of them) from the planetary
theory VSOP87 for the heliocentric longitude at the equinox of date (taken from
the 'D' solution). In Meeus' book a shortened version can be found in pages
435-440."""


VSOP87_B = [
    # B0
    [
        [4330678.040, 3.60284428399, 213.29909543800],
        [240348.303, 2.85238489390, 426.59819087600],
        [84745.939, 0.00000000000, 0.00000000000],
        [30863.357, 3.48441504465, 220.41264243880],
        [34116.063, 0.57297307844, 206.18554843720],
        [14734.070, 2.11846597870, 639.89728631400],
        [9916.668, 5.79003189405, 419.48464387520],
        [6993.564, 4.73604689179, 7.11354700080],
        [4807.587, 5.43305315602, 316.39186965660],
        [4788.392, 4.96512927420, 110.20632121940],
        [3432.125, 2.73255752123, 433.71173787680],
        [1506.129, 6.01304536144, 103.09277421860],
        [1060.298, 5.63099292414, 529.69096509460],
        [969.071, 5.20434966103, 632.78373931320],
        [942.050, 1.39646678088, 853.19638175200],
        [707.645, 3.80302329547, 323.50541665740],
        [552.313, 5.13149109045, 202.25339517410],
        [399.675, 3.35891413961, 227.52618943960],
        [316.063, 1.99716764199, 647.01083331480],
        [319.380, 3.62571550980, 209.36694217490],
        [284.494, 4.88648481625, 224.34479570190],
        [314.225, 0.46510272410, 217.23124870110],
        [236.442, 2.13887472281, 11.04570026390],
        [215.354, 5.94982610103, 846.08283475120],
        [208.522, 2.12003893769, 415.55249061210],
        [178.958, 2.95361514672, 63.73589830340],
        [207.213, 0.73021462851, 199.07200143640],
        [139.140, 1.99821990940, 735.87651353180],
        [134.884, 5.24500819605, 742.99006053260],
        [140.585, 0.64417620299, 490.33408917940],
        [121.669, 3.11537140876, 522.57741809380],
        [139.240, 4.59535168021, 14.22709400160],
        [115.524, 3.10891547171, 216.48048917570],
        [114.218, 0.96261442133, 210.11770170030],
        [96.376, 4.48164339766, 117.31986822020],
        [80.593, 1.31692750150, 277.03499374140],
        [72.952, 3.05988482370, 536.80451209540],
        [69.261, 4.92378633635, 309.27832265580],
        [74.302, 2.89376539620, 149.56319713460],
        [68.040, 2.18002263974, 351.81659230870],
        [61.734, 0.67728106562, 1066.49547719000],
        [56.598, 2.60963391288, 440.82528487760],
        [48.864, 5.78725874107, 95.97922721780],
        [48.243, 2.18211837430, 74.78159856730],
        [38.304, 5.29151303843, 1059.38193018920],
        [36.323, 1.63348365121, 628.85158605010],
        [35.055, 1.71279210041, 1052.26838318840],
        [34.270, 2.45740470599, 422.66603761290],
        [34.313, 5.97994514798, 412.37109687440],
        [33.787, 1.14073392951, 949.17560896980],
        [31.633, 4.14722153007, 437.64389113990],
        [36.833, 6.27769966148, 1162.47470440780],
        [26.980, 1.27154816810, 860.30992875280],
        [23.516, 2.74936525342, 838.96928775040],
        [23.460, 0.98962849901, 210.85141488320],
        [23.600, 4.11386961467, 3.93215326310],
        [23.631, 3.07427204313, 215.74677599280],
        [20.813, 3.51084686918, 330.61896365820],
        [19.509, 2.81857577372, 127.47179660680],
        [17.103, 3.89784279922, 214.26230328450],
        [17.635, 6.19715516746, 703.63318461740],
        [17.824, 2.28524493886, 388.46515523820],
        [20.935, 0.14356167048, 430.53034413910],
        [16.551, 1.66649120724, 38.13303563780],
        [19.100, 2.97699096081, 137.03302416240],
        [15.517, 4.54798410406, 956.28915597060],
        [17.065, 0.16611115812, 212.33588759150],
        [14.169, 0.48937283445, 213.34727954780],
        [19.027, 6.27326062836, 423.41679713830],
        [13.344, 2.37136126257, 429.77958461370],
        [12.565, 1.03178071173, 563.63121503840],
        [14.173, 3.57477564831, 213.25091132820],
        [11.374, 1.45300927024, 1368.66025284500],
        [10.585, 6.17633425930, 200.76892246580],
        [10.600, 3.84358958373, 138.51749687070],
        [10.263, 2.17423692422, 76.26607127560],
        [10.072, 1.33197220789, 565.11568774670],
        [12.058, 0.44149242700, 222.86032299360],
        [10.367, 1.85278552549, 350.33211960040],
        [8.706, 2.58144528603, 1155.36115740700],
        [8.470, 1.97890349826, 625.67019231240],
        [8.518, 4.51649648578, 3.18139373770],
        [7.439, 4.92597321442, 212.77783057620],
        [7.409, 2.03506679104, 288.08069400530],
        [8.137, 3.98500592467, 85.82729883120],
        [7.985, 2.20794292064, 362.86229257260],
        [6.610, 6.14944028835, 417.03696332040],
        [7.753, 6.23664549070, 1478.86657406440],
        [6.318, 1.87388481013, 654.12438031560],
        [6.319, 1.17328438271, 1265.56747862640],
        [5.841, 2.35829915285, 750.10360753340],
        [5.808, 5.01602242794, 479.28838891550],
        [8.079, 0.42574715104, 554.06998748280],
        [6.014, 5.58952234348, 425.11371816770],
        [7.444, 5.41859596459, 213.82036029980],
        [7.567, 2.68446523795, 191.20769491020],
        [7.421, 4.19354269508, 9.56122755560],
        [5.466, 3.21737829505, 234.63973644040],
        [5.661, 1.46982713550, 265.98929347750],
        [5.851, 4.81776629912, 1.48447270830],
        [5.341, 3.45755372717, 203.73786788240],
        [4.960, 1.04628615559, 12.53017297220],
        [4.920, 3.85622235967, 173.94221952280],
        [4.883, 1.94823282939, 195.13984817330],
        [5.621, 0.81869581274, 52.69019803950],
        [5.200, 3.32827437636, 515.46387109300],
        [4.927, 3.81806549732, 225.82926841020],
        [5.033, 0.10756875163, 252.65597135320],
        [4.416, 5.45506938037, 408.43894361130],
        [4.169, 1.21145214135, 1685.05212250160],
        [4.066, 6.24213578122, 1279.79457262800],
        [3.972, 6.13850317719, 217.49188113200],
        [5.398, 5.67212179194, 1375.77379984580],
        [3.916, 5.96105725915, 210.37833413120],
        [4.017, 0.99840226682, 842.15068148810],
        [3.899, 4.58983662507, 1272.68102562720],
        [3.764, 3.30663337976, 212.54833591260],
        [4.345, 3.18562241830, 414.06801790380],
        [3.565, 4.75262007127, 207.88246946660],
        [3.542, 2.30814954338, 1471.75302706360],
        [3.732, 1.61040235688, 635.96513305090],
        [3.709, 2.97082943086, 223.59403617650],
        [3.576, 3.83436862558, 483.22054217860],
        [4.053, 3.72105017218, 942.06206196900],
        [3.756, 0.74987556308, 214.04985496340],
        [3.162, 3.64550741000, 207.67002114550],
        [3.149, 2.27647454229, 728.76296653100],
        [3.971, 4.37874143597, 216.21985674480],
        [3.541, 5.62281936827, 218.71572140940],
        [2.965, 3.40117024932, 650.94298657790],
        [3.949, 4.19728912450, 209.10630974400],
        [2.853, 4.81077453523, 231.45834270270],
        [2.826, 0.86682282341, 217.96496188400],
        [2.970, 5.75162134301, 160.60889739850],
        [2.724, 0.47941267764, 497.44763618020],
        [2.787, 4.03144791896, 62.25142559510],
        [2.605, 5.04152791794, 65.22037101170],
        [2.652, 0.30602610654, 424.15051032120],
        [2.543, 2.76499056123, 543.91805909620],
        [2.485, 5.78396049817, 99.16062095550],
        [3.209, 0.42853440759, 218.92816973050],
        [2.509, 2.94704589100, 70.84944530420],
        [2.308, 0.63861650866, 251.43213107580],
        [2.869, 4.31959346745, 767.36908292080],
        [2.232, 0.56929937568, 1073.60902419080],
        [2.245, 1.69945964547, 488.84961647110],
        [2.283, 1.55589787463, 601.76425067620],
        [2.384, 4.44583493310, 21.34064100240],
        [2.096, 5.77425542767, 88.86568021700],
        [2.363, 3.35786310868, 124.43341522100],
        [2.162, 6.24029269257, 1795.25844372100],
        [2.452, 3.19804814047, 208.63322899200],
        [2.033, 4.87029603776, 327.43756992050],
        [1.950, 5.56004000293, 18.15924726470],
        [2.283, 4.22375355881, 22.09140052780],
        [2.217, 4.61433094630, 302.16477565500],
        [1.888, 5.45600788064, 142.44965013380],
        [2.075, 3.55622165076, 1169.58825140860],
        [2.069, 2.75786819366, 491.81856188770],
        [1.810, 5.96568495526, 213.18722085340],
        [1.813, 1.39785500313, 211.81462272970],
        [1.843, 1.15001484281, 203.00415469950],
        [1.854, 1.41350087782, 1581.95934828300],
        [1.697, 3.23613719814, 427.56139872250],
        [1.736, 5.45933992115, 916.93228005540],
        [1.714, 6.14729146384, 643.82943957710],
        [1.948, 5.70817363392, 425.63498302950],
        [1.770, 3.36194411768, 248.72381809010],
        [1.611, 0.97888081762, 2001.44399215820],
        [1.971, 2.59654430358, 429.04587143080],
        [1.628, 0.74011617198, 177.87437278590],
        [1.564, 2.04342011485, 1788.14489672020],
        [1.574, 6.01995224314, 426.64637498580],
        [1.939, 5.48134669380, 636.71589257630],
        [1.651, 4.61629429952, 621.73803904930],
        [1.552, 2.55542734908, 692.58748435350],
        [1.484, 6.17173980637, 56.62235130260],
        [1.782, 3.26122906302, 175.16605980020],
        [1.503, 2.59953333916, 228.27694896500],
        [1.559, 0.36281050773, 776.93031047640],
        [1.799, 3.46395976970, 1258.45393162560],
        [1.521, 3.75588462293, 213.51154375910],
        [1.810, 4.37552745264, 213.41097002260],
        [1.521, 0.30214462385, 213.08664711690],
        [1.608, 4.66724132818, 269.92144674060],
        [1.525, 1.47939329423, 198.32124191100],
        [1.408, 1.38491846750, 501.37978944330],
        [1.327, 0.23760037979, 148.07872442630],
        [1.305, 2.41201772023, 275.55052103310],
        [1.578, 2.82443242444, 426.55000676620],
        [1.203, 1.36928065935, 235.39049596580],
        [1.296, 5.75203277385, 1692.16566950240],
        [1.295, 3.04090959062, 831.85574074960],
        [1.283, 1.62400181159, 643.07868005170],
        [1.617, 1.31181209458, 214.78356814630],
        [1.346, 4.01262069353, 278.51946644970],
        [1.166, 0.09590019561, 340.77089204480],
        [1.115, 2.20460481017, 221.37585028530],
        [1.176, 1.07528227869, 312.19908396260],
        [1.107, 1.50329021421, 289.56516671360],
        [1.143, 4.40125874383, 213.55972786890],
        [1.253, 0.21632769953, 404.50679034820],
        [1.074, 3.17412275025, 98.89998852460],
        [1.103, 0.23626839162, 617.80588578620],
        [1.077, 3.51670933532, 312.45971639350],
        [1.039, 0.53953974796, 778.41478318470],
        [1.195, 2.11232088496, 205.22234059070],
        [1.093, 5.16243153571, 630.33605875840],
        [1.143, 5.93977485365, 213.03846300710],
        [1.040, 4.79631324365, 106.27416795630],
        [1.100, 0.02509241739, 219.44943459230],
        [1.311, 5.93900415785, 436.15941843160],
        [1.260, 0.72481995446, 355.74874557180],
        [0.950, 2.00801292252, 1045.15483618760],
        [1.186, 1.84906064486, 151.04766984290],
        [0.974, 3.01092368346, 696.51963761660],
        [0.924, 4.88158186437, 39.35687591520],
        [0.961, 2.80113869315, 738.79727483860],
        [1.011, 6.27004359435, 121.25202148330],
        [0.904, 4.15218356485, 426.07692601420],
        [0.904, 4.24232505252, 10.29494073850],
        [0.895, 2.47587956888, 447.93883187840],
        [1.186, 0.85270715988, 525.49817940060],
        [0.824, 3.97540449663, 210.59078245230],
        [0.831, 4.05299295705, 207.14875628370],
        [0.937, 5.44353432307, 344.70304530790],
        [0.823, 2.08766677969, 358.93013930950],
        [0.971, 5.09512804595, 1589.07289528380],
        [1.037, 1.04152859909, 2.44768055480],
        [0.816, 0.62175655307, 188.92007304980],
        [0.798, 3.36396062989, 237.67811782620],
        [0.755, 5.90983584858, 284.14854074220],
        [0.870, 1.62765846893, 114.13847448250],
        [0.734, 6.23523714922, 2111.65031337760],
        [0.767, 2.73651219269, 627.36711334180],
        [0.701, 0.72526525525, 10213.28554621100],
        [0.801, 5.84130533519, 905.88657979150],
        [0.710, 1.78740818763, 2104.53676637680],
        [0.670, 0.75839374890, 2317.83586181480],
        [0.720, 4.95645178898, 638.41281360570],
        [0.796, 4.87623914638, 342.25536475310],
        [0.703, 0.35096991676, 220.46082654860],
        [0.835, 3.19492825640, 1574.84580128220],
        [0.824, 0.08813665925, 216.00740842370],
        [0.653, 4.19599635390, 247.23934538180],
        [0.826, 4.66845240923, 427.11945573780],
        [0.690, 0.41873449575, 5856.47765911540],
        [0.690, 1.34023204821, 6283.07584999140],
        [0.690, 0.58291546593, 213.45915413240],
        [0.639, 1.20304559619, 867.42347575360],
        [0.830, 1.57233214789, 1898.35121793960],
        [0.753, 1.51187970880, 576.16138801060],
        [0.629, 2.83598833891, 420.96911658350],
        [0.690, 3.48062808501, 213.13903674360],
        [0.600, 5.22938546212, 212.02707105080],
        [0.565, 5.28099337758, 423.67742956920],
        [0.552, 5.83265103738, 84.34282612290],
        [0.546, 3.56588711151, 1485.98012106520],
        [0.642, 6.15007145598, 179.35884549420],
        [0.697, 1.91977327925, 134.58534360760],
        [0.648, 5.15917450752, 8.07675484730],
        [0.551, 3.33109164145, 980.66817835880],
        [0.511, 3.87073988213, 125.98732389850],
        [0.506, 0.80261216855, 181.05576652360],
        [0.564, 4.05871107615, 831.10498122420],
        [0.681, 3.45804290093, 220.36445832900],
        [0.541, 2.39252901431, 421.93232443000],
        [0.498, 3.11515053568, 439.12836384820],
        [0.517, 2.75430004800, 1148.24761040620],
        [0.491, 4.47199498503, 558.00214074590],
        [0.467, 4.18144797677, 444.75743814070],
        [0.458, 0.60848440253, 206.23373254700],
        [0.542, 3.28613020157, 245.54242435240],
        [0.477, 5.48556902348, 35.42472265210],
        [0.474, 2.42545073874, 436.89313161450],
        [0.481, 3.72498040536, 206.13736432740],
        [0.435, 1.61732308614, 191.95845443560],
        [0.488, 0.26059969650, 416.30325013750],
        [0.493, 4.13699180247, 518.64526483070],
        [0.595, 5.11706007934, 214.57111982520],
        [0.486, 5.17710069856, 67.66805156650],
        [0.463, 5.53192185211, 418.52143602870],
        [0.421, 5.57377685121, 430.79097657000],
        [0.446, 6.20735049659, 73.29712585900],
        [0.421, 4.65837340438, 543.02428721890],
        [0.435, 2.95256554350, 5.41662597140],
        [0.416, 4.36391909218, 113.38771495710],
        [0.495, 5.38121485133, 391.17346822390],
        [0.520, 3.99939347071, 618.55664531160],
        [0.429, 3.26903461513, 144.14657116320],
        [0.435, 1.60816661416, 2214.74308759620],
        [0.398, 2.38329818919, 299.12639426920],
        [0.399, 0.06301179341, 206.70681329900],
        [0.394, 2.75496219113, 425.84743135060],
        [0.508, 2.86873328929, 337.73251065900],
        [0.475, 5.68530292289, 320.32402291970],
        [0.432, 0.68132398291, 116.42609634290],
        [0.404, 1.84249441289, 9786.68735533500],
        [0.371, 5.83382962844, 2008.55753915900],
        [0.382, 4.24995364794, 387.24131496080],
        [0.416, 3.81986946256, 429.51895218280],
        [0.491, 4.18623117082, 219.89137757700],
        [0.352, 1.65610545221, 963.40270297140],
        [0.353, 5.50209003460, 305.34616939270],
        [0.431, 4.39380503963, 353.30106501700],
        [0.375, 2.67828133567, 319.57326339430],
        [0.359, 3.54801032661, 421.18156490460],
        [0.339, 5.19074405462, 69.15252427480],
        [0.358, 1.11857595997, 1044.40407666220],
        [0.334, 1.84260994740, 1361.54670584420],
        [0.328, 6.07106596408, 710.74673161820],
        [0.328, 1.48618893585, 2420.92863603340],
        [0.359, 5.62797136991, 78.71375183040],
        [0.405, 2.91366762549, 1891.23767093880],
        [0.398, 2.08900381937, 4.66586644600],
        [0.326, 1.62373774313, 5.62907429250],
        [0.405, 1.87470341223, 114.39910691340],
        [0.393, 0.56487847337, 128.95626931510],
        [0.312, 5.29291124448, 347.88443904560],
        [0.310, 0.31232452686, 427.34895040140],
        [0.308, 2.84912656825, 487.36514376280],
        [0.291, 3.25977762780, 494.26624244250],
        [0.303, 4.93962873690, 373.90799283650],
        [0.289, 2.83591185309, 212.07525516060],
        [0.289, 2.05371431350, 214.52293571540],
        [0.398, 6.03822674845, 432.22726516850],
        [0.288, 6.16001418475, 969.62247809490],
        [0.296, 0.30332524090, 1055.44977692610],
        [0.280, 1.29728455136, 241.61027108930],
        [0.280, 5.41630221077, 1493.09366806600],
        [0.315, 6.24003908326, 465.95506679120],
        [0.274, 5.03981944861, 458.84151979040],
        [0.296, 3.04457317761, 211.60217440860],
        [0.274, 2.72607352851, 145.63104387150],
        [0.293, 1.32452002382, 159.12442469020],
        [0.263, 6.11559198968, 2428.04218303420],
        [0.323, 1.17502395659, 815.06334611420],
        [0.345, 5.37083374878, 428.08266358430],
        [0.258, 0.43205106363, 2634.22773147140],
        [0.275, 0.91628212149, 849.26422848890],
        [0.340, 1.29378813067, 329.72519178090],
        [0.294, 4.29399534634, 4.19278569400],
        [0.339, 1.03883773894, 32.24332891440],
        [0.244, 3.52504227332, 184.98791978670],
        [0.243, 3.13047989401, 525.75881183150],
        [0.260, 5.99216785208, 20.60692781950],
        [0.303, 3.96772261614, 934.94851496820],
        [0.285, 5.69474711283, 220.93390730060],
        [0.239, 2.09516779457, 292.01284726840],
        [0.242, 0.98744748894, 282.45161971280],
        [0.278, 2.81667003542, 87.31177153950],
        [0.285, 4.76303256917, 54.17467074780],
        [0.236, 5.79560286324, 280.96714700450],
        [0.246, 1.83689902078, 214.99601646740],
        [0.221, 0.48302467341, 153.49535039770],
        [0.238, 3.52738705554, 267.47376618580],
        [0.293, 5.91401607974, 14.97785352700],
        [0.235, 0.29884419224, 14.01464568050],
        [0.229, 3.91975580405, 182.27960680100],
        [0.217, 3.96328561940, 894.84087952760],
        [0.218, 1.46057992688, 2531.13495725280],
        [0.210, 2.01138855049, 211.86280683950],
        [0.210, 2.87823761610, 214.73538403650],
        [0.223, 2.49651288358, 1464.63948006280],
        [0.217, 4.03234048538, 835.03713448730],
        [0.209, 2.97542058241, 273.10284047830],
        [0.210, 2.56849303008, 593.42686339800],
        [0.218, 1.71859101510, 0.96320784650],
        [0.232, 1.86083014117, 221.16340196420],
        [0.199, 3.53454927143, 219.66188291340],
        [0.197, 2.65338829617, 864.24208201590],
        [0.219, 3.46267185338, 1182.92157353290],
        [0.199, 2.56046317223, 264.50482076920],
        [0.199, 2.03508708900, 757.21715453420],
        [0.237, 5.05443109284, 254.94359321360],
        [0.191, 2.07106909876, 756.32338265690],
        [0.192, 1.67985944172, 1677.93857550080],
        [0.191, 1.05067348453, 702.14871190910],
        [0.181, 1.89151852263, 6.15033915430],
        [0.205, 1.98151360584, 199.28444975750],
        [0.181, 1.25381796494, 2737.32050569000],
        [0.186, 2.81416016738, 569.04784100980],
        [0.215, 4.52373060846, 3060.82592234740],
        [0.222, 2.90133577623, 205.43478891180],
        [0.246, 3.55891849574, 1251.34038462480],
        [0.191, 4.20221553993, 556.51766803760],
        [0.217, 2.64509967170, 2207.62954059540],
        [0.225, 0.14271906959, 131.40394986990],
        [0.189, 1.27260556263, 192.69216761850],
        [0.179, 6.15189171649, 2.92076130680],
        [0.178, 2.01622328964, 705.11765732570],
        [0.181, 3.62483757675, 233.90602325750],
        [0.188, 2.92836809840, 227.31374111850],
        [0.164, 3.50682537694, 1382.88734684660],
        [0.166, 5.85452227121, 637.44960575920],
        [0.160, 0.13309484488, 431.26405732200],
        [0.158, 5.92242110049, 96.87299909510],
        [0.178, 4.55557913565, 46.47042291600],
        [0.157, 1.35908014451, 51.20572533120],
        [0.155, 6.24092514222, 464.73122651380],
        [0.155, 6.02684189458, 1286.90811962880],
        [0.155, 1.36669731999, 206.93630796260],
        [0.175, 4.95121713507, 1905.46476494040],
        [0.153, 6.06094547271, 561.18353448360],
        [0.208, 4.50537355579, 24.37902238820],
        [0.185, 5.49802440713, 205.66428357540],
        [0.160, 4.18196878816, 3340.61242669980],
        [0.160, 2.85370771355, 209.15449385380],
        [0.161, 4.98020340619, 2648.45482547300],
        [0.152, 0.85558875667, 570.74476203920],
        [0.156, 2.03601440129, 217.44369702220],
        [0.192, 3.98017256784, 212.40532356070],
        [0.192, 0.90945359875, 214.19286731530],
        [0.198, 3.54289000289, 533.62311835770],
        [0.160, 2.51522796187, 3127.31333126180],
        [0.145, 0.93414637377, 1994.33044515740],
        [0.141, 5.66801998888, 120.35824960600],
        [0.141, 3.88995778619, 454.90936652730],
        [0.152, 1.01453902153, 2840.41327990860],
        [0.172, 4.15145223592, 2.96894541660],
        [0.146, 5.26789260159, 7.06536289100],
        [0.177, 0.43196690516, 140.00196957900],
        [0.144, 3.95680110579, 300.61086697750],
        [0.152, 4.29475572258, 555.55446019110],
        [0.143, 4.15264164139, 31.01948863700],
        [0.138, 4.20096561019, 731.94436026870],
        [0.139, 0.79924371385, 166.82867252200],
        [0.135, 1.27192121638, 92.94084583200],
        [0.165, 1.94881873062, 107.02492748170],
        [0.153, 3.07590434707, 3480.31056622260],
        [0.125, 3.41796878361, 1802.37199072180],
        [0.128, 5.83700968658, 2324.94940881560],
        [0.129, 2.75851443754, 480.77286162380],
        [0.122, 4.75728514255, 2854.64037391020],
        [0.129, 4.67730374872, 913.96333463880],
        [0.121, 1.19957548726, 572.22923474750],
        [0.146, 3.71232877850, 546.95644048200],
        [0.120, 4.59083886034, 339.28641933650],
        [0.127, 6.19372294369, 59.80374504030],
        [0.123, 5.98484393970, 477.80391620720],
        [0.122, 5.82973501131, 990.22940591440],
        [0.151, 2.39413061881, 2524.02141025200],
        [0.147, 4.98199291770, 850.01498801430],
        [0.127, 0.47350572907, 6.59228213900],
        [0.116, 4.71488890981, 1130.23137549340],
        [0.129, 5.42665311725, 2538.24850425360],
        [0.111, 4.51520219898, 1699.27921650320],
        [0.114, 2.06120434865, 952.09637027660],
        [0.112, 4.14736642579, 422.40540518200],
        [0.111, 5.53230411085, 857.12853501510],
        [0.109, 3.00005651288, 420.44785172170],
        [0.112, 2.20667724213, 395.57870223900],
        [0.147, 1.52324472511, 552.58551477450],
        [0.118, 5.47495367121, 2957.73314812880],
        [0.113, 2.57036693965, 462.02291352810],
        [0.122, 4.96246567897, 638.93407846750],
        [0.104, 1.91139383428, 472.17484191470],
        [0.116, 2.82742160564, 450.97721326420],
        [0.115, 2.26043201622, 1781.03134971940],
        [0.110, 4.86686492403, 2914.01423582380],
        [0.109, 4.43848727280, 405.99126305650],
        [0.102, 5.90112611078, 99.91138048090],
        [0.101, 2.53392410330, 640.86049416050],
        [0.112, 5.53802838779, 381.35160823740],
        [0.099, 5.92199927896, 411.62033734900],
        [0.100, 5.21941099517, 426.48631629140],
        [0.137, 2.20269111622, 7.16173111060],
        [0.097, 1.27914551364, 2847.52682690940],
        [0.115, 5.22953781515, 1119.18567522950],
        [0.095, 4.26135357007, 540.73666535850],
        [0.098, 5.27107833435, 639.94547042380],
        [0.107, 4.38879925113, 412.58354519550],
        [0.093, 5.35954173624, 334.55111692130],
        [0.094, 1.16749092536, 5643.17856367740],
        [0.106, 4.19443004843, 486.40193591630],
        [0.096, 0.59816870672, 714.67888488130],
        [0.094, 0.54205024076, 423.62924545940],
        [0.109, 2.82817225044, 468.24268865160],
        [0.083, 6.12100285205, 380.12776796000],
        [0.084, 2.20217125255, 909.81873305460],
        [0.085, 5.20920130934, 562.14674233010],
        [0.105, 2.66415710279, 460.53844081980],
        [0.084, 0.14646013561, 681.54178408960],
        [0.080, 3.03551986945, 409.92341631960],
        [0.097, 5.09373549436, 92.04707395470],
        [0.110, 2.03622569317, 642.34496686880],
        [0.080, 5.71035549752, 361.37781986430],
        [0.084, 3.00961145133, 426.81063919710],
        [0.085, 4.28770375688, 135.54855145410],
        [0.093, 5.32943472274, 432.01481684740],
        [0.086, 1.51247258028, 760.25553592000],
        [0.084, 5.83905303748, 426.38574255490],
        [0.100, 3.62925349363, 426.71006546060],
        [0.094, 4.30151510535, 3377.21779200400],
        [0.098, 2.07334671974, 639.84910220420],
        [0.080, 4.11576173565, 774.48262992160],
        [0.075, 2.89122656610, 806.72595883600],
        [0.080, 0.88468467902, 856.37777548970],
        [0.072, 4.85259171933, 392.65794093220],
        [0.083, 0.11133738383, 402.21916848780],
    ],
    # B1
    [
        [397554.998, 5.33289992556, 213.29909543800],
        [49478.641, 3.14159265359, 0.00000000000],
        [18571.607, 6.09919206378, 426.59819087600],
        [14800.587, 2.30586060520, 206.18554843720],
        [9643.981, 1.69674660120, 220.41264243880],
        [3757.161, 1.25429514018, 419.48464387520],
        [2716.647, 5.91166664787, 639.89728631400],
        [1455.309, 0.85161616532, 433.71173787680],
        [1290.595, 2.91770857090, 7.11354700080],
        [852.630, 0.43572078997, 316.39186965660],
        [284.386, 1.61881754773, 227.52618943960],
        [292.185, 5.31574251270, 853.19638175200],
        [275.090, 3.88864137336, 103.09277421860],
        [297.726, 0.91909206723, 632.78373931320],
        [172.359, 0.05215146556, 647.01083331480],
        [127.731, 1.20711452525, 529.69096509460],
        [166.237, 2.44351613165, 199.07200143640],
        [158.220, 5.20850125766, 110.20632121940],
        [109.839, 2.45695551627, 217.23124870110],
        [81.759, 2.75839171353, 210.11770170030],
        [81.010, 2.86038377187, 14.22709400160],
        [68.658, 1.65537623146, 202.25339517410],
        [59.281, 1.82410768234, 323.50541665740],
        [65.161, 1.25527521313, 216.48048917570],
        [61.024, 1.25273412095, 209.36694217490],
        [46.386, 0.81534705304, 440.82528487760],
        [36.163, 1.81851057689, 224.34479570190],
        [34.041, 2.83971297997, 117.31986822020],
        [32.164, 1.18676132343, 846.08283475120],
        [33.114, 1.30557080010, 412.37109687440],
        [27.282, 4.64744847591, 1066.49547719000],
        [22.805, 4.12923703368, 415.55249061210],
        [27.128, 4.44228739187, 11.04570026390],
        [18.100, 5.56392353608, 860.30992875280],
        [20.851, 1.40999273740, 309.27832265580],
        [14.947, 1.34302610607, 95.97922721780],
        [15.316, 1.22393617996, 63.73589830340],
        [14.601, 1.00753704970, 536.80451209540],
        [12.842, 2.27059911053, 742.99006053260],
        [12.832, 4.88898877901, 522.57741809380],
        [13.137, 2.45991904379, 490.33408917940],
        [11.883, 1.87308666696, 423.41679713830],
        [13.027, 3.21731634178, 277.03499374140],
        [9.946, 3.11650057543, 625.67019231240],
        [12.710, 0.29501589197, 422.66603761290],
        [9.644, 1.74586356703, 330.61896365820],
        [8.079, 2.41931187953, 430.53034413910],
        [8.245, 4.68121931659, 215.74677599280],
        [8.958, 0.46482448501, 429.77958461370],
        [6.547, 3.01351967549, 949.17560896980],
        [7.251, 5.97098186912, 149.56319713460],
        [6.056, 1.49115011100, 234.63973644040],
        [5.791, 5.36720639912, 735.87651353180],
        [5.994, 0.02442871989, 654.12438031560],
        [6.647, 3.90879134581, 351.81659230870],
        [6.824, 1.52456408861, 437.64389113990],
        [5.134, 3.81149834833, 74.78159856730],
        [3.959, 5.63505813057, 210.85141488320],
        [3.811, 2.63992803111, 3.18139373770],
        [3.643, 1.73267151007, 1059.38193018920],
        [3.554, 4.98621474362, 3.93215326310],
        [4.568, 4.33599514584, 628.85158605010],
        [3.145, 2.51404811765, 1162.47470440780],
        [3.522, 1.16093567319, 223.59403617650],
        [2.933, 2.06057834252, 956.28915597060],
        [2.644, 5.62559379305, 203.73786788240],
        [2.992, 5.06312015437, 515.46387109300],
        [2.304, 2.73123930535, 21.34064100240],
        [2.168, 2.91805928238, 203.00415469950],
        [2.398, 3.99421633537, 1279.79457262800],
        [2.146, 0.87500689888, 408.43894361130],
        [2.074, 1.65731069687, 137.03302416240],
        [1.797, 1.56879308343, 124.43341522100],
        [2.088, 1.85721384366, 138.51749687070],
        [1.769, 4.82294294946, 1073.60902419080],
        [1.635, 1.20387813348, 88.86568021700],
        [2.202, 5.93027042684, 1052.26838318840],
        [1.843, 0.22126910774, 750.10360753340],
        [1.851, 2.45470409290, 340.77089204480],
        [1.890, 0.41025631859, 127.47179660680],
        [1.582, 5.63360832372, 214.26230328450],
        [1.920, 3.77935901504, 350.33211960040],
        [1.786, 5.78644477326, 635.96513305090],
        [1.497, 3.13026893210, 703.63318461740],
        [1.583, 3.46882532865, 38.13303563780],
        [1.577, 4.02973226017, 388.46515523820],
        [1.645, 5.59115773632, 483.22054217860],
        [1.405, 4.07880624509, 728.76296653100],
        [1.498, 5.87094430469, 362.86229257260],
        [1.317, 2.22386203585, 213.34727954780],
        [1.321, 2.91534782718, 1265.56747862640],
        [1.307, 5.41748323885, 217.96496188400],
        [1.483, 0.91111666841, 543.91805909620],
        [1.291, 2.62333801070, 554.06998748280],
        [1.406, 0.34582712649, 85.82729883120],
        [1.287, 2.82247279651, 231.45834270270],
        [1.563, 4.88438049382, 208.63322899200],
        [1.316, 5.30963570131, 213.25091132820],
        [1.164, 1.39531381032, 210.37833413120],
        [1.295, 2.46089213219, 200.76892246580],
        [1.236, 3.03659580871, 838.96928775040],
        [1.449, 4.00934078371, 195.13984817330],
        [1.251, 1.46674521697, 218.71572140940],
        [1.568, 1.89939852487, 212.33588759150],
        [1.067, 5.34734443894, 207.67002114550],
        [1.111, 0.70962013461, 447.93883187840],
        [1.012, 1.37217220640, 636.71589257630],
        [1.163, 6.00108996618, 191.20769491020],
        [0.887, 2.84483069917, 191.95845443560],
        [1.005, 2.72373040634, 1478.86657406440],
        [0.879, 1.19585734916, 417.03696332040],
        [0.829, 4.94182950387, 497.44763618020],
        [0.878, 6.24981924813, 265.98929347750],
        [0.781, 4.61973017912, 424.15051032120],
        [0.924, 3.64210508536, 222.86032299360],
        [0.971, 2.89404568581, 563.63121503840],
        [0.946, 5.72987725592, 1368.66025284500],
        [0.834, 6.12384852532, 209.10630974400],
        [0.911, 1.06795057723, 650.94298657790],
        [0.731, 0.81660632103, 142.44965013380],
        [0.795, 5.54574074566, 76.26607127560],
        [1.012, 5.97140626297, 643.07868005170],
        [0.703, 2.41479303782, 10.29494073850],
        [0.726, 4.52598413209, 565.11568774670],
        [0.691, 2.31682364985, 160.60889739850],
        [0.695, 0.37889317398, 212.77783057620],
        [0.676, 4.43606270900, 842.15068148810],
        [0.769, 4.47368582271, 52.69019803950],
        [0.658, 0.32331118921, 621.73803904930],
        [0.838, 4.68035046143, 288.08069400530],
        [0.674, 5.63961963995, 867.42347575360],
        [0.695, 1.32062509205, 1169.58825140860],
        [0.841, 2.46995220838, 1375.77379984580],
        [0.633, 3.46145143241, 18.15924726470],
        [0.722, 0.04119071457, 269.92144674060],
        [0.757, 2.15030650611, 207.88246946660],
        [0.692, 0.87072324976, 213.82036029980],
        [0.602, 0.85288769518, 225.82926841020],
        [0.635, 4.76109030475, 831.85574074960],
        [0.701, 1.20410854661, 479.28838891550],
        [0.624, 2.30585779534, 643.82943957710],
        [0.582, 0.18811617696, 1.48447270830],
        [0.513, 2.56525967840, 404.50679034820],
        [0.502, 4.97423814367, 212.54833591260],
        [0.644, 2.10955154583, 1272.68102562720],
        [0.467, 4.47217820662, 235.39049596580],
        [0.566, 4.44740324446, 429.04587143080],
        [0.448, 0.57491120802, 22.09140052780],
        [0.520, 1.15131866397, 337.73251065900],
        [0.476, 4.78513362967, 218.92816973050],
        [0.519, 0.61616177345, 436.89313161450],
        [0.442, 2.10204008144, 416.30325013750],
        [0.536, 1.08779067908, 344.70304530790],
        [0.477, 2.42483193385, 216.21985674480],
        [0.469, 5.22622034028, 942.06206196900],
        [0.392, 0.41137926465, 302.16477565500],
        [0.400, 5.17216478470, 414.06801790380],
        [0.383, 3.58419227076, 1045.15483618760],
        [0.443, 1.11051326413, 425.11371816770],
        [0.517, 3.44547026103, 12.53017297220],
        [0.369, 1.60095273908, 56.62235130260],
        [0.354, 2.79123486392, 1581.95934828300],
        [0.405, 5.93402105921, 173.94221952280],
        [0.319, 4.31850876624, 219.44943459230],
        [0.330, 0.62529198264, 358.93013930950],
        [0.305, 0.82404423420, 1485.98012106520],
        [0.391, 2.59385552893, 1795.25844372100],
        [0.291, 3.12019266878, 217.49188113200],
        [0.294, 2.18552193901, 444.75743814070],
        [0.281, 0.77791302266, 757.21715453420],
        [0.355, 5.44570491928, 1685.05212250160],
        [0.305, 2.65927043884, 355.74874557180],
        [0.269, 6.00323720265, 934.94851496820],
        [0.287, 2.60486363576, 113.38771495710],
        [0.348, 0.98872551635, 70.84944530420],
        [0.331, 5.62133883922, 9.56122755560],
        [0.255, 4.14086605030, 284.14854074220],
        [0.311, 6.27145060602, 207.14875628370],
        [0.267, 4.72606312146, 696.51963761660],
        [0.319, 4.68828119248, 1155.36115740700],
        [0.227, 3.10352674343, 1361.54670584420],
        [0.228, 1.19253095837, 1589.07289528380],
        [0.244, 5.36327976010, 245.54242435240],
        [0.218, 2.16069250901, 177.87437278590],
        [0.254, 4.51652534648, 1148.24761040620],
        [0.211, 2.82699627326, 106.27416795630],
        [0.230, 4.63171743406, 107.02492748170],
        [0.201, 4.52152562223, 508.35032409220],
        [0.230, 5.93560798508, 618.55664531160],
        [0.253, 2.08074949572, 252.65597135320],
        [0.234, 2.43423283339, 1692.16566950240],
        [0.196, 1.01284131123, 114.39910691340],
        [0.196, 2.73629728926, 214.04985496340],
        [0.191, 1.51642829677, 6069.77675455340],
        [0.252, 5.10097595426, 1258.45393162560],
        [0.240, 2.93712394928, 916.93228005540],
        [0.224, 4.42406538277, 251.43213107580],
        [0.223, 2.34400676548, 1677.93857550080],
        [0.228, 5.00073557208, 1574.84580128220],
        [0.183, 5.09056895026, 220.46082654860],
        [0.178, 6.05669760153, 206.13736432740],
        [0.185, 3.73859309582, 114.13847448250],
        [0.200, 1.89409546254, 2420.92863603340],
        [0.158, 2.78517362162, 2008.55753915900],
        [0.191, 2.28724146195, 2435.15573003500],
        [0.188, 4.29343910228, 1471.75302706360],
        [0.155, 5.35194123420, 576.16138801060],
        [0.174, 2.80423114755, 1781.03134971940],
        [0.166, 5.32531835813, 2001.44399215820],
        [0.165, 2.62993712087, 525.49817940060],
        [0.141, 4.73916921092, 501.37978944330],
        [0.161, 5.83368523750, 181.05576652360],
        [0.137, 1.45135867137, 131.54696222180],
        [0.157, 3.22870773657, 1493.09366806600],
        [0.136, 4.20279658293, 710.74673161820],
        [0.144, 4.61003572124, 121.25202148330],
        [0.131, 5.85409245557, 175.16605980020],
        [0.144, 0.96056164245, 214.78356814630],
        [0.146, 3.95687956474, 421.93232443000],
        [0.138, 1.03080573610, 4.66586644600],
        [0.167, 1.97508934614, 62.25142559510],
        [0.123, 2.28640485589, 1898.35121793960],
        [0.168, 4.63122081226, 1891.23767093880],
        [0.129, 0.05327999225, 211.81462272970],
        [0.134, 3.49720535944, 488.84961647110],
        [0.117, 3.43819501459, 436.15941843160],
        [0.125, 5.87007326241, 963.40270297140],
        [0.120, 0.70795300239, 81.75213321620],
        [0.125, 4.50999471095, 2317.83586181480],
        [0.116, 6.11600926571, 558.00214074590],
        [0.117, 4.78666549046, 601.76425067620],
        [0.108, 0.45464469172, 1802.37199072180],
        [0.111, 1.44669239244, 2531.13495725280],
        [0.109, 6.14289264597, 151.04766984290],
        [0.113, 4.05600865495, 1286.90811962880],
        [0.129, 5.13936946160, 849.26422848890],
        [0.127, 3.88189432056, 98.89998852460],
        [0.133, 2.38290634070, 2111.65031337760],
        [0.122, 4.40757611742, 778.41478318470],
        [0.095, 0.07909774752, 213.41097002260],
        [0.095, 1.66925524906, 213.18722085340],
        [0.103, 1.88058957173, 99.16062095550],
        [0.119, 3.62785705509, 248.72381809010],
        [0.090, 4.63029999228, 228.27694896500],
        [0.092, 5.48700119144, 767.36908292080],
        [0.089, 4.61331934339, 431.26405732200],
        [0.099, 3.60670326134, 776.93031047640],
        [0.085, 4.93878023673, 2.44768055480],
        [0.089, 6.24541644164, 661.23792731640],
        [0.085, 0.45896349060, 1382.88734684660],
        [0.088, 3.81144552178, 1788.14489672020],
        [0.103, 3.20558404998, 312.19908396260],
        [0.080, 2.28889729136, 213.08664711690],
        [0.080, 5.74264101239, 213.51154375910],
        [0.082, 3.23546757052, 198.32124191100],
        [0.078, 6.03841191050, 835.03713448730],
        [0.080, 0.22601918692, 427.56139872250],
        [0.072, 2.05164614795, 2634.22773147140],
        [0.091, 5.97938003596, 556.51766803760],
        [0.087, 2.71469794199, 617.80588578620],
    ],
    # B2
    [
        [20629.977, 0.50482422817, 213.29909543800],
        [3719.555, 3.99833475829, 206.18554843720],
        [1627.158, 6.18189939500, 220.41264243880],
        [1346.067, 0.00000000000, 0.00000000000],
        [705.842, 3.03914308836, 419.48464387520],
        [365.042, 5.09928680706, 426.59819087600],
        [329.632, 5.27899210039, 433.71173787680],
        [219.335, 3.82841533795, 639.89728631400],
        [139.393, 1.04272623499, 7.11354700080],
        [103.980, 6.15730992966, 227.52618943960],
        [92.961, 1.97994412845, 316.39186965660],
        [71.242, 4.14754353431, 199.07200143640],
        [51.927, 2.88364833898, 632.78373931320],
        [48.961, 4.43390206741, 647.01083331480],
        [41.373, 3.15927770079, 853.19638175200],
        [28.602, 4.52978327558, 210.11770170030],
        [23.969, 1.11595912146, 14.22709400160],
        [20.511, 4.35095844197, 217.23124870110],
        [19.532, 5.30779711223, 440.82528487760],
        [18.263, 0.85391476786, 110.20632121940],
        [15.742, 4.25767226302, 103.09277421860],
        [16.840, 5.68112084135, 216.48048917570],
        [13.613, 2.99904334066, 412.37109687440],
        [11.567, 2.52679928410, 529.69096509460],
        [7.963, 3.31512423920, 202.25339517410],
        [6.599, 0.28766025146, 323.50541665740],
        [6.312, 1.16121321336, 117.31986822020],
        [5.891, 3.58260177246, 309.27832265580],
        [6.648, 5.55714129949, 209.36694217490],
        [5.590, 2.47783944511, 1066.49547719000],
        [6.192, 3.61231886519, 860.30992875280],
        [4.231, 3.02212363572, 846.08283475120],
        [3.612, 4.79935735435, 625.67019231240],
        [3.398, 3.76732731354, 423.41679713830],
        [3.387, 6.04222745633, 234.63973644040],
        [2.578, 5.63610668746, 735.87651353180],
        [2.831, 4.81642822334, 429.77958461370],
        [2.817, 4.47516563908, 654.12438031560],
        [2.573, 0.22467245054, 522.57741809380],
        [2.610, 3.29126967191, 95.97922721780],
        [2.419, 0.02986335489, 415.55249061210],
        [2.112, 4.55964179603, 422.66603761290],
        [2.304, 6.25081073546, 330.61896365820],
        [1.758, 5.53430456858, 536.80451209540],
        [1.814, 5.05675881426, 277.03499374140],
        [1.550, 5.60375604692, 223.59403617650],
        [1.457, 4.47767649852, 430.53034413910],
        [1.607, 5.53599550100, 224.34479570190],
        [1.172, 4.71017775994, 203.00415469950],
        [1.231, 0.25115931880, 3.93215326310],
        [1.105, 1.01595427676, 21.34064100240],
        [0.868, 4.84623483952, 949.17560896980],
        [0.939, 1.35429452093, 742.99006053260],
        [0.693, 6.03599130692, 124.43341522100],
        [0.712, 4.45550701473, 191.95845443560],
        [0.690, 5.44243765037, 437.64389113990],
        [0.810, 0.46198177342, 515.46387109300],
        [0.694, 5.23748122403, 447.93883187840],
        [0.604, 2.95749705544, 88.86568021700],
        [0.669, 0.08457977809, 215.74677599280],
        [0.579, 0.65329445948, 3.18139373770],
        [0.712, 6.05964117622, 11.04570026390],
        [0.698, 2.91371419321, 1073.60902419080],
        [0.526, 2.24947851818, 1059.38193018920],
        [0.511, 2.86838724347, 408.43894361130],
        [0.589, 5.79268515755, 63.73589830340],
        [0.519, 1.76641574095, 1279.79457262800],
        [0.503, 5.73762297081, 728.76296653100],
        [0.482, 4.68234512154, 838.96928775040],
        [0.494, 4.04363805503, 490.33408917940],
        [0.458, 1.17998315936, 210.85141488320],
        [0.380, 5.28045750432, 1052.26838318840],
        [0.404, 4.58953258519, 302.16477565500],
        [0.377, 5.20131800999, 74.78159856730],
        [0.328, 0.11893501088, 956.28915597060],
        [0.290, 3.99300398632, 1162.47470440780],
        [0.262, 2.04320741578, 1471.75302706360],
        [0.259, 3.76206113036, 635.96513305090],
        [0.254, 0.16694559092, 195.13984817330],
        [0.309, 5.44921175960, 543.91805909620],
        [0.237, 1.27761853769, 231.45834270270],
        [0.288, 1.32449995239, 203.73786788240],
        [0.229, 4.19748765966, 1265.56747862640],
        [0.238, 4.02925601887, 643.07868005170],
        [0.238, 0.49997895983, 10.29494073850],
        [0.257, 3.69107889837, 867.42347575360],
        [0.191, 0.17807919948, 628.85158605010],
        [0.246, 5.62469599682, 351.81659230870],
        [0.183, 3.38184740572, 636.71589257630],
        [0.172, 3.83173494030, 1581.95934828300],
        [0.220, 1.03443668151, 483.22054217860],
        [0.217, 4.65210162713, 750.10360753340],
        [0.143, 2.31969979791, 18.15924726470],
        [0.137, 5.50046852846, 1169.58825140860],
        [0.120, 3.70151294359, 416.30325013750],
        [0.136, 3.38453909352, 1155.36115740700],
        [0.149, 0.85459831932, 1375.77379984580],
        [0.150, 5.71949902293, 618.55664531160],
        [0.125, 4.82446274394, 436.89313161450],
        [0.120, 3.26968058035, 1478.86657406440],
        [0.131, 0.11496484259, 1898.35121793960],
        [0.099, 4.57241894541, 643.82943957710],
        [0.095, 4.92115458463, 650.94298657790],
        [0.090, 2.09300085806, 621.73803904930],
        [0.111, 0.11975665259, 831.85574074960],
        [0.089, 2.54351587616, 85.82729883120],
        [0.080, 5.09103451442, 340.77089204480],
        [0.078, 3.17395501851, 497.44763618020],
        [0.085, 0.18997660997, 1258.45393162560],
        [0.081, 1.16732337173, 217.96496188400],
        [0.072, 5.47328223678, 337.73251065900],
    ],
    # B3
    [
        [666.252, 1.99006340181, 213.29909543800],
        [632.350, 5.69778316807, 206.18554843720],
        [398.051, 0.00000000000, 0.00000000000],
        [187.838, 4.33779804809, 220.41264243880],
        [91.884, 4.84104208217, 419.48464387520],
        [42.369, 2.38073239056, 426.59819087600],
        [51.548, 3.42149490328, 433.71173787680],
        [25.661, 4.40167213109, 227.52618943960],
        [20.551, 5.85313509872, 199.07200143640],
        [18.081, 1.99321433229, 639.89728631400],
        [10.874, 5.37344546547, 7.11354700080],
        [9.590, 2.54901825866, 647.01083331480],
        [7.085, 3.45518372721, 316.39186965660],
        [6.002, 4.80055225135, 632.78373931320],
        [5.778, 0.01680378777, 210.11770170030],
        [4.881, 5.63719730884, 14.22709400160],
        [4.501, 1.22424419010, 853.19638175200],
        [5.542, 3.51756747774, 440.82528487760],
        [3.548, 4.71299370890, 412.37109687440],
        [2.851, 0.62679207578, 103.09277421860],
        [2.173, 3.71982274459, 216.48048917570],
        [1.991, 6.10867071657, 217.23124870110],
        [1.435, 1.69177141453, 860.30992875280],
        [1.217, 4.30778838827, 234.63973644040],
        [1.157, 5.75027789902, 309.27832265580],
        [0.795, 5.69026441157, 117.31986822020],
        [0.733, 0.59842720676, 1066.49547719000],
        [0.713, 0.21700311697, 625.67019231240],
        [0.773, 5.48361981990, 202.25339517410],
        [0.897, 2.65577866867, 654.12438031560],
        [0.509, 2.86079833766, 429.77958461370],
        [0.462, 4.17742567173, 529.69096509460],
        [0.390, 6.11288036049, 191.95845443560],
        [0.505, 4.51905764563, 323.50541665740],
        [0.379, 3.74436004151, 223.59403617650],
        [0.332, 5.49370890570, 21.34064100240],
        [0.377, 5.25624813434, 95.97922721780],
        [0.384, 4.48187414769, 330.61896365820],
        [0.367, 5.03190929680, 846.08283475120],
        [0.281, 1.14133888637, 735.87651353180],
        [0.245, 5.81618253250, 423.41679713830],
        [0.241, 1.70335120180, 522.57741809380],
        [0.258, 3.69110118716, 447.93883187840],
        [0.231, 4.15697626494, 110.20632121940],
        [0.305, 5.97746884029, 302.16477565500],
        [0.284, 0.66224572127, 203.00415469950],
        [0.204, 1.54683820621, 209.36694217490],
        [0.194, 4.21193801453, 124.43341522100],
        [0.145, 4.79689259614, 88.86568021700],
        [0.151, 3.82010884134, 536.80451209540],
        [0.100, 0.03596545368, 949.17560896980],
        [0.097, 0.91303450276, 1073.60902419080],
        [0.110, 2.21197473966, 515.46387109300],
        [0.084, 2.53842533109, 422.66603761290],
        [0.085, 5.11102520704, 3.93215326310],
        [0.077, 6.04074586787, 838.96928775040],
        [0.085, 1.18898817378, 728.76296653100],
        [0.084, 4.10158366806, 224.34479570190],
    ],
    # B4
    [
        [80.384, 1.11918414679, 206.18554843720],
        [31.660, 3.12218745098, 213.29909543800],
        [17.143, 2.48073200414, 220.41264243880],
        [11.844, 3.14159265359, 0.00000000000],
        [9.005, 0.38441424927, 419.48464387520],
        [6.164, 1.56186379537, 433.71173787680],
        [4.660, 1.28235639570, 199.07200143640],
        [4.775, 2.63498295487, 227.52618943960],
        [1.487, 1.43096671616, 426.59819087600],
        [1.424, 0.66988083613, 647.01083331480],
        [1.075, 6.18092274059, 639.89728631400],
        [1.145, 1.72041928134, 440.82528487760],
        [0.682, 3.84841098180, 14.22709400160],
        [0.655, 3.49486258327, 7.11354700080],
        [0.456, 0.47338193402, 632.78373931320],
        [0.509, 0.31432285584, 412.37109687440],
        [0.343, 5.86413875355, 853.19638175200],
        [0.270, 2.50125594913, 234.63973644040],
        [0.197, 5.39156324804, 316.39186965660],
        [0.236, 2.11084590211, 210.11770170030],
        [0.172, 6.09682874401, 860.30992875280],
        [0.159, 5.95049154821, 216.48048917570],
        [0.100, 1.98534903594, 625.67019231240],
        [0.112, 0.85526419268, 654.12438031560],
        [0.115, 5.03884718594, 117.31986822020],
        [0.115, 0.44589613974, 110.20632121940],
    ],
    # B5
    [
        [7.895, 2.81927558645, 206.18554843720],
        [1.014, 0.51187210270, 220.41264243880],
        [0.772, 2.99484124049, 199.07200143640],
        [0.967, 3.14159265359, 0.00000000000],
        [0.583, 5.96456944075, 433.71173787680],
        [0.588, 0.78008666397, 227.52618943960],
        [0.445, 2.38630799074, 419.48464387520],
        [0.098, 5.10622131539, 647.01083331480],
        [0.091, 5.81659714144, 7.11354700080],
        [0.088, 6.17828532308, 440.82528487760],
        [0.089, 0.58396864530, 213.29909543800],
    ],
]
"""This table contains Saturn's periodic terms (all of them) from the planetary
theory VSOP87 for the heliocentric latitude at the equinox of date (taken from
the 'D' solution). In Meeus' book a shortened version can be found in pages
440-442."""


VSOP87_R = [
    # R0
    [
        [955758135.801, 0.00000000000, 0.00000000000],
        [52921382.465, 2.39226219733, 213.29909543800],
        [1873679.934, 5.23549605091, 206.18554843720],
        [1464663.959, 1.64763045468, 426.59819087600],
        [821891.059, 5.93520025371, 316.39186965660],
        [547506.899, 5.01532628454, 103.09277421860],
        [371684.449, 2.27114833428, 220.41264243880],
        [361778.433, 3.13904303264, 7.11354700080],
        [140617.548, 5.70406652991, 632.78373931320],
        [108974.737, 3.29313595577, 110.20632121940],
        [69007.015, 5.94099622447, 419.48464387520],
        [61053.350, 0.94037761156, 639.89728631400],
        [48913.044, 1.55733388472, 202.25339517410],
        [34143.794, 0.19518550682, 277.03499374140],
        [32401.718, 5.47084606947, 949.17560896980],
        [20936.573, 0.46349163993, 735.87651353180],
        [20839.118, 1.52102590640, 433.71173787680],
        [20746.678, 5.33255667599, 199.07200143640],
        [15298.457, 3.05943652881, 529.69096509460],
        [14296.479, 2.60433537909, 323.50541665740],
        [11993.314, 5.98051421881, 846.08283475120],
        [11380.261, 1.73105746566, 522.57741809380],
        [12884.128, 1.64892310393, 138.51749687070],
        [7752.769, 5.85191318903, 95.97922721780],
        [9796.061, 5.20475863996, 1265.56747862640],
        [6465.967, 0.17733160145, 1052.26838318840],
        [6770.621, 3.00433479284, 14.22709400160],
        [5850.443, 1.45519636076, 415.55249061210],
        [5307.481, 0.59737534050, 63.73589830340],
        [4695.746, 2.14919036956, 227.52618943960],
        [4043.988, 1.64010323863, 209.36694217490],
        [3688.132, 0.78016133170, 412.37109687440],
        [3376.457, 3.69528478828, 224.34479570190],
        [2885.348, 1.38764077631, 838.96928775040],
        [2976.033, 5.68467931117, 210.11770170030],
        [3419.551, 4.94549148887, 1581.95934828300],
        [3460.943, 1.85088802878, 175.16605980020],
        [3400.616, 0.55386747515, 350.33211960040],
        [2507.630, 3.53851863255, 742.99006053260],
        [2448.325, 6.18412386316, 1368.66025284500],
        [2406.138, 2.96559220267, 117.31986822020],
        [2881.181, 0.17960757891, 853.19638175200],
        [2173.959, 0.01508587396, 340.77089204480],
        [2024.483, 5.05411271271, 11.04570026390],
        [1740.254, 2.34657043464, 309.27832265580],
        [1861.397, 5.93361638244, 625.67019231240],
        [1888.436, 0.02968443389, 3.93215326310],
        [1610.859, 1.17302463549, 74.78159856730],
        [1462.631, 1.92588134017, 216.48048917570],
        [1474.547, 5.67670461130, 203.73786788240],
        [1395.109, 5.93669404929, 127.47179660680],
        [1781.165, 0.76314388077, 217.23124870110],
        [1817.186, 5.77713225779, 490.33408917940],
        [1472.392, 1.40064915651, 137.03302416240],
        [1304.089, 0.77235613966, 647.01083331480],
        [1149.773, 5.74021249703, 1162.47470440780],
        [1126.667, 4.46707803791, 265.98929347750],
        [1277.489, 2.98412586423, 1059.38193018920],
        [1207.053, 0.75285933160, 351.81659230870],
        [1071.399, 1.13567265104, 1155.36115740700],
        [1020.922, 5.91233512844, 1685.05212250160],
        [1315.042, 5.11202572637, 211.81462272970],
        [1295.553, 4.69184139933, 1898.35121793960],
        [1099.037, 1.81765118601, 149.56319713460],
        [998.462, 2.63131596867, 200.76892246580],
        [985.869, 2.25992849742, 956.28915597060],
        [932.434, 3.66980793184, 554.06998748280],
        [664.481, 0.60297724821, 728.76296653100],
        [659.850, 4.66635439533, 195.13984817330],
        [617.740, 5.62092000007, 942.06206196900],
        [626.382, 5.94208232590, 1478.86657406440],
        [482.230, 1.84070179496, 479.28838891550],
        [487.689, 2.79373616806, 3.18139373770],
        [470.086, 0.83847755040, 1471.75302706360],
        [451.817, 5.64468459871, 2001.44399215820],
        [553.128, 3.41088600844, 269.92144674060],
        [534.397, 1.26443331367, 275.55052103310],
        [472.572, 1.88198584660, 515.46387109300],
        [405.434, 1.64001413521, 536.80451209540],
        [517.196, 4.44310450526, 2214.74308759620],
        [452.848, 3.00349117198, 302.16477565500],
        [494.340, 2.28626675074, 278.51946644970],
        [489.825, 5.80631420383, 191.20769491020],
        [427.459, 0.05741344372, 284.14854074220],
        [339.763, 1.40198657693, 440.82528487760],
        [340.627, 0.89091104306, 628.85158605010],
        [385.974, 1.99700402508, 1272.68102562720],
        [288.298, 1.12160250272, 422.66603761290],
        [294.444, 0.42577061903, 312.19908396260],
        [262.490, 0.31753439818, 1045.15483618760],
        [295.331, 0.67144493789, 88.86568021700],
        [342.968, 5.85600322299, 1795.25844372100],
        [341.117, 2.37585247250, 525.49817940060],
        [234.018, 4.22756813216, 114.13847448250],
        [223.729, 2.28129446763, 330.61896365820],
        [275.814, 0.47832439352, 38.13303563780],
        [224.592, 0.54754005675, 1788.14489672020],
        [303.300, 0.87946670205, 6069.77675455340],
        [292.103, 6.21420611920, 210.85141488320],
        [226.121, 0.37495223398, 142.44965013380],
        [277.257, 5.31917702012, 692.58748435350],
        [242.911, 5.37187983246, 1258.45393162560],
        [205.571, 0.95755250527, 288.08069400530],
        [207.567, 5.38126259725, 2317.83586181480],
        [186.835, 6.03591766061, 404.50679034820],
        [218.536, 5.25607043545, 212.33588759150],
        [222.155, 5.94588016768, 39.35687591520],
        [179.673, 4.41045924362, 408.43894361130],
        [241.440, 1.12525868110, 388.46515523820],
        [197.093, 3.90141942850, 52.69019803950],
        [236.639, 0.90802744873, 1375.77379984580],
        [171.915, 5.56318632797, 213.34727954780],
        [169.865, 2.85667554010, 99.16062095550],
        [214.398, 4.20253525974, 2531.13495725280],
        [172.010, 2.36537801012, 213.25091132820],
        [165.707, 2.63679789706, 215.74677599280],
        [230.892, 5.49463421262, 191.95845443560],
        [177.585, 0.38155817719, 430.53034413910],
        [191.514, 2.95906900704, 437.64389113990],
        [163.250, 3.45832517280, 617.80588578620],
        [162.305, 5.73050678664, 203.00415469950],
        [175.108, 5.71404465044, 1066.49547719000],
        [183.041, 5.66851947172, 2111.65031337760],
        [150.077, 4.40663921925, 417.03696332040],
        [187.935, 6.07916265661, 563.63121503840],
        [145.127, 5.08176368814, 423.41679713830],
        [137.491, 5.43912787991, 222.86032299360],
        [172.824, 1.84920994090, 1589.07289528380],
        [165.478, 2.89132196119, 214.26230328450],
        [145.727, 1.56565192483, 831.85574074960],
        [176.864, 2.30323752987, 9999.98645077300],
        [128.877, 2.55338644107, 414.06801790380],
        [120.093, 0.04329750542, 1361.54670584420],
        [143.441, 0.99817357720, 76.26607127560],
        [108.747, 2.09282278191, 207.67002114550],
        [132.106, 2.85902597898, 312.45971639350],
        [112.238, 0.26221759151, 2104.53676637680],
        [125.186, 4.78354048063, 205.22234059070],
        [104.427, 3.63671899047, 65.22037101170],
        [107.447, 3.67064138701, 212.77783057620],
        [108.642, 2.85492389024, 21.34064100240],
        [97.743, 5.12231845599, 2634.22773147140],
        [109.097, 1.63231061493, 208.63322899200],
        [96.852, 4.19928280035, 305.34616939270],
        [96.507, 2.56002066845, 1692.16566950240],
        [85.829, 4.54545085982, 210.37833413120],
        [99.249, 5.13816222131, 1574.84580128220],
        [112.532, 5.03109281265, 703.63318461740],
        [84.023, 1.18337717265, 429.77958461370],
        [89.021, 5.38791571457, 107.02492748170],
        [110.191, 2.43656081234, 355.74874557180],
        [90.659, 4.20908809746, 213.82036029980],
        [95.885, 5.44594259071, 2428.04218303420],
        [94.109, 2.39786381418, 483.22054217860],
        [85.609, 0.03354346966, 860.30992875280],
        [88.796, 4.05766306750, 128.95626931510],
        [81.951, 1.66499731549, 62.25142559510],
        [91.240, 3.96942332591, 2847.52682690940],
        [83.961, 4.60845858022, 177.87437278590],
        [88.376, 3.86800515885, 140.00196957900],
        [93.308, 0.73846639887, 831.10498122420],
        [91.872, 2.94977605320, 35.42472265210],
        [87.077, 1.33390590052, 1905.46476494040],
        [96.584, 4.84438390997, 131.40394986990],
        [71.010, 0.99334817658, 405.25754987360],
        [95.266, 2.51506908152, 2.44768055480],
        [72.514, 4.63213873657, 245.54242435240],
        [82.580, 1.52823217919, 145.63104387150],
        [76.693, 3.15240783008, 767.36908292080],
        [70.317, 4.04253707270, 173.94221952280],
        [86.015, 2.30103727270, 85.82729883120],
        [66.529, 4.75053522835, 70.84944530420],
        [65.835, 2.46869725001, 280.96714700450],
        [64.824, 0.09343869325, 9.56122755560],
        [71.557, 0.01212415296, 565.11568774670],
        [66.533, 1.08034871114, 339.28641933650],
        [63.488, 2.01740971153, 234.63973644040],
        [60.786, 5.12026947473, 756.32338265690],
        [58.123, 6.05732868566, 1677.93857550080],
        [64.236, 1.28586474622, 1148.24761040620],
        [73.124, 4.37810889148, 425.11371816770],
        [55.012, 3.85865703217, 342.25536475310],
        [57.101, 6.26689214029, 2420.92863603340],
        [64.090, 4.09854757476, 327.43756992050],
        [55.306, 1.60456896521, 543.02428721890],
        [57.987, 5.47269124340, 347.88443904560],
        [73.581, 3.72292337326, 92.04707395470],
        [73.760, 3.57045342615, 1.48447270830],
        [64.940, 2.44739629174, 267.47376618580],
        [54.414, 3.71479080197, 344.70304530790],
        [49.783, 3.93453970179, 192.69216761850],
        [49.537, 3.22831070579, 333.65734504400],
        [47.539, 3.92925402178, 199.28444975750],
        [49.368, 4.90341763553, 217.49188113200],
        [62.711, 4.40120079629, 214.78356814630],
        [46.359, 2.09430260266, 212.54833591260],
        [46.289, 2.64038453480, 10.29494073850],
        [54.335, 1.07179534996, 362.86229257260],
        [58.742, 2.62270940799, 225.82926841020],
        [48.457, 3.15166418511, 216.21985674480],
        [46.316, 4.86226642770, 2950.61960112800],
        [45.970, 4.97297391881, 198.32124191100],
        [46.678, 2.44960215701, 207.14875628370],
        [44.905, 1.77616995803, 223.59403617650],
        [44.521, 5.55987055442, 264.50482076920],
        [55.914, 4.29520232351, 329.72519178090],
        [49.643, 5.20789299388, 2744.43405269080],
        [58.829, 4.23073947869, 700.66423920080],
        [52.629, 3.79230629070, 343.21857259960],
        [41.532, 0.74488808688, 125.98732389850],
        [47.767, 2.39260015876, 207.88246946660],
        [56.157, 2.07214273531, 124.43341522100],
        [43.345, 1.83707598036, 106.27416795630],
        [39.793, 4.00870764324, 12.53017297220],
        [53.882, 4.97905460628, 134.58534360760],
        [50.135, 5.75914508514, 320.32402291970],
        [44.960, 5.35721924134, 218.92816973050],
        [41.089, 4.92252591399, 1891.23767093880],
        [46.509, 2.06623129884, 2008.55753915900],
        [42.949, 0.39856812529, 357.44566660120],
        [37.992, 2.06495914285, 247.23934538180],
        [48.733, 5.32762223699, 3127.31333126180],
        [34.583, 5.62555932761, 99.91138048090],
        [41.092, 2.47264897370, 237.67811782620],
        [40.763, 4.08408559215, 621.73803904930],
        [34.213, 0.73077393007, 750.10360753340],
        [33.967, 5.31264617621, 206.23373254700],
        [36.509, 1.68826775750, 22.09140052780],
        [39.361, 3.45730719990, 241.61027108930],
        [34.796, 2.24780137629, 487.36514376280],
        [33.049, 4.86593901955, 209.10630974400],
        [32.584, 2.22713131846, 319.57326339430],
        [39.035, 3.73870591196, 3163.91869656600],
        [32.722, 1.06640549236, 252.65597135320],
        [38.671, 4.39617126814, 18.15924726470],
        [34.514, 1.82607500690, 380.12776796000],
        [41.539, 0.08136234251, 210.33015002140],
        [33.527, 5.80475568528, 251.43213107580],
        [31.221, 1.96489151107, 244.31858407500],
        [30.521, 2.26854188579, 1169.58825140860],
        [34.828, 5.96324553131, 217.96496188400],
        [38.481, 4.43707551964, 160.60889739850],
        [35.998, 3.83262381556, 56.62235130260],
        [31.041, 4.89914223233, 144.14657116320],
        [32.342, 3.58191018804, 231.45834270270],
        [28.838, 5.80081031514, 1994.33044515740],
        [32.175, 2.13166877923, 206.13736432740],
        [32.643, 1.93131580544, 98.89998852460],
        [34.917, 5.65276617691, 497.44763618020],
        [28.928, 2.21653288920, 14.97785352700],
        [31.569, 3.81846560564, 73.29712585900],
        [32.199, 0.99811846290, 1464.63948006280],
        [29.153, 5.98414099408, 2737.32050569000],
        [36.706, 4.75493516597, 348.84764689210],
        [28.665, 1.68732054583, 78.71375183040],
        [27.501, 6.12086395418, 214.04985496340],
        [28.795, 0.04448605904, 5.62907429250],
        [27.205, 0.24587543816, 313.21047591890],
        [32.441, 3.77921585847, 33.94024994380],
        [27.088, 5.20310098020, 148.07872442630],
        [34.956, 3.43886187587, 273.10284047830],
        [33.076, 2.44662095168, 969.62247809490],
        [27.745, 1.44598606685, 258.87574647670],
        [27.178, 4.25918596220, 179.35884549420],
        [27.872, 0.78772093522, 546.95644048200],
        [29.106, 4.83947711462, 905.88657979150],
        [27.417, 2.44930366818, 254.94359321360],
        [34.296, 6.00920969644, 166.82867252200],
        [28.859, 6.02917249910, 188.92007304980],
        [26.001, 0.65046992484, 654.12438031560],
        [33.560, 1.23732329127, 2221.85663459700],
        [24.356, 0.52248751330, 894.84087952760],
        [27.767, 5.17820678484, 5.41662597140],
        [25.568, 3.35897159622, 0.96320784650],
        [22.879, 3.51293480690, 458.84151979040],
        [24.496, 0.00976884124, 69.15252427480],
        [28.794, 0.75545700854, 488.84961647110],
        [31.228, 2.05299907796, 282.45161971280],
        [25.438, 5.29037729250, 636.71589257630],
        [25.332, 4.97007969450, 3060.82592234740],
        [23.596, 2.54766434769, 196.62432088160],
        [29.602, 3.92688207792, 206.70681329900],
        [28.255, 2.72125009693, 32.24332891440],
        [22.115, 4.75775237642, 213.18722085340],
        [22.130, 3.25436709191, 681.54178408960],
        [21.675, 4.61403328597, 3267.01147078460],
        [22.115, 3.16759500067, 213.41097002260],
        [26.912, 2.86269769133, 24.37902238820],
        [20.737, 1.66895754198, 274.06604832480],
        [28.309, 4.73122154345, 552.58551477450],
        [25.252, 5.11986371899, 168.05251279940],
        [26.364, 1.59272536419, 491.81856188770],
        [21.995, 0.88079009280, 635.96513305090],
        [27.076, 5.53694832022, 555.55446019110],
        [19.683, 2.14388519695, 54.17467074780],
        [27.266, 3.57891326986, 561.18353448360],
        [25.162, 1.78070903718, 182.27960680100],
        [21.386, 3.86030772476, 116.42609634290],
        [25.572, 1.62093861709, 2324.94940881560],
        [20.025, 2.90618582553, 120.35824960600],
        [19.882, 5.59203696008, 4.19278569400],
        [19.454, 0.10623632006, 218.71572140940],
        [25.617, 2.09931460158, 248.72381809010],
        [19.804, 2.52180124343, 1485.98012106520],
        [18.516, 2.54810951896, 213.51154375910],
        [19.831, 0.07955320843, 842.15068148810],
        [18.516, 5.37755110510, 213.08664711690],
        [23.655, 1.59974907716, 738.79727483860],
        [20.375, 2.94653107321, 59.80374504030],
        [24.247, 3.15387696867, 240.38643081190],
        [18.294, 3.18715992969, 295.05122865420],
        [17.464, 2.90471803626, 477.80391620720],
        [20.698, 1.07232100334, 494.26624244250],
        [20.400, 1.83665590916, 533.62311835770],
        [21.285, 0.63341794388, 189.72322220190],
        [16.116, 0.60069688498, 746.92221379570],
        [16.297, 3.98317294128, 2.92076130680],
        [16.922, 4.74266972033, 2207.62954059540],
        [20.479, 6.05098286202, 173.68158709190],
        [15.447, 1.49120311247, 543.91805909620],
        [19.944, 4.94086632750, 121.25202148330],
        [17.127, 0.71458025372, 1781.03134971940],
        [17.240, 0.67749766724, 151.04766984290],
        [15.574, 5.70296527381, 3053.71237534660],
        [15.036, 5.52770334605, 2310.72231481400],
        [15.928, 4.45642717299, 643.82943957710],
        [16.165, 0.63286131026, 358.93013930950],
        [14.589, 5.26158292613, 472.17484191470],
        [16.545, 3.52813228069, 3480.31056622260],
        [18.912, 0.55218675639, 4.66586644600],
        [17.595, 2.26495491189, 672.14061522840],
        [18.104, 2.71285673689, 181.80652604900],
        [15.918, 5.23446779429, 135.54855145410],
        [13.931, 3.19357128657, 213.55972786890],
        [14.058, 0.82375896652, 221.37585028530],
        [13.931, 4.73208739639, 213.03846300710],
        [14.690, 2.65882838685, 292.01284726840],
        [14.454, 0.21819892811, 235.39049596580],
        [16.168, 0.91025406068, 280.00393915800],
        [13.327, 3.54947442109, 205.66428357540],
        [16.104, 0.82547975762, 176.65053250850],
        [16.441, 5.39398801335, 424.15051032120],
        [12.747, 0.75780958758, 721.64941953020],
        [12.754, 3.55466871752, 153.49535039770],
        [14.448, 0.12049617049, 313.68355667090],
        [16.499, 3.26383140489, 6283.07584999140],
        [16.564, 1.62649604519, 5856.47765911540],
        [14.950, 1.23923264394, 2641.34127847220],
        [15.724, 1.18874754834, 486.40193591630],
        [11.893, 0.91693668558, 416.30325013750],
        [11.684, 1.11385455828, 81.75213321620],
        [12.985, 4.74373293725, 3377.21779200400],
        [11.864, 0.64411806416, 28.31117565130],
        [13.216, 4.95904024430, 1279.79457262800],
        [16.121, 0.98185208328, 2538.24850425360],
        [14.900, 1.76649832526, 569.04784100980],
        [11.337, 4.36555105334, 3583.40334044120],
        [11.253, 5.98638731448, 193.65537546500],
        [14.753, 2.92291248767, 167.08930495290],
        [13.774, 2.50808183571, 1802.37199072180],
        [11.068, 0.00471764868, 629.60234557550],
        [12.781, 3.62178749219, 67.66805156650],
        [12.238, 0.27163151602, 1044.40407666220],
        [11.021, 0.15223056578, 501.37978944330],
        [14.206, 2.63254885854, 618.55664531160],
        [14.365, 0.37819794671, 601.76425067620],
        [15.034, 2.67095006272, 46.47042291600],
        [12.248, 2.19751851112, 650.94298657790],
        [10.783, 2.86375137884, 113.38771495710],
        [11.418, 1.20874560246, 172.24529849340],
        [14.613, 6.05645353059, 468.24268865160],
        [10.580, 2.05903854864, 429.04587143080],
        [13.721, 2.20936291526, 228.27694896500],
        [12.180, 1.82585577726, 241.87090352020],
        [10.787, 5.06924118186, 162.89651925890],
        [12.056, 3.20018724042, 72.07328558160],
        [12.233, 4.50741930970, 425.63498302950],
        [12.101, 4.14977794161, 1108.13997496560],
        [9.843, 1.49451039604, 226.63241756230],
        [10.287, 2.10680007784, 1033.35837639830],
        [9.975, 2.81640446254, 518.64526483070],
        [9.597, 4.80028087522, 426.64637498580],
        [10.746, 4.66838299108, 129.91947716160],
        [12.961, 5.11568581806, 219.44943459230],
        [12.302, 5.33568547700, 776.93031047640],
        [9.484, 4.85702954575, 820.05928096030],
        [11.441, 3.85769732764, 405.99126305650],
        [9.625, 1.60280478656, 426.55000676620],
        [9.164, 0.70204567980, 403.02231763990],
        [10.112, 2.76486875630, 210.59078245230],
        [10.816, 1.36864298163, 170.76082578510],
        [10.187, 2.36063948382, 685.47393735270],
        [12.397, 6.06349943525, 875.83029900100],
        [12.146, 2.04060386262, 508.35032409220],
        [9.574, 3.19555214859, 286.59622129700],
        [10.193, 4.01123146905, 381.35160823740],
        [8.900, 3.63260235880, 319.31263096340],
        [10.052, 5.16107251040, 216.00740842370],
        [8.528, 3.88076551354, 630.33605875840],
        [8.875, 5.46623776078, 3370.10424500320],
        [8.401, 5.65557131026, 213.45915413240],
        [10.033, 5.97497644283, 6.15033915430],
        [8.401, 2.27009862215, 213.13903674360],
        [11.661, 0.95163302252, 694.07195706180],
        [8.696, 2.33868966556, 220.36445832900],
        [8.141, 5.54059747150, 220.46082654860],
        [9.615, 2.75755414306, 556.51766803760],
        [9.120, 0.44322374149, 2097.42321937600],
        [8.109, 5.53989498262, 181.05576652360],
        [10.763, 0.05616402982, 691.10301164520],
        [9.579, 2.84979792871, 184.09414790940],
        [9.958, 2.38581008546, 945.24345570670],
        [8.526, 0.17821781104, 289.56516671360],
        [7.700, 0.00481375410, 7.16173111060],
        [8.613, 0.82900327241, 2957.73314812880],
        [9.517, 2.27516458273, 8.07675484730],
        [9.049, 3.37335025790, 731.94436026870],
        [7.888, 5.78452089815, 230.82520325630],
        [7.684, 3.10462250617, 7.06536289100],
        [7.391, 5.29648701813, 2627.11418447060],
        [9.875, 4.69411059509, 10213.28554621100],
        [7.328, 0.09051133382, 100.64509366380],
        [7.999, 1.60928374337, 696.51963761660],
        [7.600, 4.90078510977, 51.20572533120],
        [9.928, 5.25713005643, 699.70103135430],
        [7.333, 5.61982406824, 31.49256938900],
        [7.376, 4.52737009022, 616.32141307790],
        [9.422, 2.44475274779, 2118.76386037840],
        [7.300, 4.01885475010, 212.02707105080],
        [7.502, 4.71301501745, 436.15941843160],
        [9.071, 2.75662160229, 130.44074202340],
        [8.913, 2.19608557019, 427.56139872250],
        [8.801, 4.26655882704, 141.22580985640],
        [6.853, 5.69082635009, 480.77286162380],
        [7.765, 3.27218537808, 3796.70243587920],
        [9.159, 3.04713671650, 9786.68735533500],
        [9.034, 2.04165937353, 204.70107572890],
        [6.902, 4.61962635489, 2524.02141025200],
        [6.728, 0.58794595002, 739.80866679490],
        [6.741, 0.52362906624, 135.33610313300],
        [8.201, 5.03994203224, 411.62033734900],
        [7.240, 3.90653111215, 214.57111982520],
        [6.887, 4.11954799957, 662.53120356300],
        [6.566, 2.67659365854, 194.17664032680],
        [6.539, 6.25585361704, 31.01948863700],
        [7.484, 5.56871021201, 271.40591944890],
        [8.078, 3.09955817560, 353.30106501700],
        [7.072, 1.10066698352, 282.66406803390],
        [6.089, 0.79684364835, 593.42686339800],
        [6.140, 3.79672343724, 180.16199464630],
        [6.703, 3.82854248620, 412.58354519550],
        [6.024, 5.46288776207, 724.83081326790],
        [8.102, 4.51051495778, 268.43697403230],
        [6.033, 1.24326252021, 447.93883187840],
        [7.425, 2.29394888999, 532.61172640140],
        [6.007, 2.87215425398, 426.07692601420],
        [7.568, 0.79147591036, 2854.64037391020],
        [5.816, 1.70824982811, 50.40257617910],
        [7.534, 5.38598292680, 953.10776223290],
        [5.863, 2.04201456623, 454.90936652730],
        [7.291, 0.88044346877, 457.61767951300],
        [6.235, 4.51960341418, 3693.60966166060],
        [6.359, 6.27053660532, 313.94418910180],
        [5.722, 0.47241118592, 610.69233878540],
        [5.704, 0.45599464136, 643.07868005170],
        [6.176, 3.98739420856, 835.03713448730],
        [5.730, 0.50814242490, 1038.04128918680],
        [6.812, 4.20463385690, 938.12990870590],
        [5.620, 4.08049141112, 3899.79521009780],
        [6.770, 4.22172125738, 916.93228005540],
        [6.069, 3.46762401122, 278.25883401880],
        [5.875, 5.51773010551, 1073.60902419080],
        [5.558, 3.29478679376, 20.60692781950],
        [6.274, 4.88767368263, 0.52126486180],
        [5.794, 4.09991767938, 391.17346822390],
        [5.442, 2.79802608247, 397.39324334740],
        [5.754, 4.42718264879, 165.60483224460],
        [5.879, 6.17871525366, 291.26208774300],
        [6.716, 2.18663847730, 627.36711334180],
        [5.761, 0.62536160332, 114.39910691340],
        [5.359, 2.29390692216, 331.20966448920],
        [6.210, 4.92273259045, 450.97721326420],
        [6.686, 2.13438181268, 285.63301345050],
        [5.173, 4.10128119721, 6.59228213900],
        [5.707, 3.48716972669, 230.56457082540],
        [6.363, 5.64626069194, 518.38463239980],
        [5.241, 3.78081098206, 418.52143602870],
        [5.191, 4.39595146262, 84.34282612290],
        [6.710, 2.96748528229, 624.91943278700],
        [4.931, 2.71959451867, 558.00214074590],
        [5.225, 4.65463431385, 310.71461125430],
        [4.857, 2.61373582429, 66.70484372000],
        [4.847, 3.76991373317, 423.67742956920],
        [5.284, 1.96024672163, 1182.92157353290],
        [5.933, 2.74003948393, 219.89137757700],
        [6.373, 1.41296346460, 606.76018552230],
        [4.728, 0.23421038001, 1063.31408345230],
        [6.408, 1.16687419680, 268.95823889410],
        [4.782, 1.56813683227, 420.96911658350],
        [5.399, 4.57611449409, 238.90195810360],
        [5.161, 0.14436456585, 2413.81508903260],
        [4.858, 5.21378840436, 3686.49611465980],
        [5.086, 1.73392381835, 337.73251065900],
        [4.650, 3.71029121290, 305.08553696180],
        [4.896, 3.67786531840, 240.12579838100],
        [5.949, 0.29956165181, 524.01370669230],
        [4.968, 2.53258931342, 980.66817835880],
        [4.944, 2.16189522746, 104.05598206510],
        [5.366, 3.54867806985, 107.28555991260],
        [4.917, 0.48641512683, 3274.12501778540],
        [5.617, 6.27593478237, 112.65400177420],
        [4.524, 5.09539085552, 103.14095832840],
        [5.643, 1.52724336480, 105.54045477340],
        [4.504, 1.68251875362, 196.03362005060],
        [4.897, 4.90011892854, 102.12956637210],
        [4.525, 1.88735156553, 103.04459010880],
        [4.327, 1.45407229380, 409.92341631960],
        [5.095, 3.40640608336, 427.11945573780],
        [5.782, 3.55197606731, 25874.60404613620],
        [4.192, 0.16603430914, 958.57677783100],
        [4.976, 0.50639895683, 511.53171782990],
        [4.167, 5.94725762070, 316.44005376640],
        [4.353, 3.78587101731, 1171.87587326900],
        [5.387, 2.03693287651, 2435.15573003500],
        [4.067, 4.66592603130, 106.01353552540],
        [4.817, 3.53529781673, 960.22130923370],
        [4.048, 3.20024146722, 775.23338944700],
        [4.016, 6.00569143107, 945.99421523210],
        [3.989, 3.15130319196, 115.62294719080],
        [4.559, 5.59555355771, 778.41478318470],
        [4.153, 2.75042736587, 316.34368554680],
        [3.983, 2.00842137744, 597.35901666110],
        [4.212, 4.16852690218, 823.99143422340],
        [5.193, 0.71717111984, 810.65811209910],
        [3.927, 5.04361736754, 2943.50605412720],
        [4.225, 0.02571003853, 0.75075952540],
        [4.926, 1.12994881124, 526.98265210890],
        [4.170, 3.94116290117, 422.40540518200],
        [4.432, 3.99599046800, 393.46109008430],
        [3.840, 1.21234108241, 212.07525516060],
        [3.866, 4.20930793423, 97.67614824720],
        [4.440, 1.35536679738, 211.60217440860],
        [3.840, 0.43014354282, 214.52293571540],
        [4.724, 3.62039208608, 638.41281360570],
        [4.870, 5.75874599620, 1246.65747183630],
        [4.449, 1.43384065964, 184.98791978670],
        [3.931, 2.36660272585, 909.81873305460],
        [3.787, 5.98932416906, 325.95309721220],
        [3.665, 0.72917314141, 20.44686912510],
        [4.243, 2.29103096797, 453.42489381900],
        [3.730, 6.24831601183, 159.12442469020],
        [3.900, 1.31013240315, 850.01498801430],
        [5.134, 1.97348901289, 526.50957135690],
        [3.621, 2.77435773661, 123.53964334370],
        [3.607, 5.32058842710, 406.95447090300],
        [3.802, 1.94444523548, 421.18156490460],
        [3.906, 3.02475451573, 317.35507750310],
        [4.473, 0.80804073855, 838.21852822500],
        [4.081, 2.08732468180, 988.53248488500],
        [3.843, 4.99347148246, 7.63481186260],
        [4.467, 6.09037793116, 760.25553592000],
        [3.514, 3.97285766412, 426.48631629140],
        [3.504, 0.85064201666, 299.12639426920],
        [4.397, 1.68577228317, 824.74219374880],
        [3.581, 2.35235960566, 337.80194662820],
        [4.606, 3.48411642192, 913.96333463880],
        [3.790, 3.64538213705, 216.26804085460],
        [3.496, 0.95035381131, 436.89313161450],
        [4.422, 0.82822191292, 43.28902917830],
        [3.688, 2.83785443800, 739.05790726950],
        [4.439, 1.19409419107, 421.93232443000],
        [3.572, 2.77298538478, 444.75743814070],
        [4.420, 5.44308967028, 963.40270297140],
        [4.443, 3.73070830296, 37.87240320690],
        [4.322, 4.75680702521, 40.84134862350],
        [3.724, 0.59005210557, 256.42806592190],
        [4.471, 2.22367643527, 318.83955021140],
        [4.184, 1.52719196640, 298.23262239190],
        [3.534, 5.01937599570, 386.98068252990],
        [3.400, 3.22663067085, 4113.09430553580],
        [4.556, 1.35715974815, 495.75071515080],
        [4.453, 1.80417064247, 829.62050851590],
        [3.617, 1.51036385224, 41.64449777560],
        [3.789, 4.80357656146, 238.42887735160],
        [3.514, 2.38272766645, 426.71006546060],
        [3.208, 1.74465274123, 952.35700270750],
        [4.398, 2.65839000906, 832.58945393250],
        [4.092, 3.07954777295, 60.76695288680],
        [3.813, 5.63047104819, 315.42866181010],
        [3.219, 6.22278803635, 754.83890994860],
        [3.102, 2.69222024257, 343.73983746140],
        [4.045, 4.02463772100, 376.19561469690],
        [3.576, 0.38532787280, 214.99601646740],
        [3.291, 5.49542015261, 143.93412284210],
        [3.981, 5.75449411958, 239.16259053450],
        [3.249, 0.58789568678, 619.29035849450],
        [3.114, 0.02831060137, 221.16340196420],
        [3.555, 3.12207684735, 1048.33622992530],
        [3.010, 1.91180343491, 93.53154666300],
        [3.384, 1.81702854004, 443.86366626340],
        [3.222, 2.37342117781, 429.51895218280],
        [3.431, 5.92099840679, 570.74476203920],
        [3.271, 2.04947945059, 806.72595883600],
        [3.207, 5.44018976766, 402.21916848780],
        [3.091, 0.10717557454, 3590.51688744200],
        [2.951, 1.76365810296, 426.81063919710],
        [2.958, 0.23653889192, 1354.43315884340],
        [2.948, 4.59289832104, 426.38574255490],
        [3.506, 5.43222584214, 84.93352695390],
        [2.894, 5.69678330542, 1262.38608488870],
        [3.682, 1.07122313007, 395.57870223900],
        [2.983, 5.25093816048, 313.47110834980],
        [3.117, 4.18767239237, 366.79444583570],
        [2.873, 4.45472312727, 361.37781986430],
        [3.521, 2.05528981993, 1261.63532536330],
        [3.496, 1.87950759078, 439.12836384820],
        [3.012, 0.64439385874, 263.02034806090],
        [2.849, 1.12491777974, 262.05714021440],
        [2.910, 2.76192171681, 541.53981451060],
        [3.322, 6.08893948791, 108.72184851110],
        [3.181, 1.01419299056, 418.00017116690],
        [2.793, 0.38781777981, 211.86280683950],
        [3.091, 2.18216748751, 306.83064210100],
        [3.748, 3.89145855821, 220.93390730060],
        [2.982, 1.99831689446, 117.91056905120],
        [2.793, 1.25466684542, 214.73538403650],
        [3.512, 1.50965040301, 885.43971066640],
        [2.716, 0.80710391613, 757.21715453420],
        [3.137, 2.09889265033, 2751.54759969160],
        [2.738, 4.89270330923, 464.73122651380],
        [2.875, 4.28436709414, 4010.00153131720],
        [3.313, 3.01452486457, 336.83873878170],
        [2.746, 2.69963506928, 380.38840039090],
        [3.132, 2.19562786872, 2.96894541660],
        [3.233, 0.60809684558, 3171.03224356680],
        [3.034, 0.93246285284, 205.43478891180],
        [3.006, 5.91067479448, 2.70831298570],
        [3.643, 5.58302397259, 423.62924545940],
        [2.625, 1.07042050691, 23.57587323610],
        [3.495, 0.19887562030, 576.16138801060],
        [2.590, 0.21252773750, 110.25450532920],
        [2.704, 6.12908233599, 572.22923474750],
        [2.568, 0.17571588314, 1056.20053645150],
        [2.583, 2.96927378731, 384.05992122310],
        [2.555, 3.96441052072, 430.79097657000],
        [2.786, 2.54945911818, 195.89060769870],
        [2.869, 4.82964665921, 710.74673161820],
        [2.534, 5.31005598763, 427.34895040140],
        [2.618, 1.22081401503, 36.64856292950],
        [3.464, 2.32811328200, 285.37238101960],
        [3.374, 3.34109586766, 162.09337010680],
        [2.694, 4.64149271687, 140.96517742550],
        [2.603, 0.99527295832, 92.30770638560],
        [3.140, 5.40790277580, 328.24071907260],
        [2.603, 4.31532790880, 561.93429400900],
        [2.987, 0.82758128867, 45.57665103870],
        [2.959, 3.35623851523, 273.85360000370],
        [2.561, 0.54160683162, 107.75864066460],
        [2.900, 1.20691455948, 462.02291352810],
        [2.648, 1.90547819027, 88.11492069160],
        [2.461, 4.68211868869, 2840.41327990860],
        [2.534, 5.00311256556, 431.26405732200],
        [2.521, 3.32160472310, 136.06981631590],
        [2.945, 1.06452531856, 732.69511979410],
        [2.654, 1.36744710395, 460.53844081980],
        [3.297, 1.33975572602, 305.60680182360],
        [2.634, 2.29995533800, 519.39602435610],
        [2.805, 5.62255444533, 1699.27921650320],
        [2.439, 5.14733660159, 303.86169668440],
        [2.434, 3.71460437051, 4216.18707975440],
        [2.416, 3.76296045457, 77.75054398390],
        [2.803, 2.55280894914, 505.31194270640],
        [2.592, 3.32836551071, 110.15813710960],
        [3.074, 1.71462387764, 256.58812461630],
        [3.295, 0.81766682522, 705.11765732570],
        [3.183, 6.15742006608, 109.24311337290],
        [2.908, 5.38534195293, 315.16802937920],
        [2.326, 1.42604031905, 131.54696222180],
        [2.427, 2.04627850740, 124.50285119020],
        [2.632, 1.41253794767, 211.65456403530],
        [2.297, 1.38016674676, 425.84743135060],
        [2.318, 6.27716072818, 317.14262918200],
        [2.310, 4.86442292404, 3259.89792378380],
        [2.873, 1.10206537875, 7.86430652620],
        [2.616, 0.11849676899, 133.10087089930],
        [3.213, 4.49320461690, 432.22726516850],
        [2.276, 6.04688191978, 214.10224459010],
        [2.276, 1.96882478275, 212.49594628590],
        [2.917, 0.96774661857, 100.38446123290],
        [2.890, 5.72610904534, 322.02094394910],
        [2.829, 2.66887892162, 141.48644228730],
        [2.695, 1.47488571070, 42.53826965290],
        [2.697, 5.34002228297, 432.01481684740],
        [2.229, 4.40717937246, 540.73666535850],
        [2.214, 2.43714413196, 426.85882330690],
        [2.512, 4.68291916658, 1596.18644228460],
        [2.202, 5.91122030007, 867.42347575360],
        [2.856, 0.94736445171, 41.05379694460],
        [2.396, 0.10516628717, 206.93630796260],
        [2.345, 1.16685267780, 640.86049416050],
        [2.263, 4.62327588198, 188.02630117250],
        [2.214, 3.97566024178, 426.33755844510],
        [2.274, 4.94055830720, 4002.88798431640],
        [2.541, 0.83705615200, 12352.85260454480],
        [2.145, 3.40120044084, 111.16952906590],
        [2.895, 6.07389082608, 2914.01423582380],
        [2.520, 1.05396310009, 184.72728735580],
        [2.448, 3.49820841117, 481.73606947030],
        [2.343, 2.86472924644, 207.07932031450],
        [2.964, 5.96264929181, 465.95506679120],
        [2.122, 4.04560058177, 118.07062774560],
        [2.452, 4.35251210402, 1382.88734684660],
        [2.131, 0.61211416273, 335.14181775230],
        [2.116, 4.76181734842, 765.88461021250],
        [2.245, 5.67078632283, 6467.92575796160],
        [2.425, 0.84789834075, 550.13783421970],
        [2.227, 1.15684015463, 227.31374111850],
        [2.314, 6.13104971819, 2730.20695868920],
        [2.344, 0.35844885568, 217.44369702220],
        [2.529, 3.07975959821, 774.48262992160],
        [2.523, 1.75026771081, 1578.02719501990],
        [2.111, 4.24637589094, 96.87299909510],
        [2.826, 1.01974994073, 87.31177153950],
        [2.906, 3.75374302356, 428.08266358430],
        [2.113, 2.19787343926, 449.23210812500],
        [2.142, 1.19671359858, 209.15449385380],
        [2.882, 2.59371585952, 39.61750834610],
        [2.078, 2.87503604503, 14.01464568050],
        [2.090, 1.99032225653, 441.57604440300],
        [2.519, 2.99001165551, 745.27768239300],
        [2.035, 5.37147785849, 1041.22268292450],
        [2.054, 1.11817372961, 842.90144101350],
        [2.023, 2.94559148702, 668.20846196530],
        [2.529, 4.34280159004, 221.89711514710],
        [2.554, 5.56906955622, 214.19286731530],
        [2.712, 1.60469055827, 1050.99635880120],
        [2.350, 3.36541706919, 220.20019411770],
        [2.015, 2.60446576036, 315.64111013120],
        [2.158, 1.64945261993, 219.66188291340],
        [2.120, 2.93968038721, 304.12232911530],
        [2.357, 1.94433441808, 233.90602325750],
        [2.579, 4.54124062411, 484.70501488690],
        [2.046, 5.45531068264, 200.55647414470],
        [2.040, 2.04492641594, 1097.09427470170],
        [2.675, 1.20234167733, 28.57180808220],
        [2.528, 4.69268465973, 637.44960575920],
        [2.314, 2.81193072994, 25.12978191360],
        [2.689, 5.03868493349, 1269.49963188950],
        [2.115, 3.10772296248, 1276.61317889030],
        [2.712, 1.49710379127, 3340.61242669980],
        [2.138, 4.52114042624, 378.90392768260],
        [2.708, 0.01014338204, 389.94962794650],
        [2.560, 5.75783882561, 544.50875992720],
        [2.028, 0.24331359951, 146.59425171800],
        [2.096, 1.44475430956, 864.24208201590],
        [1.897, 4.63194412388, 220.30076785420],
        [1.901, 1.78319572727, 192.85222631290],
        [2.011, 4.11578325523, 315.87060479480],
        [2.014, 2.26726355818, 198.10879358990],
        [1.905, 2.15527255015, 326.68681039510],
        [1.949, 3.95440811214, 103.61403908040],
        [2.098, 5.24613314798, 175.42669223110],
        [1.884, 5.66018186202, 1310.39337013970],
        [1.911, 2.60665446600, 301.41401612960],
        [2.160, 3.42888079793, 420.00590873700],
        [2.325, 5.89632178100, 815.06334611420],
        [1.914, 0.22451332248, 171.65459766240],
        [1.854, 0.04278915026, 233.74596456310],
        [1.973, 2.68640259480, 769.81676347560],
        [1.941, 0.61369890353, 3487.42411322340],
        [1.836, 3.41496980986, 195.77298761970],
        [2.554, 2.35660179716, 212.40532356070],
        [1.822, 4.05510232882, 639.94547042380],
        [1.883, 6.27079329518, 16.67477455640],
        [1.865, 0.17460226411, 244.79166482700],
        [2.097, 4.58369520569, 316.91313451840],
        [1.879, 2.76480572708, 28.45418800320],
        [2.111, 2.92457831824, 328.92204262880],
        [2.077, 0.35943033580, 589.49471013490],
        [1.825, 4.04945265223, 190.40454575810],
        [1.895, 3.89414880651, 334.55111692130],
        [2.425, 3.76754213762, 25558.21217647960],
        [2.218, 1.85341154236, 635.23141986800],
        [1.782, 0.86927461254, 92.79783348010],
        [2.432, 3.78026263567, 1254.52177836250],
        [2.106, 0.17285594964, 354.26427286350],
        [1.791, 6.23892012939, 1670.82502850000],
        [1.996, 1.40940081042, 230.70758317730],
        [1.753, 1.86660297451, 241.75328344120],
        [2.093, 2.39366777880, 187.43560034150],
        [2.007, 3.54083120293, 226.79247625670],
        [1.949, 1.36533052698, 1385.17496870700],
        [1.737, 2.67583588366, 6.36278747540],
        [1.868, 4.24454204649, 1119.18567522950],
        [1.896, 3.81474515719, 310.97524368520],
        [1.860, 3.67887919389, 1321.43907040360],
        [2.305, 3.53252557028, 1570.91364801910],
        [2.008, 3.88504783546, 638.93407846750],
        [1.928, 2.64393870433, 525.75881183150],
        [1.807, 0.76057354967, 66.18357885820],
        [1.824, 0.85822155861, 639.84910220420],
        [2.221, 4.82210413830, 1585.89150154610],
        [2.227, 4.58488941022, 271.61836777000],
        [1.897, 0.64334758250, 55.65914345610],
        [1.792, 0.00514800434, 827.92358748650],
        [1.831, 3.69768852728, 172.45774681450],
        [1.790, 4.80062977720, 3576.28979344040],
        [1.698, 0.72475212282, 295.19424100610],
        [1.744, 3.45256183290, 238.57188970350],
        [2.334, 1.51845210531, 170.01006625970],
        [1.669, 4.44620549295, 4319.27985397300],
        [1.939, 3.43927826945, 102.57150935680],
        [1.733, 5.96815907422, 837.69726336320],
        [1.686, 2.12870436615, 491.55792945680],
        [1.651, 3.20586624475, 281.17959532560],
        [1.884, 4.24447812450, 13.49338081870],
        [1.880, 0.33845094634, 214.94362684070],
        [1.817, 3.07678560214, 220.52451702340],
        [1.872, 3.79328892492, 392.65794093220],
        [2.195, 1.93786776664, 259.76951835400],
        [1.881, 4.49314034712, 199.80571461930],
        [1.662, 0.29659841675, 314.90739694830],
        [1.626, 2.29697402942, 3067.93946934820],
        [1.927, 1.00706624515, 26.82670294300],
        [1.621, 0.01495920679, 1379.70595310890],
        [1.655, 4.15494230496, 4326.39340097380],
        [1.719, 5.97443860990, 152.53214255120],
        [1.653, 5.65633302949, 448.68959140380],
        [1.790, 3.73709604091, 10.03430830760],
        [1.600, 2.28430251221, 749.20983565610],
        [1.770, 0.03256515992, 364.34676528090],
        [1.939, 5.93191442558, 249.94765836750],
        [1.988, 4.78810872536, 101.86893394120],
        [1.667, 5.52149899431, 229.97386999440],
        [2.215, 3.55121116323, 594.65070367540],
        [1.950, 0.80878923373, 1049.08698945070],
        [1.773, 2.55608046714, 9985.75935677140],
        [2.088, 2.33165208016, 420.44785172170],
        [2.107, 2.43624356568, 453.68552624990],
        [1.907, 4.72569972805, 857.12853501510],
        [1.609, 4.96540433043, 285.11174858870],
        [2.152, 4.87024306306, 186.21176006410],
        [1.685, 5.68609178354, 200.03520928290],
        [1.752, 5.21532265401, 25448.00585526019],
        [1.870, 6.14683645342, 347.36317418380],
        [1.731, 1.95944272122, 934.94851496820],
        [1.680, 5.56246697700, 170.97327410620],
        [1.652, 1.24521351050, 398.14400287280],
        [1.548, 0.38524522125, 17.26547538740],
        [1.577, 0.23430114545, 434.67494572330],
        [1.652, 2.84480428863, 385.54439393140],
        [1.770, 1.15057996280, 199.96577331370],
        [1.528, 4.02240583348, 236.19364511790],
        [2.118, 5.47803429266, 369.08206769610],
        [1.543, 2.54353574089, 632.73555520340],
        [1.504, 5.94300855424, 280.21638747910],
        [1.495, 2.19380669867, 407.47573576480],
        [1.894, 1.88797605501, 598.84348936940],
        [1.515, 4.70072295492, 211.29335786790],
        [1.767, 3.11910667879, 2921.12778282460],
        [1.515, 3.31498374761, 215.30483300810],
        [1.728, 5.28095966912, 219.51887056150],
        [1.906, 5.24236020775, 248.46318565920],
        [1.607, 0.80041605735, 642.34496686880],
        [1.640, 2.93387205029, 1364.72809958190],
        [1.585, 0.77219822539, 661.23792731640],
        [1.458, 5.67666822477, 632.83192342300],
        [1.866, 4.40562835971, 971.10695080320],
        [1.838, 0.48492190760, 1127.04998175570],
        [1.903, 5.18692835916, 2015.67108615980],
        [1.590, 2.80043208070, 633.74694715970],
        [1.489, 4.10155671855, 77837.11123384659],
        [1.573, 4.23741356107, 203.89792657680],
        [1.690, 0.65475720351, 2700.71514038580],
        [1.872, 3.54376036064, 354.52490529440],
        [1.419, 4.53129346734, 224.60542813280],
        [1.575, 3.52476647615, 373.90799283650],
        [1.817, 3.95203885550, 6076.89030155420],
        [1.446, 5.41423319377, 317.87634236490],
        [1.552, 1.89030720529, 1304.92435454160],
        [1.394, 1.86243646383, 913.75088631770],
        [1.940, 4.36562864826, 432.74853003030],
        [1.655, 1.31748248488, 25668.41849769900],
        [1.438, 5.12958189872, 71.81265315070],
        [1.929, 4.90721606846, 206.39799675830],
        [1.533, 2.10965059985, 378.64329525170],
        [1.421, 6.22897936063, 904.40210708320],
        [1.450, 1.98065714692, 205.97310011610],
        [1.584, 5.94110940455, 1226.21060271120],
        [1.510, 3.77771207288, 222.70026429920],
        [1.670, 2.51954641624, 976.73602509570],
        [1.823, 1.60093991502, 1141.13406340540],
        [1.564, 3.94804398924, 9566.27471289620],
        [1.406, 3.65940442223, 316.50374424120],
        [1.653, 3.62394476466, 968.13800538660],
        [1.495, 0.77832279170, 5959.57043333400],
        [1.408, 5.24970924980, 316.27999507200],
        [1.724, 0.25483952164, 125.18417474640],
        [1.426, 2.26871672572, 17.40848773930],
        [1.881, 4.12588105003, 562.14674233010],
        [1.370, 4.74474866880, 1357.61455258110],
        [1.485, 2.18712736768, 9889.78012955360],
        [1.369, 1.31846306523, 1160.83017300510],
        [1.394, 5.66924517860, 1736.99156101570],
        [1.375, 0.16983286236, 346.39996633730],
        [1.525, 2.40244831911, 419.43645976540],
        [1.359, 2.68668516027, 310.76279536410],
        [1.396, 3.67055397940, 253.45912050530],
        [1.454, 5.49356262440, 504.56118318100],
        [1.813, 0.21882066860, 263.70167161710],
        [1.330, 3.01736059716, 254.14044406150],
        [1.498, 0.17578085903, 155.78297225810],
        [1.509, 5.13129901759, 768.85355562910],
        [1.550, 1.44225397189, 1894.41906467650],
        [1.447, 1.44933809994, 893.35640681930],
        [1.306, 5.76425101758, 714.67888488130],
        [1.660, 1.34160526151, 322.61164478010],
        [1.347, 3.00388920953, 843.63515419640],
        [1.432, 3.43786149731, 251.17149864490],
        [1.510, 4.39762427873, 25.27279426550],
        [1.587, 5.17106904014, 141.69889060840],
        [1.477, 5.47518377610, 226.04171673130],
        [1.356, 4.24406292182, 332.17287233570],
        [1.644, 1.74367211793, 67.88049988760],
        [1.407, 5.79229630947, 188.16931352440],
        [1.575, 0.09808372057, 702.14871190910],
        [1.765, 4.93410889383, 201.51968199120],
        [1.318, 4.91605557404, 17.52610781830],
        [1.701, 4.47360878108, 384.27236954420],
        [1.304, 6.04155032791, 25.86349509650],
        [1.269, 4.91035989349, 354.99798604640],
        [1.620, 5.54960841244, 260.36021918500],
        [1.263, 1.16521999431, 255.83736509090],
        [1.744, 3.70453251764, 147.11551657980],
        [1.579, 4.31561365365, 2228.97018159780],
        [1.557, 0.57740217353, 3178.14579056760],
        [1.302, 0.32055726013, 119.76754877500],
        [1.405, 3.20407486040, 395.10562148700],
        [1.234, 2.34766954239, 318.67949151700],
        [1.519, 4.74629629688, 100.17201291180],
        [1.573, 4.00132484524, 1264.29545423920],
        [1.222, 0.10709243166, 1372.59240610810],
        [1.205, 0.24105241435, 466.75821594330],
        [1.188, 1.27112537278, 1184.40604624120],
        [1.280, 4.85454052139, 535.91074021810],
        [1.482, 4.47166692910, 763.43692965770],
        [1.233, 1.64803509193, 433.66355376700],
        [1.476, 3.76634399110, 272.58157561650],
        [1.199, 1.78020373551, 102.34201469320],
        [1.186, 4.72162748523, 795.68025857210],
        [1.291, 4.42697938285, 10220.39909321180],
        [1.576, 2.40263038916, 348.63519857100],
        [1.497, 3.44614317326, 3024.22055704320],
        [1.175, 4.85145058205, 433.75992198660],
        [1.262, 5.79416346069, 531.97858695500],
        [1.486, 4.39599352105, 1055.44977692610],
        [1.351, 0.46461977407, 707.56533788050],
        [1.230, 5.18147817992, 752.39122939380],
        [1.175, 4.35535063059, 3892.68166309700],
        [1.581, 5.49361132323, 419.53282798500],
        [1.247, 4.22328749428, 113.12708252620],
        [1.145, 2.26067253357, 199.12018554620],
        [1.385, 0.89711064123, 6073.70890781650],
        [1.228, 4.55057016747, 680.05731138130],
        [1.470, 1.58708185256, 409.18970313670],
        [1.366, 3.99684537321, 6065.84460129030],
        [1.131, 1.56474593118, 196.83676920270],
        [1.163, 5.75528918663, 2303.60876781320],
        [1.142, 4.35845106342, 1834.61531963620],
        [1.244, 1.93187654929, 623.22251175760],
        [1.136, 3.13253323524, 611.44309831080],
        [1.425, 2.65840274172, 1253.77101883710],
        [1.114, 3.43048279234, 771.30123618390],
        [1.314, 0.15326588489, 493.30303459600],
        [1.109, 2.95808402860, 1091.62525910360],
        [1.177, 3.88159541809, 128.36556848410],
        [1.150, 4.69140569690, 1.27202438720],
        [1.527, 1.09983755253, 683.02625679790],
        [1.304, 5.24544813643, 5650.29211067820],
        [1.347, 4.11616699496, 97.46369992610],
        [1.085, 0.70231952018, 1166.40685767090],
        [1.224, 4.22994822530, 827.17282796110],
        [1.142, 5.36157631813, 199.02381732660],
        [1.373, 0.89371361814, 799.61241183520],
        [1.085, 1.15969472512, 398.28701522470],
        [1.098, 3.75659421786, 318.39760722670],
        [1.136, 1.35127769399, 205.92491600630],
        [1.108, 6.03201954623, 206.44618086810],
        [1.146, 0.18236094571, 6386.16862421000],
        [1.216, 0.49809632153, 1178.98942026980],
        [1.295, 2.32056477953, 10003.91860403610],
        [1.082, 1.96611069200, 3700.72320866140],
        [1.094, 5.12244388591, 314.38613208650],
        [1.464, 4.54056066665, 1248.14194454460],
        [1.277, 5.42029902662, 9996.05429750990],
        [1.248, 0.21625135029, 101.60830151030],
        [1.055, 5.53725373260, 1578.77795454530],
        [1.265, 5.85587479852, 82.85835341460],
        [1.058, 0.73824266822, 670.91677495100],
        [1.127, 3.63458498010, 582.38116313410],
        [1.188, 2.10062104535, 423.88987789030],
        [1.217, 2.49656109071, 311.72600321060],
        [1.080, 3.96349373526, 118.87377689770],
        [1.175, 0.91096377814, 740.06929922580],
        [1.087, 5.46774426742, 494.47869076360],
        [1.080, 3.49168860514, 847.04604259770],
        [1.095, 0.97418295319, 1159.29331067010],
        [1.078, 1.75579678521, 1457.52593306200],
        [1.087, 4.81206824168, 109.68505635760],
        [1.293, 0.02397468965, 2723.09341168840],
        [1.190, 4.49552956868, 429.30650386170],
        [1.127, 0.84707518843, 48.75804477640],
        [1.047, 4.58416926615, 89.75945209430],
        [1.251, 1.16783030789, 455.16999895820],
        [1.122, 5.72161306640, 78263.70942472259],
        [1.027, 0.16330222064, 229.76142167330],
        [1.069, 2.37188773221, 848.53051530600],
        [1.252, 5.28238896229, 6080.82245481730],
        [1.181, 5.22299379363, 1459.95656727430],
        [1.382, 0.51603096285, 774.00954916960],
        [1.064, 5.95222326171, 1144.31545714310],
        [1.083, 5.04862249728, 629.86297800640],
        [1.061, 3.38604454777, 27.08733537390],
        [1.415, 4.85796248007, 2332.06295581640],
        [1.082, 4.07686503205, 1245.17299912800],
        [1.112, 6.07617329506, 870.46185713940],
        [1.072, 0.73647405514, 1482.79872732750],
        [1.322, 2.81015928946, 223.33340374560],
        [1.398, 1.55232715558, 25771.51127191760],
        [1.065, 3.98884050015, 683.18631549230],
        [1.008, 5.19594380826, 316.13123722570],
        [1.197, 0.79148395839, 9580.50180689780],
        [1.402, 4.82957073563, 883.79517926370],
        [1.064, 4.65334775068, 201.99276274320],
        [1.108, 5.88857586823, 657.16276170140],
        [0.996, 4.99081076034, 426.75824957040],
        [0.996, 1.60533807224, 426.43813218160],
        [1.028, 5.92128319450, 108.50940019000],
        [1.004, 3.54259597860, 754.03576079650],
        [1.214, 0.81213286478, 1773.91780271860],
        [1.318, 0.60269176130, 1123.11782849260],
        [1.327, 4.75885008900, 321.80849562800],
        [1.014, 5.79119811472, 6460.81221096080],
        [1.178, 0.47169015111, 495.96316347190],
        [0.999, 5.95030119388, 3906.90875709860],
        [1.033, 3.75433174131, 414.81877742920],
        [1.002, 1.39171012432, 1251.34038462480],
        [0.990, 1.32621236288, 1268.74887236410],
        [1.275, 1.95417923977, 757.80785536520],
        [1.174, 6.04352585298, 225.07850888480],
        [1.174, 5.43253033568, 849.26422848890],
        [1.004, 5.33434806968, 46.20979048510],
        [1.108, 1.28177889943, 294.30046912880],
        [0.976, 1.83523959034, 5.88970672340],
        [0.971, 3.38563950019, 306.09692891810],
        [1.050, 3.88449467091, 632.26247445140],
        [1.050, 3.26096036982, 159.71512552120],
        [1.041, 2.36429894351, 821.54375366860],
        [1.218, 4.61739999906, 990.22940591440],
        [1.342, 1.55614528399, 498.93210888850],
        [0.967, 3.84645372731, 604.47256366190],
        [1.171, 0.42265751679, 10011.03215103690],
        [0.965, 0.05396772193, 962.50893109410],
        [1.096, 3.04685199735, 608.40471692500],
        [1.166, 6.14999706886, 737.31280213030],
        [0.986, 3.71830385737, 1235.61177157240],
        [0.953, 0.79704964354, 16.46232623530],
        [0.976, 1.05304205075, 8.33738727820],
        [1.142, 1.06057209808, 369.97583957340],
        [1.060, 4.36236322604, 633.30500417500],
        [1.138, 4.31859245106, 98.42690777260],
        [1.006, 5.89037944896, 10007.09999777380],
        [0.967, 1.56852913547, 157.63995198190],
        [1.082, 0.99548769517, 4.14460158420],
        [1.009, 6.15037679495, 401.32539661050],
        [0.981, 2.37620383333, 35.21227433100],
        [1.011, 5.42685471402, 110.72758608120],
        [0.989, 4.11575312649, 413.85556958270],
        [1.024, 1.90798238649, 1175.80802653210],
        [1.079, 4.95981991427, 631.82053146670],
        [0.982, 3.02842195594, 347.41135829360],
        [0.979, 3.74615289445, 700.45179087970],
        [0.928, 5.22236951137, 1173.52040467170],
        [0.912, 4.14451390992, 469.72716135990],
        [0.938, 1.18273838991, 254.35289238260],
        [0.977, 1.26684849112, 104.57724692690],
        [1.134, 5.87478488618, 6058.73105428950],
        [1.092, 4.51789158271, 532.13864564940],
        [1.132, 4.55420027150, 1912.57831194120],
        [0.915, 4.87266214195, 18.91000679010],
        [0.915, 5.96624579967, 1987.21689815660],
        [1.039, 2.40020830681, 6475.03930496240],
        [0.965, 4.98927479154, 394.35486196160],
        [0.964, 3.60903715900, 3281.23856478620],
        [0.897, 2.81660605059, 316.23181096220],
        [1.008, 5.78024010734, 502.86426215160],
        [1.005, 6.12431717236, 6275.96230299060],
        [0.895, 6.09245508513, 316.55192835100],
        [1.224, 1.73535287415, 5120.60114558360],
        [1.138, 2.92901543353, 1037.29052966140],
        [0.924, 5.70601816488, 614.83694036960],
        [0.893, 5.25155704274, 475.35623565240],
        [1.042, 1.10027795842, 1518.22344997960],
        [0.890, 4.03192782386, 1314.32552340280],
        [0.926, 3.35110915055, 635.70450062000],
        [1.130, 5.49282680494, 92.94084583200],
        [0.890, 2.12933822393, 3384.33133900480],
        [0.967, 5.86215202069, 13.33332212430],
        [1.004, 1.73116475997, 10316.37832042960],
        [1.226, 3.52834223937, 80.41067285980],
        [1.029, 4.90620832171, 19.12245511120],
        [0.996, 0.76740358631, 733.42883297700],
        [0.910, 4.08904906301, 3333.49887969900],
        [1.235, 1.23871819142, 357.23321828010],
        [1.218, 0.18349810348, 1090.40141882620],
        [0.864, 4.71917415767, 620.25356634100],
        [0.935, 1.45887009044, 1042.91960395390],
        [0.866, 4.04792682992, 522.52923398400],
        [1.158, 5.43322209110, 1089.12939443900],
        [0.860, 0.49220052417, 64.95973858080],
        [1.190, 5.58965369650, 2810.92146160520],
        [0.957, 3.12914047010, 628.59095361920],
        [0.861, 5.69790389801, 103.84353374400],
        [1.037, 5.91424823262, 11.30633269480],
        [0.918, 0.21424702155, 373.01422095920],
        [0.836, 3.02501867546, 387.24131496080],
        [1.158, 3.33343863758, 6290.18939699220],
        [0.856, 0.81593288669, 907.37105249980],
        [1.036, 3.11936047271, 5429.87946823940],
        [0.853, 6.23618175592, 938.88066823130],
        [0.982, 3.39082880963, 521.61421024730],
        [0.851, 5.14029961564, 802.79380557290],
        [0.828, 0.28399876908, 338.48327018440],
        [0.868, 1.89151676387, 627.15466502070],
        [0.878, 2.67671626912, 688.65533109040],
        [0.921, 0.18441593712, 3803.81598288000],
        [0.821, 0.74694467095, 1152.17976366930],
        [0.841, 2.42616504698, 625.88264063350],
        [0.862, 3.35273419872, 425.32616648880],
        [0.887, 3.46938383985, 1748.78802080500],
        [0.815, 5.95428642326, 321.05773610260],
        [0.958, 1.35525670354, 1201.83158032300],
        [0.905, 5.43093361027, 236.87496867410],
        [0.862, 3.24167644516, 427.87021526320],
        [0.793, 5.01131700831, 109.99387289830],
        [0.791, 2.21809404489, 110.41876954050],
        [0.842, 5.04957483651, 444.12429869430],
        [0.887, 4.23752031714, 3553.91152213780],
        [0.851, 4.64229745145, 4105.98075853500],
        [1.067, 4.69271921916, 559.69906177530],
        [0.966, 4.55013458162, 9360.08916445900],
        [0.977, 1.50925667031, 186.47239249500],
        [1.092, 0.58153747940, 203.26478713040],
        [0.997, 0.24589891452, 439.93151300030],
        [0.786, 3.84829878956, 194.38908864790],
        [0.774, 3.76050639310, 219.14061805160],
        [0.845, 4.21602090805, 2648.45482547300],
        [0.962, 0.30590569897, 229.34073054800],
        [0.763, 0.05577842075, 846.13101886100],
        [0.839, 5.68124142701, 2620.00063746980],
        [0.818, 2.52609626540, 26301.20223701220],
        [0.929, 0.71906225883, 740.28174754690],
        [0.895, 5.83218231202, 4539.69249641180],
        [0.939, 3.68460642343, 817.77165909990],
        [0.755, 4.90024080821, 532.87235883230],
        [0.784, 1.14096100609, 551.03160609700],
        [0.899, 1.85252071775, 835.78789401270],
        [0.753, 0.04254534997, 1534.73816584160],
        [1.033, 1.00137993270, 134.11226285560],
        [0.851, 6.12272864540, 1475.68518032670],
        [0.797, 5.14322789256, 473.65931462300],
        [0.729, 4.94931618796, 476.10699517780],
        [0.852, 3.11058720799, 232.42155054920],
        [0.783, 4.50805467439, 1151.42900414390],
        [0.751, 0.92289775523, 1884.12412393800],
        [0.832, 4.19686348297, 29.20494752860],
        [0.720, 0.40961041030, 522.62560220360],
        [0.722, 3.96121088528, 1474.93442080130],
        [0.788, 0.99170388242, 121.84272231430],
        [0.722, 4.16734185316, 232.20910222810],
        [0.970, 2.32204039048, 566.60016045500],
        [0.814, 5.19337022083, 948.21240112330],
        [0.724, 2.30837674225, 949.12742486000],
        [0.770, 5.89605163084, 156.67674413540],
        [0.705, 5.40102118863, 1193.96727379680],
        [0.809, 3.56474059969, 845.33207522580],
        [0.755, 3.94580797273, 451.72797278960],
        [0.731, 6.11847213487, 1239.54392483550],
        [0.747, 1.40599730465, 782.34693644780],
        [0.861, 1.84312374221, 984.60033162190],
        [0.695, 4.78088165969, 10419.47109464820],
        [0.804, 1.07998437400, 89.00869256890],
        [0.828, 5.86080569334, 845.11962690470],
        [0.692, 3.38736418117, 6489.26139842860],
        [0.694, 3.07863807714, 316.60431797770],
        [0.690, 5.86681311380, 316.17942133550],
        [0.714, 5.38707933404, 567.82400073240],
        [0.767, 0.12081849650, 485.88067105450],
        [0.820, 1.87877245664, 499.89531673500],
        [0.705, 0.57839934869, 1053.75285589670],
        [0.684, 4.88442270630, 2545.36205125440],
        [0.689, 6.14296395253, 622.48879857470],
        [0.823, 3.50224755884, 877.57540414020],
        [0.827, 2.59300433753, 232.94281541100],
        [0.735, 3.05650026582, 66.91729204110],
    ],
    # R1
    [
        [6182981.282, 0.25843515034, 213.29909543800],
        [506577.574, 0.71114650941, 206.18554843720],
        [341394.136, 5.79635773960, 426.59819087600],
        [188491.375, 0.47215719444, 220.41264243880],
        [186261.540, 3.14159265359, 0.00000000000],
        [143891.176, 1.40744864239, 7.11354700080],
        [49621.111, 6.01744469580, 103.09277421860],
        [20928.189, 5.09245654470, 639.89728631400],
        [19952.612, 1.17560125007, 419.48464387520],
        [18839.639, 1.60819563173, 110.20632121940],
        [12892.827, 5.94330258435, 433.71173787680],
        [13876.565, 0.75886204364, 199.07200143640],
        [5396.699, 1.28852405908, 14.22709400160],
        [4869.308, 0.86793894213, 323.50541665740],
        [4247.455, 0.39299384543, 227.52618943960],
        [3252.084, 1.25853470491, 95.97922721780],
        [2856.006, 2.16731405366, 735.87651353180],
        [2909.411, 4.60679154788, 202.25339517410],
        [3081.408, 3.43662557418, 522.57741809380],
        [1987.689, 2.45054204795, 412.37109687440],
        [1941.309, 6.02393385142, 209.36694217490],
        [1581.446, 1.29191789712, 210.11770170030],
        [1339.511, 4.30801821806, 853.19638175200],
        [1315.590, 1.25296446023, 117.31986822020],
        [1203.085, 1.86654673794, 316.39186965660],
        [1091.088, 0.07527246854, 216.48048917570],
        [954.403, 5.15173410519, 647.01083331480],
        [966.012, 0.47991379141, 632.78373931320],
        [881.827, 1.88471724478, 1052.26838318840],
        [874.215, 1.40224683864, 224.34479570190],
        [897.512, 0.98343776092, 529.69096509460],
        [784.866, 3.06377517461, 838.96928775040],
        [739.892, 1.38225356694, 625.67019231240],
        [612.961, 3.03307306767, 63.73589830340],
        [658.210, 4.14362930980, 309.27832265580],
        [649.600, 1.72489486160, 742.99006053260],
        [599.236, 2.54924174765, 217.23124870110],
        [502.886, 2.12958819475, 3.93215326310],
        [413.017, 4.59334402271, 415.55249061210],
        [356.117, 2.30312127651, 728.76296653100],
        [344.777, 5.88787577835, 440.82528487760],
        [395.004, 0.53349091102, 956.28915597060],
        [335.526, 1.61614647174, 1368.66025284500],
        [362.772, 4.70691652867, 302.16477565500],
        [321.611, 0.97931764923, 3.18139373770],
        [277.783, 0.26007031431, 195.13984817330],
        [291.173, 2.83129427918, 1155.36115740700],
        [264.971, 2.42670902733, 88.86568021700],
        [264.864, 5.82860588985, 149.56319713460],
        [316.777, 3.58395655749, 515.46387109300],
        [294.324, 2.81632778983, 11.04570026390],
        [244.864, 1.04493438899, 942.06206196900],
        [215.368, 3.56535574833, 490.33408917940],
        [264.047, 1.28547685567, 1059.38193018920],
        [246.245, 0.90730313861, 191.95845443560],
        [222.077, 5.13193212050, 269.92144674060],
        [194.973, 4.56665009915, 846.08283475120],
        [182.802, 2.67913220473, 127.47179660680],
        [181.645, 4.93431600689, 74.78159856730],
        [174.651, 3.44560172182, 137.03302416240],
        [165.515, 5.99775895715, 536.80451209540],
        [154.809, 1.19720845085, 265.98929347750],
        [169.743, 4.63464467495, 284.14854074220],
        [151.526, 0.52928231044, 330.61896365820],
        [152.461, 5.43886711695, 422.66603761290],
        [157.687, 2.99559914619, 340.77089204480],
        [140.630, 2.02069760726, 1045.15483618760],
        [139.834, 1.35282959390, 1685.05212250160],
        [140.977, 1.27099900689, 203.00415469950],
        [136.013, 5.01678984678, 351.81659230870],
        [153.391, 0.26968607873, 1272.68102562720],
        [129.476, 1.14344730612, 21.34064100240],
        [127.831, 2.53876158952, 1471.75302706360],
        [126.538, 3.00310970076, 277.03499374140],
        [100.277, 3.61360169153, 1066.49547719000],
        [103.169, 0.38175114761, 203.73786788240],
        [107.527, 4.31870663477, 210.85141488320],
        [95.934, 0.79463744168, 1258.45393162560],
        [82.663, 0.28181414606, 234.63973644040],
        [97.986, 2.56085956186, 191.20769491020],
        [97.389, 3.26245865063, 831.85574074960],
        [72.227, 4.37984630380, 860.30992875280],
        [70.639, 0.73191513920, 437.64389113990],
        [70.447, 0.87698401733, 423.41679713830],
        [72.057, 5.58013290518, 429.77958461370],
        [73.332, 0.62505906432, 1375.77379984580],
        [66.433, 2.68414462465, 405.25754987360],
        [63.812, 1.75051498180, 1361.54670584420],
        [61.601, 1.09332288242, 2001.44399215820],
        [67.006, 0.06872766216, 408.43894361130],
        [68.945, 2.47127505057, 949.17560896980],
        [60.456, 2.25094790113, 1788.14489672020],
        [67.074, 5.45365870159, 200.76892246580],
        [65.579, 0.05539079332, 1589.07289528380],
        [49.320, 4.17243429807, 138.51749687070],
        [50.648, 6.26867505289, 223.59403617650],
        [55.166, 4.59491533823, 628.85158605010],
        [47.916, 0.83929741626, 10.29494073850],
        [46.691, 2.17322569098, 312.19908396260],
        [54.179, 0.28360076018, 124.43341522100],
        [49.511, 3.79960349195, 215.74677599280],
        [40.136, 5.18161452756, 1478.86657406440],
        [39.302, 0.56257369109, 1574.84580128220],
        [34.962, 4.68487505703, 38.13303563780],
        [42.770, 2.98582069454, 1148.24761040620],
        [36.521, 0.63453270366, 52.69019803950],
        [39.752, 0.28412706854, 131.40394986990],
        [31.777, 5.19036499973, 76.26607127560],
        [33.041, 1.97964846430, 142.44965013380],
        [42.053, 4.83017951800, 288.08069400530],
        [30.757, 1.47903923433, 1677.93857550080],
        [42.829, 3.38225543528, 208.63322899200],
        [29.245, 5.09869866956, 654.12438031560],
        [29.165, 4.95664881649, 1795.25844372100],
        [29.136, 2.74747553685, 404.50679034820],
        [32.689, 6.12099521344, 145.63104387150],
        [28.008, 0.83185907283, 2317.83586181480],
        [27.725, 2.24364073545, 430.53034413910],
        [29.939, 1.96415498448, 2104.53676637680],
        [32.982, 3.28236160491, 222.86032299360],
        [31.772, 6.02453027348, 1905.46476494040],
        [26.959, 5.24308283338, 388.46515523820],
        [26.514, 0.99638302878, 107.02492748170],
        [25.421, 2.87336642463, 703.63318461740],
        [24.908, 1.07713811775, 99.91138048090],
        [24.955, 6.23974037842, 106.27416795630],
        [24.894, 0.81040976807, 312.45971639350],
        [24.340, 0.54867402916, 214.26230328450],
        [28.441, 0.82630052794, 1692.16566950240],
        [23.219, 5.07995629354, 479.28838891550],
        [24.362, 3.10643455533, 212.33588759150],
        [21.951, 6.06688237952, 85.82729883120],
        [22.046, 3.89863665506, 563.63121503840],
        [22.596, 4.86725457223, 295.05122865420],
        [21.256, 5.10797617452, 333.65734504400],
        [25.985, 2.20813879137, 1265.56747862640],
        [20.904, 3.28855303434, 70.84944530420],
        [21.505, 3.79541155976, 347.88443904560],
        [22.067, 4.22716352578, 217.96496188400],
        [20.629, 1.68732248608, 231.45834270270],
        [21.429, 3.08914428467, 554.06998748280],
        [21.310, 0.38868340861, 319.57326339430],
        [20.521, 2.45651851283, 18.15924726470],
        [26.055, 4.27554951169, 483.22054217860],
        [20.703, 5.12057936320, 362.86229257260],
        [22.047, 5.51249354809, 343.21857259960],
        [19.443, 2.02441679295, 313.21047591890],
        [20.163, 5.08481373110, 750.10360753340],
        [20.125, 3.42997916125, 213.34727954780],
        [24.196, 0.64787472796, 207.88246946660],
        [21.977, 0.72894956852, 99.16062095550],
        [21.120, 2.69286728009, 1464.63948006280],
        [17.192, 4.71525117969, 2111.65031337760],
        [18.540, 0.04817255506, 245.54242435240],
        [17.521, 3.83662880684, 497.44763618020],
        [16.107, 4.22374822303, 565.11568774670],
        [21.607, 4.16647257628, 2.44768055480],
        [15.979, 0.27376396113, 225.82926841020],
        [16.831, 1.41134653939, 114.13847448250],
        [15.626, 2.82768623405, 81.75213321620],
        [15.499, 1.20606390539, 1994.33044515740],
        [15.168, 3.84591816174, 1162.47470440780],
        [16.436, 3.04752365976, 134.58534360760],
        [15.870, 0.33026420429, 1891.23767093880],
        [20.370, 0.23170286692, 213.25091132820],
        [16.291, 1.70643197929, 2420.92863603340],
        [16.280, 4.94159427320, 357.44566660120],
        [18.076, 5.69515344123, 56.62235130260],
        [13.724, 0.57240190030, 2634.22773147140],
        [17.355, 3.55311137444, 218.92816973050],
        [13.740, 5.70545527289, 92.04707395470],
        [15.328, 1.31338692850, 216.21985674480],
        [12.538, 5.19222019427, 635.96513305090],
        [12.815, 1.60151130870, 320.32402291970],
        [13.043, 0.45068441373, 1169.58825140860],
        [11.984, 5.94916123570, 543.91805909620],
        [11.753, 2.80279347133, 217.49188113200],
        [14.746, 5.56520105813, 344.70304530790],
        [12.762, 1.63557330778, 273.10284047830],
        [11.855, 2.46234840263, 721.64941953020],
        [13.309, 5.75641013916, 2221.85663459700],
        [14.471, 0.45316163629, 2008.55753915900],
        [11.840, 1.75720772380, 160.60889739850],
        [12.374, 1.01456317602, 329.72519178090],
        [10.747, 1.58065203003, 212.77783057620],
        [12.758, 1.91952373240, 1581.95934828300],
        [11.944, 4.44720922423, 32.24332891440],
        [11.865, 5.10696147162, 4.66586644600],
        [11.861, 4.30847607078, 618.55664531160],
        [10.036, 0.48709852137, 305.34616939270],
        [12.777, 3.74412991331, 508.35032409220],
        [10.677, 0.76645916273, 218.71572140940],
        [11.351, 3.00009819697, 198.32124191100],
        [10.249, 2.40923650192, 546.95644048200],
        [9.984, 2.63882014753, 416.30325013750],
        [9.345, 5.45917317860, 414.06801790380],
        [9.317, 4.46380159546, 2428.04218303420],
        [9.928, 4.04821559448, 62.25142559510],
        [12.767, 3.43273835457, 258.87574647670],
        [9.733, 1.61066324680, 327.43756992050],
        [11.163, 2.40665325234, 1781.03134971940],
        [10.608, 2.07480020830, 213.82036029980],
        [9.125, 2.92369523159, 1279.79457262800],
        [9.525, 1.10338403136, 113.38771495710],
        [9.805, 3.28427768485, 275.55052103310],
        [11.263, 1.89402915826, 561.18353448360],
        [8.572, 2.17858055966, 425.11371816770],
        [8.577, 1.95484887975, 35.42472265210],
        [10.157, 0.09037368733, 182.27960680100],
        [11.807, 3.71278037583, 350.33211960040],
        [8.595, 1.83382454431, 629.60234557550],
        [8.396, 3.76782674303, 251.43213107580],
        [8.460, 0.35676476459, 617.80588578620],
        [8.250, 5.31140994372, 65.22037101170],
        [8.342, 1.38307663880, 1.48447270830],
        [7.987, 5.13622898170, 22.09140052780],
        [8.377, 0.91817077859, 1485.98012106520],
        [7.980, 0.94199750915, 2310.72231481400],
        [8.898, 0.54037636841, 168.05251279940],
        [8.233, 3.45785310349, 424.15051032120],
        [8.034, 3.38451795597, 144.14657116320],
        [7.871, 5.14041888473, 358.93013930950],
        [8.868, 6.13541788772, 621.73803904930],
        [7.523, 5.75475671698, 447.93883187840],
        [7.515, 2.18967849979, 264.50482076920],
        [8.083, 1.42661116937, 2737.32050569000],
        [8.199, 0.96419579079, 767.36908292080],
        [8.232, 0.35471613534, 278.51946644970],
        [8.226, 5.44467204721, 254.94359321360],
        [6.779, 1.19567671732, 5.41662597140],
        [8.928, 4.88240256153, 120.35824960600],
        [7.845, 4.56376829397, 280.96714700450],
        [6.566, 3.50152072308, 9.56122755560],
        [6.398, 0.33471834269, 2950.61960112800],
        [6.881, 3.39438820076, 98.89998852460],
        [7.418, 4.52451404934, 5.62907429250],
        [8.021, 0.94470052446, 636.71589257630],
        [6.134, 0.18013315689, 2207.62954059540],
        [7.153, 3.85218295688, 214.04985496340],
        [6.046, 4.66733263196, 543.02428721890],
        [6.365, 2.12000811873, 274.06604832480],
        [6.481, 5.31032923608, 6076.89030155420],
        [5.935, 6.16808119163, 650.94298657790],
        [5.752, 3.55773840903, 1073.60902419080],
        [6.438, 0.44934410249, 10007.09999777380],
        [6.283, 3.20942251433, 219.44943459230],
        [5.542, 3.61193204407, 125.98732389850],
        [5.522, 3.84217355164, 181.05576652360],
        [5.777, 3.00590926498, 121.25202148330],
        [6.670, 1.65236689367, 1898.35121793960],
        [7.591, 0.10483002359, 2324.94940881560],
        [5.881, 1.04006410206, 9992.87290377220],
        [5.609, 4.83142709229, 643.07868005170],
        [5.569, 2.23863483508, 1038.04128918680],
        [5.755, 5.91598458372, 6062.66320755260],
        [5.845, 6.10234689502, 209.10630974400],
        [5.577, 0.81426649853, 472.17484191470],
        [5.247, 0.56496127013, 192.69216761850],
        [5.493, 5.81071309534, 237.67811782620],
        [5.148, 4.85160826999, 267.47376618580],
        [6.122, 2.11480301005, 2097.42321937600],
        [6.188, 4.59441762166, 207.67002114550],
        [6.303, 0.75806431119, 210.37833413120],
        [5.102, 4.01017179605, 205.22234059070],
        [6.583, 1.79054357427, 12.53017297220],
        [4.902, 0.85099521860, 247.23934538180],
        [4.918, 4.03512681632, 487.36514376280],
        [5.818, 5.48495503489, 2538.24850425360],
        [4.855, 4.18197778083, 2744.43405269080],
        [4.885, 0.25103933716, 129.91947716160],
        [5.748, 0.55968589618, 116.42609634290],
        [4.901, 4.48628916012, 291.26208774300],
        [4.720, 5.57686152365, 342.25536475310],
        [5.962, 5.12885837444, 692.58748435350],
        [5.629, 4.39847572369, 196.62432088160],
        [5.596, 0.94874135403, 1802.37199072180],
        [6.197, 3.80364010966, 339.28641933650],
        [4.668, 3.16816375033, 148.07872442630],
        [4.891, 2.67234862638, 417.03696332040],
        [4.959, 1.63453587065, 166.82867252200],
        [4.408, 4.95179678525, 184.09414790940],
        [4.449, 5.69134789394, 252.65597135320],
        [4.943, 0.85358212806, 46.47042291600],
        [5.153, 3.82176885491, 842.15068148810],
        [5.930, 5.95484153666, 486.40193591630],
        [4.206, 2.97664198894, 380.12776796000],
        [4.467, 0.24914978400, 128.95626931510],
        [5.419, 6.19106890918, 337.73251065900],
        [4.499, 4.71434958315, 151.04766984290],
        [4.233, 4.18702525973, 685.47393735270],
        [4.695, 1.54881559549, 214.78356814630],
        [4.084, 4.87173226400, 14.97785352700],
        [4.321, 5.42615168860, 436.89313161450],
        [5.145, 0.49931857511, 248.72381809010],
        [3.897, 0.74661138504, 2627.11418447060],
        [3.995, 3.07750371135, 710.74673161820],
        [4.089, 5.81996977038, 491.81856188770],
        [4.532, 3.67494714028, 189.72322220190],
        [3.690, 1.26565281569, 211.81462272970],
        [4.036, 1.15473702593, 3053.71237534660],
        [3.672, 4.52661018437, 488.84961647110],
        [3.662, 2.87243745783, 411.62033734900],
        [3.653, 3.06205147988, 409.92341631960],
        [3.908, 3.45947158106, 220.46082654860],
        [4.989, 3.36376245705, 824.74219374880],
        [3.677, 3.55713278092, 244.31858407500],
        [3.580, 1.57825591891, 643.82943957710],
        [3.546, 2.19846245030, 135.33610313300],
        [3.560, 4.51362022045, 601.76425067620],
        [3.843, 0.98567531677, 271.40591944890],
        [3.559, 1.11005765159, 6283.07584999140],
        [4.266, 6.19696005871, 268.43697403230],
        [3.442, 4.27628882392, 867.42347575360],
        [4.844, 3.73706907228, 235.39049596580],
        [3.659, 2.21859531609, 2.92076130680],
        [3.958, 5.17084570945, 114.39910691340],
        [3.609, 5.54387488088, 458.84151979040],
        [4.470, 3.74256930900, 699.70103135430],
        [3.293, 4.48068043469, 289.56516671360],
        [3.240, 5.94728881707, 131.54696222180],
        [3.477, 3.54553285172, 963.40270297140],
        [3.838, 4.77967877681, 175.16605980020],
        [3.223, 1.95410765469, 212.02707105080],
        [4.053, 4.19011281964, 501.37978944330],
        [3.100, 2.11956558345, 916.93228005540],
        [3.183, 1.93201605379, 1354.43315884340],
        [3.301, 1.80825506815, 756.32338265690],
        [4.187, 5.96622666047, 212.54833591260],
        [3.716, 3.70660462807, 204.70107572890],
        [3.000, 6.15443664698, 3267.01147078460],
        [2.993, 4.20888489881, 533.62311835770],
        [4.125, 6.09715151219, 2641.34127847220],
        [3.145, 2.55483540896, 905.88657979150],
        [2.982, 1.52760656472, 945.99421523210],
        [3.015, 1.76012152992, 28.31117565130],
        [3.453, 1.42473508236, 2214.74308759620],
        [2.926, 5.50138147476, 24.37902238820],
        [2.978, 4.27440059910, 195.89060769870],
        [3.526, 3.63935401565, 229.97386999440],
        [2.860, 4.52551886503, 241.61027108930],
        [3.059, 5.68165832697, 282.66406803390],
        [3.415, 5.26311934884, 67.66805156650],
        [2.819, 5.42053027567, 305.08553696180],
        [3.503, 1.31670335802, 69.15252427480],
        [2.746, 0.82597971627, 444.75743814070],
        [2.796, 0.07021047160, 681.54178408960],
        [3.366, 4.03843228994, 6.15033915430],
        [3.242, 2.63461047831, 739.80866679490],
        [2.718, 3.40899287465, 188.92007304980],
        [2.741, 3.22092213412, 776.93031047640],
        [2.793, 3.39766347322, 431.26405732200],
        [2.966, 3.91429372950, 526.50957135690],
        [2.693, 3.38996413068, 778.41478318470],
        [2.680, 3.82192393959, 3060.82592234740],
        [2.954, 2.69669880207, 426.64637498580],
        [2.681, 1.04615621583, 28.45418800320],
        [3.182, 2.72333374876, 432.22726516850],
        [2.633, 2.55029306465, 10213.28554621100],
        [2.923, 0.85695094024, 2435.15573003500],
        [2.596, 5.42890752137, 207.14875628370],
        [3.225, 0.96538615730, 2118.76386037840],
        [2.774, 0.33260844270, 326.68681039510],
        [2.550, 5.88893697427, 439.12836384820],
        [2.716, 3.15505406487, 170.76082578510],
        [2.942, 4.88555233562, 397.39324334740],
        [3.121, 1.87815629157, 2413.81508903260],
        [3.263, 2.59868619716, 213.03846300710],
        [2.518, 0.15471130491, 945.24345570670],
        [3.169, 5.70993651497, 381.35160823740],
        [2.515, 0.06248441393, 427.56139872250],
        [3.279, 4.95751323467, 313.94418910180],
        [2.595, 5.13169797457, 299.12639426920],
        [2.572, 3.42558391509, 4.19278569400],
        [2.580, 2.03280916494, 319.31263096340],
        [3.294, 6.24566168486, 421.18156490460],
        [2.580, 2.62721090534, 213.18722085340],
        [2.879, 0.45679876898, 285.63301345050],
        [2.406, 4.57473098758, 228.27694896500],
        [2.518, 2.55500085830, 140.00196957900],
        [2.422, 2.36310658303, 84.34282612290],
        [2.374, 2.25544718932, 17.26547538740],
        [2.627, 1.26370339212, 724.83081326790],
        [2.346, 3.77641630157, 206.23373254700],
        [2.463, 5.42094278240, 395.57870223900],
        [2.352, 0.63041319237, 210.59078245230],
        [3.166, 0.26273580642, 201.51968199120],
        [2.405, 0.78919759458, 426.07692601420],
        [2.390, 5.89523812458, 738.79727483860],
        [2.515, 0.70044371265, 2943.50605412720],
        [2.332, 4.06963624306, 519.39602435610],
        [3.132, 2.79331632190, 732.69511979410],
        [2.658, 3.34020209714, 1141.13406340540],
        [2.258, 0.12403309730, 2524.02141025200],
        [2.697, 2.58404587754, 425.63498302950],
        [2.416, 3.85003724506, 696.51963761660],
        [2.597, 2.54164936697, 436.15941843160],
        [2.192, 3.07202313269, 203.26478713040],
        [2.424, 2.60715310452, 511.53171782990],
        [2.126, 0.14811901148, 405.99126305650],
        [2.306, 1.25068142377, 427.11945573780],
        [2.121, 0.43505808954, 184.98791978670],
        [2.755, 3.02380019321, 468.24268865160],
        [2.333, 3.02634928771, 216.00740842370],
        [2.182, 4.27912012069, 7.16173111060],
        [2.101, 4.31781498012, 572.22923474750],
        [2.362, 4.82914341110, 556.51766803760],
        [2.218, 0.82936075453, 3370.10424500320],
        [2.103, 5.25950154713, 661.23792731640],
        [2.580, 1.03705340380, 213.41097002260],
        [2.366, 6.14368355608, 205.43478891180],
        [2.042, 0.21462094901, 3259.89792378380],
        [2.547, 4.69969204009, 221.37585028530],
        [1.987, 3.22670561632, 1382.88734684660],
        [2.213, 0.89932827487, 286.59622129700],
        [2.191, 0.08759174058, 259.76951835400],
        [1.968, 0.57824086026, 180.16199464630],
        [2.037, 2.35713099759, 610.69233878540],
        [1.959, 2.18553775379, 72.07328558160],
        [2.061, 1.68041202479, 1670.82502850000],
        [1.940, 0.62951066481, 406.95447090300],
        [2.043, 4.39130144045, 576.16138801060],
        [1.936, 1.05286934530, 1262.38608488870],
        [1.975, 0.31945835160, 938.12990870590],
        [2.015, 1.66410213484, 193.65537546500],
        [1.971, 0.72639439054, 200.55647414470],
        [1.952, 6.25320630177, 241.75328344120],
        [1.976, 1.31263772699, 135.54855145410],
        [2.448, 0.52850194172, 429.51895218280],
        [1.977, 3.13944703383, 421.93232443000],
        [1.853, 0.17184530353, 196.03362005060],
        [2.552, 5.39764879348, 2854.64037391020],
        [1.830, 1.47821899466, 638.41281360570],
        [2.245, 6.00427164270, 230.70758317730],
        [1.822, 6.08626100417, 1261.63532536330],
        [2.168, 0.41741136149, 213.51154375910],
        [1.869, 3.67791368036, 403.02231763990],
        [1.866, 1.59662677545, 391.17346822390],
        [2.034, 1.21814866092, 3046.59882834580],
        [1.929, 4.93193335031, 420.96911658350],
        [1.746, 5.09757251683, 107.75864066460],
        [2.168, 3.24685294764, 213.08664711690],
        [2.178, 5.09777299346, 558.00214074590],
        [1.992, 2.29524873043, 1773.91780271860],
        [1.761, 2.88655624670, 141.69889060840],
        [1.769, 5.47051542758, 206.13736432740],
        [1.734, 2.11941015901, 430.79097657000],
        [2.377, 1.07633521570, 59.80374504030],
        [1.797, 2.90984583978, 92.79783348010],
        [1.725, 5.22827286197, 757.21715453420],
        [2.305, 5.88235807192, 426.55000676620],
        [1.751, 5.28990803470, 87.31177153950],
        [2.202, 1.28096946505, 624.91943278700],
        [2.043, 0.46193065602, 831.10498122420],
        [1.931, 1.26974971942, 219.89137757700],
        [1.953, 2.96900002385, 398.14400287280],
        [1.676, 4.81683149512, 181.80652604900],
        [1.902, 2.74426125465, 4952.06359328620],
        [2.133, 5.37177705284, 627.36711334180],
        [1.962, 3.52111949662, 213.45915413240],
        [1.709, 6.14073761844, 952.35700270750],
        [1.784, 1.05243716682, 353.30106501700],
        [1.700, 1.17418864170, 739.05790726950],
        [1.609, 1.35009554392, 84.93352695390],
        [2.038, 2.47570829812, 26.82670294300],
        [1.870, 5.61729116529, 2957.73314812880],
        [1.962, 0.13564680851, 213.13903674360],
        [2.041, 3.31354526279, 1596.18644228460],
        [1.612, 6.19495100885, 432.01481684740],
        [1.742, 2.87947098602, 179.35884549420],
        [1.964, 2.84253666387, 429.04587143080],
        [1.805, 0.60932632638, 532.61172640140],
        [1.647, 0.82347900016, 214.57111982520],
        [1.893, 4.33962647901, 173.94221952280],
        [1.689, 1.13037158144, 586.31331639720],
        [1.523, 2.71561930244, 73.29712585900],
        [1.524, 5.26558677448, 5429.87946823940],
        [1.582, 2.79533721474, 842.90144101350],
        [1.608, 2.33230359324, 418.52143602870],
        [1.579, 1.15182102801, 731.94436026870],
        [1.689, 1.91915438546, 630.33605875840],
        [1.990, 5.23790221176, 550.13783421970],
        [1.772, 2.95372411478, 172.24529849340],
        [1.596, 0.99004701777, 953.10776223290],
        [1.784, 3.91391032360, 159.12442469020],
        [1.592, 2.99690086808, 45.57665103870],
        [1.968, 0.23073879009, 220.36445832900],
        [1.549, 5.88699595922, 60.55450456570],
        [1.459, 5.51999778036, 273.85360000370],
        [1.909, 2.78415262815, 418.00017116690],
        [1.445, 3.25530914937, 453.42489381900],
        [1.454, 0.16250693313, 115.62294719080],
        [1.566, 2.24077018103, 1056.20053645150],
        [1.412, 3.45442909885, 354.99798604640],
        [1.564, 3.38591337689, 409.18970313670],
        [1.631, 1.06286709889, 213.55972786890],
        [1.415, 1.32091877590, 373.90799283650],
        [1.389, 0.40584159469, 9360.08916445900],
        [1.663, 2.33357114562, 188.02630117250],
        [1.426, 5.44677783737, 864.24208201590],
        [1.716, 3.96056093028, 1699.27921650320],
        [1.682, 0.39747670320, 17.40848773930],
        [1.368, 5.83289186692, 569.04784100980],
        [1.416, 3.65464640816, 6.85291456990],
        [1.469, 3.49069193830, 934.94851496820],
        [1.330, 4.41794310534, 3914.02230409940],
        [1.309, 1.29979865382, 428.08266358430],
        [1.300, 1.57748627871, 238.57188970350],
        [1.389, 1.31202796503, 6275.96230299060],
        [1.384, 0.67585082323, 2751.54759969160],
        [1.471, 1.21149871903, 2531.13495725280],
        [1.334, 4.11154515525, 206.93630796260],
        [1.601, 0.93356520728, 355.74874557180],
        [1.259, 2.56678207309, 850.01498801430],
        [1.277, 0.41764451386, 100.64509366380],
        [1.436, 4.06045514506, 177.87437278590],
        [1.308, 1.01324076289, 423.67742956920],
        [1.541, 6.03020449918, 292.01284726840],
        [1.307, 5.83815678434, 5863.59120611620],
        [1.613, 2.45074803642, 1049.08698945070],
        [1.249, 3.01518429832, 464.73122651380],
        [1.250, 6.23516728885, 823.99143422340],
        [1.275, 2.68217384213, 637.44960575920],
        [1.249, 2.97028182853, 51749.20809227239],
        [1.240, 2.66940683813, 2700.71514038580],
        [1.456, 1.85558224828, 96.87299909510],
        [1.491, 4.98649587341, 295.19424100610],
        [1.230, 4.27283851216, 12139.55350910680],
        [1.292, 2.73196017809, 10206.17199921020],
        [1.247, 3.77399749791, 504.56118318100],
        [1.408, 1.02955773079, 518.38463239980],
        [1.223, 1.12202093840, 221.16340196420],
        [1.420, 4.39795293289, 606.76018552230],
        [1.190, 1.57292553631, 820.05928096030],
        [1.247, 0.99102599652, 9793.80090233580],
        [1.234, 1.10826361423, 2303.60876781320],
        [1.186, 4.55984967028, 9808.53818466140],
        [1.346, 4.94456950019, 384.05992122310],
        [1.514, 3.60392291730, 2015.67108615980],
        [1.432, 2.28704432909, 525.49817940060],
        [1.129, 0.87100340620, 162.09337010680],
        [1.196, 5.13485214850, 227.31374111850],
        [1.339, 2.48923887712, 206.70681329900],
        [1.421, 1.65379789078, 857.12853501510],
        [1.162, 1.92099315083, 220.93390730060],
        [1.277, 4.85435999187, 54.17467074780],
        [1.153, 5.33028034679, 233.90602325750],
        [1.214, 4.11324721963, 3377.21779200400],
        [1.109, 5.68915582674, 162.89651925890],
        [1.068, 4.85383480876, 611.44309831080],
        [1.119, 1.40805686363, 1987.21689815660],
        [1.085, 0.64208148190, 731.68372783780],
        [1.435, 3.20880139888, 835.78789401270],
        [1.184, 2.99776919968, 199.28444975750],
        [1.281, 3.12245339510, 427.34895040140],
        [1.058, 5.17851282929, 306.09692891810],
        [1.152, 4.39244449554, 199.96577331370],
        [1.036, 3.68027119804, 597.35901666110],
        [1.055, 3.25561743426, 394.35486196160],
        [1.127, 4.33255371960, 552.58551477450],
        [1.213, 6.21447612110, 42.53826965290],
        [1.117, 3.74367882111, 214.19286731530],
        [1.023, 3.84199833949, 894.84087952760],
        [1.042, 5.30120078590, 450.97721326420],
        [1.290, 3.96221234564, 318.83955021140],
        [1.073, 4.10012122884, 188.16931352440],
        [1.204, 0.37702365750, 393.46109008430],
        [1.214, 2.01826978554, 401.32539661050],
        [1.018, 0.02946649279, 2840.41327990860],
        [1.237, 5.41088851225, 425.84743135060],
        [1.187, 5.16511890602, 838.21852822500],
        [1.276, 2.93572146232, 1457.52593306200],
        [0.994, 3.40079885702, 211.60217440860],
        [1.042, 2.42209320898, 361.37781986430],
        [1.093, 3.66289018246, 226.63241756230],
        [0.978, 3.76334208607, 5856.47765911540],
        [1.263, 2.09195268609, 78.71375183040],
        [1.009, 5.85963705048, 1268.74887236410],
        [1.148, 4.39543895068, 570.74476203920],
        [1.051, 3.27272240682, 153.49535039770],
        [0.975, 3.42924642244, 105.54045477340],
        [0.997, 4.30943991893, 212.40532356070],
        [0.954, 3.88548755058, 171.65459766240],
        [0.960, 1.90180005280, 1159.29331067010],
        [0.953, 3.40787141587, 244.79166482700],
        [0.969, 1.93369993197, 525.75881183150],
        [0.918, 1.73738789723, 223.33340374560],
        [1.164, 5.05392864346, 263.70167161710],
        [0.951, 4.23581224839, 92.94084583200],
        [1.160, 5.80630916592, 460.53844081980],
        [1.186, 4.46262000755, 465.95506679120],
        [0.931, 2.09868057209, 205.66428357540],
        [0.942, 3.86810837922, 238.42887735160],
        [1.020, 5.53181822898, 0.04818410980],
        [0.995, 2.03457885490, 6290.18939699220],
        [0.888, 2.60957592990, 1912.57831194120],
        [0.873, 5.78433393020, 480.77286162380],
        [0.854, 1.63255087291, 328.24071907260],
        [1.123, 4.07401922216, 3693.60966166060],
        [0.934, 3.52355235083, 10220.39909321180],
        [0.833, 3.03302227840, 532.87235883230],
        [1.007, 2.73615455688, 4841.85727206680],
        [0.870, 1.06968760644, 51.20572533120],
        [0.891, 1.36817544763, 700.45179087970],
        [0.833, 5.39754715806, 159.71512552120],
        [0.854, 1.91765015557, 622.48879857470],
        [0.976, 3.09106001923, 2332.06295581640],
        [0.819, 5.55683690482, 462.02291352810],
        [0.800, 1.47042677460, 969.62247809490],
        [0.933, 1.40166917666, 287.93768165340],
        [0.784, 1.69235162770, 477.80391620720],
        [0.782, 3.98153416585, 702.14871190910],
        [0.810, 5.87689161549, 561.93429400900],
        [0.858, 4.02964773169, 41.64449777560],
        [0.819, 0.98885784755, 960.22130923370],
        [0.882, 1.49559638306, 760.25553592000],
        [0.767, 2.46787654531, 402.21916848780],
        [0.851, 3.64678001195, 348.63519857100],
        [0.852, 1.03470840672, 2620.00063746980],
        [0.770, 4.67090683753, 16.67477455640],
        [0.849, 5.27994730935, 74.63858621540],
        [0.854, 2.56811257488, 432.74853003030],
        [0.767, 1.11806243753, 2847.52682690940],
        [0.729, 0.44171990710, 898.77303279070],
        [0.776, 5.54603607568, 3171.03224356680],
        [0.721, 6.05392551158, 91.78644152380],
        [0.739, 3.57172839746, 775.23338944700],
        [0.730, 4.98865345688, 705.11765732570],
        [0.705, 0.44942445750, 219.66188291340],
        [0.708, 0.69014726046, 1048.33622992530],
        [0.706, 2.22974712805, 29.20494752860],
        [0.711, 0.00981284716, 2115.58246664070],
        [0.722, 4.14075205197, 225.07850888480],
        [0.735, 1.54083195463, 201.99276274320],
        [0.746, 4.08997409526, 849.26422848890],
        [0.729, 2.81779134868, 419.53282798500],
        [0.837, 1.00844734156, 4127.32139953740],
        [0.939, 0.46037763932, 5488.86810538160],
        [0.939, 1.85473712038, 5062.26991450560],
        [0.721, 1.62872794201, 2200.51599359460],
    ],
    # R2
    [
        [436902.464, 4.78671673044, 213.29909543800],
        [71922.760, 2.50069994874, 206.18554843720],
        [49766.792, 4.97168150870, 220.41264243880],
        [43220.894, 3.86940443794, 426.59819087600],
        [29645.554, 5.96310264282, 7.11354700080],
        [4141.650, 4.10670940823, 433.71173787680],
        [4720.909, 2.47527992423, 199.07200143640],
        [3789.370, 3.09771025067, 639.89728631400],
        [2963.990, 1.37206248846, 103.09277421860],
        [2556.363, 2.85065721526, 419.48464387520],
        [2208.457, 6.27588858707, 110.20632121940],
        [2187.621, 5.85545832218, 14.22709400160],
        [1956.896, 4.92448618045, 227.52618943960],
        [2326.801, 0.00000000000, 0.00000000000],
        [923.840, 5.46392422737, 323.50541665740],
        [705.936, 2.97081280098, 95.97922721780],
        [546.115, 4.12854181522, 412.37109687440],
        [373.838, 5.83435991809, 117.31986822020],
        [360.882, 3.27703082368, 647.01083331480],
        [356.350, 3.19152043942, 210.11770170030],
        [390.627, 4.48106176893, 216.48048917570],
        [431.485, 5.17825414612, 522.57741809380],
        [325.598, 2.26867601656, 853.19638175200],
        [405.018, 4.17294157872, 209.36694217490],
        [204.494, 0.08774848590, 202.25339517410],
        [206.854, 4.02188336738, 735.87651353180],
        [178.474, 4.09716541453, 440.82528487760],
        [180.143, 3.59704903955, 632.78373931320],
        [153.656, 3.13470530382, 625.67019231240],
        [147.779, 0.13614300541, 302.16477565500],
        [123.189, 4.18895309647, 88.86568021700],
        [133.076, 2.59350469420, 191.95845443560],
        [100.367, 5.46056190585, 3.18139373770],
        [131.975, 5.93293968941, 309.27832265580],
        [97.235, 4.01832604356, 728.76296653100],
        [110.709, 4.77853798276, 838.96928775040],
        [119.053, 5.55385105975, 224.34479570190],
        [93.852, 4.38395529912, 217.23124870110],
        [108.701, 5.29310899841, 515.46387109300],
        [78.609, 5.72525447528, 21.34064100240],
        [81.468, 5.10897365253, 956.28915597060],
        [96.412, 6.25859229567, 742.99006053260],
        [69.228, 4.04901237761, 3.93215326310],
        [65.168, 3.77713343518, 1052.26838318840],
        [64.088, 5.81235002453, 529.69096509460],
        [62.541, 2.18445116349, 195.13984817330],
        [56.987, 3.14666549033, 203.00415469950],
        [55.979, 4.84108422860, 234.63973644040],
        [52.940, 5.07780548444, 330.61896365820],
        [50.635, 2.77318570728, 942.06206196900],
        [41.649, 4.79014211005, 63.73589830340],
        [44.858, 0.56460613593, 269.92144674060],
        [41.357, 3.73496404402, 316.39186965660],
        [52.847, 3.92623831484, 949.17560896980],
        [38.398, 3.73966157784, 1045.15483618760],
        [37.583, 4.18924633757, 536.80451209540],
        [35.285, 2.90795856092, 284.14854074220],
        [33.576, 3.80465978802, 149.56319713460],
        [41.073, 4.57870454147, 1155.36115740700],
        [30.412, 2.48140171991, 860.30992875280],
        [31.373, 4.84075951849, 1272.68102562720],
        [30.218, 4.35186294470, 405.25754987360],
        [39.430, 3.50858482049, 422.66603761290],
        [29.658, 1.58886982096, 1066.49547719000],
        [35.202, 5.94478241578, 1059.38193018920],
        [25.829, 3.54946335477, 1368.66025284500],
        [26.283, 4.81567477177, 124.43341522100],
        [29.963, 3.66312205813, 429.77958461370],
        [33.011, 4.96879544579, 831.85574074960],
        [24.305, 5.31133255082, 10.29494073850],
        [26.332, 4.45253273390, 223.59403617650],
        [22.108, 2.76092021113, 415.55249061210],
        [27.187, 1.66347897738, 277.03499374140],
        [21.639, 1.03836302307, 11.04570026390],
        [19.713, 2.52194629263, 1258.45393162560],
        [17.062, 3.27669927228, 654.12438031560],
        [17.261, 3.49414816663, 1361.54670584420],
        [16.097, 1.73396878598, 490.33408917940],
        [21.099, 3.62102032955, 1265.56747862640],
        [17.692, 4.31141612385, 1471.75302706360],
        [13.458, 0.32327889681, 295.05122865420],
        [12.586, 3.13794576887, 74.78159856730],
        [12.023, 2.32917797741, 210.85141488320],
        [15.120, 3.59558424278, 265.98929347750],
        [12.959, 4.62359706368, 1589.07289528380],
        [15.424, 5.01335704925, 127.47179660680],
        [11.193, 4.54981248285, 81.75213321620],
        [13.449, 4.88710089777, 437.64389113990],
        [10.673, 5.05234757424, 191.20769491020],
        [13.963, 3.04990968366, 423.41679713830],
        [10.614, 5.02845923229, 137.03302416240],
        [14.382, 4.68720080027, 1148.24761040620],
        [13.470, 1.90280407135, 408.43894361130],
        [10.077, 5.20426583827, 340.77089204480],
        [10.323, 3.34460279759, 1685.05212250160],
        [9.563, 3.17317920222, 351.81659230870],
        [11.295, 5.47808960704, 1375.77379984580],
        [8.617, 2.81294528041, 99.91138048090],
        [8.460, 3.22691940753, 1677.93857550080],
        [7.914, 2.35624291874, 1574.84580128220],
        [7.587, 6.08171425316, 231.45834270270],
        [9.175, 3.40072244924, 1581.95934828300],
        [7.337, 2.00393601815, 131.40394986990],
        [8.240, 4.04095881407, 1788.14489672020],
        [7.579, 3.68311134272, 846.08283475120],
        [6.691, 4.37253800717, 145.63104387150],
        [7.539, 3.29482043104, 750.10360753340],
        [6.367, 4.00239137708, 447.93883187840],
        [6.249, 4.55603671940, 106.27416795630],
        [6.489, 1.33782087599, 215.74677599280],
        [6.501, 3.78204726337, 313.21047591890],
        [5.978, 0.55276980086, 18.15924726470],
        [6.171, 2.84712795642, 138.51749687070],
        [6.837, 4.83481646949, 319.57326339430],
        [6.678, 5.43046031699, 508.35032409220],
        [7.175, 4.37855723752, 1464.63948006280],
        [5.753, 4.14268749228, 543.91805909620],
        [5.727, 4.35383078313, 1905.46476494040],
        [5.101, 2.63866058897, 288.08069400530],
        [5.311, 3.62520849510, 6076.89030155420],
        [5.498, 4.19972735173, 721.64941953020],
        [5.089, 5.04845206653, 10007.09999777380],
        [5.505, 1.13479635941, 56.62235130260],
        [4.820, 3.30043078578, 76.26607127560],
        [4.915, 6.17790518458, 483.22054217860],
        [5.048, 2.44627820757, 628.85158605010],
        [4.534, 1.19648682598, 200.76892246580],
        [4.817, 3.11549733365, 2001.44399215820],
        [4.712, 1.26507812515, 6062.66320755260],
        [4.811, 5.78388270496, 184.84490743480],
        [4.775, 0.76197795755, 333.65734504400],
        [4.514, 0.95293919611, 343.21857259960],
        [4.525, 2.68827745072, 9992.87290377220],
        [4.378, 0.80241129896, 222.86032299360],
        [4.873, 5.92092913946, 618.55664531160],
        [4.142, 1.91878383159, 497.44763618020],
        [5.112, 4.50449287745, 416.30325013750],
        [4.125, 1.98204847532, 347.88443904560],
        [4.045, 2.87666810085, 38.13303563780],
        [4.133, 2.90478811425, 107.02492748170],
        [4.035, 2.92972681787, 1994.33044515740],
        [4.916, 3.12316267561, 1898.35121793960],
        [3.657, 3.24680246734, 362.86229257260],
        [3.753, 0.87719890943, 703.63318461740],
        [3.576, 3.48080143501, 388.46515523820],
        [3.555, 4.08436297683, 430.53034413910],
        [3.598, 0.05255328597, 32.24332891440],
        [3.561, 5.46414552453, 6283.07584999140],
        [3.480, 1.81622589595, 70.84944530420],
        [3.827, 3.12041228490, 635.96513305090],
        [3.399, 0.54882815021, 10213.28554621100],
        [3.399, 3.51833356080, 629.60234557550],
        [3.364, 3.27821747958, 357.44566660120],
        [3.260, 1.97623748027, 203.73786788240],
        [3.118, 2.18465627368, 1891.23767093880],
        [3.163, 1.26040995242, 134.58534360760],
        [4.004, 5.45434102599, 1692.16566950240],
        [3.180, 2.46319174788, 867.42347575360],
        [3.389, 4.20503159673, 337.73251065900],
        [3.026, 2.19331614526, 217.96496188400],
        [3.573, 5.55097240810, 113.38771495710],
        [3.682, 3.78966280284, 2104.53676637680],
        [3.125, 4.09203641264, 1478.86657406440],
        [2.881, 3.90810650240, 312.19908396260],
        [3.199, 3.92123638342, 1038.04128918680],
        [4.014, 5.17826893553, 404.50679034820],
        [3.907, 4.11767191780, 1781.03134971940],
        [3.144, 1.61185684069, 1073.60902419080],
        [3.072, 5.00675625396, 312.45971639350],
        [3.034, 5.46288652854, 258.87574647670],
        [2.884, 2.38477237305, 181.05576652360],
        [2.986, 0.88783591586, 1279.79457262800],
        [2.683, 0.00956197492, 195.89060769870],
        [3.081, 5.60034737330, 216.21985674480],
        [2.626, 6.12701960244, 273.10284047830],
        [2.665, 2.31576422128, 565.11568774670],
        [3.245, 3.87540558646, 85.82729883120],
        [2.740, 5.73784096806, 160.60889739850],
        [2.876, 4.74720607366, 213.25091132820],
        [2.523, 5.30458920892, 444.75743814070],
        [2.752, 5.08984539930, 1169.58825140860],
        [2.889, 1.66674437398, 213.34727954780],
        [2.923, 4.21481009033, 650.94298657790],
        [3.036, 2.55426675350, 6069.77675455340],
        [3.116, 2.67220972004, 52.69019803950],
        [2.371, 0.89591351822, 121.25202148330],
        [2.993, 3.96957827454, 9999.98645077300],
        [3.088, 0.40656113014, 561.18353448360],
        [2.385, 4.74063881551, 218.71572140940],
        [2.632, 1.12706218927, 344.70304530790],
        [2.316, 4.08445262041, 131.54696222180],
        [2.214, 3.37726228553, 22.09140052780],
        [2.129, 3.32497715011, 358.93013930950],
        [2.679, 1.68971401870, 208.63322899200],
        [2.607, 5.10250482155, 824.74219374880],
        [2.250, 2.60474848767, 305.34616939270],
        [2.087, 3.37293958793, 320.32402291970],
        [2.693, 3.62159456470, 436.89313161450],
        [2.492, 2.96129217279, 2214.74308759620],
        [2.704, 2.88483697319, 643.07868005170],
        [2.124, 1.61210282593, 218.92816973050],
        [2.037, 4.63481160778, 188.02630117250],
        [2.394, 3.46386258552, 6275.96230299060],
        [1.973, 2.28886138203, 2627.11418447060],
        [1.937, 5.67082364247, 28.45418800320],
        [1.920, 4.25647211328, 546.95644048200],
        [2.498, 3.57572154405, 2420.92863603340],
        [1.898, 1.30987536388, 212.33588759150],
        [1.852, 1.58508015515, 424.15051032120],
        [1.850, 3.57830449726, 329.72519178090],
        [2.128, 3.95329215734, 1795.25844372100],
        [2.236, 4.22073549375, 2221.85663459700],
        [1.933, 1.68771499202, 350.33211960040],
        [1.799, 2.06541260431, 144.14657116320],
        [1.904, 4.60953896857, 182.27960680100],
        [2.236, 5.17945392885, 99.16062095550],
        [1.755, 2.73425330428, 291.26208774300],
        [2.231, 5.42548168745, 207.88246946660],
        [1.848, 2.24194286719, 168.05251279940],
        [1.726, 1.31878655393, 219.44943459230],
        [1.709, 5.55913931846, 92.79783348010],
        [1.693, 1.95360003617, 129.91947716160],
        [2.064, 4.84900344498, 1141.13406340540],
        [1.758, 5.05088656436, 214.26230328450],
        [1.781, 2.85880153340, 636.71589257630],
        [1.900, 2.90295578617, 2310.72231481400],
        [1.759, 5.34657858395, 45.57665103870],
        [1.654, 6.14450664508, 554.06998748280],
        [1.578, 4.50941374663, 210.37833413120],
        [1.681, 3.55136706992, 1354.43315884340],
        [1.862, 3.01276783582, 2317.83586181480],
        [1.589, 1.15773448350, 235.39049596580],
        [1.551, 2.15558953807, 207.67002114550],
        [1.874, 4.12861627986, 225.82926841020],
        [1.621, 3.29992957653, 1670.82502850000],
        [1.911, 0.17724319140, 12.53017297220],
        [1.477, 5.90270260570, 1.48447270830],
        [1.618, 5.72513459206, 1485.98012106520],
        [1.446, 1.78104589920, 1382.88734684660],
        [1.683, 3.43534671475, 2428.04218303420],
        [1.542, 5.51223038941, 204.70107572890],
        [1.420, 2.07339356364, 198.32124191100],
        [1.444, 5.56032454849, 128.36556848410],
        [1.476, 6.12782257368, 212.77783057620],
        [1.474, 0.33626790634, 213.82036029980],
        [1.428, 3.25039966249, 945.99421523210],
        [1.410, 0.68747644676, 429.04587143080],
        [1.752, 2.70090942746, 12.74262129330],
        [1.681, 4.97526853273, 2008.55753915900],
        [1.408, 0.80461100746, 1585.14074202070],
        [1.485, 0.49674043855, 120.35824960600],
        [1.490, 2.68459799437, 207.14875628370],
        [1.411, 4.36399216092, 5863.59120611620],
        [1.315, 4.73430848989, 241.75328344120],
        [1.516, 4.99488503706, 1162.47470440780],
        [1.310, 1.98714265058, 563.63121503840],
        [1.286, 2.12891372062, 251.43213107580],
        [1.271, 5.70165238307, 2.92076130680],
        [1.312, 1.68811514551, 2207.62954059540],
        [1.259, 0.35924965717, 334.55111692130],
        [1.252, 2.14513440216, 1055.44977692610],
        [1.401, 6.13250261735, 1802.37199072180],
        [1.343, 5.79995727295, 9793.80090233580],
        [1.228, 3.29059284057, 661.23792731640],
        [1.202, 2.88792018909, 2413.81508903260],
        [1.286, 5.72360160371, 298.23262239190],
        [1.357, 0.93175963411, 217.49188113200],
        [1.356, 2.28121627817, 601.76425067620],
        [1.190, 1.94993809928, 501.37978944330],
        [1.304, 0.37337923280, 3473.19701922180],
        [1.350, 2.87235622320, 142.44965013380],
        [1.349, 3.21102203937, 175.16605980020],
        [1.312, 3.70149813509, 2111.65031337760],
        [1.129, 1.08860603834, 842.15068148810],
        [1.237, 0.08698781252, 526.50957135690],
        [1.217, 3.89835349840, 209.10630974400],
        [1.467, 1.16228775027, 621.73803904930],
        [1.044, 0.30512759901, 436.15941843160],
        [1.140, 5.33720637097, 114.13847448250],
        [1.295, 4.70261675421, 9786.68735533500],
        [1.037, 4.07846687083, 156.67674413540],
        [1.391, 4.73554028436, 398.14400287280],
        [1.167, 5.68899703631, 479.28838891550],
        [1.035, 5.34279429465, 327.43756992050],
        [0.997, 1.19323192891, 710.74673161820],
        [1.193, 5.17722376816, 98.89998852460],
        [1.165, 4.58588490135, 732.69511979410],
        [1.161, 4.90854984994, 10206.17199921020],
        [1.144, 0.50394784140, 3906.90875709860],
        [1.182, 3.69482624364, 2854.64037391020],
        [0.970, 2.89031410383, 1987.21689815660],
        [1.039, 0.48694895443, 525.49817940060],
        [1.079, 3.61750956217, 2097.42321937600],
        [1.148, 3.31015591733, 5856.47765911540],
        [1.241, 4.31971543677, 230.70758317730],
        [0.910, 4.59825926062, 380.12776796000],
        [0.907, 1.34912454077, 685.47393735270],
        [1.166, 1.61085609717, 5849.36411211460],
        [0.882, 6.12045540405, 519.39602435610],
        [0.963, 4.96065454054, 699.70103135430],
        [1.062, 5.13323858077, 2751.54759969160],
        [0.865, 6.12821112133, 245.54242435240],
        [1.100, 2.18435744407, 1699.27921650320],
        [0.822, 5.55083534581, 739.05790726950],
        [0.926, 2.01158276144, 417.03696332040],
        [0.813, 5.18401872205, 214.78356814630],
        [1.033, 5.48677848094, 3995.77443731560],
        [0.872, 3.02363724703, 306.09692891810],
        [0.796, 0.44343664540, 486.40193591630],
        [0.878, 1.82164034386, 135.33610313300],
        [0.791, 2.14989417962, 2620.00063746980],
        [0.881, 2.39697554334, 289.56516671360],
        [0.782, 4.50471317138, 980.66817835880],
        [0.783, 1.14229319753, 540.73666535850],
        [0.831, 0.69937251013, 421.93232443000],
        [0.770, 2.40292326155, 576.16138801060],
        [0.950, 5.97621460162, 196.62432088160],
        [0.814, 4.19303098086, 831.10498122420],
        [0.969, 4.78071024754, 326.68681039510],
        [0.760, 0.44860533530, 425.63498302950],
        [0.907, 0.94781730418, 525.75881183150],
        [0.788, 0.14287187051, 916.93228005540],
        [0.801, 1.86383119100, 3039.48528134500],
        [0.801, 0.46947170994, 3466.08347222100],
        [0.747, 6.05374861925, 211.81462272970],
        [0.968, 3.02618272726, 2634.22773147140],
        [0.739, 2.27110740297, 2303.60876781320],
        [0.750, 5.48554383902, 173.94221952280],
        [1.024, 1.91564925560, 229.97386999440],
        [0.816, 4.98990432666, 4209.07353275360],
        [0.728, 1.30997967935, 511.53171782990],
        [0.716, 3.74192651696, 3053.71237534660],
        [0.727, 0.39191881243, 1493.09366806600],
        [0.717, 2.68899513085, 228.27694896500],
        [0.739, 2.12749199443, 1176.70179840940],
        [0.805, 0.07187193910, 556.51766803760],
        [0.835, 3.48287855700, 84.93352695390],
        [0.790, 0.48073040004, 4017.11507831800],
        [0.725, 1.96643065215, 220.46082654860],
        [0.683, 2.68825142163, 151.04766984290],
        [0.739, 3.33688408107, 953.10776223290],
        [0.745, 6.22304530635, 1269.49963188950],
    ],
    # R3
    [
        [20315.005, 3.02186626038, 213.29909543800],
        [8923.581, 3.19144205755, 220.41264243880],
        [6908.677, 4.35174889353, 206.18554843720],
        [4087.129, 4.22406927376, 7.11354700080],
        [3879.041, 2.01056445995, 426.59819087600],
        [1070.788, 4.20360341236, 199.07200143640],
        [907.332, 2.28344368029, 433.71173787680],
        [606.121, 3.17458570534, 227.52618943960],
        [596.639, 4.13455753351, 14.22709400160],
        [483.181, 1.17345973258, 639.89728631400],
        [393.174, 0.00000000000, 0.00000000000],
        [229.472, 4.69838526383, 419.48464387520],
        [188.250, 4.59003889007, 110.20632121940],
        [149.508, 3.20199444400, 103.09277421860],
        [121.442, 3.76831374104, 323.50541665740],
        [101.215, 5.81884137755, 412.37109687440],
        [102.146, 4.70974422803, 95.97922721780],
        [93.078, 1.43531270909, 647.01083331480],
        [72.601, 4.15395598507, 117.31986822020],
        [84.347, 2.63462379693, 216.48048917570],
        [62.198, 2.31239345505, 440.82528487760],
        [45.145, 4.37317047297, 191.95845443560],
        [49.536, 2.38854232908, 209.36694217490],
        [54.829, 0.30526468471, 853.19638175200],
        [40.498, 1.83836569765, 302.16477565500],
        [38.089, 5.94455115525, 88.86568021700],
        [32.243, 4.01146349387, 21.34064100240],
        [40.671, 0.68845183210, 522.57741809380],
        [28.209, 5.77193013961, 210.11770170030],
        [24.976, 3.06249709014, 234.63973644040],
        [20.824, 4.92570695678, 625.67019231240],
        [25.070, 0.73137425284, 515.46387109300],
        [17.485, 5.73135068691, 728.76296653100],
        [18.009, 1.45593152612, 309.27832265580],
        [16.927, 3.52771580455, 3.18139373770],
        [13.437, 3.36479898106, 330.61896365820],
        [11.090, 3.37212682914, 224.34479570190],
        [11.082, 3.41719974793, 956.28915597060],
        [9.978, 1.58791582772, 202.25339517410],
        [11.551, 5.99093726182, 735.87651353180],
        [10.500, 6.06911092266, 405.25754987360],
        [9.144, 2.93557421591, 124.43341522100],
        [8.737, 4.65432480769, 632.78373931320],
        [10.023, 0.58247011625, 860.30992875280],
        [7.482, 4.50669216436, 942.06206196900],
        [10.091, 0.28268774007, 838.96928775040],
        [9.243, 2.57034547708, 223.59403617650],
        [8.652, 1.75808100881, 429.77958461370],
        [7.564, 1.45635107202, 654.12438031560],
        [7.058, 5.47394786065, 1045.15483618760],
        [6.970, 1.51811695028, 422.66603761290],
        [8.067, 4.48457709292, 742.99006053260],
        [6.817, 4.83084424818, 316.39186965660],
        [7.693, 0.43769724671, 831.85574074960],
        [7.934, 4.20112367712, 195.13984817330],
        [6.119, 2.33960392135, 269.92144674060],
        [5.589, 1.14518720694, 284.14854074220],
        [5.564, 4.18123189068, 529.69096509460],
        [5.034, 2.12020038657, 295.05122865420],
        [6.556, 3.42459866876, 10.29494073850],
        [5.544, 2.46823271699, 536.80451209540],
        [6.189, 6.01433827520, 1066.49547719000],
        [5.649, 0.82784598388, 217.23124870110],
        [4.264, 3.23245736673, 1272.68102562720],
        [4.450, 0.92477808590, 203.00415469950],
        [3.268, 4.32777516976, 1258.45393162560],
        [3.655, 0.05832123987, 81.75213321620],
        [3.951, 0.11124996745, 1155.36115740700],
        [3.773, 6.01157059552, 1052.26838318840],
        [2.915, 5.64342950039, 3.93215326310],
        [3.019, 2.19411778004, 447.93883187840],
        [2.977, 1.89387342550, 149.56319713460],
        [3.146, 0.19215180096, 1148.24761040620],
        [2.763, 0.92363342001, 508.35032409220],
        [2.790, 4.97199778427, 1677.93857550080],
        [2.608, 2.99591016813, 1589.07289528380],
        [2.881, 5.40535671721, 1361.54670584420],
        [2.337, 1.30362271569, 184.84490743480],
        [2.536, 3.71412120849, 408.43894361130],
        [2.450, 3.22118361135, 319.57326339430],
        [2.585, 2.31346415454, 543.91805909620],
        [2.324, 5.87500715503, 721.64941953020],
        [1.990, 0.51565577383, 416.30325013750],
        [2.490, 4.24017800021, 1059.38193018920],
        [1.935, 2.41463084855, 337.73251065900],
        [1.886, 0.53809070779, 635.96513305090],
        [1.893, 5.62352727352, 11.04570026390],
        [2.389, 5.73399981234, 313.21047591890],
        [1.900, 2.41000566465, 131.54696222180],
        [1.743, 4.57646237847, 1994.33044515740],
        [1.913, 5.17436386408, 2854.64037391020],
        [1.946, 6.23355845623, 1471.75302706360],
        [1.963, 6.17814558628, 1464.63948006280],
        [1.838, 5.59464577559, 1038.04128918680],
        [1.541, 0.60765337379, 210.85141488320],
        [1.617, 1.75479346067, 195.89060769870],
        [1.577, 0.55789908488, 2324.94940881560],
        [1.492, 0.26624235633, 497.44763618020],
        [1.659, 2.57526072926, 2090.30967237520],
        [1.809, 1.82317819973, 436.89313161450],
        [1.566, 6.15328100324, 490.33408917940],
        [1.771, 6.11741716855, 1073.60902419080],
        [1.456, 0.85374460914, 415.55249061210],
        [1.645, 2.95335775161, 437.64389113990],
        [1.391, 4.12025028560, 1574.84580128220],
        [1.585, 5.96841377266, 1781.03134971940],
        [1.507, 3.84895122542, 1251.34038462480],
        [1.442, 5.32547705924, 2538.24850425360],
        [1.805, 1.50973093681, 750.10360753340],
        [1.462, 3.28599831588, 1884.12412393800],
        [1.482, 0.99340744053, 643.07868005170],
        [1.312, 3.79347668996, 1567.73225428140],
        [1.665, 0.02551523913, 423.41679713830],
        [1.469, 5.35285153471, 1354.43315884340],
        [1.352, 0.69945139243, 867.42347575360],
        [1.124, 1.79624810407, 618.55664531160],
        [1.126, 4.70052329245, 113.38771495710],
        [1.122, 3.95537224270, 1891.23767093880],
        [1.458, 1.50198846753, 430.53034413910],
        [1.145, 5.13093399117, 25.27279426550],
        [1.178, 2.97062300389, 241.75328344120],
        [1.274, 2.29089799814, 2420.92863603340],
        [1.071, 0.04888943982, 63.73589830340],
        [1.377, 5.58271514873, 1382.88734684660],
        [1.145, 3.10797488346, 2200.51599359460],
        [1.076, 0.79465514815, 127.47179660680],
        [1.046, 5.85060227045, 215.74677599280],
        [1.082, 3.72589445510, 131.40394986990],
        [1.218, 0.47504349592, 824.74219374880],
        [1.116, 3.78039049056, 1375.77379984580],
        [0.969, 5.90752273481, 265.98929347750],
        [1.230, 1.41325962069, 2634.22773147140],
        [1.070, 4.80334493874, 1987.21689815660],
        [0.946, 6.25968535931, 2015.67108615980],
        [1.030, 1.08973644893, 362.86229257260],
        [1.072, 5.41838042079, 1279.79457262800],
        [0.880, 1.92224908504, 483.22054217860],
        [0.878, 2.96591300878, 934.94851496820],
        [0.879, 2.65659265685, 145.63104387150],
        [0.872, 6.26261969664, 2.44768055480],
        [1.082, 4.48298283322, 2214.74308759620],
        [0.959, 0.74479087918, 16.67477455640],
        [1.035, 4.05664979327, 231.45834270270],
        [0.851, 0.09360495322, 628.85158605010],
        [0.888, 5.98816755324, 2524.02141025200],
        [0.866, 3.16259265630, 2207.62954059540],
        [0.843, 1.23731248821, 74.78159856730],
        [0.809, 2.89742868175, 2008.55753915900],
        [0.779, 2.28434811609, 1478.86657406440],
        [0.990, 5.32604038017, 2428.04218303420],
        [0.795, 2.38178135810, 2228.97018159780],
        [0.765, 4.70033674940, 1670.82502850000],
        [1.024, 4.23352869513, 1802.37199072180],
        [0.831, 5.87457134912, 1368.66025284500],
        [0.717, 5.92144324994, 1685.05212250160],
        [0.772, 1.15596098579, 3053.71237534660],
        [0.691, 3.13193109668, 56.62235130260],
    ],
    # R4
    [
        [1202.050, 1.41499446465, 220.41264243880],
        [707.796, 1.16153570102, 213.29909543800],
        [516.121, 6.23973568330, 206.18554843720],
        [426.664, 2.46924890293, 7.11354700080],
        [267.736, 0.18659206741, 426.59819087600],
        [170.171, 5.95926972384, 199.07200143640],
        [145.113, 1.44211060143, 227.52618943960],
        [150.339, 0.47970167140, 433.71173787680],
        [121.033, 2.40527320817, 14.22709400160],
        [47.332, 5.56857488676, 639.89728631400],
        [15.745, 2.90112466278, 110.20632121940],
        [16.668, 0.52920774279, 440.82528487760],
        [18.954, 5.85626429118, 647.01083331480],
        [14.074, 1.30343550656, 412.37109687440],
        [12.708, 2.09349305926, 323.50541665740],
        [14.724, 0.29905316786, 419.48464387520],
        [11.133, 2.46304825990, 117.31986822020],
        [11.320, 0.21785507019, 95.97922721780],
        [9.233, 2.28127318068, 21.34064100240],
        [9.246, 1.56496312830, 88.86568021700],
        [8.970, 0.68301278041, 216.48048917570],
        [7.674, 3.59367715368, 302.16477565500],
        [7.823, 4.48688804175, 853.19638175200],
        [8.360, 1.27239488455, 234.63973644040],
        [9.552, 3.14159265359, 0.00000000000],
        [4.834, 2.58836294602, 515.46387109300],
        [6.059, 5.16774448740, 103.09277421860],
        [4.410, 0.02211643085, 191.95845443560],
        [4.364, 1.59622746023, 330.61896365820],
        [3.676, 3.29899839673, 210.11770170030],
        [4.364, 5.97349927933, 654.12438031560],
        [4.447, 4.97415112184, 860.30992875280],
        [3.220, 2.72684237392, 522.57741809380],
        [4.005, 1.59858435636, 405.25754987360],
        [3.099, 0.75235436533, 209.36694217490],
        [2.464, 1.19167306488, 124.43341522100],
        [3.088, 1.32258934286, 728.76296653100],
        [2.220, 3.28087994088, 203.00415469950],
        [2.127, 6.14648095022, 429.77958461370],
        [2.110, 0.75462855247, 295.05122865420],
        [2.020, 3.89394929749, 1066.49547719000],
        [2.248, 0.49319150178, 447.93883187840],
        [2.180, 0.72761059998, 625.67019231240],
        [1.809, 0.09057839517, 942.06206196900],
        [1.672, 1.39635398184, 224.34479570190],
        [1.641, 3.02468307550, 184.84490743480],
        [1.772, 0.81879250825, 223.59403617650],
        [1.902, 2.00472814984, 831.85574074960],
        [1.600, 5.41185167676, 824.74219374880],
        [1.505, 5.95520747253, 422.66603761290],
        [1.133, 1.11512973946, 838.96928775040],
        [1.190, 1.89600567803, 956.28915597060],
        [1.487, 2.11906469507, 529.69096509460],
        [1.409, 0.72254420236, 536.80451209540],
        [1.125, 0.89062692183, 721.64941953020],
        [1.301, 1.64867038984, 17.40848773930],
        [1.164, 5.96957981840, 195.13984817330],
        [0.950, 5.36080713290, 316.39186965660],
        [0.985, 3.05768671768, 1574.84580128220],
        [1.050, 1.59202481523, 735.87651353180],
        [0.817, 4.92838813598, 56.62235130260],
        [0.780, 2.72125404102, 508.35032409220],
        [0.969, 1.00708261792, 1045.15483618760],
        [0.716, 1.11042181341, 1169.58825140860],
    ],
    # R5
    [
        [128.612, 5.91282565136, 220.41264243880],
        [32.273, 0.69256228602, 7.11354700080],
        [26.698, 5.91428528629, 227.52618943960],
        [19.923, 0.67370653385, 14.22709400160],
        [20.223, 4.95136801768, 433.71173787680],
        [13.537, 1.45669521408, 199.07200143640],
        [14.097, 2.67074280191, 206.18554843720],
        [13.364, 4.58826996370, 426.59819087600],
        [7.257, 4.62966127155, 213.29909543800],
        [4.876, 3.61448275002, 639.89728631400],
        [3.136, 4.65661021909, 191.95845443560],
        [2.917, 0.48665273315, 323.50541665740],
        [3.759, 4.89624165044, 440.82528487760],
        [3.303, 4.07190859545, 647.01083331480],
        [2.883, 3.18003019204, 419.48464387520],
        [2.338, 3.69553554327, 88.86568021700],
        [1.950, 5.32729247780, 302.16477565500],
        [2.052, 3.31663577368, 95.97922721780],
        [1.591, 2.67009215574, 853.19638175200],
        [2.028, 0.56025552769, 117.31986822020],
        [1.560, 0.85608042681, 515.46387109300],
        [1.678, 0.00000000000, 0.00000000000],
        [1.102, 5.98011943842, 3.18139373770],
        [1.285, 5.82563377753, 234.63973644040],
        [0.896, 5.22791858719, 216.48048917570],
        [1.141, 0.15741228205, 412.37109687440],
        [0.798, 0.37452846153, 28.45418800320],
        [0.837, 5.04769794123, 124.43341522100],
    ],
]
"""This table contains Saturn's periodic terms (all of them) from the planetary
theory VSOP87 for the radius vector at the equinox of date (taken from the 'D'
solution). In Meeus' book a shortened version can be found in pages 442-445."""


ORBITAL_ELEM = [
    [50.077444, 1223.5110686, 0.00051908, -0.00000003],         # L
    [9.554909192, -0.000002139, 0.000000004, 0.0],              # a
    [0.05554814, -0.000346641, -0.0000006436, 0.0000000034],    # e
    [2.488879, -0.0037362, -0.00001519, 0.000000087],           # i
    [113.665503, 0.877088, -0.00012176, -0.000002249],          # Omega
    [93.057237, 1.9637613, 0.00083753, 0.000004928]             # pie
]
"""This table contains the parameters to compute Saturn's orbital elements for
the mean equinox of date. Based in Table 31.A, page 213"""


ORBITAL_ELEM_J2000 = [
    [50.077444, 1222.1138488, 0.00021004, -0.000000046],    # L
    [2.488879, 0.0025514, -0.00004906, 0.000000017],        # i
    [113.665503, -0.2566722, -0.00018399, 0.00000048],      # Omega
    [93.057237, 0.5665415, 0.0005285, 0.000004912]          # pie
]
"""This table contains the parameters to compute Saturn's orbital elements for
the standard equinox J2000.0. Based on Table 31.B, page 215"""


class Saturn(object):
    """
    Class Saturn models that planet.
    """

    @staticmethod
    def geometric_heliocentric_position(epoch, tofk5=True):
        """This method computes the geometric heliocentric position of planet
        Saturn for a given epoch, using the VSOP87 theory.

        :param epoch: Epoch to compute Saturn position, as an Epoch object
        :type epoch: :py:class:`Epoch`
        :param tofk5: Whether or not the small correction to convert to the FK5
            system will be applied or not
        :type tofk5: bool

        :returns: A tuple with the heliocentric longitude and latitude (as
            :py:class:`Angle` objects), and the radius vector (as a float,
            in astronomical units), in that order
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2018, 10, 27.0)
        >>> l, b, r = Saturn.geometric_heliocentric_position(epoch)
        >>> print(round(l.to_positive(), 4))
        279.5108
        >>> print(round(b, 4))
        0.6141
        >>> print(round(r, 5))
        10.06266
        """

        return geometric_vsop_pos(epoch, VSOP87_L, VSOP87_B, VSOP87_R, tofk5)

    @staticmethod
    def apparent_heliocentric_position(epoch):
        """This method computes the apparent heliocentric position of planet
        Saturn for a given epoch, using the VSOP87 theory.

        :param epoch: Epoch to compute Saturn position, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple with the heliocentric longitude and latitude (as
            :py:class:`Angle` objects), and the radius vector (as a float,
            in astronomical units), in that order
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.
        """

        return apparent_vsop_pos(epoch, VSOP87_L, VSOP87_B, VSOP87_R)

    @staticmethod
    def orbital_elements_mean_equinox(epoch):
        """This method computes the orbital elements of Saturn for the mean
        equinox of the date for a given epoch.

        :param epoch: Epoch to compute orbital elements, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the following six orbital elements:
            - Mean longitude of the planet (Angle)
            - Semimajor axis of the orbit (float, astronomical units)
            - eccentricity of the orbit (float)
            - inclination on the plane of the ecliptic (Angle)
            - longitude of the ascending node (Angle)
            - argument of the perihelion (Angle)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2065, 6, 24.0)
        >>> l, a, e, i, ome, arg = Saturn.orbital_elements_mean_equinox(epoch)
        >>> print(round(l, 6))
        131.196871
        >>> print(round(a, 8))
        9.55490779
        >>> print(round(e, 7))
        0.0553209
        >>> print(round(i, 6))
        2.486426
        >>> print(round(ome, 5))
        114.23974
        >>> print(round(arg, 6))
        -19.896331
        """

        return orbital_elements(epoch, ORBITAL_ELEM, ORBITAL_ELEM)

    @staticmethod
    def orbital_elements_j2000(epoch):
        """This method computes the orbital elements of Saturn for the
        standard equinox J2000.0 for a given epoch.

        :param epoch: Epoch to compute orbital elements, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the following six orbital elements:
            - Mean longitude of the planet (Angle)
            - Semimajor axis of the orbit (float, astronomical units)
            - eccentricity of the orbit (float)
            - inclination on the plane of the ecliptic (Angle)
            - longitude of the ascending node (Angle)
            - argument of the perihelion (Angle)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2065, 6, 24.0)
        >>> l, a, e, i, ome, arg = Saturn.orbital_elements_j2000(epoch)
        >>> print(round(l, 6))
        130.28188
        >>> print(round(a, 8))
        9.55490779
        >>> print(round(e, 7))
        0.0553209
        >>> print(round(i, 6))
        2.490529
        >>> print(round(ome, 5))
        113.49736
        >>> print(round(arg, 6))
        -20.068943
        """

        return orbital_elements(epoch, ORBITAL_ELEM, ORBITAL_ELEM_J2000)

    @staticmethod
    def geocentric_position(epoch):
        """This method computes the geocentric position of Saturn (right
        ascension and declination) for the given epoch, as well as the
        elongation angle.

        :param epoch: Epoch to compute geocentric position, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the right ascension, the declination and
            the elongation angle as Angle objects
        :rtype: tuple
        :raises: TypeError if input value is of wrong type.

        >>> epoch = Epoch(1992, 12, 20.0)
        >>> ra, dec, elon = Saturn.geocentric_position(epoch)
        >>> print(ra.ra_str(n_dec=1))
        21h 11' 41.8''
        >>> print(dec.dms_str(n_dec=1))
        -17d 15' 40.8''
        >>> print(elon.dms_str(n_dec=1))
        46d 51' 47.7''
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Compute the heliocentric position of Saturn
        l, b, r = Saturn.geometric_heliocentric_position(epoch, tofk5=False)
        # Compute the heliocentric position of the Earth
        l0, b0, r0 = Earth.geometric_heliocentric_position(epoch, tofk5=False)
        # Convert to radians
        lr = l.rad()
        br = b.rad()
        l0r = l0.rad()
        b0r = b0.rad()
        # Compute first iteration
        x = r * cos(br) * cos(lr) - r0 * cos(b0r) * cos(l0r)
        y = r * cos(br) * sin(lr) - r0 * cos(b0r) * sin(l0r)
        z = r * sin(br) - r0 * sin(b0r)
        delta = sqrt(x * x + y * y + z * z)
        tau = 0.0057755183 * delta
        # Adjust the epoch for light-time
        epoch -= tau
        # Compute again Saturn coordinates with this correction
        l, b, r = Saturn.geometric_heliocentric_position(epoch, tofk5=False)
        # Compute second iteration
        lr = l.rad()
        br = b.rad()
        x = r * cos(br) * cos(lr) - r0 * cos(b0r) * cos(l0r)
        y = r * cos(br) * sin(lr) - r0 * cos(b0r) * sin(l0r)
        z = r * sin(br) - r0 * sin(b0r)
        # Compute longitude and latitude
        lamb = atan2(y, x)
        beta = atan2(z, sqrt(x * x + y * y))
        # Now, let's compute the aberration effect
        t = (epoch - JDE2000) / 36525
        e = 0.016708634 + t * (-0.000042037 - t * 0.0000001267)
        pie = 102.93735 + t * (1.71946 + t * 0.00046)
        pie = radians(pie)
        lon = l0 + 180.0
        lon = lon.rad()
        k = 20.49552    # The constant of aberration
        deltal1 = k * (-cos(lon - lamb) + e * cos(pie - lamb)) / cos(beta)
        deltab1 = -k * sin(beta) * (sin(lon - lamb) - e * sin(pie - lamb))
        deltal1 = Angle(0, 0, deltal1)
        deltab1 = Angle(0, 0, deltab1)
        # Correction to FK5 system
        lamb = Angle(lamb, radians=True)
        lamb = lamb.to_positive()
        beta = Angle(beta, radians=True)
        l_prime = lamb - t * (1.397 + t * 0.00031)
        deltal2 = Angle(0, 0, -0.09033)
        a = 0.03916 * (cos(l_prime.rad()) + sin(l_prime.rad()))
        a = a * tan(b.rad())
        deltal2 += Angle(0, 0, a)
        deltab2 = 0.03916 * (cos(l_prime.rad()) - sin(l_prime.rad()))
        deltab2 = Angle(0, 0, deltab2)
        # Apply the corrections
        lamb = lamb + deltal1 + deltal2
        beta = beta + deltab1 + deltab2
        # Correction for nutation
        dpsi = nutation_longitude(epoch)
        lamb += dpsi
        e = true_obliquity(epoch)
        ra, dec = ecliptical2equatorial(lamb, beta, e)
        # Let's compute the elongation angle
        lons, lats, rs = Sun.apparent_geocentric_position(epoch)
        lambr = lamb.rad()
        lsr = lons.rad()
        betar = beta.rad()
        elon = acos(cos(betar) * cos(lambr - lsr))
        elon = Angle(elon, radians=True)
        return ra, dec, elon

    @staticmethod
    def conjunction(epoch):
        """This method computes the time of the conjunction closest to the
        given epoch.

        :param epoch: Epoch close to the desired conjunction
        :type epoch: :py:class:`Epoch`

        :returns: The time when the conjunction happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(2125, 6, 1.0)
        >>> conj = Saturn.conjunction(epoch)
        >>> y, m, d = conj.get_date()
        >>> print(y)
        2125
        >>> print(m)
        8
        >>> print(round(d, 4))
        26.4035
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Saturn's conjunction
        a = 2451681.124
        b = 378.091904
        m0 = 131.6934
        m1 = 12.647487
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        # Compute auxiliary angles
        aa = 82.74 + 40.76 * t
        bb = 29.86 + 1181.36 * t
        cc = 14.13 + 590.68 * t
        dd = 220.02 + 1262.87 * t
        # Convert to radians
        aa = Angle(aa).rad()
        bb = Angle(bb).rad()
        cc = Angle(cc).rad()
        dd = Angle(dd).rad()
        corr = (0.0172 + t * (-0.0006 + t * 0.00023) +
                sin(m) * (-8.5885 + t * (0.0411 + t * 0.0002)) +
                cos(m) * (-1.147 + t * (0.0352 - t * 0.00011)) +
                sin(2.0 * m) * (0.3331 + t * (-0.0034 - t * 0.00001)) +
                cos(2.0 * m) * (0.1145 + t * (-0.0045 + t * 0.00002)) +
                sin(3.0 * m) * (-0.0169 + t * 0.0002) +
                cos(3.0 * m) * (-0.0109 + t * 0.0004) +
                sin(aa) * (0.0 + t * (-0.0337 + t * 0.00018)) +
                cos(aa) * (-0.851 + t * (0.0044 + t * 0.00068)) +
                sin(bb) * (0.0 + t * (-0.0064 + t * 0.00004)) +
                cos(bb) * (0.2397 + t * (-0.0012 - t * 0.00008)) +
                sin(cc) * (0.0 - t * 0.001) +
                cos(cc) * (0.1245 + t * 0.0006) +
                sin(dd) * (0.0 + t * (0.0024 - t * 0.00003)) +
                cos(dd) * (0.0477 + t * (-0.0005 - t * 0.00006)))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def opposition(epoch):
        """This method computes the time of the opposition closest to the given
        epoch.

        :param epoch: Epoch close to the desired opposition
        :type epoch: :py:class:`Epoch`

        :returns: The time when the opposition happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(-6, 9, 1.0)
        >>> oppo = Saturn.opposition(epoch)
        >>> y, m, d = oppo.get_date()
        >>> print(y)
        -6
        >>> print(m)
        9
        >>> print(round(d, 4))
        14.3709
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Saturn's opposition
        a = 2451870.17
        b = 378.091904
        m0 = 318.0172
        m1 = 12.647487
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        # Compute an auxiliary angle
        aa = 82.74 + 40.76 * t
        bb = 29.86 + 1181.36 * t
        cc = 14.13 + 590.68 * t
        dd = 220.02 + 1262.87 * t
        # Convert to radians
        aa = Angle(aa).rad()
        bb = Angle(bb).rad()
        cc = Angle(cc).rad()
        dd = Angle(dd).rad()
        corr = (-0.0209 + t * (0.0006 + t * 0.00023) +
                sin(m) * (4.5795 + t * (-0.0312 - t * 0.00017)) +
                cos(m) * (1.1462 + t * (-0.0351 + t * 0.00011)) +
                sin(2.0 * m) * (0.0985 - t * 0.0015) +
                cos(2.0 * m) * (0.0733 + t * (-0.0031 + t * 0.00001)) +
                sin(3.0 * m) * (0.0025 - t * 0.0001) +
                cos(3.0 * m) * (0.005 - t * 0.0002) +
                sin(aa) * (0.0 + t * (-0.0337 + t * 0.00018)) +
                cos(aa) * (-0.851 + t * (0.0044 + t * 0.00068)) +
                sin(bb) * (0.0 + t * (-0.0064 + t * 0.00004)) +
                cos(bb) * (0.2397 + t * (-0.0012 - t * 0.00008)) +
                sin(cc) * (0.0 - t * 0.001) +
                cos(cc) * (0.1245 + t * 0.0006) +
                sin(dd) * (0.0 + t * (0.0024 - t * 0.00003)) +
                cos(dd) * (0.0477 + t * (-0.0005 - t * 0.00006)))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def station_longitude_1(epoch):
        """This method computes the time of the 1st station in longitude
        (i.e. when the planet is stationary and begins to move westward -
        retrograde - among the starts) closest to the given epoch.

        :param epoch: Epoch close to the desired opposition
        :type epoch: :py:class:`Epoch`

        :returns: Time when the 1st station in longitude happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(2018, 11, 1.0)
        >>> sta1 = Saturn.station_longitude_1(epoch)
        >>> y, m, d = sta1.get_date()
        >>> print(y)
        2018
        >>> print(m)
        4
        >>> print(round(d, 4))
        17.9433
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Saturn's opposition
        a = 2451870.17
        b = 378.091904
        m0 = 318.0172
        m1 = 12.647487
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        # Compute an auxiliary angle
        aa = 82.74 + 40.76 * t
        bb = 29.86 + 1181.36 * t
        cc = 14.13 + 590.68 * t
        dd = 220.02 + 1262.87 * t
        # Convert to radians
        aa = Angle(aa).rad()
        bb = Angle(bb).rad()
        cc = Angle(cc).rad()
        dd = Angle(dd).rad()
        corr = (-68.884 + t * (0.0009 + t * 0.00023) +
                sin(m) * (5.5452 + t * (-0.0279 - t * 0.0002)) +
                cos(m) * (3.0727 + t * (-0.043 + t * 0.00007)) +
                sin(2.0 * m) * (0.1101 + t * (-0.0006 - t * 0.00001)) +
                cos(2.0 * m) * (0.1654 + t * (-0.0043 + t * 0.00001)) +
                sin(3.0 * m) * (0.001 + t * 0.0001) +
                cos(3.0 * m) * (0.0095 - t * 0.0003) +
                sin(aa) * (0.0 + t * (-0.0337 + t * 0.00018)) +
                cos(aa) * (-0.851 + t * (0.0044 + t * 0.00068)) +
                sin(bb) * (0.0 + t * (-0.0064 + t * 0.00004)) +
                cos(bb) * (0.2397 + t * (-0.0012 - t * 0.00008)) +
                sin(cc) * (0.0 - t * 0.001) +
                cos(cc) * (0.1245 + t * 0.0006) +
                sin(dd) * (0.0 + t * (0.0024 - t * 0.00003)) +
                cos(dd) * (0.0477 + t * (-0.0005 - t * 0.00006)))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def station_longitude_2(epoch):
        """This method computes the time of the 2nd station in longitude
        (i.e. when the planet is stationary and begins to move eastward -
        prograde - among the starts) closest to the given epoch.

        :param epoch: Epoch close to the desired opposition
        :type epoch: :py:class:`Epoch`

        :returns: Time when the 2nd station in longitude happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(2018, 11, 1.0)
        >>> sta2 = Saturn.station_longitude_2(epoch)
        >>> y, m, d = sta2.get_date()
        >>> print(y)
        2018
        >>> print(m)
        9
        >>> print(round(d, 4))
        6.4175
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Saturn's opposition
        a = 2451870.17
        b = 378.091904
        m0 = 318.0172
        m1 = 12.647487
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        # Compute an auxiliary angle
        aa = 82.74 + 40.76 * t
        bb = 29.86 + 1181.36 * t
        cc = 14.13 + 590.68 * t
        dd = 220.02 + 1262.87 * t
        # Convert to radians
        aa = Angle(aa).rad()
        bb = Angle(bb).rad()
        cc = Angle(cc).rad()
        dd = Angle(dd).rad()
        corr = (68.872 + t * (-0.0007 + t * 0.00023) +
                sin(m) * (5.9399 + t * (-0.04 - t * 0.00015)) +
                cos(m) * (-0.7998 + t * (-0.0266 + t * 0.00014)) +
                sin(2.0 * m) * (0.1738 - t * 0.0032) +
                cos(2.0 * m) * (-0.0039 + t * (-0.0024 + t * 0.00001)) +
                sin(3.0 * m) * (0.0073 - t * 0.0002) +
                cos(3.0 * m) * (0.002 - t * 0.0002) +
                sin(aa) * (0.0 + t * (-0.0337 + t * 0.00018)) +
                cos(aa) * (-0.851 + t * (0.0044 + t * 0.00068)) +
                sin(bb) * (0.0 + t * (-0.0064 + t * 0.00004)) +
                cos(bb) * (0.2397 + t * (-0.0012 - t * 0.00008)) +
                sin(cc) * (0.0 - t * 0.001) +
                cos(cc) * (0.1245 + t * 0.0006) +
                sin(dd) * (0.0 + t * (0.0024 - t * 0.00003)) +
                cos(dd) * (0.0477 + t * (-0.0005 - t * 0.00006)))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def perihelion_aphelion(epoch, perihelion=True):
        """This method computes the time of Perihelion (or Aphelion) closer to
        a given epoch.

        :param epoch: Epoch close to the desired Perihelion (or Aphelion)
        :type epoch: :py:class:`Epoch`
        :param peihelion: If True, the epoch of the closest Perihelion is
            computed, if False, the epoch of the closest Aphelion is found.
        :type bool:

        :returns: The epoch of the desired Perihelion (or Aphelion)
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(1944, 1, 1.0)
        >>> e = Saturn.perihelion_aphelion(epoch)
        >>> y, m, d, h, mi, s = e.get_full_date()
        >>> print(y)
        1944
        >>> print(m)
        9
        >>> print(d)
        8
        >>> print(h)
        1
        >>> epoch = Epoch(2047, 1, 1.0)
        >>> e = Saturn.perihelion_aphelion(epoch, perihelion=False)
        >>> y, m, d, h, mi, s = e.get_full_date()
        >>> print(y)
        2047
        >>> print(m)
        7
        >>> print(d)
        15
        >>> print(h)
        0
        """

        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input value")
        # First approximation
        k = 0.03393 * (epoch.year() - 2003.52)
        if perihelion:
            k = round(k)
        else:
            k = round(k + 0.5) - 0.5
        jde = 2452830.12 + k * (10764.21676 - k * 0.000827)
        # Compute the epochs three months before and after
        jde_before = jde - 90.0
        jde_after = jde + 90.0
        # Compute the Sun-Saturn distance for each epoch
        l, b, r_b = Saturn.geometric_heliocentric_position(Epoch(jde_before))
        l, b, r = Saturn.geometric_heliocentric_position(Epoch(jde))
        l, b, r_a = Saturn.geometric_heliocentric_position(Epoch(jde_after))
        # Call an interpolation object
        m = Interpolation([jde_before, jde, jde_after], [r_b, r, r_a])
        sol = m.minmax()
        return Epoch(sol)

    @staticmethod
    def passage_nodes(epoch, ascending=True):
        """This function computes the time of passage by the nodes (ascending
        or descending) of Saturn, nearest to the given epoch.

        :param epoch: Epoch closest to the node passage
        :type epoch: :py:class:`Epoch`
        :param ascending: Whether the time of passage by the ascending (True)
            or descending (False) node will be computed
        :type ascending: bool

        :returns: Tuple containing:
            - Time of passage through the node (:py:class:`Epoch`)
            - Radius vector when passing through the node (in AU, float)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(2019, 1, 1)
        >>> time, r = Saturn.passage_nodes(epoch)
        >>> year, month, day = time.get_date()
        >>> print(year)
        2034
        >>> print(month)
        5
        >>> print(round(day, 1))
        30.2
        >>> print(round(r, 4))
        9.0546
        """

        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input types")
        # Get the orbital parameters
        l, a, e, i, ome, arg = Saturn.orbital_elements_mean_equinox(epoch)
        # Compute the time of passage through perihelion
        t = Saturn.perihelion_aphelion(epoch)
        # Get the time of passage through the node
        time, r = passage_nodes_elliptic(arg, e, a, t, ascending)
        return time, r

    @staticmethod
    def magnitude(sun_dist, earth_dist, delta_u, b):
        """This function computes the approximate magnitude of Saturn.

        :param sun_dist: Distance from Saturn to the Sun, in Astronomical Units
        :type sun_dist: float
        :param earth_dist: Distance from Saturn to Earth, in Astronomical Units
        :type earth_dist: float
        :param delta_u: Difference between the Saturnicentric longitudes of the
            Sun and the Earth, measured in the plane of the ring
        :type delta_u: float, :py:class:`Angle`
        :param b: Saturnicentric latitude of the Earth refered to the plane of
            the ring, positive towards the north
        :type b: float, :py:class:`Angle`

        :returns: Saturn's magnitude
        :rtype: float
        :raises: TypeError if input values are of wrong type.

        >>> sun_dist = 9.867882
        >>> earth_dist = 10.464606
        >>> delta_u = Angle(16.442)
        >>> b = Angle(4.198)
        >>> m = Saturn.magnitude(sun_dist, earth_dist, delta_u, b)
        >>> print(m)
        1.9
        """

        # WARNING: According to Example 41.d in page 286 of Meeus book, the
        # result for the example above is 0.9 (instead of 1.9). However, after
        # carefully checking the formula implemented here, I'm sure that the
        # book has an error
        if not (isinstance(sun_dist, float) and isinstance(earth_dist, float)
                and isinstance(delta_u, (float, Angle))
                and isinstance(b, (float, Angle))):
            raise TypeError("Invalid input types")
        delta_u = float(delta_u)
        b = Angle(b).rad()
        m = (-8.68 + 5.0 * log10(sun_dist * earth_dist) + 0.044 * abs(delta_u)
             - 2.6 * sin(abs(b)) + 1.25 * sin(b) * sin(b))
        return round(m, 1)


def main():

    # Let's define a small helper function
    def print_me(msg, val):
        print("{}: {}".format(msg, val))

    # Let's show some uses of Saturn class
    print("\n" + 35 * "*")
    print("*** Use of Saturn class")
    print(35 * "*" + "\n")

    # Let's now compute the heliocentric position for a given epoch
    epoch = Epoch(2018, 10, 27.0)
    lon, lat, r = Saturn.geometric_heliocentric_position(epoch)
    print_me("Geometric Heliocentric Longitude", lon.to_positive())
    print_me("Geometric Heliocentric Latitude", lat)
    print_me("Radius vector", r)

    print("")

    # Compute the geocentric position for 1992/12/20:
    epoch = Epoch(1992, 12, 20.0)
    ra, dec, elon = Saturn.geocentric_position(epoch)
    print_me("Right ascension", ra.ra_str(n_dec=1))
    print_me("Declination", dec.dms_str(n_dec=1))
    print_me("Elongation", elon.dms_str(n_dec=1))

    print("")

    # Print mean orbital elements for Saturn at 2065.6.24
    epoch = Epoch(2065, 6, 24.0)
    l, a, e, i, ome, arg = Saturn.orbital_elements_mean_equinox(epoch)
    print_me("Mean longitude of the planet", round(l, 6))       # 131.196871
    print_me("Semimajor axis of the orbit (UA)", round(a, 8))   # 9.55490779
    print_me("Eccentricity of the orbit", round(e, 7))          # 0.0553209
    print_me("Inclination on plane of the ecliptic", round(i, 6))   # 2.486426
    print_me("Longitude of the ascending node", round(ome, 5))  # 114.23974
    print_me("Argument of the perihelion", round(arg, 6))       # -19.896331

    print("")

    # Compute the time of the conjunction close to 2125/6/1
    epoch = Epoch(2125, 6, 1.0)
    conj = Saturn.conjunction(epoch)
    y, m, d = conj.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Conjunction date", date)

    # Compute the time of the opposition close to -6/9/1
    epoch = Epoch(-6, 9, 1.0)
    oppo = Saturn.opposition(epoch)
    y, m, d = oppo.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Opposition date", date)

    print("")

    # Compute the time of the station in longitude #1 close to 2018/11/1
    epoch = Epoch(2018, 11, 1.0)
    sta1 = Saturn.station_longitude_1(epoch)
    y, m, d = sta1.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Date of station in longitude #1", date)

    # Compute the time of the station in longitude #2 close to 2018/11/1
    epoch = Epoch(2018, 11, 1.0)
    sta2 = Saturn.station_longitude_2(epoch)
    y, m, d = sta2.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Date of station in longitude #2", date)

    print("")

    # Find the epoch of the Perihelion closer to 2000/1/1
    epoch = Epoch(2000, 1, 1.0)
    e = Saturn.perihelion_aphelion(epoch)
    y, m, d, h, mi, s = e.get_full_date()
    peri = str(y) + '/' + str(m) + '/' + str(d) + ' at ' + str(h) + ' hours'
    print_me("The Perihelion closest to 2000/1/1 happened on", peri)

    print("")

    # Compute the time of passage through an ascending node
    epoch = Epoch(2019, 1, 1)
    time, r = Saturn.passage_nodes(epoch)
    y, m, d = time.get_date()
    d = round(d, 1)
    print("Time of passage through ascending node: {}/{}/{}".format(y, m, d))
    # 2034/5/30.2
    print("Radius vector at ascending node: {}".format(round(r, 4)))  # 9.0546


if __name__ == "__main__":

    main()
