/*
 * Copyright (C) 2014 Canonical Ltd.
 *
 * This file is part of unity-chromium-extension
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "service.h"

#include <QDebug>
#include <QSignalSpy>
#include <QTest>

class Handler1: public QObject
{
    Q_OBJECT

public:
    Handler1() {}

public Q_SLOTS:
    QVariantMap sayHello(const QVariantMap &params);
    QVariantMap countArguments(const QVariantMap &params);
};

QVariantMap Handler1::sayHello(const QVariantMap &params)
{
    QVariantMap reply;
    reply.insert("greeting",
                 QString("Hello %1").arg(params.value("name").toString()));
    return reply;
}

QVariantMap Handler1::countArguments(const QVariantMap &params)
{
    QVariantMap reply;
    // We subtract one, because that's the "method" field
    reply.insert("count", params.count() - 1);
    return reply;
}

class Handler2: public QObject
{
    Q_OBJECT

public:
    Handler2() {}

public Q_SLOTS:
    QVariantMap listArguments(const QVariantMap &params);
};

QVariantMap Handler2::listArguments(const QVariantMap &params)
{
    QVariantMap reply;
    QStringList arguments = params.keys();
    arguments.removeAll("method");
    reply.insert("arguments", arguments);
    return reply;
}

class ServiceTest: public QObject
{
    Q_OBJECT

public:
    ServiceTest();

private Q_SLOTS:
    void testNoHandlers();
    void testNonExisting();
    void testHandling();
};

ServiceTest::ServiceTest():
    QObject(0)
{
}

void ServiceTest::testNoHandlers()
{
    UnityWebapps::Service service;
    QSignalSpy messageHandled(&service, SIGNAL(messageHandled(const QVariantMap &)));

    QVariantMap message;
    message.insert("method", QString("sayHello"));
    message.insert("name", QString("Tom"));
    service.handleMessage(message);

    QCOMPARE(messageHandled.count(), 0);
    QVERIFY(service.isIdle());
}

void ServiceTest::testNonExisting()
{
    UnityWebapps::Service service;
    QSignalSpy messageHandled(&service, SIGNAL(messageHandled(const QVariantMap &)));

    Handler1 handler1;
    service.addHandler(&handler1);

    QVariantMap message;
    message.insert("method", QString("nonExistingMethod"));
    service.handleMessage(message);

    QCOMPARE(messageHandled.count(), 0);
    QVERIFY(service.isIdle());

    // Add the second service (it still should fail)
    Handler2 handler2;
    service.addHandler(&handler2);

    service.handleMessage(message);

    QCOMPARE(messageHandled.count(), 0);
    QVERIFY(service.isIdle());
}

void ServiceTest::testHandling()
{
    UnityWebapps::Service service;
    QSignalSpy messageHandled(&service, SIGNAL(messageHandled(const QVariantMap &)));
    QSignalSpy isIdleChanged(&service, SIGNAL(isIdleChanged()));
    QVERIFY(service.isIdle());

    Handler1 handler1;
    service.addHandler(&handler1);

    QVariantMap message;
    message.insert("method", QString("sayHello"));
    message.insert("name", QString("Tom"));
    service.handleMessage(message);

    QCOMPARE(messageHandled.count(), 1);
    QVariantMap reply = messageHandled.at(0).at(0).toMap();
    QCOMPARE(reply.value("greeting").toString(), QString("Hello Tom"));
    messageHandled.clear();
    QCOMPARE(isIdleChanged.count(), 2);
    isIdleChanged.clear();
    QVERIFY(service.isIdle());

    // Make sure it continues to work when we add a second handler
    Handler2 handler2;
    service.addHandler(&handler2);
    service.handleMessage(message);

    QCOMPARE(messageHandled.count(), 1);
    reply = messageHandled.at(0).at(0).toMap();
    QCOMPARE(reply.value("greeting").toString(), QString("Hello Tom"));
    messageHandled.clear();
    QCOMPARE(isIdleChanged.count(), 2);
    isIdleChanged.clear();
    QVERIFY(service.isIdle());

    // Test a method from the second handler
    message.clear();
    message.insert("method", QString("listArguments"));
    message.insert("number", 1);
    message.insert("string", QString("Hello"));
    message.insert("boolean", false);
    service.handleMessage(message);

    QCOMPARE(messageHandled.count(), 1);
    reply = messageHandled.at(0).at(0).toMap();
    QVERIFY(reply.contains("arguments"));
    QStringList expectedArguments;
    expectedArguments << "number" << "string" << "boolean";
    QCOMPARE(reply.value("arguments").toStringList().toSet(),
             expectedArguments.toSet());
    messageHandled.clear();
    QCOMPARE(isIdleChanged.count(), 2);
    isIdleChanged.clear();
    QVERIFY(service.isIdle());
}

QTEST_MAIN(ServiceTest);

#include "tst_service.moc"
