/*--------------------------------------------------------------------*//*:Ignore this sentence.
Copyright (C) 1999, 2001 SIL International. All rights reserved.

Distributable under the terms of either the Common Public License or the
GNU Lesser General Public License, as specified in the LICENSING.txt file.

File: GrcGlyphAttrMatrix.h
Responsibility: Sharon Correll
Last reviewed: Not yet.

Description:
    A matrix to hold the glyph attribute assignments for each individual glyph.
-------------------------------------------------------------------------------*//*:End Ignore*/
#ifdef _MSC_VER
#pragma once
#endif
#ifndef GRC_GAMATRIX_INCLUDED
#define GRC_GAMATRIX_INCLUDED

/*----------------------------------------------------------------------------------------------
Class: GrcAssignment
Description: An expression, the statement number in which that expression was assigned,
	and the current values of relevant directives.
	Note that this class, used by the glyph attribute matrix, is not responsible for
	deleting the expression; however the subclass GdlAssignment, used by the master tables, is.
Hungarian: asgnx
----------------------------------------------------------------------------------------------*/
class GrcAssignment
{
	friend class GrcMasterValueList;
	friend class GrcMasterTable;
	friend class GrcGlyphAttrMatrix;
	friend class GdlAssignment;

public:
	//	Constructors & destructor:
	GrcAssignment()
		:	m_pexp(NULL),
			m_nPointRadius(0),
			m_mPrUnits(0),
			m_fOverride(false),
			m_fShadow(false)
	{
	}

	GrcAssignment(GdlExpression * pexp, int nPR, int mPrUnits, bool f, GrpLineAndFile const& lnf)
		:	m_pexp(pexp),
			m_nPointRadius(nPR),
			m_mPrUnits(mPrUnits),
			m_fOverride(f),
			m_fShadow(false),
			m_lnf(lnf)
	{
	}

	virtual ~GrcAssignment()
	{
		// Not responsible for deleting the expression.
	}

	//	Getters:
	GdlExpression * Expression()	{ return m_pexp; }
	int PointRadius()				{ return m_nPointRadius; }
	int PointRadiusUnits()			{ return m_mPrUnits; }
	bool Override()					{ return m_fOverride; }
	GrpLineAndFile & LineAndFile()	{ return m_lnf; }
	int PreProcessedLine()			{ return m_lnf.PreProcessedLine(); }

	//	Setters:
	virtual void Set(GdlExpression * pexp, int nPR, int mPrUnits, bool fOverride,
		GrpLineAndFile const& lnf)
	{
		m_pexp = pexp;
		m_nPointRadius = nPR;
		m_mPrUnits = mPrUnits;
		m_fOverride = fOverride;
		m_fShadow = false;
		m_lnf = lnf;
	}

protected:
	GdlExpression *	m_pexp;
	int				m_nPointRadius;
	int				m_mPrUnits;
	bool			m_fOverride;
	bool			m_fShadow;	// eg, x/y attributes generated by the compiler to
								// shadow or duplicate gpoint
	GrpLineAndFile	m_lnf;
};

/*----------------------------------------------------------------------------------------------
Class: GrcGlyphAttrMatrix
Description: A matrix to hold the glyph attribute assignments for each individual glyph.
Hungarian: gax
----------------------------------------------------------------------------------------------*/

class GrcGlyphAttrMatrix
{
public:
	GrcGlyphAttrMatrix(size_t cGlyphIDs, size_t cGlyphAttrs, size_t cStyles)
	{
		m_cGlyphAttrs = cGlyphAttrs;
		m_cGlyphIDs = cGlyphIDs;
		m_cStyles = cStyles;
		m_prgasgnx = new GrcAssignment[cGlyphIDs * cGlyphAttrs * cStyles];
	}
	~GrcGlyphAttrMatrix()
	{
		delete[] m_prgasgnx;
	}

	int Index(gid16 wGlyphID, unsigned int nAttrID, unsigned int nStyle)
	{
		Assert(wGlyphID < m_cGlyphIDs);
		Assert(nAttrID < m_cGlyphAttrs);
		Assert(nStyle < m_cStyles);
		return int(wGlyphID * m_cGlyphAttrs * m_cStyles +
			(nAttrID * m_cStyles) +
			nStyle);
	}

	void Get(gid16 wGlyphID, int nAttrID,
		GdlExpression ** ppexp, int * pnPR, int * pmunitPR, bool * pfOverride, bool * pfShadow,
		GrpLineAndFile * plnf)
	{
		Get(wGlyphID, nAttrID, 0, ppexp, pnPR, pmunitPR, pfOverride, pfShadow, plnf);
	}

	void Get(gid16 wGlyphID, int nAttrID, int nStyle,
		GdlExpression ** ppexp, int * pnPR, int * pmunitPR, bool * pfOverride, bool * pfShadow,
		GrpLineAndFile * plnf);

	GdlExpression * GetExpression(gid16 wGlyphID, int nAttrID)
	{
		return GetExpression(wGlyphID, nAttrID, 0);
	}
	GdlExpression * GetExpression(gid16 wGlyphID, int nAttrID, int nStyle);

	void Set(gid16 wGlyphID, int nAttrID,
		GdlExpression * pexp, int nPR, int munitPR, bool fOverride, bool fShadow,
		GrpLineAndFile const& lnf)
	{
		Set(wGlyphID, nAttrID, 0, pexp, nPR, munitPR, fOverride, fShadow, lnf);
	}

	void Set(gid16 wGlyphID, int nAttrID, int nStyle,
		GdlExpression * pexp, int nPR, int munitPR, bool fOverride, bool fShadow,
		GrpLineAndFile const& lnf);

	bool Defined(gid16 wGlyphID, int nAttrID, int nStyle = 0)
	{
		if (m_cGlyphAttrs == 0 || m_cGlyphIDs == 0 || m_cStyles == 0)
			return false;
		GrcAssignment * pasgnx = m_prgasgnx + Index(wGlyphID, nAttrID, nStyle);
		return (pasgnx->Expression() != NULL);
	}

	bool DefinedButMaybeShadow(gid16 wGlyphID, int nAttrID, bool *pfShadow, int nStyle = 0)
	{
		if (m_cGlyphAttrs == 0 || m_cGlyphIDs == 0 || m_cStyles == 0)
			return false;
		GrcAssignment * pasgnx = m_prgasgnx + Index(wGlyphID, nAttrID, nStyle);
		*pfShadow = pasgnx->m_fShadow;
		return (pasgnx->Expression());
	}

	bool GpointDefined(gid16 wGlyphID, int nAttrID = 0, int nStyle = 0);

	void Clear(gid16 wGlyphID, int nAttrID, int nStyle = 0);

	GdlExpression * Expression(gid16 wGlyphID, int nAttrID, int nStyle = 0)
	{
		GrcAssignment * pasgnx = m_prgasgnx + Index(wGlyphID, nAttrID, nStyle);
		return pasgnx->Expression();
	}

protected:	
	GrcAssignment * m_prgasgnx;	// matrix
	size_t m_cGlyphAttrs;
	size_t m_cGlyphIDs;
	size_t m_cStyles;
};

/*----------------------------------------------------------------------------------------------
Class: GrcLigComponentList
Description: Ligature components for each glyph.

	First, note that the first batch of (global) glyph attributes are a set of special
	system attributes indicating whether or not a given component is defined for the glyph.
	
	GrcLigComponentList contains an array indicating which components are defined for each glyph.
	For glyphs that are not ligatures, the array holds NULL. For glyphs that are ligatures,
	it holds in essence vector of the special system attribute IDs.
Hungarian: lclist
----------------------------------------------------------------------------------------------*/
class GrcLigComponentList
{
	struct LigCompMap	// lcmap
	{
		//	Indices of component symbols in m_vpsymDefinedComponents that are defined
		//	for a given glyph.
		std::vector<int> m_vinIDs;
	};
public:
	GrcLigComponentList(size_t cvGlyphIDs);
	~GrcLigComponentList();

	int AddComponentFor(gid16 wGlyphID, Symbol psymComponent, GdlRenderer * prndr);
	bool FindComponentFor(gid16 wGlyphID, int nID);
//	int FindComponentID(Symbol psymComponent);

//	int NumberOfComponents()
//	{
//		return m_vpsymDefinedComponents.Size();
//	}

protected:
//	int AddComponent(Symbol psymComponent);
	
protected:
	//	Global list of all ligature components that have been defined for all glyphs;
	//	the index serves as an internal ID.
	//	OBSOLETE--this is now covered by the fact that the components are listed first
	//	in the global glyph attribute list.
//	std::vector<Symbol> m_vpsymDefinedComponents;

	//	List of defined items for each glyph ID.
	std::vector<LigCompMap *> m_vgplcmap;
};


#endif // GRC_GAMATRIX_INCLUDED
