/* -*- Mode: c++ -*- */
/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#ifndef _GRMC4020SOURCE_H_
#define _GRMC4020SOURCE_H_

extern "C" {
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <assert.h>
}

#include <VrSource.h>
#include <GrMC4020Buffer.h>
#include <mc4020.h>

static const char 	  *GRMC_DEVICE_NAME = "/dev/mc4020_0";
static const unsigned long GRMC_DEFAULT_CONFIG_BITMASK = MCC_CH0_EN | MCC_ALL_5V;
static const double 	   GRMC_DEFAULT_SAMPLE_FREQ = 20e6;

#define	SAMPLES_PER_PAGE   (PAGE_SIZE / sizeof (oType))


// FIXME ought to be configurable
#define	MC4020_BUFFER_SIZE	(16L << 20)	// 16 MB


template<class oType> 
class GrMC4020Source: public VrSource<oType> {

public: 
  virtual const char *name() { return "GrMC4020Source"; }

  virtual float memoryTouched() {
    return 0; //no outputs are cached
  }

  virtual int work2(VrSampleRange output, void *o[]);
  virtual void initOutputBuffer(int n);

  GrMC4020Source(double sample_freq = GRMC_DEFAULT_SAMPLE_FREQ,
		 unsigned long a_bitmask = GRMC_DEFAULT_CONFIG_BITMASK);

protected:
  int 		device_fd;
  unsigned long	config_bitmask;
  unsigned long buffersize_pages;

  // we own num_pages starting at page_index
  
  unsigned long	page_index;	// driver page index of first page we own
  VrSampleIndex	sample_index;	// sample index that corresponds to page_index
  unsigned long	num_pages;	// number of driver pages we own

  unsigned long	npages_to_free ();
  unsigned long index_sub (unsigned long a, unsigned long b);
};

template<class oType> unsigned long 
GrMC4020Source<oType>::index_sub (unsigned long a, unsigned long b)
{
  long s = a - b;

  if (s < 0)
    s += buffersize_pages;

  assert (s >= 0 && (unsigned long) s < buffersize_pages);
  return s;
}

/*!
 * Determine how many pages at the beginning of the region that
 * the driver has allocated to us we no longer need.  The earliest
 * VrSampleIndex in use is given by proc_minRP(). We use this and
 * sample_index (the VrSampleIndex of the beginning of the active portion
 * of the driver buffer) to see if we can return some pages to the driver.
 */
template<class oType> unsigned long 
GrMC4020Source<oType>::npages_to_free ()
{
  VrSampleIndex		minRP = proc_minRP ();

  // round down to page boundary
  minRP &= ~((VrSampleIndex) SAMPLES_PER_PAGE - 1);   

  assert (minRP != (VrSampleIndex) -1);
  assert (minRP >= sample_index);

  return (unsigned long) (minRP - sample_index) / SAMPLES_PER_PAGE;
}

template<class oType> int
GrMC4020Source<oType>::work2(VrSampleRange output, void *ao[])
{
  struct mc4020_status 	status;
  VrSampleIndex		target_index = output.index + output.size;
  unsigned long		npgs;
  int			last_lost = 0;

  sync(output.index);

  // fprintf (stderr, "@");

  while ((sample_index + num_pages * SAMPLES_PER_PAGE) < target_index){
    npgs = npages_to_free ();
    
    // fprintf (stderr, "f: %lu\n", npgs);
    
    status.num = npgs;		// free npgs pages
    status.index = page_index;	// starting here

    // free the pages and get new active region
    if (ioctl(device_fd, GIOCSETGETSTATUS, &status) < 0) {
      perror("GrMC4020Source: failed to get mc4020 status");
      exit(-1);
    }

    // how many pages did the beginning of our buffer advance?
    //  	    		       new_index - old_index
    unsigned long delta = index_sub (status.index, page_index);
    assert (npgs == delta);

    sample_index += delta * SAMPLES_PER_PAGE;

    // fprintf (stderr, "G: %lu\n", index_sub (status.index + status.num, page_index + num_pages));

    // remember new range
    page_index = status.index;
    num_pages = status.num;

    // do something about overruns
    if(status.lost && !last_lost) {
      // fprintf(stderr,"GrMC4020Source: overrun\n");
      fputc ('O', stderr);
    }
    last_lost = status.lost;
  }

  return output.size;
}


template<class oType> void
GrMC4020Source<oType>::initOutputBuffer (int n)
{
  if (n != 0) {
    fprintf(stderr,"GrMC4020Source can only have one output buffer.\n");
    exit(-1);
  }
  outBuffer[0] = new GrMC4020Buffer (this, device_fd,
				     buffersize_pages * PAGE_SIZE);
}

template<class oType>
GrMC4020Source<oType>::GrMC4020Source(double sample_freq, unsigned long bitmask)
  : device_fd(-1), page_index(0), sample_index(0), num_pages(0)
{
  struct mc4020_config	c;
  
  buffersize_pages = MC4020_BUFFER_SIZE / PAGE_SIZE;
  
  if ((device_fd = open(GRMC_DEVICE_NAME, O_RDONLY)) < 0) {
    perror(GRMC_DEVICE_NAME);
    exit(1);
  }

  if ((bitmask & MCC_CLK_MASK) == MCC_CLK_INTERNAL)
    c.scan_rate = (unsigned long) sample_freq;
  else
    c.scan_rate = 2;	// minimum divisor
  
  config_bitmask = (bitmask & ~MCC_ASRC_MASK) | MCC_ASRC_BNC;	// ensure some sanity

  if ((bitmask & (MCC_CH0_EN | MCC_CH1_EN | MCC_CH2_EN | MCC_CH3_EN)) == 0){
    fprintf (stderr, "GrMC4020Source: you must enable at least one channel\n");
    exit (1);
  }

  c.bitmask = config_bitmask;

  if (ioctl (device_fd, GIOCSETCONFIG, &c) < 0){
    perror ("can't set GrMC4020Source configuration (GIOCSETCONFIG)");
    exit (1);
  }

  if (ioctl (device_fd, GIOCSETBUFSIZE, buffersize_pages * PAGE_SIZE) < 0) {
    fprintf (stderr, "buffersize = %ld (%#lx)\n", MC4020_BUFFER_SIZE, MC4020_BUFFER_SIZE);
    perror("GrMC4020Buffer(allocateBuffer): Failed to set buffersize");
    exit(-1);
  }

  if (ioctl (device_fd, GIOCSTART) < 0){
    perror ("GIOCSTART failed");
    exit (1);
  }


  setSamplingFrequency (sample_freq);

  setOutputSize (SAMPLES_PER_PAGE);
}

#endif 
