/*
    $Id: vcd_files.c,v 1.18.2.7 2001/05/10 22:29:43 hvr Exp $

    Copyright (C) 2000 Herbert Valerio Riedel <hvr@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <string.h>
#include <assert.h>
#include <stdlib.h>
#include <math.h>

#include "vcd_files.h"
#include "vcd_files_private.h"
#include "vcd_bytesex.h"
#include "vcd_obj.h"
#include "vcd_logging.h"
#include "vcd_util.h"
#include "vcd_mpeg_stream.h"

static const char _rcsid[] = "$Id: vcd_files.c,v 1.18.2.7 2001/05/10 22:29:43 hvr Exp $";

void
set_entries_vcd (VcdObj *obj, void *buf)
{
  VcdListNode *node = NULL;
  int n = 0;
  EntriesVcd entries_vcd;

  assert(sizeof(EntriesVcd) == 2048);

  assert(_vcd_list_length (obj->mpeg_track_list) <= 509);
  assert(_vcd_list_length (obj->mpeg_track_list) > 0);

  memset(&entries_vcd, 0, sizeof(entries_vcd)); /* paranoia / fixme */

  switch (obj->type)
    {
    case VCD_TYPE_VCD11:
      strncpy(entries_vcd.ID, ENTRIES_ID_VCD, 8);
      entries_vcd.version = ENTRIES_VERSION_VCD11;
      entries_vcd.sys_prof_tag = ENTRIES_SPTAG_VCD11;
      break;

    case VCD_TYPE_VCD2:
      strncpy(entries_vcd.ID, ENTRIES_ID_VCD, 8);
      entries_vcd.version = ENTRIES_VERSION_VCD2;
      entries_vcd.sys_prof_tag = ENTRIES_SPTAG_VCD2;
      break;

    case VCD_TYPE_SVCD:
      strncpy(entries_vcd.ID, ENTRIES_ID_SVCD, 8);
      if (obj->broken_svcd_mode_flag)
        {
          vcd_warn ("broken SVCD mode: setting non-conforming ENTRYSVD id");
          strncpy(entries_vcd.ID, "ENTRYSVD", 8);
        }
      entries_vcd.version = ENTRIES_VERSION_SVCD;
      entries_vcd.sys_prof_tag = ENTRIES_SPTAG_SVCD;
      break;
      
    default:
      assert (0);
      break;
    }

  entries_vcd.tracks = UINT16_TO_BE(_vcd_list_length (obj->mpeg_track_list));

  for (n = 0, node = _vcd_list_begin (obj->mpeg_track_list);
       node != NULL;
       n++, node = _vcd_list_node_next (node))
    {
      mpeg_track_t *track = _vcd_list_node_data (node);
      uint32_t lsect = track->relative_start_extent;

      lsect += obj->iso_size;

      entries_vcd.entry[n].n = to_bcd8(n+2);

      lba_to_msf(lsect + 150, &(entries_vcd.entry[n].msf));
    }

  memcpy(buf, &entries_vcd, sizeof(entries_vcd));
}


static void
_set_bit (uint8_t bitset[], unsigned bitnum)
{
  unsigned _byte = bitnum / 8;
  unsigned _bit  = bitnum % 8;

  bitset[_byte] |= (1 << _bit);
}


uint32_t 
get_psd_size (VcdObj *obj)
{
  uint32_t psd_size;
  
  psd_size = _vcd_list_length (obj->mpeg_track_list)*16; /* 2<<3 */
  psd_size += 8; /* stop descriptor */

  return psd_size;
}

void
set_psd_vcd (VcdObj *obj, void *buf)
{
  int n;
  char psd_buf[ISO_BLOCKSIZE] = { 0, };

  assert (obj->type == VCD_TYPE_SVCD
          || obj->type == VCD_TYPE_VCD2);

  for (n = 0; n < _vcd_list_length (obj->mpeg_track_list); n++)
    {
      const int noi = 1;
      int descriptor_size = sizeof (PsdPlayListDescriptor) + (noi * sizeof (uint16_t));
      PsdPlayListDescriptor *_md = _vcd_malloc (descriptor_size);

      _md->type = PSD_TYPE_PLAY_LIST;
      _md->noi = noi;
      _md->lid = UINT16_TO_BE (n+1);
      _md->prev_ofs = UINT16_TO_BE (n ? (n - 1) << 1 : 0xffff);
      _md->next_ofs = UINT16_TO_BE ((n + 1) << 1);
      _md->return_ofs = UINT16_TO_BE ((get_psd_size (obj) - 8) >> 3);
      _md->ptime = UINT16_TO_BE (0x0000);
      _md->wtime = 0x05;
      _md->atime = 0x00;
      _md->itemid[0] = UINT16_TO_BE (n+2);

      memcpy (psd_buf+(n << 4), _md, descriptor_size);
      free (_md);
    }

  {
    PsdEndOfListDescriptor _sd;
    
    memset (&_sd, 0, sizeof (_sd));
    _sd.type = PSD_TYPE_END_OF_LIST;
    memcpy (psd_buf + (n << 4), &_sd, sizeof (_sd));
  }

  memcpy (buf, psd_buf, sizeof (psd_buf));
}

void
set_lot_vcd(VcdObj *obj, void *buf)
{
  LotVcd *lot_vcd = NULL;
  int n;

  assert (obj->type == VCD_TYPE_SVCD
          || obj->type == VCD_TYPE_VCD2);

  lot_vcd = _vcd_malloc (sizeof (LotVcd));
  memset(lot_vcd, 0xff, sizeof(LotVcd));

  lot_vcd->reserved = 0x0000;

  for(n = 0;n < _vcd_list_length (obj->mpeg_track_list)+1;n++)
    lot_vcd->offset[n] = UINT16_TO_BE(n << 1); /* quick'n'dirty */

  memcpy(buf, lot_vcd, sizeof(LotVcd));
  free(lot_vcd);
}

void
set_info_vcd(VcdObj *obj, void *buf)
{
  InfoVcd info_vcd;
  VcdListNode *node = NULL;
  int n = 0;

  assert(sizeof(InfoVcd) == 2048);
  assert(_vcd_list_length (obj->mpeg_track_list) <= 98);
  
  memset(&info_vcd, 0, sizeof(info_vcd));

  switch (obj->type)
    {
    case VCD_TYPE_VCD11:
      strncpy(info_vcd.ID, INFO_ID_VCD, sizeof(info_vcd.ID));
      info_vcd.version = INFO_VERSION_VCD11;
      info_vcd.sys_prof_tag = INFO_SPTAG_VCD11;
      break;

    case VCD_TYPE_VCD2:
      strncpy(info_vcd.ID, INFO_ID_VCD, sizeof(info_vcd.ID));
      info_vcd.version = INFO_VERSION_VCD2;
      info_vcd.sys_prof_tag = INFO_SPTAG_VCD2;
      break;

    case VCD_TYPE_SVCD:
      strncpy(info_vcd.ID, INFO_ID_SVCD, sizeof(info_vcd.ID));
      info_vcd.version = INFO_VERSION_SVCD;
      info_vcd.sys_prof_tag = INFO_SPTAG_SVCD;
      break;
      
    default:
      assert (0);
      break;
    }
  
  _vcd_strncpy_pad (info_vcd.album_desc, 
                    obj->info_album_id,
                    sizeof(info_vcd.album_desc));

  info_vcd.vol_count = UINT16_TO_BE(obj->info_volume_count);
  info_vcd.vol_id = UINT16_TO_BE(obj->info_volume_number);

  switch (obj->type)
    {
    case VCD_TYPE_VCD2:
    case VCD_TYPE_SVCD:
      /* NTSC/PAL bitset */

      for (n = 0, node = _vcd_list_begin (obj->mpeg_track_list);
           node != NULL;
           n++, node = _vcd_list_node_next (node))
        {
          mpeg_track_t *track = _vcd_list_node_data (node);
          
          if(track->info->norm == MPEG_NORM_PAL 
             || track->info->norm == MPEG_NORM_PAL_S)
            _set_bit(info_vcd.pal_flags, n);
          else if (track->info->vsize == 288 || track->info->vsize == 576)
            {
              vcd_warn ("INFO.{VCD,SVD}: assuming PAL resolution type for track #%d"
                        " -- are we creating a X(S)VCD?", n);
              _set_bit(info_vcd.pal_flags, n);
            }
           
        }
      
      info_vcd.psd_size = UINT32_TO_BE(get_psd_size (obj));
      info_vcd.offset_mult = INFO_OFFSET_MULT;
      info_vcd.lot_entries = 
        UINT16_TO_BE(_vcd_list_length (obj->mpeg_track_list) + 1);
      info_vcd.item_count = UINT16_TO_BE(0x0000); /* no items in /SEGMENT
                                                     supported yet */
      break;

    case VCD_TYPE_VCD11:
      /* keep 0ed */
      break;

    default:
      assert (0);
      break;
    }

  memcpy(buf, &info_vcd, sizeof(info_vcd));
}

void
set_tracks_svd (VcdObj *obj, void *buf)
{
  char tracks_svd[ISO_BLOCKSIZE] = { 0, };
  TracksSVD *tracks_svd1 = (void *) tracks_svd;
  TracksSVD2 *tracks_svd2;
  VcdListNode *node;
  int n;
  
  assert (obj->type == VCD_TYPE_SVCD);
  assert (sizeof (SVDTrackContent) == 1);

  strncpy (tracks_svd1->file_id, TRACKS_SVD_FILE_ID, sizeof (TRACKS_SVD_FILE_ID));
  tracks_svd1->version = TRACKS_SVD_VERSION;

  tracks_svd1->tracks = _vcd_list_length (obj->mpeg_track_list);

  tracks_svd2 = (void *) &(tracks_svd1->playing_time[tracks_svd1->tracks]);

  for (n = 0, node = _vcd_list_begin (obj->mpeg_track_list);
       node != NULL;
       n++, node = _vcd_list_node_next (node))
    {
      mpeg_track_t *track = _vcd_list_node_data (node);
      double playtime = track->info->playing_time;
       
      switch (track->info->norm)
        {
        case MPEG_NORM_PAL:
        case MPEG_NORM_PAL_S:
          tracks_svd2->contents[n].video = 0x07;
          break;

        case MPEG_NORM_NTSC:
        case MPEG_NORM_NTSC_S:
          tracks_svd2->contents[n].video = 0x03;
          break;
          
        default:
          if (track->info->vsize == 240 || track->info->vsize == 480)
            {
              vcd_warn ("TRACKS.SVD: assuming NTSC resolution type for track #%d"
                        " -- are we creating a X(S)VCD?", n);
              tracks_svd2->contents[n].video = 0x03;
            }
          else if (track->info->vsize == 288 || track->info->vsize == 576)
            {
              vcd_warn ("TRACKS.SVD: assuming PAL resolution type for track #%d"
                        " -- are we creating a X(S)VCD?", n);
              tracks_svd2->contents[n].video = 0x07;
            }
          else
            vcd_warn("SVCD/TRACKS.SVCD: No MPEG video for track #%d?", n);
          break;
        }

      tracks_svd2->contents[n].audio = (int) track->info->audio_type;

      /* setting playtime */
      
      {
        double i, f;

        f = modf(playtime, &i);

        if (playtime >= 6000.0)
          {
            vcd_warn ("SVCD/TRACKS.SVD: playing time value (%d seconds) to great,"
                      " clipping to 100 minutes", (int) i);
            i = 5999.0;
            f = 74.0 / 75.0;
          }

        lba_to_msf (i * 75, &(tracks_svd1->playing_time[n]));
        tracks_svd1->playing_time[n].f = to_bcd8 (floor (f * 75.0));
      }
    }  
  
  memcpy (buf, &tracks_svd, sizeof(tracks_svd));
}

static double
_get_cumulative_playing_time (const VcdObj *obj, unsigned up_to_track_no)
{
  double result = 0;
  VcdListNode *node;

  for (node = _vcd_list_begin (obj->mpeg_track_list);
       node != NULL;
       node = _vcd_list_node_next (node))
    {
      mpeg_track_t *track = _vcd_list_node_data (node);

      if (!up_to_track_no)
        break;

      result += track->info->playing_time;
      up_to_track_no--;
    }
  
  if (up_to_track_no)
    vcd_warn ("internal error...");

  return result;
}

static unsigned 
_get_scanpoint_count (const VcdObj *obj)
{
  double total_playing_time;

  total_playing_time = _get_cumulative_playing_time (obj, _vcd_list_length (obj->mpeg_track_list));

  return ceil (total_playing_time * 2.0);
}

uint32_t 
get_search_dat_size (const VcdObj *obj)
{
  return sizeof (SearchDat) 
    + (_get_scanpoint_count (obj) * sizeof (msf_t));
}

static VcdList *
_make_track_scantable (const VcdObj *obj)
{
  VcdList *all_aps = _vcd_list_new ();
  VcdList *scantable = _vcd_list_new ();
  unsigned scanpoints = _get_scanpoint_count (obj);
  unsigned track_no;
  VcdListNode *node;

  for (node = _vcd_list_begin (obj->mpeg_track_list), track_no = 0;
       node != NULL;
       node = _vcd_list_node_next (node), track_no++)
    {
      mpeg_track_t *track = _vcd_list_node_data (node);
      VcdListNode *node = _vcd_list_begin (vcd_mpeg_source_get_info (track->source)->aps_list);
      
      for (; node; node = _vcd_list_node_next (node))
        {
          struct aps_data *_data = _vcd_malloc (sizeof (struct aps_data));
          
          _data->timestamp = ((struct aps_data *)_vcd_list_node_data (node))-> timestamp;
          _data->packet_no = ((struct aps_data *)_vcd_list_node_data (node))-> packet_no;

          _data->timestamp += _get_cumulative_playing_time (obj, track_no);
          _data->packet_no += obj->iso_size + track->relative_start_extent;
          _data->packet_no += obj->pre_data_gap;

          _vcd_list_append (all_aps, _data);
        }
    }
  
  {
    VcdListNode *aps_node = _vcd_list_begin (all_aps);
    VcdListNode *n;
    struct aps_data *_data;
    double aps_time;
    double playing_time;
    int aps_packet;
    double t;

    playing_time = scanpoints;
    playing_time /= 2;

    _data = _vcd_list_node_data (aps_node);
    aps_time = _data->timestamp;
    aps_packet = _data->packet_no;

    for (t = 0; t < playing_time; t += 0.5)
      {
	for(n = _vcd_list_node_next (aps_node); n; n = _vcd_list_node_next (n))
	  {
	    _data = _vcd_list_node_data (n);

	    if (fabs (_data->timestamp - t) < fabs (aps_time - t))
	      {
		aps_node = n;
		aps_time = _data->timestamp;
		aps_packet = _data->packet_no;
	      }
	    else 
	      break;
	  }

        {
          uint32_t *lsect = _vcd_malloc (sizeof (uint32_t));
          
          *lsect = aps_packet;
          _vcd_list_append (scantable, lsect);
        }
        
      }

  }

  _vcd_list_free (all_aps, true);

  assert (scanpoints == _vcd_list_length (scantable));

  return scantable;
}

void
set_search_dat (VcdObj *obj, void *buf)
{
  VcdList *scantable;
  VcdListNode *node;
  SearchDat search_dat;
  unsigned n;

  assert (obj->type == VCD_TYPE_SVCD);
  /* assert (sizeof (SearchDat) == ?) */

  memset (&search_dat, 0, sizeof (search_dat));

  strncpy (search_dat.file_id, SEARCH_FILE_ID, sizeof (SEARCH_FILE_ID));
  
  search_dat.version = SEARCH_VERSION;
  search_dat.scan_points = UINT16_TO_BE (_get_scanpoint_count (obj));
  search_dat.time_interval = SEARCH_TIME_INTERVAL;

  memcpy (buf, &search_dat, sizeof (search_dat));
  
  scantable = _make_track_scantable (obj);

  for (node = _vcd_list_begin (scantable), n = 0;
       node != NULL;
       node = _vcd_list_node_next (node), n++)
    {
      SearchDat *search_dat2 = buf;
      uint32_t sect = *(uint32_t *) _vcd_list_node_data (node);
          
      lba_to_msf(sect + 150, &(search_dat2->points[n]));
    }

  assert (n = _get_scanpoint_count (obj));

  _vcd_list_free (scantable, true);
}

static uint32_t 
_get_scandata_count (const struct vcd_mpeg_source_info *info)
{ 
  return ceil (info->playing_time * 2.0);
}

static uint32_t *
_get_scandata_table (const struct vcd_mpeg_source_info *info)
{
  VcdListNode *n, *aps_node = _vcd_list_begin (info->aps_list);
  struct aps_data *_data;
  double aps_time, t;
  int aps_packet;
  uint32_t *retval;
  unsigned i;
  
  retval = _vcd_malloc (_get_scandata_count (info) * sizeof (uint32_t));

  _data = _vcd_list_node_data (aps_node);
  aps_time = _data->timestamp;
  aps_packet = _data->packet_no;

  for (t = 0, i = 0; t < info->playing_time; t += 0.5, i++)
    {
      for(n = _vcd_list_node_next (aps_node); n; n = _vcd_list_node_next (n))
        {
          _data = _vcd_list_node_data (n);

          if (fabs (_data->timestamp - t) < fabs (aps_time - t))
            {
              aps_node = n;
              aps_time = _data->timestamp;
              aps_packet = _data->packet_no;
            }
          else 
            break;
        }

      /* vcd_debug ("%f %f %d", t, aps_time, aps_packet); */

      assert (i < _get_scandata_count (info));

      retval[i] = aps_packet;
    }

  assert (i = _get_scandata_count (info));

  return retval;
}

uint32_t 
get_scandata_dat_size (const VcdObj *obj)
{
  uint32_t retval = 0;

  /* struct 1 */
  retval += sizeof (ScandataDat1);
  retval += sizeof (msf_t) * _vcd_list_length (obj->mpeg_track_list);

  /* struct 2 */
  assert (sizeof (ScandataDat2) == 0);
  retval += sizeof (ScandataDat2);
  retval += sizeof (uint16_t) * 0;

  /* struct 3 */
  retval += sizeof (ScandataDat3);
  retval += (sizeof (uint8_t) + sizeof (uint16_t)) * _vcd_list_length (obj->mpeg_track_list);

  /* struct 4 */
  assert (sizeof (ScandataDat4) == 0);
  retval += sizeof (ScandataDat4);
  {
    VcdListNode *node;
    for (node = _vcd_list_begin (obj->mpeg_track_list);
         node != NULL;
         node = _vcd_list_node_next (node))
      {
        const mpeg_track_t *track = _vcd_list_node_data (node);
        
        retval += sizeof (msf_t) * _get_scandata_count (track->info);
      }
    
  }

  return retval;
}

void
set_scandata_dat (VcdObj *obj, void *buf)
{
  unsigned tracks = _vcd_list_length (obj->mpeg_track_list);
  VcdListNode *node;
  ScandataDat1 *scandata_dat1 = (ScandataDat1 *) buf;
  ScandataDat2 *scandata_dat2 = (ScandataDat2 *) &(scandata_dat1->cum_playtimes[tracks]);
  ScandataDat3 *scandata_dat3 = (ScandataDat3 *) &(scandata_dat2->spi_indexes[0]);
  ScandataDat4 *scandata_dat4 = (ScandataDat4 *) &(scandata_dat3->mpeg_track_offsets[tracks]);
  unsigned n;
  uint16_t _tmp_offset;

  assert (obj->type == VCD_TYPE_SVCD);

  /* memset (buf, 0, get_scandata_dat_size (obj)); */

  /* struct 1 */
  strncpy (scandata_dat1->file_id, SCANDATA_FILE_ID, sizeof (SCANDATA_FILE_ID));
  
  scandata_dat1->version = 0x01;
  scandata_dat1->reserved = 0x00;
  scandata_dat1->scandata_count = UINT16_TO_BE (_get_scanpoint_count (obj));

  scandata_dat1->track_count = UINT16_TO_BE (tracks);
  scandata_dat1->spi_count = UINT16_TO_BE (0);

  for (n = 0; n < tracks; n++)
    {
      double playtime = _get_cumulative_playing_time (obj, n + 1);
      double i = 0, f = 0;

      f = modf(playtime, &i);

      while (i >= (60 * 100))
        i -= (60 * 100);

      assert (i >= 0);

      lba_to_msf (i * 75, &(scandata_dat1->cum_playtimes[n]));
      scandata_dat1->cum_playtimes[n].f = to_bcd8 (floor (f * 75.0));
    }

  /* struct 2 -- nothing */

  /* struct 3/4 */

  scandata_dat3->mpegtrack_start_index = UINT16_TO_BE (0);

  _tmp_offset = 0;

  for (n = 0, node = _vcd_list_begin (obj->mpeg_track_list);
       node != NULL;
       n++, node = _vcd_list_node_next (node))
    {
      const mpeg_track_t *track = _vcd_list_node_data (node);
      uint32_t *_table;
      int point, scanpoints = _get_scandata_count (track->info);
      

      scandata_dat3->mpeg_track_offsets[n].track_num = n + 2;
      scandata_dat3->mpeg_track_offsets[n].table_offset 
        = UINT16_TO_BE (_tmp_offset * sizeof (msf_t));

      _table = _get_scandata_table (track->info);

      for (point = 0; point < scanpoints; point++)
        {
          uint32_t lsect = _table[point];

          lsect += track->relative_start_extent;
          lsect += obj->iso_size;
          lsect += obj->pre_data_gap;

          /* vcd_debug ("lsect %d %d", point, lsect); */

          lba_to_msf(lsect + 150, &(scandata_dat4->scandata_table[_tmp_offset + point]));
        }

      free (_table);

      _tmp_offset += scanpoints;
    }

  /* struct 4 */

  
}

/* eof */


/* 
 * Local variables:
 *  c-file-style: "gnu"
 *  tab-width: 8
 *  indent-tabs-mode: nil
 * End:
 */
