
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: filelist.c 2664 2007-08-22 17:24:17Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <ctype.h>
#include <crypt.h>
#include <errno.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "filelist.h"
#include "disc.h"
#include "disc_image.h"
#include "environment.h"
#include "i18n.h"
#include "heap.h"
#include "list.h"
#include "mutex.h"
#include "logger.h"
#include "mediamarks.h"
#include "playlist_m3u.h"
#include "playlist_xml.h"
#include "shoutcast.h"
#include "utils.h"
#include "vdr.h"
#include "youtube.h"
#include "tvlinks.h"

extern oxine_t *oxine;

static bool (*pwd_callback) (const char *mrl, const char *pwd) = NULL;

static char *xine_extensions = NULL;

static l_list_t *allowed_xine_extensions;
static l_list_t *allowed_audio_extensions;
static l_list_t *allowed_video_extensions;
static l_list_t *allowed_subtitle_extensions;

static char *audio_extensions[] = {
    "16sv", "16sv", "669", "8svx", "8svx", "aac", "abs", "ac3", "aif",
    "aiff", "amf", "anx", "au", "aud", "axa", "axv", "cak", "cpk", "dat",
    "dps", "dts", "film", "flac", "iff", "ik2", "iki", "it", "m4a", "mdl",
    "med", "mka", "mod", "mp+", "mp2", "mp3", "mpa", "mpc", "mpega", "nsf",
    "ogg", "ra", "ram", "rpm", "s3m", "shn", "snd", "spx", "stm", "str",
    "svx", "tta", "voc", "vox", "wav", "wma", "wve", "xa", "xa1", "xa2",
    "xap", "xas", "xm", NULL
};

static char *video_extensions[] = {
    "4xm", "anim", "anim", "anim3", "anim5", "anim7", "anim8", "anx", "asf",
    "asp", "asx", "avi", "axa", "axv", "bin", "cak", "cin", "cpk", "cue",
    "dat", "dif", "divx", "dv", "film", "flc", "fli", "flv", "iff", "iki",
    "ik2", "img", "iso", "dps", "m2p", "m2t", "m2v", "m4b", "mdf", "mjpg",
    "mkv", "mng", "mov", "mp4", "mpe", "mpeg", "mpg", "mpv", "mv8", "mve",
    "nsv", "nrg", "ogm", "pes", "pva", "qt", "ram", "rm", "rmvb", "roq",
    "rpm", "rv", "spx", "str", "toc", "trp", "ts", "vmd", "vob", "vqa",
    "wax", "wma", "wmv", "wva", "wvx", "xa", "xa1", "xa2", "xap", "xas",
    "y4m", NULL
};

static char *image_extensions[] = {
    "gif", "ham", "ham6", "ham8", "iff", "ilbm", "jpeg", "jpg", "png",
    NULL
};

static char *subtitle_extensions[] = {
    "asc", "smi", "srt", "ssa", "sub", "txt",
    NULL
};

static char *playlist_extensions[] = {
    "m3u", "pls", "oxp",
    NULL
};

static char *title_prefixes[] = {
    "The ", "Der ", "Die ", "Das ", "Les ",
    NULL
};


static bool
is_audio (const char *ext)
{
    int i = 0;
    for (; audio_extensions[i]; i++) {
        if (strcasecmp (audio_extensions[i], ext) == 0) {
            return true;
        }
    }
    return false;
}


static bool
is_video (const char *ext)
{
    int i = 0;
    for (; video_extensions[i]; i++) {
        if (strcasecmp (video_extensions[i], ext) == 0) {
            return true;
        }
    }
    return false;
}


static bool
is_image (const char *ext)
{
    int i = 0;
    for (; image_extensions[i]; i++) {
        if (strcasecmp (image_extensions[i], ext) == 0) {
            return true;
        }
    }
    return false;
}


static bool
is_subtitle (const char *ext)
{
    int i = 0;
    for (; subtitle_extensions[i]; i++) {
        if (strcasecmp (subtitle_extensions[i], ext) == 0) {
            return true;
        }
    }
    return false;
}


void
filelist_extensions_init (void)
{
    allowed_xine_extensions = l_list_new ();
    allowed_audio_extensions = l_list_new ();
    allowed_video_extensions = l_list_new ();
    allowed_subtitle_extensions = l_list_new ();

    char *exts = xine_get_file_extensions (oxine->xine);
    xine_extensions = ho_strdup (exts);

    char *p = xine_extensions;
    while (p) {
        char *q = index (p, ' ');
        if (q) {
            q[0] = '\0';
            q++;
        }

        l_list_append (allowed_xine_extensions, p);

        /* It is possible, that some extensions are valid for more than one
         * media type. Thus we have to test all extensions against all media
         * types (audio, video, image, subtitle). */
        bool found = false;
        if (is_audio (p)) {
            l_list_append (allowed_audio_extensions, p);
            found = true;
        }
        if (is_video (p)) {
            l_list_append (allowed_video_extensions, p);
            found = true;
        }
        if (is_image (p)) {
            /* As xine-lib does not really make it easy to display images, we
             * ignore all images for now. */
            found = true;
        }
        if (is_subtitle (p)) {
            l_list_append (allowed_subtitle_extensions, p);
            found = true;
        }
        if (!found) {
            debug ("Unknown file extension: '%s'.", p);
        }

        p = q;
    }

    free (exts);
}


void
filelist_extensions_free (void)
{
    l_list_free (allowed_xine_extensions, NULL);
    l_list_free (allowed_audio_extensions, NULL);
    l_list_free (allowed_video_extensions, NULL);
    l_list_free (allowed_subtitle_extensions, NULL);
    ho_free (xine_extensions);
}


bool
is_file_allowed (const char *mrl, fileitem_allowed_t allowed)
{
    const char *suffix = get_extension (mrl);

    if (!mrl) {
        return false;
    }

    if (!config_get_bool ("media.files.show_hidden_files")
        && is_file_hidden (mrl)) {
        return false;
    }

    switch (allowed) {
    case ALLOW_FILES_ALL:
        {
            return true;
        }
        break;
    case ALLOW_FILES_MULTIMEDIA:
        if (starts_with (mrl, "cdda:/")) {
            return true;
        }
        else if (starts_with (mrl, "dvd:/")) {
            return true;
        }
        else if (starts_with (mrl, "vcd:/")) {
            return true;
#ifdef HAVE_IMAGE_PLAYBACK
        }
        else if (is_file_image (mrl)) {
            return true;
#endif
        }
        else if (suffix) {
            char *ext = l_list_first (allowed_xine_extensions);
            while (ext) {
                if (strcasecmp (ext, suffix) == 0) {
                    return true;
                }
                ext = l_list_next (allowed_xine_extensions, ext);
            }
        }
        break;
    case ALLOW_FILES_MUSIC:
        if (starts_with (mrl, "cdda:/")) {
            return true;
        }
        else if (suffix) {
            char *ext = l_list_first (allowed_audio_extensions);
            while (ext) {
                if (strcasecmp (ext, suffix) == 0) {
                    return true;
                }
                ext = l_list_next (allowed_audio_extensions, ext);
            }
        }
        break;
    case ALLOW_FILES_VIDEO:
        if (starts_with (mrl, "dvd:/")) {
            return true;
        }
        else if (starts_with (mrl, "vcd:/")) {
            return true;
        }
        else if (suffix) {
            char *ext = l_list_first (allowed_video_extensions);
            while (ext) {
                if (strcasecmp (ext, suffix) == 0) {
                    return true;
                }
                ext = l_list_next (allowed_video_extensions, ext);
            }
        }
        break;
#ifdef HAVE_IMAGE_PLAYBACK
    case ALLOW_FILES_IMAGE:
        if (is_file_image (mrl)) {
            return true;
        }
        break;
#endif
    case ALLOW_FILES_SUBTITLE:
        if (suffix) {
            char *ext = l_list_first (allowed_subtitle_extensions);
            while (ext) {
                if (strcasecmp (ext, suffix) == 0) {
                    return true;
                }
                ext = l_list_next (allowed_subtitle_extensions, ext);
            }
        }
        break;
    default:
        break;
    }

    return false;
}


static fileitem_type_t
fileitem_detect_type (const char *mrl)
{
    fileitem_type_t type = FILE_TYPE_UNKNOWN;

    /* It's a directory. */
    if (is_directory (mrl)) {
        type = FILE_TYPE_DIRECTORY;
    }

    /* The entry is a playlist (M3U). */
    else if (is_playlist_m3u (mrl)) {
        type = FILE_TYPE_PLAYLIST_M3U;
    }

    /* The entry is a playlist (PLS). */
    else if (is_playlist_pls (mrl)) {
        type = FILE_TYPE_PLAYLIST_PLS;
    }

    /* The entry is a playlist (OXP). */
    else if (is_playlist_oxp (mrl)) {
        type = FILE_TYPE_PLAYLIST_OXP;
    }

    /* We assume that it's a regular file. */
    else {
        type = FILE_TYPE_REGULAR;
    }

    return type;
}


/**
 * Callback used to free a fileitem. This is the callback passed to
 * <code>l_list_free</code> and <code>l_list_clear</code> to free the memory
 * of a fileitem.
 */
static void
fileitem_free_cb (void *data)
{
    fileitem_t *fileitem = (fileitem_t *) data;

    assert (fileitem);

    if (fileitem->child_list) {
        fileitem->child_list->parent_item = NULL;
        fileitem->child_list->parent_list = NULL;
    }
    filelist_ref_set (&fileitem->child_list, NULL);

    ho_free (fileitem->title);
    ho_free (fileitem->mrl);
    ho_free (fileitem->thumbnail_mrl);
    ho_free (fileitem->sort_by);
#ifdef HAVE_HAL
    ho_free (fileitem->device);
    ho_free (fileitem->volume_udi);
    ho_free (fileitem->drive_udi);
#endif

    ho_free (fileitem);
}


/**
 * Callback to swap fileitems. This is the callback passed to
 * <code>l_list_sort</code>, to determin, if two fileitems have to be
 * reordered.
 */
static bool
fileitem_swap_cb (void *d1, void *d2)
{
    fileitem_t *f1 = (fileitem_t *) d1;
    fileitem_t *f2 = (fileitem_t *) d2;

    assert (f1);
    assert (f2);

    if ((f2->type == FILE_TYPE_CDDA_VFOLDER)
        && (f1->type != FILE_TYPE_CDDA_VFOLDER))
        return true;
    if ((f1->type == FILE_TYPE_CDDA_VFOLDER)
        && (f2->type != FILE_TYPE_CDDA_VFOLDER))
        return false;

    if (strcmp (f2->mrl, get_file_favorites ()) == 0)
        return true;
    if (strcmp (f1->mrl, get_file_favorites ()) == 0)
        return false;

    if (strcmp (f2->mrl, get_dir_oxine_playlists ()) == 0)
        return true;
    if (strcmp (f1->mrl, get_dir_oxine_playlists ()) == 0)
        return false;

#ifdef HAVE_VDR
    if (f2->type == FILE_TYPE_VDR_RECORDINGS_VFOLDER)
        return true;
    if (f1->type == FILE_TYPE_VDR_RECORDINGS_VFOLDER)
        return false;
    if (f2->type == FILE_TYPE_VDR_TIMERS_VFOLDER)
        return true;
    if (f1->type == FILE_TYPE_VDR_TIMERS_VFOLDER)
        return false;
#endif
#ifdef HAVE_SHOUTCAST
    if (f2->type == FILE_TYPE_SHOUTCAST_VFOLDER)
        return true;
    if (f1->type == FILE_TYPE_SHOUTCAST_VFOLDER)
        return false;
#endif

    if (f2->type == FILE_TYPE_FAVORITES)
        return true;
    if (f1->type == FILE_TYPE_FAVORITES)
        return false;

    if (f2->type == FILE_TYPE_MEDIAMARKS)
        return true;
    if (f1->type == FILE_TYPE_MEDIAMARKS)
        return false;

    if (strcmp (f2->mrl, get_dir_home ()) == 0)
        return true;
    if (strcmp (f1->mrl, get_dir_home ()) == 0)
        return false;

    if (strcmp (f2->mrl, "/") == 0)
        return true;
    if (strcmp (f1->mrl, "/") == 0)
        return false;

    if ((f2->type == FILE_TYPE_DIRECTORY)
        && (f1->type != FILE_TYPE_DIRECTORY))
        return true;
    if ((f1->type == FILE_TYPE_DIRECTORY)
        && (f2->type != FILE_TYPE_DIRECTORY))
        return false;

    if ((f2->type == FILE_TYPE_MOUNTPOINT)
        && (f1->type != FILE_TYPE_MOUNTPOINT))
        return false;
    if ((f1->type == FILE_TYPE_MOUNTPOINT)
        && (f2->type != FILE_TYPE_MOUNTPOINT))
        return true;

    if ((f2->type == FILE_TYPE_PLAYLIST_PLS)
        && (f1->type != FILE_TYPE_PLAYLIST_PLS))
        return true;
    if ((f1->type == FILE_TYPE_PLAYLIST_PLS)
        && (f2->type != FILE_TYPE_PLAYLIST_PLS))
        return false;

    if ((f2->type == FILE_TYPE_PLAYLIST_M3U)
        && (f1->type != FILE_TYPE_PLAYLIST_M3U))
        return true;
    if ((f1->type == FILE_TYPE_PLAYLIST_M3U)
        && (f2->type != FILE_TYPE_PLAYLIST_M3U))
        return false;

#ifdef HAVE_LIBEXIF
    if (is_file_image (f1->mrl) && is_file_image (f2->mrl)
        && config_get_bool ("image.sort.by_date")) {
        if (!f1->sort_by) {
            f1->sort_by = exif_info_get (EXIF_INFO_SORT, f1->mrl);
        }
        if (!f2->sort_by) {
            f2->sort_by = exif_info_get (EXIF_INFO_SORT, f2->mrl);
        }
    }
#endif /* HAVE_LIBEXIF */

    if (!f1->sort_by) {
        f1->sort_by = ho_strdup (f1->title);
    }
    if (!f2->sort_by) {
        f2->sort_by = ho_strdup (f2->title);
    }

    return swap_strings (f1->sort_by, f2->sort_by);
}


filelist_t *
filelist_new (fileitem_t * parent_item,
              const char *title, const char *mrl,
              fileitem_allowed_t allowed_filetypes)
{
    filelist_t *filelist = ho_new (filelist_t);

    filelist->list = l_list_new ();
    filelist->allowed_filetypes = allowed_filetypes;

    if (mrl) {
        filelist->mrl = ho_strdup (mrl);
    }
    else {
        filelist->mrl = ho_strdup (FILELIST_TOPLEVEL_MRL);
    }
    if (title) {
        filelist->title = ho_strdup (title);
    }
    else {
        filelist->title = ho_strdup (FILELIST_TOPLEVEL_TITLE);
    }
    filelist->thumbnail_mrl = NULL;

    filelist->parent_item = parent_item;
    filelist->parent_list = parent_item ? parent_item->parent_list : NULL;

    pthread_mutexattr_init (&filelist->mutex_attr);
    pthread_mutexattr_settype (&filelist->mutex_attr,
                               PTHREAD_MUTEX_RECURSIVE);
    pthread_mutex_init (&filelist->mutex, &filelist->mutex_attr);

    filelist->top_position = 0;
    filelist->cur_position = 0;
    filelist->reference_count = 0;
    filelist->wait_for_password = false;

    return filelist;
}


void
filelist_clear (filelist_t * filelist)
{
    assert (filelist);

    filelist_lock (filelist);

    l_list_clear (filelist->list, fileitem_free_cb);
    filelist->top_position = 0;
    filelist->cur_position = 0;
    ho_free (filelist->error);

    filelist_unlock (filelist);
}


/// Removes all fileitems and frees the list.
static void
filelist_free (filelist_t * filelist)
{
    assert (filelist);

    l_list_free (filelist->list, fileitem_free_cb);
    pthread_mutex_destroy (&filelist->mutex);
    pthread_mutexattr_destroy (&filelist->mutex_attr);

    ho_free (filelist->title);
    ho_free (filelist->mrl);
    ho_free (filelist->thumbnail_mrl);
    ho_free (filelist->error);

    ho_free (filelist);
}


void
filelist_ref_set (filelist_t ** p, filelist_t * d)
{
    assert (p);

    if (*p) {
        if ((*p)->reference_count == 1) {
            filelist_free (*p);
        }
        else {
            (*p)->reference_count--;
        }
    }
    *p = d;
    if (*p) {
        (*p)->reference_count++;
    }
}


void
filelist_sort (filelist_t * filelist, l_swap_cb_t swap_cb)
{
    assert (filelist);

    filelist_lock (filelist);

    if (swap_cb) {
        l_list_sort (filelist->list, swap_cb);
    }
    else {
        l_list_sort (filelist->list, fileitem_swap_cb);
    }

    filelist_unlock (filelist);
}


fileitem_t *
filelist_add (filelist_t * filelist, const char *title, const char *mrl,
              fileitem_type_t type)
{
    assert (filelist);
    assert (title);
    assert (mrl);

    filelist_lock (filelist);

    if (type == FILE_TYPE_UNKNOWN) {
        type = fileitem_detect_type (mrl);
    }

    fileitem_t *fileitem = ho_new (fileitem_t);

    fileitem->title = ho_strdup (title);
    fileitem->mrl = ho_strdup (mrl);
    fileitem->thumbnail_mrl = NULL;
    fileitem->sort_by = NULL;
    fileitem->type = type;
    filelist_ref_set (&fileitem->child_list, NULL);
    fileitem->parent_list = filelist;
#ifdef HAVE_HAL
    fileitem->device = NULL;
    fileitem->volume_udi = NULL;
    fileitem->drive_udi = NULL;
#endif

    l_list_append (filelist->list, fileitem);

    filelist_unlock (filelist);

    return fileitem;
}


/// Replaces underscores with spaces.
static char *
underscore_to_space (char *title)
{
    int i = 0;
    int l = strlen (title);

    for (i = 0; i < l; i++) {
        if (title[i] == '_')
            title[i] = ' ';
    }

    return title;
}


/// Removes any extension we know about from the filename.
static char *
remove_extension (char *title)
{
    int i = 0;
    int l = strlen (title);

    for (i = 0; video_extensions[i]; i++) {
        char *ext = video_extensions[i];
        if (has_extension (title, ext)) {
            title[l - strlen (ext) - 1] = '\0';
            return title;
        }
    }

    for (i = 0; audio_extensions[i]; i++) {
        char *ext = audio_extensions[i];
        if (has_extension (title, ext)) {
            title[l - strlen (ext) - 1] = '\0';
            return title;
        }
    }

    for (i = 0; image_extensions[i]; i++) {
        char *ext = image_extensions[i];
        if (has_extension (title, ext)) {
            title[l - strlen (ext) - 1] = '\0';
            return title;
        }
    }

    for (i = 0; subtitle_extensions[i]; i++) {
        char *ext = subtitle_extensions[i];
        if (has_extension (title, ext)) {
            title[l - strlen (ext) - 1] = '\0';
            return title;
        }
    }

    for (i = 0; playlist_extensions[i]; i++) {
        char *ext = playlist_extensions[i];
        if (has_extension (title, ext)) {
            title[l - strlen (ext) - 1] = '\0';
            return title;
        }
    }

    return title;
}


/**
 * Changes a title. If the title starts with one of the suffixes, this
 * suffix is cut from the beginning of the title and pasted to the end.
 */
static char *
rework_title (const char *title)
{
    int i = 0;

    for (; title_prefixes[i]; i++) {
        char *prefix = title_prefixes[i];
        if (starts_with (title, prefix)) {
            return ho_strdup_printf ("%s, %c%c%c", title + 4,
                                     title[0], title[1], title[2]);
        }
    }

    return ho_strdup (title);
}


static char *
_create_title (const char *mrl, fileitem_type_t type)
{
    char *title = NULL;

    char *filename_bn = get_basename (mrl);
    underscore_to_space (filename_bn);
    char *filename_bnw = rework_title (filename_bn);

    char *filename_sf = ho_strdup (filename_bn);
    remove_extension (filename_sf);

    char *filename_sfw = rework_title (filename_sf);

    switch (type) {
    case FILE_TYPE_DIRECTORY:
    case FILE_TYPE_MOUNTPOINT:
        if (strcmp (mrl, get_dir_home ()) == 0) {
            title = ho_strdup_printf ("[%s]", _("Home"));
        }
        else if (strcmp (mrl, "/") == 0) {
            title = ho_strdup_printf ("[%s]", _("Filesystem"));
        }
        else if (strcmp (mrl, get_dir_oxine_playlists ()) == 0) {
            title = ho_strdup_printf ("[%s]", _("Playlists"));
        }
        else {
            title = ho_strdup_printf ("[%s]", filename_bnw);
        }
        break;
    case FILE_TYPE_PLAYLIST_M3U:
    case FILE_TYPE_PLAYLIST_PLS:
    case FILE_TYPE_PLAYLIST_OXP:
        if (strstr (mrl, get_dir_oxine_playlists ())) {
            title = ho_strdup_printf ("%s", filename_sf);
        }
        else {
            title = ho_strdup_printf ("[%s: %s]", _("Playlist"), filename_sf);
        }
        break;
    default:
        if (starts_with (mrl, "dvd:/")) {
            title = ho_strdup_printf ("%s", _("DVD"));
        }
        else if (starts_with (mrl, "vcd:/")) {
            title = ho_strdup_printf ("%s", _("Video CD"));
        }
        else if (starts_with (mrl, "cdda:/")) {
            title = ho_strdup_printf ("%s", _("Audio CD"));
        }
        else if (strcmp (mrl, get_file_mediamarks_music ()) == 0) {
            title = ho_strdup_printf ("[%s]", _("Music"));
        }
        else if (strcmp (mrl, get_file_mediamarks_video ()) == 0) {
            title = ho_strdup_printf ("[%s]", _("Films"));
        }
        else if (strcmp (mrl, get_file_favorites ()) == 0) {
            title = ho_strdup_printf ("[%s]", _("Favorites"));
        }
        else {
            title = ho_strdup_printf ("%s", filename_sfw);
        }
        break;
    }

    ho_free (filename_bn);
    ho_free (filename_bnw);
    ho_free (filename_sf);
    ho_free (filename_sfw);

    return title;
}


char *
create_title (const char *mrl)
{
    if (!mrl)
        return NULL;

    return _create_title (mrl, fileitem_detect_type (mrl));
}


/**
 * Reads the titles of an audio CD. All titles of the CD are added to the
 * list.
 */
static void
audiocd_read (filelist_t * filelist)
{
    char *mrl = filelist->mrl;
    if (strncmp (mrl, "cdda://", 7) != 0)
        return;

    int num_mrls;
    char **mrls = xine_get_autoplay_mrls (oxine->xine, "CD", &num_mrls);

    if (!mrls || (num_mrls <= 0))
        return;

    int i;
    bool use_cddb = true;
    for (i = 0; i < num_mrls; i++) {
        char *track_mrl = ho_strdup_printf ("%s/%d", mrl, i + 1);
        char *track_title = NULL;

        if (use_cddb) {
            char *title = meta_info_get (XINE_META_INFO_TITLE, track_mrl);

            if (title) {
                track_title = ho_strdup_printf ("%02d - %s", i + 1, title);
                ho_free (title);
            }

            /* If retrieving information from CDDB fails for one of the
             * track, we do not try to retrieve the information for any of
             * the other tracks. This is necessary because of the long
             * timeouts that may occur. */
            else {
                use_cddb = false;
            }
        }

        if (!track_title) {
            track_title = ho_strdup_printf ("%s - %s %02d", _("Audio CD"),
                                            _("Title"), i + 1);
        }

        fileitem_t *item = filelist_add (filelist, track_title, track_mrl,
                                         FILE_TYPE_REGULAR);

#ifdef HAVE_HAL
        fileitem_t *p = filelist->parent_item;
        item->device = p->device ? ho_strdup (p->device) : NULL;
        item->volume_udi = p->volume_udi ? ho_strdup (p->volume_udi) : NULL;
        item->drive_udi = p->drive_udi ? ho_strdup (p->drive_udi) : NULL;
#endif /* HAVE_HAL */

        ho_free (track_mrl);
        ho_free (track_title);
    }
}


/**
 * Reads a directory. All entries that are currently allowed are added to
 * the list.
 */
static void
directory_read (filelist_t * filelist)
{
    assert (filelist);
    assert (filelist->mrl);

    char *pwdfile = ho_strdup_printf ("%s/.password", filelist->mrl);
    if (file_exists (pwdfile)) {
        /* Get the password from the file. */
        FILE *f = fopen (pwdfile, "r");
        if (!f) {
            error (_("Could not open '%s': %s!"), pwdfile, strerror (errno));
            info (_("Not granting access to directory '%s'!"), filelist->mrl);
            ho_free (pwdfile);
            return;
        }

        char pwd_exp[128];
        if (fgets (pwd_exp, 128, f)) {
            trim_whitespace (pwd_exp);
        }
        else {
            pwd_exp[0] = '\0';
        }
        fclose (f);

        /* Check the password against a user provided password. */
        if (!pwd_callback || !pwd_callback (filelist->mrl, pwd_exp)) {
            filelist->wait_for_password = true;
            error (_("Incorrect password provided for directory '%s'!"),
                   filelist->mrl);
            info (_("Not granting access to directory '%s'!"), filelist->mrl);
            ho_free (pwdfile);
            return;
        }
    }
    ho_free (pwdfile);
    filelist->wait_for_password = false;

    DIR *dirp = opendir (filelist->mrl);
    if (dirp == NULL) {
        error (_("Unable to list contents of directory '%s': %s!"),
               filelist->mrl, strerror (errno));
        if (errno == EACCES) {
            ho_free (filelist->error);
            filelist->error = ho_strdup (_("Permission denied..."));
        }
        return;
    }

    struct dirent *entp;
    while ((entp = readdir (dirp))) {
        char *sub_mrl = NULL;
        if (filelist->mrl[strlen (filelist->mrl) - 1] == '/') {
            sub_mrl = ho_strdup_printf ("%s%s", filelist->mrl, entp->d_name);
        }
        else {
            sub_mrl = ho_strdup_printf ("%s/%s", filelist->mrl, entp->d_name);
        }

        /* The current and the parent directory are always ignored. */
        if ((strcmp (entp->d_name, ".") == 0)
            || (strcmp (entp->d_name, "..") == 0)) {
            /* Do nothing */
        }

        /* The user can specify if hidden files are ignored. */
        else if (!config_get_bool ("media.files.show_hidden_files")
                 && is_file_hidden (sub_mrl)) {
            /* Do nothing */
        }

        /* The directory we found is a DVD copied to disk. */
        else if (is_directory_dvd (sub_mrl)) {
            char *mrl = ho_strdup_printf ("dvd://%s", sub_mrl);
            if (is_file_allowed (mrl, filelist->allowed_filetypes)) {
                char *title = ho_strdup_printf ("[%s: %s]", _("DVD"),
                                                entp->d_name);
                underscore_to_space (title);
                filelist_add (filelist, title, mrl, FILE_TYPE_REGULAR);
                ho_free (title);
            }
            ho_free (mrl);
        }

        /* The directory we found is video CD copied to disk. */
        else if (is_directory_vcd (sub_mrl)) {
            char *mrl = ho_strdup_printf ("vcd://%s", sub_mrl);
            if (is_file_allowed (mrl, filelist->allowed_filetypes)) {
                char *title = ho_strdup_printf ("[%s: %s]", _("Video CD"),
                                                entp->d_name);
                underscore_to_space (title);
                filelist_add (filelist, title, mrl, FILE_TYPE_REGULAR);
                ho_free (title);
            }
            ho_free (mrl);
        }

        /* The directory is a normal directory. */
        else if (is_directory (sub_mrl)) {
            char *title = _create_title (sub_mrl, FILE_TYPE_DIRECTORY);
            filelist_add (filelist, title, sub_mrl, FILE_TYPE_DIRECTORY);
            ho_free (title);
        }

        /* The entry is a playlist (M3U). */
        else if (is_playlist_m3u (sub_mrl)) {
            char *title = _create_title (sub_mrl, FILE_TYPE_PLAYLIST_M3U);
            filelist_add (filelist, title, sub_mrl, FILE_TYPE_PLAYLIST_M3U);
            ho_free (title);
        }

        /* The entry is a playlist (PLS). */
        else if (is_playlist_pls (sub_mrl)) {
            char *title = _create_title (sub_mrl, FILE_TYPE_PLAYLIST_PLS);
            filelist_add (filelist, title, sub_mrl, FILE_TYPE_PLAYLIST_PLS);
            ho_free (title);
        }

        /* The entry is a playlist (oxine's format). */
        else if (is_playlist_oxp (sub_mrl)) {
            char *title = _create_title (sub_mrl, FILE_TYPE_PLAYLIST_OXP);
            filelist_add (filelist, title, sub_mrl, FILE_TYPE_PLAYLIST_OXP);
            ho_free (title);
        }

        /* The entry is an ISO-image of some sort. */
        else if (is_iso_image (sub_mrl)) {
#ifdef HAVE_LIBCDIO
            disc_image_detect (filelist, sub_mrl);
#endif /* HAVE_LIBCDIO */
        }

        /* The entry is a regular file. */
        else if (is_file_allowed (sub_mrl, filelist->allowed_filetypes)) {
            char *title = _create_title (sub_mrl, FILE_TYPE_REGULAR);
            filelist_add (filelist, title, sub_mrl, FILE_TYPE_REGULAR);
            ho_free (title);
        }

        ho_free (sub_mrl);
    }
    closedir (dirp);

    filelist_sort (filelist, NULL);
}


filelist_t *
filelist_expand (fileitem_t * fileitem)
{
    assert (fileitem);
    assert (fileitem->mrl);
    assert (fileitem->type != FILE_TYPE_UNKNOWN);

    filelist_lock (fileitem->parent_list);

    int top_position = 0;
    int cur_position = 0;

    switch (fileitem->type) {
    case FILE_TYPE_REGULAR:
    case FILE_TYPE_PLAYLIST_M3U:
    case FILE_TYPE_PLAYLIST_PLS:
    case FILE_TYPE_PLAYLIST_OXP:
#ifdef HAVE_VDR
    case FILE_TYPE_VDR_RECORDING:
    case FILE_TYPE_VDR_TIMER:
#endif
        /* These types cannot have a child list, so there is nothing we have
         * to do. */
        goto out_free;
    case FILE_TYPE_DIRECTORY:
    case FILE_TYPE_MOUNTPOINT:
    case FILE_TYPE_FAVORITES:
#ifdef HAVE_VDR
    case FILE_TYPE_VDR_RECORDINGS_VFOLDER:
    case FILE_TYPE_VDR_TIMERS_VFOLDER:
#endif
#ifdef HAVE_SHOUTCAST
    case FILE_TYPE_SHOUTCAST_VFOLDER:
#endif
#ifdef HAVE_YOUTUBE
    case FILE_TYPE_YOUTUBE_VFOLDER:
#endif
#ifdef HAVE_TVLINKS
    case FILE_TYPE_TVLINKS_VFOLDER:
#endif
        /* If one of these already has a child list, we clear it and save
         * the position in the list. */
        if (fileitem->child_list) {
            top_position = fileitem->child_list->top_position;
            cur_position = fileitem->child_list->cur_position;
            filelist_clear (fileitem->child_list);
        }
        break;
    default:
        /* If any of the others already has a child list, there is nothing
         * we have to do. */
        if (fileitem->child_list) {
            goto out_free;
        }
        break;
    }

    filelist_t *child_list = fileitem->child_list;
    if (!fileitem->child_list) {
        child_list = filelist_new (fileitem, fileitem->title, fileitem->mrl,
                                   fileitem->parent_list->allowed_filetypes);
        if (fileitem->thumbnail_mrl) {
            child_list->thumbnail_mrl = ho_strdup (fileitem->thumbnail_mrl);
        }
        filelist_ref_set (&fileitem->child_list, child_list);
    }

    switch (fileitem->type) {
    case FILE_TYPE_MOUNTPOINT:
#ifdef HAVE_HAL
        if (fileitem->device && volume_mount (fileitem->device)) {
            /* After mounting the volume we have to get the mountpoint. This
             * is necessary because HAL does not give us a mountpoint before
             * the disc is mounted. */
            ho_free (fileitem->mrl);
            fileitem->mrl = volume_get_mountpoint (fileitem->device);
            ho_free (child_list->mrl);
            child_list->mrl = ho_strdup (fileitem->mrl);
            directory_read (child_list);
        }
        else
#endif
        if (volume_mount (fileitem->mrl)) {
            directory_read (child_list);
        }
        break;
    case FILE_TYPE_DIRECTORY:
        directory_read (child_list);
        break;
    case FILE_TYPE_FAVORITES:
        favorites_read (child_list);
        break;
    case FILE_TYPE_MEDIAMARKS:
        mediamarks_read (child_list);
        break;
    case FILE_TYPE_CDDA_VFOLDER:
        audiocd_read (child_list);
        break;
#ifdef HAVE_VDR
    case FILE_TYPE_VDR_RECORDINGS_VFOLDER:
        vdr_recordings_read (child_list);
        break;
    case FILE_TYPE_VDR_TIMERS_VFOLDER:
        vdr_timers_read (child_list);
        break;
#endif
#ifdef HAVE_SHOUTCAST
    case FILE_TYPE_SHOUTCAST_VFOLDER:
        shoutcast_stationlist_read (child_list);
        break;
    case FILE_TYPE_SHOUTCAST_STATION:
        shoutcast_station_read (child_list);
        break;
#endif
#ifdef HAVE_YOUTUBE
    case FILE_TYPE_YOUTUBE_VFOLDER:
    case FILE_TYPE_YOUTUBE_VIDEO:
        youtube_read (child_list);
        break;
#endif
#ifdef HAVE_TVLINKS
    case FILE_TYPE_TVLINKS_VFOLDER:
    case FILE_TYPE_TVLINKS_VIDEO:
        tvlinks_read (child_list);
        break;
#endif
    default:
        break;
    }

    if (fileitem->child_list) {
        fileitem->child_list->top_position = top_position;
        fileitem->child_list->cur_position = cur_position;
    }

  out_free:
    filelist_unlock (fileitem->parent_list);

    return fileitem->child_list;
}


void
filelist_remove (filelist_t * filelist, fileitem_t * fileitem)
{
    assert (filelist);
    assert (fileitem);
    assert (filelist == fileitem->parent_list);

    filelist_lock (filelist);

    l_list_remove (filelist->list, fileitem);
    fileitem_free_cb (fileitem);

    filelist_unlock (filelist);
}


fileitem_t *
filelist_first (filelist_t * filelist)
{
    assert (filelist);

    return (fileitem_t *) l_list_first (filelist->list);
}


fileitem_t *
filelist_prev (filelist_t * filelist, fileitem_t * fileitem)
{
    assert (filelist);
    assert (fileitem);
    assert (filelist == fileitem->parent_list);

    return (fileitem_t *) l_list_prev (filelist->list, fileitem);
}


fileitem_t *
filelist_next (filelist_t * filelist, fileitem_t * fileitem)
{
    assert (filelist);
    assert (fileitem);
    assert (filelist == fileitem->parent_list);

    return (fileitem_t *) l_list_next (filelist->list, fileitem);
}


fileitem_t *
filelist_get_by_mrl (filelist_t * filelist, const char *mrl)
{
    assert (filelist);

    if (!mrl)
        return NULL;

    filelist_lock (filelist);

    fileitem_t *founditem = NULL;
    fileitem_t *fileitem = filelist_first (filelist);
    while (fileitem) {
        if (strcasecmp (fileitem->mrl, mrl) == 0) {
            founditem = fileitem;
            break;
        }
        fileitem = filelist_next (filelist, fileitem);
    }

    filelist_unlock (filelist);

    return founditem;
}


fileitem_t *
filelist_get_by_pos (filelist_t * filelist, int pos)
{
    assert (filelist);

    if (pos >= filelist_length (filelist))
        return NULL;

    filelist_lock (filelist);

    int i = 0;
    fileitem_t *founditem = NULL;
    fileitem_t *fileitem = filelist_first (filelist);
    while (fileitem) {
        if (i == pos) {
            founditem = fileitem;
            break;
        }
        fileitem = filelist_next (filelist, fileitem);
    }

    filelist_unlock (filelist);

    return founditem;
}


bool
filelist_contains (filelist_t * filelist, const char *mrl)
{
    assert (filelist);

    filelist_lock (filelist);

    bool contains = false;
    if (mrl) {
        fileitem_t *fileitem = filelist_first (filelist);
        while (fileitem) {
            if (strcasecmp (fileitem->mrl, mrl) == 0) {
                contains = true;
                break;
            }
            fileitem = filelist_next (filelist, fileitem);
        }
    }

    filelist_unlock (filelist);

    return contains;
}


int
filelist_length (filelist_t * filelist)
{
    assert (filelist);

    return l_list_length (filelist->list);
}


void
filelist_set_pwd_callback (bool (*callback)
                           (const char *mrl, const char *pwd))
{
    pwd_callback = callback;
}


const char *
filelist_get_thumbnail (filelist_t * filelist)
{
    const char *mrl = NULL;

    if (!filelist) {
        /* Nothing to do. */
    }
    else if (filelist->thumbnail_mrl) {
        mrl = filelist->thumbnail_mrl;
    }
    else {
        filelist->thumbnail_mrl = get_thumbnail (filelist->mrl);
        mrl = filelist->thumbnail_mrl;
    }

    return mrl;
}


const char *
fileitem_get_thumbnail (fileitem_t * fileitem)
{
    const char *mrl = NULL;

    if (!fileitem) {
        /* Nothing to do. */
    }
    else if (fileitem->thumbnail_mrl) {
        mrl = fileitem->thumbnail_mrl;
    }
    else if (is_file_image (fileitem->mrl)) {
        fileitem->thumbnail_mrl = ho_strdup (fileitem->mrl);
        mrl = fileitem->thumbnail_mrl;
    }
    else {
        fileitem->thumbnail_mrl = get_thumbnail (fileitem->mrl);
        mrl = fileitem->thumbnail_mrl;
    }

    return mrl;
}


#ifdef DEBUG
void
filelist_print (filelist_t * filelist)
{
    assert (filelist);

    debug ("=============================================");
    debug ("Filelist: %s", filelist->title);
    debug ("          %s", filelist->mrl);

    fileitem_t *item = filelist_first (filelist);
    while (item) {
        debug ("Fileitem: %s", item->title);
        debug ("          %s", item->mrl);
        item = filelist_next (filelist, item);
    }
    debug ("=============================================");
}
#endif
