
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: menu_main.c 2632 2007-08-16 14:06:45Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "disc.h"
#include "download.h"
#include "environment.h"
#include "extractor.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "utils.h"
#include "oxine.h"

#include "menu_base.h"
#include "menu_extractor.h"
#include "menu_help.h"
#include "menu_main.h"
#include "menu_filelist.h"
#include "menu_playback.h"
#include "menu_playlist.h"
#include "menu_weather.h"
#include "menu_settings.h"

extern oxine_t *oxine;

#ifdef HAVE_HAL
static otk_widget_t *eject_list;
#endif
static otk_widget_t *shutdown_list;
static char *shutdown_commands[3] = { NULL, NULL, NULL };


void
execute_cb (void *p)
{
    char *command = (char *) p;
    execute_shell (command, 0);
}


void
play_dvb_cb (void *cb_data)
{
    const char *channels_conf = get_file_dvb_channels ();

    if (!file_exists (channels_conf)) {
        error (_("Could not open '%s': %s!"), channels_conf,
               strerror (errno));
        show_message_dialog (show_menu_main, oxine, NULL, NULL, DIALOG_OK,
                             NULL, _("Failed to start TV playback!"));
        return;
    }

    set_backto_menu (show_menu_main, NULL);
    set_playback_ended_menu (show_menu_main, NULL);

    please_wait ();

    playitem_t *default_item = NULL;
    int default_channel = config_get_number ("media.dvb.last_channel");

    /* We add all channels from .xine/channels.conf to the playlist. */
    FILE *f = fopen (channels_conf, "r");
    if (f) {
        playlist_clear (oxine->ro_playlist);

        int i;
        for (i = 1; !feof (f); i++) {
            char line[256];
            char title[64];
            char mrl[16];
            if (!fgets (line, 256, f)) {
                break;
            }
            sscanf (line, "%[^:]", title);
            snprintf (mrl, 16, "dvb://%d", (i - 1));
            playitem_t *item = playlist_add (oxine->ro_playlist, title, mrl);

            if (default_channel == i) {
                default_item = item;
            }
        }
        fclose (f);

        /* If we found the last channel we play that. */
        if (default_item) {
            playlist_play_item (oxine->ro_playlist, default_item);
        }
        else {
            playlist_play_first (oxine->ro_playlist);
        }
    }
}


void
play_v4l_cb (void *cb_data)
{
    set_backto_menu (show_menu_main, NULL);
    set_playback_ended_menu (show_menu_main, NULL);

    please_wait ();

    playlist_clear (oxine->ro_playlist);
    playlist_add (oxine->ro_playlist, "Analog Television", "v4l://");
    playlist_play_first (oxine->ro_playlist);
}


#ifdef HAVE_VDR
void
play_vdr_cb (void *cb_data)
{
    set_backto_menu (show_menu_main, NULL);
    set_playback_ended_menu (show_menu_main, NULL);

    please_wait ();

    int num_mrls;
    char **mrls = xine_get_autoplay_mrls (oxine->xine, "VDR", &num_mrls);
    if (mrls) {
        playlist_clear (oxine->ro_playlist);
        int i;
        for (i = 0; i < num_mrls; i++) {
            playlist_add (oxine->ro_playlist, "Video Disc Recorder", mrls[i]);
        }
        playlist_play_first (oxine->ro_playlist);
    }
    else {
        error (_("Failed to retrieve MRL for VDR playback!"));
        show_message_dialog (show_menu_main, oxine, NULL, NULL, DIALOG_OK,
                             NULL, _("Failed to start TV playback!"));
    }
}
#endif


void
play_tv_cb (void *cb_data)
{
    int type = config_get_number ("television.type");

    switch (type) {
    case 0:
        play_v4l_cb (cb_data);
        break;
    case 1:
        play_dvb_cb (cb_data);
        break;
#ifdef HAVE_VDR
    case 2:
        play_vdr_cb (cb_data);
        break;
#endif
    default:
        error (_("Unknown type '%d' for television."), type);
        break;
    }
}


void
play_mrl_cb (void *mrl_p)
{
    char *mrl = (char *) mrl_p;

    set_backto_menu (show_menu_main, NULL);
    set_playback_ended_menu (show_menu_main, NULL);

    please_wait ();

    playlist_clear (oxine->ro_playlist);
    playlist_add (oxine->ro_playlist, NULL, mrl);
    playlist_play_first (oxine->ro_playlist);
}


static void
playlist_add_audiocd (playlist_t * playlist, const char *mrl)
{
    filelist_t *filelist = NULL;
    filelist_ref_set (&filelist, filelist_new (NULL, NULL, NULL,
                                               ALLOW_FILES_MULTIMEDIA));
    playlist_add_fileitem (playlist,
                           filelist_add (filelist, "Audio CD", mrl,
                                         FILE_TYPE_CDDA_VFOLDER));
    filelist_ref_set (&filelist, NULL);
}


void
play_cdda_cb (void *device_p)
{
    set_backto_menu (show_menu_main, NULL);
    set_playback_ended_menu (show_menu_main, NULL);

    please_wait ();

    char *device = NULL;
    if (device_p) {
        device = ho_strdup ((char *) device_p);
    }
    else {
        device = ho_strdup (config_get_string ("media.audio_cd.device"));
    }

    /* If the drive is ejected we inject the drive. */
    if (drive_is_ejected (device)) {
        drive_inject (device);
    }

    volume_type_t type = volume_get_type (device);
    /* If we are certain that the disc in the specified audio CD drive is
     * indeed an audio CD we play this disc. */
    if (type == VOLUME_TYPE_AUDIO_CD) {
        char mrl[1024];
        snprintf (mrl, 1024, "cdda:/%s", device);

        playlist_clear (oxine->ro_playlist);
        playlist_add_audiocd (oxine->ro_playlist, mrl);
        playlist_play_first (oxine->ro_playlist);
    }
    /* If there is no disc in the drive or if there is a disc in the drive but
     * we know what kind of disc this is we tell the user to insert a audio CD. */
    else if (type != VOLUME_TYPE_UNKNOWN) {
        show_message_dialog (play_cdda_cb, device,
                             backto_menu_cb, oxine,
                             DIALOG_OK_CANCEL, NULL,
                             _("Please insert an Audio CD!"));
    }
    /* If none of the above apply we just try our luck and hope xine-lib can
     * solve this problem for us. */
    else {
        playlist_clear (oxine->ro_playlist);
        playlist_add_audiocd (oxine->ro_playlist, "cdda:/");
        playlist_play_first (oxine->ro_playlist);
    }

    ho_free (device);
}


void
play_vcd_cb (void *device_p)
{
    set_backto_menu (show_menu_main, NULL);
    set_playback_ended_menu (show_menu_main, NULL);

    please_wait ();

    char *device = NULL;
    if (device_p) {
        device = ho_strdup ((char *) device_p);
    }
    else {
        device = ho_strdup (config_get_string ("media.vcd.device"));
    }

    /* If the drive is ejected we inject the drive. */
    if (drive_is_ejected (device)) {
        drive_inject (device);
    }

    volume_type_t type = volume_get_type (device);
    /* If we are certain that the disc in the specified VCD drive is indeed a
     * VCD we play this disc. */
    if (type == VOLUME_TYPE_VIDEO_CD) {
        char mrl[1024];
        snprintf (mrl, 1024, "vcd://%s", device);

        playlist_clear (oxine->ro_playlist);
        playlist_add (oxine->ro_playlist, _("Video CD"), mrl);
        playlist_play_first (oxine->ro_playlist);
    }
    /* If there is no disc in the drive or if there is a disc in the drive but
     * we know what kind of disc this is we tell the user to insert a VCD. */
    else if (type != VOLUME_TYPE_UNKNOWN) {
        show_message_dialog (play_vcd_cb, device,
                             backto_menu_cb, oxine,
                             DIALOG_OK_CANCEL, NULL,
                             _("Please insert a Video CD!"));
    }
    /* If none of the above apply we just try our luck and hope xine-lib can
     * solve this problem for us. */
    else {
        playlist_clear (oxine->ro_playlist);
        playlist_add (oxine->ro_playlist, _("Video CD"), "vcd://");
        playlist_play_first (oxine->ro_playlist);
    }

    ho_free (device);
}


void
play_dvd_cb (void *device_p)
{
    set_backto_menu (show_menu_main, NULL);
    set_playback_ended_menu (show_menu_main, NULL);

    please_wait ();

    char *device = NULL;
    if (device_p) {
        device = ho_strdup ((char *) device_p);
    }
    else {
        device = ho_strdup (config_get_string ("media.dvd.device"));
    }

    /* If the drive is ejected we inject the drive. */
    if (drive_is_ejected (device)) {
        drive_inject (device);
    }

    volume_type_t type = volume_get_type (device);
    /* If we are certain that the disc in the specified DVD drive is indeed a
     * DVD we play this disc. */
    if (type == VOLUME_TYPE_VIDEO_DVD) {
        char mrl[1024];
        snprintf (mrl, 1024, "dvd://%s", device);

        playlist_clear (oxine->ro_playlist);
        playlist_add (oxine->ro_playlist, _("DVD"), mrl);
        playlist_play_first (oxine->ro_playlist);
    }
    /* If there is no disc in the drive or if there is a disc in the drive but
     * we know what kind of disc this is we tell the user to insert a DVD. */
    else if (type != VOLUME_TYPE_UNKNOWN) {
        show_message_dialog (play_dvd_cb, device,
                             backto_menu_cb, oxine,
                             DIALOG_OK_CANCEL, NULL,
                             _("Please insert a DVD!"));
    }
    /* If none of the above apply we just try our luck and hope xine-lib can
     * solve this problem for us. */
    else {
        playlist_clear (oxine->ro_playlist);
        playlist_add (oxine->ro_playlist, _("DVD"), "dvd://");
        playlist_play_first (oxine->ro_playlist);
    }

    ho_free (device);
}


static void
shutdown_execute_cb (void *cb_data)
{
    int p = otk_list_get_pos (shutdown_list);

    xine_cfg_entry_t config;
    config_lookup_entry ("shutdown.last", &config);
    config.num_value = p;
    config_update_entry (&config);

    if (p > 0) {
        const char *cmd = shutdown_commands[p - 1];

        if (cmd && strlen (cmd) > 0) {
            oxine->shutdown_command = ho_strdup (cmd);
        }
    }

    odk_exit (oxine->odk);
}


void
shutdown_cb (void *cb_data)
{
    if (!config_get_bool ("shutdown.ask")) {
        odk_exit (oxine->odk);
        return;
    }

    create_new_window (false, true);

    int w = 600;
    int h = 200;
    int x = (odk_osd_get_width (oxine->odk) - w) / 2;
    int y = (odk_osd_get_height (oxine->odk) - h) / 2;

    otk_border_new (oxine->otk, x, y, w, h);
    otk_label_new (oxine->otk, x + 20, y + 50, 560,
                   OTK_ALIGN_LEFT | OTK_ALIGN_BOTTOM,
                   _("How do you want to proceed?"));

    shutdown_list = otk_selector_new (oxine->otk, x + 20, y + 70, 560,
                                      NULL, NULL);

    ho_free (shutdown_commands[0]);
    shutdown_commands[0] = NULL;
    ho_free (shutdown_commands[1]);
    shutdown_commands[1] = NULL;
    ho_free (shutdown_commands[2]);
    shutdown_commands[2] = NULL;

    {
        otk_selectoritem_new (shutdown_list, _("Quit oxine"),
                              shutdown_execute_cb, oxine);
    }

    int i = 0;
    const char *command;

    command = config_get_string ("shutdown.shutdown_command");
    if (command && (strlen (command) > 0)) {
        shutdown_commands[i++] = ho_strdup (command);
        otk_selectoritem_new (shutdown_list, _("Shutdown the computer"),
                              shutdown_execute_cb, oxine);
    }

    command = config_get_string ("shutdown.standby_command");
    if (command && (strlen (command) > 0)) {
        shutdown_commands[i++] = ho_strdup (command);
        otk_selectoritem_new (shutdown_list, _("Switch computer to standby"),
                              shutdown_execute_cb, oxine);
    }

    command = config_get_string ("shutdown.reboot_command");
    if (command && (strlen (command) > 0)) {
        shutdown_commands[i++] = ho_strdup (command);
        otk_selectoritem_new (shutdown_list, _("Reboot the computer"),
                              shutdown_execute_cb, oxine);
    }

    int last_command = config_get_number ("shutdown.last");
    otk_list_set_pos (shutdown_list, last_command);
    otk_list_set_focus (shutdown_list, last_command);

    otk_widget_t *button;
    button = otk_text_button_new (oxine->otk, x + 130, y + 140, 160, 40,
                                  _("OK"), shutdown_execute_cb, oxine);
    otk_widget_set_alignment (button, OTK_ALIGN_CENTER);

    button = otk_text_button_new (oxine->otk, x + 310, y + 140, 160, 40,
                                  _("Cancel"), backto_menu_cb, oxine);
    otk_widget_set_alignment (button, OTK_ALIGN_CENTER);

    set_current_menu (shutdown_cb, NULL);

    show_user_interface (NULL);
    show_menu_background ("menu_quit.png");
}


static void
eject_execute_cb (void *device_p)
{
#ifdef HAVE_HAL
    filelist_lock (oxine->hal_volume_list);
#endif

    char *device = NULL;
    if (device_p) {
        device = (char *) device_p;
    }
    else {
#ifdef HAVE_HAL
        int p = otk_list_get_pos (eject_list);
        int i = 0;
        fileitem_t *item = filelist_first (oxine->hal_volume_list);
        while (item) {
            if (i == p) {
                device = item->device;
                break;
            }
            item = filelist_next (oxine->hal_volume_list, item);
            i++;
        }
#endif
    }

    assert (device);

    /* If the tray of the drive is currently open (ejected) we try to close
     * the tray. */
    if (drive_is_ejected (device)) {
        otk_cb_t cb = backto_menu_cb;
        please_wait ();

        if (!drive_inject (device)) {
            show_message_dialog (eject_execute_cb, device, cb, oxine,
                                 DIALOG_RETRY_CANCEL, NULL,
                                 _("Could not inject device!"));
        }
        else {
            cb (oxine);
        }
    }

    /* If the tray of the drive is currently closed we try to open the tray. */
    else {
        otk_cb_t cb = backto_menu_cb;
        please_wait ();

        /* If we're playing from the disc we're about to eject, we stop the
         * stream first. */
        if (odk_current_is_playback_mode (oxine->odk)) {
            const char *current_mrl = odk_current_get_mrl (oxine->odk);
            char *mountpoint = volume_get_mountpoint (device);
            if ((mountpoint && (strstr (current_mrl, mountpoint) != NULL))
                || (odk_current_is_dvd (oxine->odk)
                    && (strstr (current_mrl, device) != NULL))
                || (odk_current_is_vcd (oxine->odk)
                    && (strstr (current_mrl, device) != NULL))
                || (odk_current_is_cdda (oxine->odk)
                    && (strstr (current_mrl, device) != NULL))) {
                cb = playback_ended_menu_cb;
                odk_stop_stream (oxine->odk);
            }
            ho_free (mountpoint);
        }

#ifdef HAVE_EXTRACTOR
        /* If we're extracting from the disc we're about to eject, we stop
         * the extractor. */
        if (extractor_is_running ()) {
            int pos = extractor_get_progress_all () - 1;
            filelist_t *list = extractor_get_list ();
            fileitem_t *item = filelist_get_by_pos (list, pos);

            if (strstr (item->mrl, device) != NULL) {
                extractor_stop ();
            }
        }
#endif

        /* Now we first try to unmount and then try to eject the device. */
        if (volume_is_mounted (device) && !volume_umount (device)) {
            show_message_dialog (eject_execute_cb, device, cb, oxine,
                                 DIALOG_RETRY_CANCEL, NULL,
                                 _("Could not unmount device!"));
        }
        else if (!drive_eject (device)) {
            show_message_dialog (eject_execute_cb, device, cb, oxine,
                                 DIALOG_RETRY_CANCEL, NULL,
                                 _("Could not eject device!"));
        }
        else {
            cb (oxine);
        }
    }

#ifdef HAVE_HAL
    filelist_unlock (oxine->hal_volume_list);
#endif
}


void
eject_cb (void *cb_data)
{
#ifdef HAVE_HAL
    if (filelist_length (oxine->hal_volume_list) == 0) {
        return;
    }
    else if (filelist_length (oxine->hal_volume_list) == 1) {
        if (!config_get_bool ("misc.eject.one_device.confirm")) {
            fileitem_t *item = filelist_first (oxine->hal_volume_list);
            eject_execute_cb (item->device);
            return;
        }
    }

    create_new_window (false, true);

    int w = 600;
    int h = 200;
    int x = (odk_osd_get_width (oxine->odk) - w) / 2;
    int y = (odk_osd_get_height (oxine->odk) - h) / 2;

    otk_border_new (oxine->otk, x, y, w, h);
    otk_label_new (oxine->otk, x + 20, y + 50, 560,
                   OTK_ALIGN_LEFT | OTK_ALIGN_BOTTOM,
                   _("Select device to eject:"));

    eject_list = otk_selector_new (oxine->otk, x + 20, y + 70, 560,
                                   NULL, NULL);

    filelist_lock (oxine->hal_volume_list);
    fileitem_t *item = filelist_first (oxine->hal_volume_list);
    while (item) {
        otk_selectoritem_new (eject_list, item->title,
                              eject_execute_cb, item->device);
        item = filelist_next (oxine->hal_volume_list, item);
    }
    filelist_unlock (oxine->hal_volume_list);

    otk_list_set_pos (eject_list, 0);
    otk_list_set_focus (eject_list, 0);

    otk_widget_t *button;
    button = otk_text_button_new (oxine->otk, x + 130, y + 140, 160, 40,
                                  _("Eject"), eject_execute_cb, NULL);
    otk_widget_set_alignment (button, OTK_ALIGN_CENTER);

    button = otk_text_button_new (oxine->otk, x + 310, y + 140, 160, 40,
                                  _("Cancel"), backto_menu_cb, oxine);
    otk_widget_set_alignment (button, OTK_ALIGN_CENTER);

    set_current_menu (eject_cb, NULL);

    show_user_interface (NULL);
    show_menu_background ("menu_eject.png");
#else
    const char *device = config_get_string ("media.audio_cd.device");
    eject_execute_cb ((void *) device);
#endif
}


static void
menulist_focus_enter (void *p)
{
    otk_cb_t cb = (otk_cb_t) p;

    if (cb == play_tv_cb) {
        show_menu_background ("menu_main_tv.png");
    }
    else if (cb == play_dvd_cb) {
        show_menu_background ("menu_main_dvd.png");
    }
    else if (cb == play_vcd_cb) {
        show_menu_background ("menu_main_vcd.png");
    }
    else if (cb == play_cdda_cb) {
        show_menu_background ("menu_main_cdda.png");
    }
    else if (cb == show_menu_music) {
        show_menu_background ("menu_main_music.png");
    }
    else if (cb == show_menu_video) {
        show_menu_background ("menu_main_video.png");
    }
#ifdef HAVE_IMAGE_PLAYBACK
    else if (cb == show_menu_image) {
        show_menu_background ("menu_main_image.png");
    }
#endif
    else if (cb == show_menu_playlist) {
        show_menu_background ("menu_main_playlist.png");
    }
#ifdef HAVE_WEATHER
    else if (cb == show_menu_weather) {
        show_menu_background ("menu_main_weather.png");
    }
#endif
#ifdef HAVE_EXTRACTOR
    else if (cb == show_menu_extractor) {
        show_menu_background ("menu_main_extractor.png");
    }
#endif
    else if (cb == show_menu_settings) {
        show_menu_background ("menu_main_settings.png");
    }
    else if (cb == show_menu_help) {
        show_menu_background ("menu_main_help.png");
    }
    else if (cb == shutdown_cb) {
        show_menu_background ("menu_main_quit.png");
    }
    else if (cb == show_menu_playback) {
        show_menu_background ("menu_main_playback.png");
    }
    else {
        show_menu_background ("menu_main.png");
    }
}


static void
menulist_fill (otk_widget_t * menu, menulist_t * menulist)
{
    menuitem_t *item = l_list_first (menulist->list);
    while (item) {
        if ((item->cb == show_menu_playback)
            && !odk_current_is_playback_mode (oxine->odk)) {
            /* If we're not in playback mode we hide any menu item pointing
             * to the playback menu. */
        }
#ifdef HAVE_EXTRACTOR
        else if ((item->cb == show_menu_extractor)
                 && !extractor_is_running ()) {
            /* If the extractor is currently not running we hide any menu
             * item pointing to the extractor menu. */
        }
#endif
        else {
            otk_widget_t *w = otk_menuitem_new (menu, _(item->title),
                                                item->cb, item->cb_data);
            otk_widget_set_focus_callbacks (w, menulist_focus_enter, item->cb,
                                            NULL, NULL);
        }

        item = l_list_next (menulist->list, item);
    }
}


static otk_widget_t *
menulist_show (menulist_t * menulist)
{
    otk_widget_t *l;

    int w = 370;
    if (menulist->w != -1) {
        w = menulist->w;
    }
    int x = (odk_osd_get_width (oxine->odk) - w) / 2;
    if (menulist->x != -1) {
        x = menulist->x;
    }
    int y = 100;
    if (menulist->y != -1) {
        y = menulist->y;
    }
    int h = (odk_osd_get_height (oxine->odk) - y - 20);
    if (menulist->h != -1) {
        h = menulist->h;
    }

    l = otk_menulist_new (oxine->otk, x, y, w, h, 45, 55, NULL);
    menulist_fill (l, menulist);

    return l;
}


static void
main_menu_event_handler (void *cb_data, oxine_event_t * event)
{
    if (!is_current_menu (show_menu_main)
        && !is_current_menu (show_menu_sub)) {
        return;
    }
    if (event->type != OXINE_EVENT_KEY) {
        return;
    }

    switch (event->source.key) {
    case OXINE_KEY_BACK:
        if (is_current_menu (show_menu_sub)) {
            show_menu_main (oxine);
        }
        else if (!odk_current_is_logo_mode (oxine->odk)) {
            show_menu_playback (oxine);
        }
        event->source.key = OXINE_KEY_NULL;
        break;
    default:
        break;
    }
}


void
show_menu_sub (void *menulist_p)
{
    menulist_t *menulist = (menulist_t *) menulist_p;

    create_new_window (false, true);

    menulist_show (menulist);

    if (odk_current_is_logo_mode (oxine->odk)) {
        set_playback_ended_menu (show_menu_sub, menulist);
    }

    set_backto_menu (show_menu_sub, menulist);
    set_current_menu (show_menu_sub, menulist);

    show_user_interface (NULL);
    show_menu_background ("menu_main.png");
}


void
show_menu_main (void *cb_data)
{
    static otk_widget_t *main_menu_list = NULL;
    static otk_widget_t *main_menu_window = NULL;
    if (main_menu_window) {
        otk_set_current_window (oxine->otk, main_menu_window);
        int top_pos = otk_list_get_pos (main_menu_list);
        int cur_pos = otk_list_get_focus (main_menu_list);
        int length = otk_list_get_length (main_menu_list);
        otk_list_clear (main_menu_list);
        menulist_fill (main_menu_list, oxine->main_menu);
        top_pos += otk_list_get_length (main_menu_list) - length;
        cur_pos += otk_list_get_length (main_menu_list) - length;
        otk_list_set_focus (main_menu_list, cur_pos);
        otk_list_set_pos (main_menu_list, top_pos);
        goto out_show;
    }

    odk_add_event_handler (oxine->odk, main_menu_event_handler, oxine,
                           EVENT_HANDLER_PRIORITY_NORMAL);
    main_menu_window = create_new_window (true, true);

    /* These entries are needed so the titles used in the standard
     * mainmenu.xml file appear in the translation files. */
    _("Listen to Music");
    _("Watch Films");
    _("View Images");
    _("Watch Television");
    _("Play a DVD");
    _("Play a Video CD");
    _("Play an Audio CD");
    _("Show Help");
    _("Show Weather Report");
    _("Edit Settings");
    _("Shutdown");

    main_menu_list = menulist_show (oxine->main_menu);

  out_show:
    if (!odk_current_is_playback_mode (oxine->odk)) {
        set_playback_ended_menu (show_menu_main, NULL);
    }
    set_backto_menu (show_menu_main, NULL);
    set_current_menu (show_menu_main, NULL);

    show_user_interface (NULL);
}


void
init_menu_main ()
{
    char *menu_xml = NULL;

    menu_xml = ho_strdup_printf ("%s/mainmenu.xml", get_dir_oxine ());
    oxine->main_menu = menulist_new (menu_xml, NULL);
    if (oxine->main_menu) {
        goto out_free;
    }
    ho_free (menu_xml);

    menu_xml = ho_strdup_printf ("%s/mainmenu.xml", get_dir_oxine_skin ());
    oxine->main_menu = menulist_new (menu_xml, NULL);
    if (oxine->main_menu) {
        goto out_free;
    }
    ho_free (menu_xml);

    menu_xml = ho_strdup_printf ("%s/mainmenu.xml", OXINE_DATADIR);
    oxine->main_menu = menulist_new (menu_xml, NULL);
    if (oxine->main_menu) {
        goto out_free;
    }
    ho_free (menu_xml);

    fatal (_("Could not load main menu!"));
    abort ();

  out_free:
    ho_free (menu_xml);
}


void
free_menu_main ()
{
    if (oxine->main_menu) {
        menulist_free (oxine->main_menu);
        oxine->main_menu = NULL;
    }

    ho_free (shutdown_commands[0]);
    ho_free (shutdown_commands[1]);
    ho_free (shutdown_commands[2]);
}
