
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_keymap.c 2413 2007-07-03 08:18:06Z mschwerin $
 *
 */

#include "config.h"

#include <stdio.h>
#include <string.h>
#include <limits.h>

#include "event.h"
#include "heap.h"
#include "logger.h"
#include "odk_keymap.h"
#include "odk_xk.h"


typedef struct {
    struct {
        /// The keyboard key-code.
        KeySym key;
        /// The keyboard modifier (e.g. CTRL, ALT).
        int modifier;
    } keyboard;

    struct {
        /// The LIRC config code.
        char *config;

        /**
         * Defines what happens if a key is repeated. A value of zero tells us
         * to ignore the repeated key. Any other positive value 'n' tells us
         * to use the config string every 'n'-th time.
         */
        int repeat;
    } lirc;

    /// The oxine key-event ID.
    oxine_key_id_t key_id;

    /**
     * A parameter defining how much to change a value. This is used e.g. to
     * define by how much the volume is changed when the volume key is pressed
     * once. */
    int how;

    /// A human readable description of the action.
    char *description;
} odk_keymap_entry_t;


static odk_keymap_entry_t default_keymap_table[] = {
    /* Menus */
    {{XK_F1, ControlMask}, {"menu1", 0}, OXINE_KEY_MENU1, 0, NULL},
    {{XK_F2, ControlMask}, {"menu2", 0}, OXINE_KEY_MENU2, 0, NULL},
    {{XK_F3, ControlMask}, {"menu3", 0}, OXINE_KEY_MENU3, 0, NULL},
    {{XK_F4, ControlMask}, {"menu4", 0}, OXINE_KEY_MENU4, 0, NULL},
    {{XK_F5, ControlMask}, {"menu5", 0}, OXINE_KEY_MENU5, 0, NULL},
    {{XK_F6, ControlMask}, {"menu6", 0}, OXINE_KEY_MENU6, 0, NULL},
    {{XK_F7, ControlMask}, {"menu7", 0}, OXINE_KEY_MENU7, 0, NULL},
    {{XK_F8, ControlMask}, {"menu8", 0}, OXINE_KEY_MENU8, 0, NULL},

    {{XK_F1, 0}, {"menu_help", 0}, OXINE_KEY_MENU_HELP, 0, NULL},
    {{XK_Help, 0}, {"menu_help", 0}, OXINE_KEY_MENU_HELP, 0, NULL},
    {{XK_h, 0}, {"menu_help", 0}, OXINE_KEY_MENU_HELP, 0, NULL},
    {{XK_H, 0}, {"menu_help", 0}, OXINE_KEY_MENU_HELP, 0, NULL},
    {{XK_question, 0}, {"menu_help", 0}, OXINE_KEY_MENU_HELP, 0, NULL},

    {{XK_o, ControlMask}, {"menu_settings", 0}, OXINE_KEY_MENU_SETTINGS, 0, "show the settings menu"},
    {{XK_F2, 0}, {"menu_settings", 0}, OXINE_KEY_MENU_SETTINGS, 0, "show the settings menu"},

    {{XK_F3, 0}, {"menu_playback", 0}, OXINE_KEY_MENU_PLAYBACK, 0, "show the playback menu"},
    {{XK_F4, 0}, {"television", 0}, OXINE_KEY_TELEVISION, 0, "start television playback"},

    {{XK_F5, 0}, {"menu_main", 0}, OXINE_KEY_MENU_MAIN, 0, "show the main menu"},
    {{XK_F6, 0}, {"menu_music", 0}, OXINE_KEY_MENU_MUSIC, 0, "show the music menu"},
    {{XK_F7, 0}, {"menu_video", 0}, OXINE_KEY_MENU_VIDEO, 0, "show the video menu"},
    {{XK_F8, 0}, {"menu_image", 0}, OXINE_KEY_MENU_IMAGE, 0, "show the image menu"},
    {{XK_F9, 0}, {"menu_playlist", 0}, OXINE_KEY_MENU_PLAYLIST, 0, "show the playlist menu"},

    {{XK_F10, 0}, {"menu_weather", 0}, OXINE_KEY_MENU_WEATHER, 0, "show the weather menu"},

    /* Numbers */
    {{XK_0, 0}, {"0", 2}, OXINE_KEY_0, 0, NULL},
    {{XK_KP_0, 0}, {"0", 2}, OXINE_KEY_0, 0, NULL},
    {{XK_1, 0}, {"1", 2}, OXINE_KEY_1, 0, NULL},
    {{XK_KP_1, 0}, {"1", 2}, OXINE_KEY_1, 0, NULL},
    {{XK_2, 0}, {"2", 2}, OXINE_KEY_2, 0, NULL},
    {{XK_KP_2, 0}, {"2", 2}, OXINE_KEY_2, 0, NULL},
    {{XK_3, 0}, {"3", 2}, OXINE_KEY_3, 0, NULL},
    {{XK_KP_3, 0}, {"3", 2}, OXINE_KEY_3, 0, NULL},
    {{XK_4, 0}, {"4", 2}, OXINE_KEY_4, 0, NULL},
    {{XK_KP_4, 0}, {"4", 2}, OXINE_KEY_4, 0, NULL},
    {{XK_5, 0}, {"5", 2}, OXINE_KEY_5, 0, NULL},
    {{XK_KP_5, 0}, {"5", 2}, OXINE_KEY_5, 0, NULL},
    {{XK_6, 0}, {"6", 2}, OXINE_KEY_6, 0, NULL},
    {{XK_KP_6, 0}, {"6", 2}, OXINE_KEY_6, 0, NULL},
    {{XK_7, 0}, {"7", 2}, OXINE_KEY_7, 0, NULL},
    {{XK_KP_7, 0}, {"7", 2}, OXINE_KEY_7, 0, NULL},
    {{XK_8, 0}, {"8", 2}, OXINE_KEY_8, 0, NULL},
    {{XK_KP_8, 0}, {"8", 2}, OXINE_KEY_8, 0, NULL},
    {{XK_9, 0}, {"9", 2}, OXINE_KEY_9, 0, NULL},
    {{XK_KP_9, 0}, {"9", 2}, OXINE_KEY_9, 0, NULL},

    /* Zoom */
    {{XK_z, ControlMask | Mod1Mask}, {"zoom_reset", 0}, OXINE_KEY_ZOOM_RESET, 0, NULL},
    {{XK_y, ControlMask | Mod1Mask}, {"zoom_reset", 0}, OXINE_KEY_ZOOM_RESET, 0, NULL},
    {{XK_Z, ControlMask | Mod1Mask}, {"zoom_reset", 0}, OXINE_KEY_ZOOM_RESET, 0, NULL},
    {{XK_Y, ControlMask | Mod1Mask}, {"zoom_reset", 0}, OXINE_KEY_ZOOM_RESET, 0, NULL},
    {{XK_z, ControlMask}, {"zoom_x", 0}, OXINE_KEY_ZOOM_X, +1, NULL},
    {{XK_y, ControlMask}, {"zoom_x", 0}, OXINE_KEY_ZOOM_X, +1, NULL},
    {{XK_Z, ControlMask}, {"zoom_x", 0}, OXINE_KEY_ZOOM_X, -1, NULL},
    {{XK_Y, ControlMask}, {"zoom_x", 0}, OXINE_KEY_ZOOM_X, -1, NULL},
    {{XK_z, Mod1Mask}, {"zoom_y", 0}, OXINE_KEY_ZOOM_Y, +1, NULL},
    {{XK_y, Mod1Mask}, {"zoom_y", 0}, OXINE_KEY_ZOOM_Y, +1, NULL},
    {{XK_Z, Mod1Mask}, {"zoom_y", 0}, OXINE_KEY_ZOOM_Y, -1, NULL},
    {{XK_Y, Mod1Mask}, {"zoom_y", 0}, OXINE_KEY_ZOOM_Y, -1, NULL},
    {{XK_z, 0}, {"zoom", 0}, OXINE_KEY_ZOOM, +1, NULL},
    {{XK_y, 0}, {"zoom", 0}, OXINE_KEY_ZOOM, +1, NULL},
    {{XK_Z, 0}, {"zoom", 0}, OXINE_KEY_ZOOM, -1, NULL},
    {{XK_Y, 0}, {"zoom", 0}, OXINE_KEY_ZOOM, -1, NULL},

    /* Audio Channel, Audio Offset */
    {{XK_x, 0}, {"audio_channel", 0}, OXINE_KEY_AUDIO_CHANNEL, +1, NULL},
    {{XK_X, 0}, {"audio_channel", 0}, OXINE_KEY_AUDIO_CHANNEL, -1, NULL},
    {{XK_c, 0}, {"audio_offset", 2}, OXINE_KEY_AUDIO_OFFSET, +9, NULL},
    {{XK_C, 0}, {"audio_offset", 2}, OXINE_KEY_AUDIO_OFFSET, -9, NULL},

    /* Subtitle Channel, Subtitle Offset */
    {{XK_r, 0}, {"spu_channel", 0}, OXINE_KEY_SPU_CHANNEL, +1, NULL},
    {{XK_R, 0}, {"spu_channel", 0}, OXINE_KEY_SPU_CHANNEL, -1, NULL},
    {{XK_t, 0}, {"spu_offset", 2}, OXINE_KEY_SPU_OFFSET, +9, NULL},
    {{XK_T, 0}, {"spu_offset", 2}, OXINE_KEY_SPU_OFFSET, -9, NULL},

    /* Volume */
    {{XK_v, 0}, {"volume", 2}, OXINE_KEY_VOLUME, +5, "increase volume by 5%"},
    {{XK_plus, 0}, {"volume", 2}, OXINE_KEY_VOLUME, +5, "increase volume by 5%"},
    {{XK_KP_Add, 0}, {"volume", 2}, OXINE_KEY_VOLUME, +5, "increase volume by 5%"},
    {{XF86XK_AudioRaiseVolume, 0}, {"volume", 2}, OXINE_KEY_VOLUME, +5, "increase volume by 5%"},

    {{XK_V, 0}, {"volume", 2}, OXINE_KEY_VOLUME, -5, "decrease volume by 5%"},
    {{XK_minus, 0}, {"volume", 2}, OXINE_KEY_VOLUME, -5, "decrease volume by 5%"},
    {{XK_KP_Subtract, 0}, {"volume", 2}, OXINE_KEY_VOLUME, -5, "decrease volume by 5%"},
    {{XF86XK_AudioLowerVolume, 0}, {"volume", 2}, OXINE_KEY_VOLUME, -5, "decrease volume by 5%"},

    {{XK_m, 0}, {"toggle_mute", 0}, OXINE_KEY_VOLUME_MUTE, 0, "mute volume"},
    {{XF86XK_AudioMute, 0}, {"toggle_mute", 0}, OXINE_KEY_VOLUME_MUTE, 0, "mute volume"},

    /* Stream control */
    {{0, 0}, {"play", 0}, OXINE_KEY_PLAY, 0, "play"},
    {{0, 0}, {"pause", 0}, OXINE_KEY_PAUSE, 0, "pause"},

    {{XF86XK_AudioPlay, 0}, {"toggle_pause_play", 0}, OXINE_KEY_PPLAY, 0, "play/ pause"},
    {{XF86XK_AudioPause, 0}, {"toggle_pause_play", 0}, OXINE_KEY_PPLAY, 0, "play/ pause"},
    {{XK_Pause, 0}, {"toggle_pause_play", 0}, OXINE_KEY_PPLAY, 0, "play/ pause"},

    {{XK_s, 0}, {"stop", 0}, OXINE_KEY_STOP, 0, "stop"},
    {{XK_S, 0}, {"stop", 0}, OXINE_KEY_STOP, 0, "stop"},
    {{XF86XK_AudioStop, 0}, {"stop", 0}, OXINE_KEY_STOP, 0, "stop"},

    {{XK_Left, Mod1Mask}, {"prev", 0}, OXINE_KEY_PREV, 0, NULL},
    {{XK_Page_Up, ControlMask}, {"prev", 0}, OXINE_KEY_PREV, 0, "previous title/ chapter"},
    {{XK_p, 0}, {"prev", 0}, OXINE_KEY_PREV, 0, "previous title/ chapter"},
    {{XK_P, 0}, {"prev", 0}, OXINE_KEY_PREV, 0, "previous title/ chapter"},
    {{XF86XK_AudioPrev, 0}, {"prev", 0}, OXINE_KEY_PREV, 0, "previous title/ chapter"},

    {{XK_Right, Mod1Mask}, {"next", 0}, OXINE_KEY_NEXT, 0, NULL},
    {{XK_Page_Down, ControlMask}, {"next", 0}, OXINE_KEY_NEXT, 0, "next title/ chapter"},
    {{XK_n, 0}, {"next", 0}, OXINE_KEY_NEXT, 0, "next title/ chapter"},
    {{XK_N, 0}, {"next", 0}, OXINE_KEY_NEXT, 0, "next title/ chapter"},
    {{XF86XK_AudioNext, 0}, {"next", 0}, OXINE_KEY_NEXT, 0, "next title/ chapter"},

    {{XK_Left, ControlMask | Mod1Mask}, {"skip", 2}, OXINE_KEY_SKIP, -30, "skip backward 30 secs"},
    {{XK_Right, ControlMask | Mod1Mask}, {"skip", 2}, OXINE_KEY_SKIP, +30, "skip foreward 30 secs"},

    {{XK_Left, ControlMask}, {"fast_rewind", 0}, OXINE_KEY_FASTRWD, 0, "fast rewind"},
    {{XK_Right, ControlMask}, {"fast_forward", 0}, OXINE_KEY_FASTFWD, 0, "fast foreward"},

    {{XK_Up, ControlMask}, {"speed", 0}, OXINE_KEY_SPEED, +1, "increase playback speed"},
    {{XK_Down, ControlMask}, {"speed", 0}, OXINE_KEY_SPEED, -1, "decrease playback speed"},

    {{XK_e, 0}, {"eject", 0}, OXINE_KEY_EJECT, 0, NULL},
    {{XK_E, 0}, {"eject", 0}, OXINE_KEY_EJECT, 0, NULL},
    {{XF86XK_Eject, 0}, {"eject", 0}, OXINE_KEY_EJECT, 0, NULL},

    {{XK_a, 0}, {"toggle_aspect_ratio", 0}, OXINE_KEY_ASPECT_RATIO, 0, NULL},
    {{XK_A, 0}, {"toggle_aspect_ratio", 0}, OXINE_KEY_ASPECT_RATIO, 0, NULL},

    {{XK_i, 0}, {"toggle_deinterlace", 0}, OXINE_KEY_DEINTERLACE, 0, NULL},
    {{XK_I, 0}, {"toggle_deinterlace", 0}, OXINE_KEY_DEINTERLACE, 0, NULL},

    {{0, 0}, {"saturation", 2}, OXINE_KEY_SATURATION, +500, NULL},
    {{0, 0}, {"saturation", 2}, OXINE_KEY_SATURATION, -500, NULL},
    {{0, 0}, {"brightness", 2}, OXINE_KEY_BRIGHTNESS, +500, NULL},
    {{0, 0}, {"brightness", 2}, OXINE_KEY_BRIGHTNESS, -500, NULL},
    {{0, 0}, {"contrast", 2}, OXINE_KEY_CONTRAST, +500, NULL},
    {{0, 0}, {"contrast", 2}, OXINE_KEY_CONTRAST, -500, NULL},
    {{0, 0}, {"hue", 2}, OXINE_KEY_HUE, +500, NULL},
    {{0, 0}, {"hue", 2}, OXINE_KEY_HUE, -500, NULL},

    /* Navigation */
    {{XK_o, 0}, {"show_osd", 0}, OXINE_KEY_SHOW_OSD, 0, "show OSD"},
    {{XK_O, 0}, {"show_osd", 0}, OXINE_KEY_SHOW_OSD, 0, "show OSD"},

    {{XK_Left, 0}, {"left", 2}, OXINE_KEY_LEFT, 0, NULL},
    {{XK_Right, 0}, {"right", 2}, OXINE_KEY_RIGHT, 0, NULL},
    {{XK_Up, 0}, {"up", 2}, OXINE_KEY_UP, 0, NULL},
    {{XK_Down, 0}, {"down", 2}, OXINE_KEY_DOWN, 0, NULL},

    {{XK_Home, 0}, {"first", 0}, OXINE_KEY_FIRST, 0, NULL},
    {{XK_End, 0}, {"last", 0}, OXINE_KEY_LAST, 0, NULL},

    {{XK_Page_Up, 0}, {"pageup", 0}, OXINE_KEY_PAGE_UP, 0, NULL},
    {{XK_Page_Down, 0}, {"pagedown", 0}, OXINE_KEY_PAGE_DOWN, 0, NULL},

    {{XK_space, 0}, {"select", 0}, OXINE_KEY_SELECT, 0, NULL},
    {{XK_space, ControlMask}, {"select", 0}, OXINE_KEY_SELECT, 0, NULL},

    {{XK_Return, 0}, {"activate", 0}, OXINE_KEY_ACTIVATE, 0, NULL},
    {{XK_Down, Mod1Mask}, {"activate", 0}, OXINE_KEY_ACTIVATE, 0, NULL},

    {{0xFE20, 0}, {"prev_widget", 0}, OXINE_KEY_PREV_WIDGET, 0, "previous widget"},
    {{XK_Tab, 0}, {"next_widget", 0}, OXINE_KEY_NEXT_WIDGET, 0, "next widget"},

    {{XK_Up, Mod1Mask}, {"back", 0}, OXINE_KEY_BACK, 0, NULL},
    {{XK_Escape, 0}, {"back", 0}, OXINE_KEY_BACK, 0, NULL},
    {{XK_BackSpace, 0}, {"back", 0}, OXINE_KEY_BACK, 0, NULL},

    {{XK_Home, Mod1Mask}, {"home", 0}, OXINE_KEY_HOME, 0, NULL},

    /* Misc */
    {{XK_f, 0}, {"toggle_fullscreen", 0}, OXINE_KEY_FULLSCREEN, 0, NULL},
    {{XK_F, 0}, {"toggle_fullscreen", 0}, OXINE_KEY_FULLSCREEN, 0, NULL},

    {{XK_M, 0}, {"toggle_playmode", 0}, OXINE_KEY_PLAYMODE, 0, NULL},

    {{XK_q, 0}, {"shutdown", 0}, OXINE_KEY_SHUTDOWN, 0, NULL},
    {{XK_q, ControlMask}, {"quit", 0}, OXINE_KEY_QUIT, 0, NULL},

    {{XK_Delete, 0}, {"remove", 0}, OXINE_KEY_REMOVE, 0, "remove"},
    {{XK_Insert, 0}, {"insert", 0}, OXINE_KEY_INSERT, 0, "insert"},
    {{XK_s, ControlMask}, {"save", 0}, OXINE_KEY_SAVE, 0, "save"},
    {{XK_c, ControlMask}, {"copy", 0}, OXINE_KEY_COPY, 0, "copy"},
    {{XK_x, ControlMask}, {"cut", 0}, OXINE_KEY_CUT, 0, "cut"},
    {{XK_v, ControlMask}, {"paste", 0}, OXINE_KEY_PASTE, 0, "paste"},

    {{XK_r, ControlMask}, {"rotate_right", 0}, OXINE_KEY_ROTATE_RIGHT, +90, "rotate clockwise"},
    {{XK_l, ControlMask}, {"rotate_left", 0}, OXINE_KEY_ROTATE_LEFT, -90, "rotate counter-clockwise"},

    {{XF86XK_AudioRecord, 0}, {"record", 0}, OXINE_KEY_RECORD, 0, NULL},

    /* When viewing an image or playing television the events MENU5, MENU6,
     * MENU7 and MENU8 are mapped to the events RED, GREEN, YELLOW and BLUE. */
    {{0, 0}, {"red", 2}, OXINE_KEY_RED, 0, NULL},
    {{0, 0}, {"green", 2}, OXINE_KEY_GREEN, 0, NULL},
    {{0, 0}, {"yellow", 2}, OXINE_KEY_YELLOW, 0, NULL},
    {{0, 0}, {"blue", 2}, OXINE_KEY_BLUE, 0, NULL},

    /* The End */
    {{0, 0}, {0, 0}, 0, 0, NULL}
};


#ifndef CREATE_LIRCRC
int
odk_keymap_kbd2action (oxine_event_t * event, KeySym kbd_key,
                       int kbd_modifier)
{
    odk_keymap_entry_t *entry = &default_keymap_table[0];

#ifdef DEBUG_MODIFIERS
    if (kbd_modifier & ControlMask)
        debug ("detected modifier: CTRL");
    if (kbd_modifier & Mod1Mask)
        debug ("detected modifier: MOD1");
    if (kbd_modifier & Mod2Mask)
        debug ("detected modifier: MOD2");
    if (kbd_modifier & Mod3Mask)
        debug ("detected modifier: MOD3");
#endif

    /* We only care about Control (Control_L) and Mod1 (Meta_L Meta_R) */
    int kbd_modifier_saved = kbd_modifier;
    kbd_modifier &= ControlMask | Mod1Mask;

    event->source.key = OXINE_KEY_NULL;
    event->how = 0;
    while (entry->key_id != 0) {
        if (entry->keyboard.modifier != kbd_modifier) {
            /* Do nothing */
        }
        else if (entry->keyboard.key != kbd_key) {
            /* Do nothing */
        }
        else {
            event->source.key = entry->key_id;
            event->how = entry->how;
            event->data.keyboard.key = kbd_key;
            event->data.keyboard.modifier = kbd_modifier_saved;
#ifdef DEBUG
            if (entry->description) {
                debug ("command: %s", entry->description);
            }
            else if (entry->lirc.config) {
                debug ("command: %s", entry->lirc.config);
            }
            if (event->how != 0) {
                debug ("    how: %+d", event->how);
            }
#endif
            return 1;
        }
        entry++;
    }

    return 0;
}


int
odk_keymap_lirc2action (oxine_event_t * event, char *lirc_config)
{
    odk_keymap_entry_t *entry = &default_keymap_table[0];

    char *config = ho_strdup (lirc_config);
    char *plus = index (config, '+');
    char *minus = index (config, '-');

    int how = INT_MAX;
    if (plus) {
        if (plus[1] != '\0') {
            how = atoi (plus);
        }
        else {
            how = INT_MAX;
        }
    }
    if (minus) {
        if (minus[1] != '\0') {
            how = atoi (minus);
        }
        else {
            how = INT_MAX;
        }
    }

    event->source.key = OXINE_KEY_NULL;
    event->how = 0;
    while (entry->key_id != 0) {
        int len = strlen (entry->lirc.config);
        if (!entry->lirc.config) {
            /* If the current entry does not have a LIRC config we can forget
             * about it. */
        }
        else if (plus && (entry->how <= 0)) {
            /* If the config we got from the LIRC client had a plus at the end
             * but the current entry does not have a positive 'how' value we
             * forget about it. */
        }
        else if (minus && (entry->how >= 0)) {
            /* If the config we got from the LIRC client had a minus at the
             * end but the current entry does not have a negative 'how' value
             * we forget about it. */
        }
        else if (strncasecmp (entry->lirc.config, config, len) == 0) {
            /* If the config we got from the LIRC client only had a plus or a
             * minus without a number behind it, we use the 'how' value of
             * this entry. */
            if (how == INT_MAX) {
                event->how = entry->how;
            }
            /* Otherwise we use the how value that the config brought along. */
            else {
                event->how = how;
            }

            event->source.key = entry->key_id;

#ifdef DEBUG
            if (entry->description) {
                debug ("command: %s", entry->description);
            }
            else if (entry->lirc.config) {
                debug ("command: %s", entry->lirc.config);
            }
            if (event->how != 0) {
                debug ("    how: %+d", event->how);
            }
#endif

            /* This is neccessary because the standard behaviour of the list
             * is, that on a select without Ctrl only one item is selected. As
             * people using a remote control should also be able to select
             * multiple entries in a list, we simulate the Ctrl modifier. */
            if (event->source.key == OXINE_KEY_SELECT) {
                event->data.keyboard.modifier = ControlMask;
            }
            ho_free (config);
            return 1;
        }
        entry++;
    }

    ho_free (config);

    return 0;
}

#else /* CREATE_LIRCRC */

int
main (int argc, char **argv)
{
    odk_keymap_entry_t *entry = &default_keymap_table[0];

    if (argc != 2) {
        printf ("Usage: %s <lircrc-filename>\n", argv[0]);
        return 1;
    }

    FILE *f = fopen (argv[1], "w");
    if (!f) {
        printf ("Usage: %s <lircrc-filename>\n", argv[0]);
        return 1;
    }

    char *last_config = "";
    int last_how = 0;

    while (entry->key_id != 0) {
        if (entry->lirc.config) {
            if ((last_how == entry->how)
                && (strcmp (last_config, entry->lirc.config) == 0)) {
                /* Ignore it. */
            }
            else {
                fprintf (f, "begin\n");
                fprintf (f, "    prog   = oxine\n");

                if (entry->how > 0) {
                    fprintf (f, "    button = %s+\n", entry->lirc.config);
                }
                else if (entry->how < 0) {
                    fprintf (f, "    button = %s-\n", entry->lirc.config);
                }
                else {
                    fprintf (f, "    button = %s\n", entry->lirc.config);
                }

                if (entry->how != 0) {
                    fprintf (f, "    config = %s%+d\n", entry->lirc.config,
                             entry->how);
                }
                else {
                    fprintf (f, "    config = %s\n", entry->lirc.config);
                }

                if (entry->lirc.repeat) {
                    fprintf (f, "    repeat = %d\n", entry->lirc.repeat);
                }

                fprintf (f, "end\n");

                last_config = entry->lirc.config;
                last_how = entry->how;
            }
        }
        entry++;
    }

    fclose (f);

    return 0;
}

#endif /* CREATE_LIRCRC */
