
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Thanks goes to Florian Echtler for adding Xinerama support.
 *
 * $Id: odk_x11.c 2664 2007-08-22 17:24:17Z mschwerin $
 *
 */

#include "config.h"

#ifdef HAVE_LIBX11

#include <assert.h>
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <xine.h>

#include <sys/types.h>

#include <X11/Xlib.h>
#include <X11/cursorfont.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/Xatom.h>

#ifdef HAVE_XTEST
#include <X11/extensions/XTest.h>
#endif

#ifdef HAVE_XINERAMA
#include <X11/extensions/Xinerama.h>
#endif

/* If buttons 6 and 7 are not defined in X.h we define them here. */
#ifndef Button6
#define Button6     6
#endif
#ifndef Button7
#define Button7     7
#endif

#include "event.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "odk_keymap.h"
#include "odk_plugin.h"
#include "odk_private.h"
#include "scheduler.h"
#include "utils.h"

#define WINDOW_TITLE                PACKAGE_NAME" version "PACKAGE_VERSION

#define SCREENSAVER_TIMEOUT         (1000 * 60 * 2)
#define MOUSECURSOR_TIMEOUT         (1000 * 10)

typedef struct {
    odk_window_t window_class;
    x11_visual_t visual;

    bool show_video;
    bool keep_going;
    bool wm_is_ewmh_compliant;
    bool is_fullscreen;

    bool with_border;
    bool stay_on_top;

    /* The event handler. */
    odk_cb event_handler;
    void *event_handler_data;

    /* The X display and screen number */
    Display *display;
    Window rootwindow;

    int screen;
    int xinerama_screen;

    /* The output window. */
    Window window;

    /* The pixel aspect ratio of the monitor. */
    double monitor_pixel_aspect;

    /* The real width and height of the display. */
    int display_width;
    int display_height;

    /* The position and size of the window. */
    int window_xpos;
    int window_ypos;

    int window_width;
    int window_height;

    /* The saved position and size of the non-fullscreen window. */
    int saved_xpos;
    int saved_ypos;

    int saved_width;
    int saved_height;

    /* Stuff needed to hide the mouse cursor. */
    Cursor no_cursor;
    Cursor on_cursor;
    int mousecursor_job;

    /* Stuff needed to disable the screensaver. */
    bool have_xtest;
    int xtest_keycode;

    bool keep_screensaver;
    int screensaver_job;
} x11_window_t;

static Atom XA_WIN_LAYER = None;
static Atom XA_DEL_WINDOW = None;

#define MWM_HINTS_DECORATIONS   (1L << 1)
#define PROP_MWM_HINTS_ELEMENTS 5
typedef struct {
    uint32_t flags;
    uint32_t functions;
    uint32_t decorations;
    int32_t input_mode;
    uint32_t status;
} MWMHints;


/*
 * ***************************************************************************
 * Window manager stuff
 * ***************************************************************************
 */
static int
x11_error_eater (Display * display, XErrorEvent * xevent)
{
    return 0;
}


static void
wm_check_ewmh_compliancy (x11_window_t * win)
{
    win->wm_is_ewmh_compliant = 0;

    Atom atom0 = XInternAtom (win->display, "_NET_SUPPORTING_WM_CHECK", True);
    Atom atom1 = XInternAtom (win->display, "_NET_WORKAREA", True);

    /* Check for Extended Window Manager Hints (EWMH) Compliant */
    if ((atom0 != None) && (atom1 != None)) {
        unsigned char *prop_return = NULL;
        unsigned long nitems_return;
        unsigned long bytes_after_return;
        Atom type_return;
        int format_return;
        Status status;

        /* We ignore errors because some windows managers have problems with
         * the following code. */
        int (*eh) (Display *, XErrorEvent *);
        eh = XSetErrorHandler (x11_error_eater);

        if (XGetWindowProperty (win->display, win->rootwindow, atom0,
                                0, 1, False, XA_WINDOW, &type_return,
                                &format_return, &nitems_return,
                                &bytes_after_return,
                                &prop_return) == Success) {

            if ((type_return != None) && (type_return == XA_WINDOW) &&
                (format_return == 32) && (nitems_return == 1)
                && (bytes_after_return == 0)) {
                unsigned char *prop_return2 = NULL;
                Window win_id;
                win_id = *(long *) prop_return;

                status = XGetWindowProperty (win->display, win_id, atom0, 0,
                                             1, False, XA_WINDOW,
                                             &type_return, &format_return,
                                             &nitems_return,
                                             &bytes_after_return,
                                             &prop_return2);

                if ((status == Success) && (type_return != None)
                    && (type_return == XA_WINDOW) && (format_return == 32)
                    && (nitems_return == 1) && (bytes_after_return == 0)) {

                    if (win_id == *(long *) prop_return) {
                        info (_("The current window manager supports "
                                "'Extended Window Manager Hints'."));
                        win->wm_is_ewmh_compliant = 1;
                    }
                }

                if (prop_return2)
                    XFree (prop_return2);
            }

            if (prop_return)
                XFree (prop_return);
        }

        /* Unset the error eater. */
        XSetErrorHandler (eh);
    }
}


static void
wm_set_state (x11_window_t * win, Atom atom, bool enable)
{
    if (!win->wm_is_ewmh_compliant) {
        return;
    }

    Atom prop = XInternAtom (win->display, "_NET_WM_STATE", False);

    XEvent xev;
    xev.xclient.type = ClientMessage;
    xev.xclient.message_type = prop;
    xev.xclient.display = win->display;
    xev.xclient.window = win->window;
    xev.xclient.format = 32;
    xev.xclient.data.l[0] = enable ? 1 : 0;
    xev.xclient.data.l[1] = atom;
    xev.xclient.data.l[2] = 0l;
    xev.xclient.data.l[3] = 0l;
    xev.xclient.data.l[4] = 0l;

    XSendEvent (win->display, win->rootwindow, True,
                SubstructureRedirectMask, &xev);
}


static void
wm_set_above (x11_window_t * win, bool enable)
{
    Atom prop = XInternAtom (win->display, "_NET_WM_STATE_ABOVE", False);

    wm_set_state (win, prop, enable);
}


static void
wm_set_fullscreen (x11_window_t * win, bool enable)
{
    Atom prop = XInternAtom (win->display, "_NET_WM_STATE_FULLSCREEN", False);

    wm_set_state (win, prop, enable);
}


static void
wm_set_border (x11_window_t * win, bool enable)
{
    Atom prop = XInternAtom (win->display, "_MOTIF_WM_HINTS", False);

    MWMHints mwmhints;
    mwmhints.flags = MWM_HINTS_DECORATIONS;
    mwmhints.decorations = enable;

    XChangeProperty (win->display, win->window, prop, prop, 32,
                     PropModeReplace, (unsigned char *) &mwmhints,
                     PROP_MWM_HINTS_ELEMENTS);
}


/*
 * ***************************************************************************
 * xine-lib callbacks
 * ***************************************************************************
 */
static void
dest_size_cb (void *x11_win_p,
              int video_width, int video_height,
              double video_pixel_aspect,
              int *dest_width, int *dest_height, double *dest_pixel_aspect)
{
    x11_window_t *win = (x11_window_t *) x11_win_p;
    assert (win);

    if (win->show_video) {
        *dest_width = win->window_width;
        *dest_height = win->window_height;
    }
    else {
        *dest_width = 1;
        *dest_height = 1;
    }

    *dest_pixel_aspect = win->monitor_pixel_aspect;
}


static void
frame_output_cb (void *x11_win_p,
                 int video_width, int video_height,
                 double video_pixel_aspect,
                 int *dest_x, int *dest_y,
                 int *dest_width, int *dest_height,
                 double *dest_pixel_aspect, int *win_x, int *win_y)
{
    x11_window_t *win = (x11_window_t *) x11_win_p;
    assert (win);

    *win_x = win->window_xpos;
    *win_y = win->window_ypos;

    *dest_x = 0;
    *dest_y = 0;

    if (win->show_video) {
        *dest_width = win->window_width;
        *dest_height = win->window_height;
    }
    else {
        *dest_width = 1;
        *dest_height = 1;
    }

    *dest_pixel_aspect = win->monitor_pixel_aspect;
}


/*
 * Connect to the X server.
 * returns 1 on success, 0 on error
 */
static int
connect_x11_server (x11_window_t * win)
{
    if (!XInitThreads ()) {
        error (_("Could not initialize X threads!"));
        return 0;
    }

    /* Try to open the X display. */
    win->display = XOpenDisplay (NULL);
    if (!win->display) {
        error (_("Could not open display!"));
        return 0;
    }

    XLockDisplay (win->display);

    /* Get the default screen. */
    win->screen = DefaultScreen (win->display);
    win->rootwindow = RootWindow (win->display, win->screen);

    win->display_width = DisplayWidth (win->display, win->screen);
    win->display_height = DisplayHeight (win->display, win->screen);

    /* Get information about monitor and screen sizes. */
    int dw = DisplayWidth (win->display, win->screen);
    int dh = DisplayHeight (win->display, win->screen);
    int dwmm = DisplayWidthMM (win->display, win->screen);
    int dhmm = DisplayHeightMM (win->display, win->screen);
    debug ("display dimensions: %dx%d (%dx%d mm)", dw, dh, dwmm, dhmm);

    /* Calculate the horizontal and vertical resolution. */
    double res_h = (double) dw / (double) dwmm;
    double res_v = (double) dh / (double) dhmm;
    debug ("display resolution: %dx%d dots per inch",
           (int) round (res_h * 25.4), (int) round (res_v * 25.4));

    /* Determine display aspect ratio. */
    win->monitor_pixel_aspect = res_v / res_h;

    /* If we're using xinerama, we assume square pixels. */
#ifdef HAVE_XINERAMA
    int event_base;
    int error_base;

    int screen_num;
    XineramaScreenInfo *screeninfo;

    if (XineramaQueryExtension (win->display, &event_base, &error_base)
        && (screeninfo = XineramaQueryScreens (win->display, &screen_num))
        && (XineramaIsActive (win->display))) {

        info (_("Found %d Xinerama screens:"), screen_num);
        int i = 0;
        for (; i < screen_num; i++) {
            info ("    screen[%2d]: %dx%d, +%d+%d", i,
                  screeninfo[i].width, screeninfo[i].height,
                  screeninfo[i].x_org, screeninfo[i].y_org);
        }

        if (screen_num > 1)
            win->monitor_pixel_aspect = 1.0;
    }
#endif

    debug ("calculated pixel aspect ratio: %.02f", win->monitor_pixel_aspect);

    XA_DEL_WINDOW = XInternAtom (win->display, "WM_DELETE_WINDOW", False);
    XA_WIN_LAYER = XInternAtom (win->display, "_WIN_LAYER", False);

    /* Check for EWMH compliancy. */
    wm_check_ewmh_compliancy (win);

    XUnlockDisplay (win->display);

    return 1;
}


/*
 * Close the connection to the X server.
 */
static int
disconnect_x11_server (x11_window_t * win)
{

    if (win->display) {
        XLockDisplay (win->display);
        XCloseDisplay (win->display);
    }

    win->display = NULL;
    win->screen = 0;
    win->monitor_pixel_aspect = 0;

    return 1;
}


/*
 * Job to hide the mouse cursor.
 */
static void
hide_mouse_cursor (void *x11_win_p)
{
    x11_window_t *win = (x11_window_t *) x11_win_p;

    XLockDisplay (win->display);
    XDefineCursor (win->display, win->window, win->no_cursor);
    XFlush (win->display);
    XUnlockDisplay (win->display);
}


/*
 * Shows the mouse cursor.
 */
static void
show_mouse_cursor (void *x11_win_p)
{
    x11_window_t *win = (x11_window_t *) x11_win_p;

    XLockDisplay (win->display);
    XDefineCursor (win->display, win->window, win->on_cursor);
    XFlush (win->display);
    XUnlockDisplay (win->display);

    cancel_job (win->mousecursor_job);
    win->mousecursor_job = schedule_job (MOUSECURSOR_TIMEOUT,
                                         hide_mouse_cursor, win);
}


/*
 * Job to disable the screensaver.
 */
static void
screensaver_job (void *x11_win_p)
{
    x11_window_t *win = (x11_window_t *) x11_win_p;

    XLockDisplay (win->display);
#ifdef HAVE_XTEST
    if (win->have_xtest == True) {
        XTestFakeKeyEvent (win->display, win->xtest_keycode,
                           True, CurrentTime);
        XTestFakeKeyEvent (win->display, win->xtest_keycode,
                           False, CurrentTime);
        XSync (win->display, False);
    }
    else
#endif
    {
        XResetScreenSaver (win->display);
    }
    XUnlockDisplay (win->display);

    execute_shell ("gnome-screensaver-command --poke", 0);

    win->screensaver_job = schedule_job (SCREENSAVER_TIMEOUT,
                                         screensaver_job, win);
}


/**
 * Creates a new X window.
 *
 * @param x                     The initial x offset.
 * @param y                     The initial y offset.
 * @param w                     The initial width.
 * @param h                     The initial height.
 */
static bool
create_x11_window (x11_window_t * win, int x, int y, int w, int h)
{
    win->window_xpos = x;
    win->window_ypos = x;
    win->window_width = w;
    win->window_height = h;

    XLockDisplay (win->display);

    /* Create the window */
    unsigned long black = BlackPixel (win->display, win->screen);
    win->window = XCreateSimpleWindow (win->display, win->rootwindow,
                                       win->window_xpos, win->window_ypos,
                                       win->window_width, win->window_height,
                                       0, black, black);
    if (!win->window) {
        XUnlockDisplay (win->display);
        return false;
    }

    /* Set window title. */
    XmbSetWMProperties (win->display, win->window,
                        WINDOW_TITLE, WINDOW_TITLE,
                        NULL, 0, NULL, NULL, NULL);

    /* Set window manager properties. */
    XSizeHints hint;
    hint.flags = PWinGravity | PPosition | PSize | USSize | USPosition;
    hint.x = 0;
    hint.y = 0;
    hint.width = win->window_width;
    hint.height = win->window_height;
    hint.win_gravity = StaticGravity;
    XSetWMNormalHints (win->display, win->window, &hint);

    /* Set the window layer. */
    long propvalue[1];
    propvalue[0] = 10;
    XChangeProperty (win->display, win->window,
                     XA_WIN_LAYER, XA_CARDINAL, 32, PropModeReplace,
                     (unsigned char *) propvalue, 1);

    /* Set the WM_TRANSIENT_FOR property. */
    XSetTransientForHint (win->display, win->window, None);

    /* Set the input masks */
    XSelectInput (win->display, win->window,
                  StructureNotifyMask | ExposureMask |
                  ButtonPressMask | PointerMotionMask | KeyPressMask);

    /* Set up the window delete message */
    XSetWMProtocols (win->display, win->window, &XA_DEL_WINDOW, 1);

    /* Create the mouse cursors (on and off) */
    static char bm_no_data[] = { 0, 0, 0, 0, 0, 0, 0, 0 };
    Pixmap bm_no = XCreateBitmapFromData (win->display, win->window,
                                          bm_no_data, 8, 8);
    win->no_cursor = XCreatePixmapCursor (win->display, bm_no, bm_no,
                                          (XColor *) & black,
                                          (XColor *) & black, 0, 0);
    win->on_cursor = XCreateFontCursor (win->display, XC_left_ptr);

    XFlush (win->display);
    XUnlockDisplay (win->display);

    win->visual.display = win->display;
    win->visual.screen = win->screen;
    win->visual.d = win->window;
    win->visual.dest_size_cb = dest_size_cb;
    win->visual.frame_output_cb = frame_output_cb;
    win->visual.user_data = win;

    show_mouse_cursor (win);

    return 1;
}


static int
destroy_x11_window (x11_window_t * win)
{
    XLockDisplay (win->display);
    XDestroyWindow (win->display, win->window);
    XUnlockDisplay (win->display);

    win->visual.display = NULL;
    win->visual.screen = 0;
    win->visual.dest_size_cb = NULL;
    win->visual.frame_output_cb = NULL;
    win->visual.user_data = NULL;

    return 1;
}


static void
move_resize_window (odk_window_t * this, int x, int y, int width, int height)
{
    x11_window_t *win = (x11_window_t *) this;

    if (!win->is_fullscreen) {
        if (x > 0) {
            win->window_xpos = x;
        }
        if (y > 0) {
            win->window_ypos = y;
        }
        if (width > 0) {
            win->window_width = width;
        }
        if (height > 0) {
            win->window_height = height;
        }

        XLockDisplay (win->display);
        XMoveResizeWindow (win->display, win->window,
                           win->window_xpos, win->window_ypos,
                           win->window_width, win->window_height);
        XFlush (win->display);
        XUnlockDisplay (win->display);

        /* Send size change event to xine. */
        xine_port_send_gui_data (win->window_class.video_port,
                                 XINE_GUI_SEND_DRAWABLE_CHANGED,
                                 (void *) win->window);
    }

    else {
        if (x > 0) {
            win->saved_xpos = x;
        }
        if (y > 0) {
            win->saved_ypos = y;
        }
        if (width > 0) {
            win->saved_width = width;
        }
        if (height > 0) {
            win->saved_height = height;
        }
    }
}


static void
move_window (odk_window_t * this, int x, int y)
{
    x11_window_t *win = (x11_window_t *) this;

    int width = win->window_width;
    int height = win->window_height;

    move_resize_window (this, x, y, width, height);
}


static void
resize_window (odk_window_t * this, int width, int height)
{
    x11_window_t *win = (x11_window_t *) this;

    int x = win->window_xpos;
    int y = win->window_ypos;

    move_resize_window (this, x, y, width, height);
}


static int
get_window_width (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;

    return win->window_width;
}


static int
get_window_height (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;

    return win->window_height;
}


static int
get_window_id (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;

    return win->window;
}


static void
show_video (odk_window_t * this, bool show)
{
    x11_window_t *win = (x11_window_t *) this;

    win->show_video = show;
}


static bool
is_fullscreen (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;

    return win->is_fullscreen;
}


static bool
is_viewable (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;

    XWindowAttributes wa;
    XGetWindowAttributes (win->display, win->window, &wa);

    return (wa.map_state == IsViewable);
}


static void
set_fullscreen (odk_window_t * this, bool want_fullscreen)
{
    x11_window_t *win = (x11_window_t *) this;

    if (win->is_fullscreen == want_fullscreen) {
        return;
    }

    XLockDisplay (win->display);

    if (want_fullscreen) {
        /* Save size and position of non-fullscreen window. */
        win->saved_xpos = win->window_xpos;
        win->saved_ypos = win->window_ypos;
        win->saved_width = win->window_width;
        win->saved_height = win->window_height;

        /* Set size and position of the output window. */
#ifdef HAVE_XINERAMA
        int event_base;
        int error_base;
        int screen_use = 0;
        int screen_num = 0;
        XineramaScreenInfo *screeninfo;

        if (XineramaQueryExtension (win->display, &event_base, &error_base)
            && (screeninfo = XineramaQueryScreens (win->display, &screen_num))
            && (XineramaIsActive (win->display))) {

            if (win->xinerama_screen < screen_num) {
                screen_use = win->xinerama_screen;
            }

            win->window_xpos = screeninfo[screen_use].x_org;
            win->window_ypos = screeninfo[screen_use].y_org;
            win->window_width = screeninfo[screen_use].width;
            win->window_height = screeninfo[screen_use].height;
        }
        else
#endif
        {
            win->window_xpos = 0;
            win->window_ypos = 0;
            win->window_width = win->display_width;
            win->window_height = win->display_height;
        }
    }

    else {
        /* Set size and position of the output window. */
        win->window_xpos = win->saved_xpos;
        win->window_ypos = win->saved_ypos;
        win->window_width = win->saved_width;
        win->window_height = win->saved_height;
    }

    /* Go to fullscreen the modern (e.g. metacity) way. */
    wm_set_fullscreen (win, want_fullscreen);

    /* Set window stacking. */
    wm_set_above (win, want_fullscreen || win->stay_on_top);

    /* Set window border. */
    wm_set_border (win, !want_fullscreen && win->with_border);

    /* Set the input masks */
    if (want_fullscreen) {
        XSelectInput (win->display, win->window,
                      ExposureMask | KeyPressMask | ButtonPressMask |
                      StructureNotifyMask | FocusChangeMask |
                      PointerMotionMask);
    }
    else {
        XSelectInput (win->display, win->window,
                      ExposureMask | KeyPressMask | ButtonPressMask |
                      StructureNotifyMask | PointerMotionMask);
    }

    /* Resize and move the window. */
    XMoveResizeWindow (win->display, win->window,
                       win->window_xpos, win->window_ypos,
                       win->window_width, win->window_height);

    /* Raise the window */
    XRaiseWindow (win->display, win->window);

    /* Set the input focus */
    if (is_viewable (this)) {
        XSetInputFocus (win->display, win->window, RevertToNone, CurrentTime);
    }

    XFlush (win->display);
    XUnlockDisplay (win->display);

    /* Send size change event to xine. */
    xine_port_send_gui_data (win->window_class.video_port,
                             XINE_GUI_SEND_DRAWABLE_CHANGED,
                             (void *) win->window);

    show_mouse_cursor (win);

    win->is_fullscreen = want_fullscreen;
}


static void
set_with_border (odk_window_t * this, bool with_border)
{
    x11_window_t *win = (x11_window_t *) this;

    win->with_border = with_border;

    if (!win->is_fullscreen) {
        XLockDisplay (win->display);
        wm_set_border (win, with_border);
        XFlush (win->display);
        XUnlockDisplay (win->display);

        /* Send size change event to xine. */
        xine_port_send_gui_data (win->window_class.video_port,
                                 XINE_GUI_SEND_DRAWABLE_CHANGED,
                                 (void *) win->window);
    }
}


static void
set_stay_on_top (odk_window_t * this, bool stay_on_top)
{
    x11_window_t *win = (x11_window_t *) this;

    win->stay_on_top = stay_on_top;

    if (!win->is_fullscreen) {
        XLockDisplay (win->display);
        wm_set_above (win, stay_on_top);
        XFlush (win->display);
        XUnlockDisplay (win->display);
    }
}


static void
handle_event (x11_window_t * win, XEvent * event)
{
    if (!win->event_handler) {
        return;
    }

    switch (event->type) {
    case Expose:
        if (event->xexpose.count == 0) {
            xine_port_send_gui_data (win->window_class.video_port,
                                     XINE_GUI_SEND_EXPOSE_EVENT, event);
        }
        break;
    case ConfigureNotify:
        if (event->xany.window == win->window) {
            XConfigureEvent *cevent = (XConfigureEvent *) event;

            debug ("window size has changed +%d+%d %dx%d",
                   cevent->x, cevent->y, cevent->width, cevent->height);

            /* We update the display width and height to keep track of changes
             * in the resolution. Otherwise changing the resolution and going
             * to fullscreen afterwards leads to errors. */
            win->display_width = DisplayWidth (win->display, win->screen);
            win->display_height = DisplayHeight (win->display, win->screen);

            win->window_width = cevent->width;
            win->window_height = cevent->height;

            if ((cevent->x == 0) && (cevent->y == 0)) {
                Window tmp_win;
                XLockDisplay (cevent->display);
                XTranslateCoordinates (cevent->display, cevent->window,
                                       win->rootwindow, 0, 0,
                                       &win->window_xpos, &win->window_ypos,
                                       &tmp_win);
                XUnlockDisplay (cevent->display);
            }
            else {
                win->window_xpos = cevent->x;
                win->window_ypos = cevent->y;
            }

            oxine_event_t ev;
            ev.type = OXINE_EVENT_OUTPUT_FORMAT_CHANGED;
            win->event_handler (win->event_handler_data, &ev);
        }
        break;
    case ClientMessage:
        {
            XClientMessageEvent *cevent = (XClientMessageEvent *) event;

            /* We got a window deletion message from our window manager. */
            if (cevent->data.l[0] == XA_DEL_WINDOW) {
                win->keep_going = 0;
            }
        }
        break;
    case MotionNotify:
        {
            XMotionEvent *mevent = (XMotionEvent *) event;

            oxine_event_t ev;
            ev.type = OXINE_EVENT_MOTION;
            ev.data.mouse.pos.x = mevent->x;
            ev.data.mouse.pos.y = mevent->y;

            win->event_handler (win->event_handler_data, &ev);

            show_mouse_cursor (win);
        }
        break;
    case ButtonPress:
        {
            XButtonEvent *bevent = (XButtonEvent *) event;

            oxine_event_t ev;
            ev.type = OXINE_EVENT_BUTTON;
            ev.data.mouse.button.num = bevent->button;
            ev.data.mouse.button.modifier = bevent->state;
            ev.data.mouse.pos.x = bevent->x;
            ev.data.mouse.pos.y = bevent->y;
            switch (bevent->button) {
            case Button1:
                ev.source.button = OXINE_MOUSE_BUTTON_1;
                break;
            case Button2:
                ev.source.button = OXINE_MOUSE_BUTTON_2;
                break;
            case Button3:
                ev.source.button = OXINE_MOUSE_BUTTON_3;
                break;
            case Button4:
                ev.source.button = OXINE_MOUSE_BUTTON_4;
                break;
            case Button5:
                ev.source.button = OXINE_MOUSE_BUTTON_5;
                break;
            case Button6:
                ev.source.button = OXINE_MOUSE_BUTTON_6;
                break;
            case Button7:
                ev.source.button = OXINE_MOUSE_BUTTON_7;
                break;
            default:
                ev.source.button = OXINE_MOUSE_BUTTON_NULL;
                break;
            }
            win->event_handler (win->event_handler_data, &ev);

            show_mouse_cursor (win);
        }
        break;
    case KeyPress:
        {
            XKeyEvent *kevent = (XKeyEvent *) event;

            KeySym key;
            char kbuf[256];
            XLockDisplay (kevent->display);
            XLookupString (kevent, kbuf, sizeof (kbuf), &key, NULL);
            XUnlockDisplay (kevent->display);

            oxine_event_t ev;
            ev.type = OXINE_EVENT_KEY;
            if (odk_keymap_kbd2action (&ev, key, kevent->state)) {
                win->event_handler (win->event_handler_data, &ev);
            }

            if (((kbuf[0] != 0) && (kbuf[1] == 0))
                || (key == XK_Escape)
                || (key == XK_Delete)
                || (key == XK_BackSpace)
                || (key == XK_Left)
                || (key == XK_Right)
                || (key == XK_Home)
                || (key == XK_End)) {
                ev.source.key = OXINE_KEY_INPUT;
                ev.data.keyboard.key = key;
                ev.data.keyboard.modifier = kevent->state;
                win->event_handler (win->event_handler_data, &ev);
            }
        }
        break;
    default:
        break;
    }
}


static void
enter_event_loop (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;

    win->keep_going = true;
    while (win->keep_going) {
        XEvent event;
        XNextEvent (win->display, &event);
        handle_event (win, &event);
    }
}


static void
leave_event_loop (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;

    XEvent event;
    event.type = Expose;
    event.xexpose.display = win->display;
    event.xexpose.window = win->window;
    event.xexpose.x = 0;
    event.xexpose.y = 0;
    event.xexpose.width = 1;
    event.xexpose.height = 1;

    win->keep_going = false;

    XLockDisplay (win->display);
    XSendEvent (win->display, win->window, False, ExposureMask, &event);
    XFlush (win->display);
    XUnlockDisplay (win->display);
}


static void
set_event_handler (odk_window_t * this, odk_cb cb, void *data)
{
    x11_window_t *win = (x11_window_t *) this;
    win->event_handler = cb;
    win->event_handler_data = data;
}


static bool
show_window (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;

    XLockDisplay (win->display);
    XClearWindow (win->display, win->window);
    XMapWindow (win->display, win->window);
    XUnlockDisplay (win->display);

    return true;
}


static bool
hide_window (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;

    XLockDisplay (win->display);
    XUnmapWindow (win->display, win->window);
    XUnlockDisplay (win->display);

    return true;
}


static bool
free_window (odk_window_t * this)
{
    x11_window_t *win = (x11_window_t *) this;

    /* Stop the event loop. */
    if (win->keep_going) {
        leave_event_loop (this);
    }

    /* We need to lock the job mutex, because we modify 
     * the screensaver and the mouse cursor jobs. */
    lock_job_mutex ();
    cancel_job (win->screensaver_job);
    cancel_job (win->mousecursor_job);
    unlock_job_mutex ();

    /* Close the video port and destroy the window. */
    xine_close_video_driver (this->xine, this->video_port);
    this->video_port = NULL;
    destroy_x11_window (win);
    disconnect_x11_server (win);

    ho_free (win);

    return true;
}

odk_window_t *x11_construct (xine_t * xine, const char *driver);

/*
 * Constructor
 */
odk_window_t *
x11_construct (xine_t * xine, const char *driver)
{
    /* Create new window data */
    x11_window_t *win = ho_new (x11_window_t);

    double monitor_pixel_aspect = config_get_number ("gui.x11.monitor_"
                                                     "pixel_aspect");
    monitor_pixel_aspect /= 100;
    win->xinerama_screen = config_get_number ("gui.x11.xinerama_screen");

    /* Try to connect to the X11 server. */
    if (!connect_x11_server (win)) {
        ho_free (win);
        return NULL;
    }

    if (monitor_pixel_aspect) {
        info (_("Overriding calculated monitor pixel aspect ratio with "
                "user provided value: %.02f"), monitor_pixel_aspect);
        win->monitor_pixel_aspect = monitor_pixel_aspect;
    }

    /* Create our X11 output window. */
    create_x11_window (win, 0, 0, 1, 1);
    win->window_class.xine = xine;

    /* Create the xine video port. */
    {
        win->window_class.video_port =
            xine_open_video_driver (xine, driver, XINE_VISUAL_TYPE_X11,
                                    (void *) &(win->visual));
    }
    if (!win->window_class.video_port) {
        warn (_("Failed to open video port '%s', "
                "falling back to 'auto'."), driver);
        win->window_class.video_port =
            xine_open_video_driver (xine, NULL, XINE_VISUAL_TYPE_X11,
                                    (void *) &(win->visual));
    }
    if (!win->window_class.video_port) {
        error (_("Failed to open video port!"));
        destroy_x11_window (win);
        ho_free (win);
        return NULL;
    }

    /* Screensaver disabling stuff */
#ifdef HAVE_XTEST
    {
        int a;
        int b;
        int c;
        int d;
        win->have_xtest = XTestQueryExtension (win->display, &a, &b, &c, &d);
        if (win->have_xtest == True) {
            win->xtest_keycode = XKeysymToKeycode (win->display, XK_Shift_L);
        }
    }
#endif

    win->screensaver_job = schedule_job (SCREENSAVER_TIMEOUT,
                                         screensaver_job, win);
    win->mousecursor_job = schedule_job (MOUSECURSOR_TIMEOUT,
                                         hide_mouse_cursor, win);

    win->show_video = true;
    win->with_border = true;
    win->stay_on_top = false;

    win->window_class.show = show_window;
    win->window_class.hide = hide_window;
    win->window_class.free = free_window;
    win->window_class.show_video = show_video;

    win->window_class.set_fullscreen = set_fullscreen;
    win->window_class.is_fullscreen = is_fullscreen;

    win->window_class.set_stay_on_top = set_stay_on_top;
    win->window_class.set_with_border = set_with_border;

    win->window_class.enter_event_loop = enter_event_loop;
    win->window_class.leave_event_loop = leave_event_loop;
    win->window_class.set_event_handler = set_event_handler;

    win->window_class.get_width = get_window_width;
    win->window_class.get_height = get_window_height;
    win->window_class.get_window_id = get_window_id;

    win->window_class.move = move_window;
    win->window_class.resize = resize_window;
    win->window_class.move_resize = move_resize_window;

    return (odk_window_t *) win;
}

#endif
