
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: otk_list.c 2593 2007-07-24 22:21:44Z mschwerin $
 */
#include "config.h"

#include "heap.h"
#include "i18n.h"
#include "list.h"
#include "logger.h"
#include "odk.h"
#include "odk_xk.h"
#include "otk.h"
#include "oxine.h"
#include "utils.h"

extern oxine_t *oxine;

typedef struct otk_listentry_s otk_listentry_t;

typedef enum {
    OTK_LIST_TYPE_NORMAL,
    OTK_LIST_TYPE_MENU,
    OTK_LIST_TYPE_SELECTOR,
} otk_list_type_t;


typedef struct {
    otk_widget_t widget;

    /// The scrollbar associated with this list.
    otk_widget_t *scrollbar;
    otk_widget_t *button_up;
    otk_widget_t *button_down;

    /// The entries of this list (otk_listentry_t).
    l_list_t *entries;

    /// The last listentry that was selected alone.
    otk_listentry_t *last_single_selected;

    /// Is it necessary to adapt the list entries.
    bool need_adaption;

    /// What type of list is this.
    otk_list_type_t type;

    /// Number of selected entries.
    int num_selected;
    /// Number of visible entries.
    int num_visible;
    /// Number of the entry at the top of the visible list.
    int first_visible;

    /// How may the user select an item.
    otk_list_selection_t allow_select;

    int entry_x;
    int entry_y;
    int entry_height;
    int entry_width;
    int entry_spacing;

    /// Userdefined data that is sent in every callback.
    void *cb_data;

    /// Callback that is called whenever the visible entry changes.

    /**
     * This is only used by the selector widget.
     */
    otk_int_set_cb_t entry_changed_cb;

    /// User-defined data for the above callback.
    void *entry_changed_cb_data;
} otk_list_t;


struct otk_listentry_s {
    otk_widget_t widget;

    otk_list_t *list;

    bool is_first;
    bool is_last;
    bool is_selected;

    /// Text to display.
    char *text;

    /// Time this item was last clicked.
    time_t doubleclicktime;

    /**
     * This callback is called whenever the list receives an activation event.
     * Activation events are OXINE_KEY_ACTIVATE or a double-click
     * (OXINE_MOUSE_BUTTON_LEFT twice).
     */
    otk_cb_t activate_cb;

    /// User-defined data for the above callback.
    void *activate_cb_data;

    /**
     * This callback is called whenever the list receives a selection event.
     * Selection events are OXINE_KEY_SELECT or a single-click
     * (OXINE_MOUSE_BUTTON_LEFT once).
     */
    otk_cb_t select_cb;

    /// User-defined data for the above callback.
    void *select_cb_data;

    /**
     * This callback is called whenever the list receives a remove event. The
     * remove event is OXINE_KEY_REMOVE. Apart from calling the callback
     * nothing is done. The user is responsible for removing the entry.
     */
    otk_cb_t remove_cb;

    /// User-defined data for the above callback.
    void *remove_cb_data;
};


static void listentries_adapt (otk_list_t * list);


static void
list_page_down (void *data)
{
    otk_list_t *list = (otk_list_t *) data;

    list->first_visible += list->num_visible;
    listentries_adapt (list);
    otk_draw (list->widget.otk);

    if (list->entry_changed_cb) {
        list->entry_changed_cb (list->entry_changed_cb_data,
                                list->first_visible);
    }
}


static void
list_page_up (void *data)
{
    otk_list_t *list = (otk_list_t *) data;

    list->first_visible -= list->num_visible;
    listentries_adapt (list);
    otk_draw (list->widget.otk);

    if (list->entry_changed_cb) {
        list->entry_changed_cb (list->entry_changed_cb_data,
                                list->first_visible);
    }
}


static void
list_scroll_down (void *data)
{
    otk_list_t *list = (otk_list_t *) data;

    list->first_visible++;
    listentries_adapt (list);
    otk_draw (list->widget.otk);

    if (list->entry_changed_cb) {
        list->entry_changed_cb (list->entry_changed_cb_data,
                                list->first_visible);
    }
}


static void
list_scroll_up (void *data)
{
    otk_list_t *list = (otk_list_t *) data;

    list->first_visible--;
    listentries_adapt (list);
    otk_draw (list->widget.otk);

    if (list->entry_changed_cb) {
        list->entry_changed_cb (list->entry_changed_cb_data,
                                list->first_visible);
    }
}


static int
list_set_position (void *data, int position)
{
    otk_list_t *list = (otk_list_t *) data;

    list->first_visible = position * l_list_length (list->entries) / 100;
    listentries_adapt (list);
    otk_draw (list->widget.otk);

    return position;
}


static void
listentry_destroy (otk_widget_t * this)
{
    otk_listentry_t *listentry = (otk_listentry_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LISTENTRY))
        return;

    ho_free (listentry->text);
    ho_free (listentry);
}


static void
listentry_draw (otk_widget_t * this)
{
    otk_listentry_t *listentry = (otk_listentry_t *) this;
    otk_list_t *list = listentry->list;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LISTENTRY))
        return;

    int palette = 0;
    if (!this->is_enabled) {
        palette = otk_get_palette (this->otk, OTK_PALETTE_BUTTON_DISABLED);
    }
    else if (this->is_focused) {
        palette = otk_get_palette (this->otk, OTK_PALETTE_BUTTON_FOCUSED);
    }
    else {
        palette = otk_get_palette (this->otk, OTK_PALETTE_BUTTON);
    }

    if ((list->type == OTK_LIST_TYPE_MENU)
        || this->is_focused || listentry->is_selected || !this->is_enabled) {
        int background_color = palette + OSD_TEXT_PALETTE_BACKGROUND;
        odk_draw_rect (this->odk, this->x, this->y, this->w, this->h, 5,
                       background_color, true);
    }

    if (listentry->text) {
        odk_osd_set_font (this->odk, this->font, this->fontsize);

        int x = this->x + this->w / 2;
        if (this->alignment & OTK_ALIGN_RIGHT) {
            x = this->x + this->w - 5;
        }
        else if (this->alignment & OTK_ALIGN_LEFT) {
            x = this->x + 5;
        }

        /* We always align the text vertically */
        this->alignment &= ~OTK_ALIGN_TOP;
        this->alignment &= ~OTK_ALIGN_BOTTOM;
        this->alignment |= OTK_ALIGN_VCENTER;

        /* We align tabstops every 20 pixels. */
        int w = this->w - 10;
        char *cpy = ho_strdup (listentry->text);
        char *str = cpy;
        char *tab = NULL;
        char *txt = NULL;
        do {
            int tw;
            tab = index (str, '\t');
            if (tab) {
                tab[0] = '\0';
                tab += 1;
            }
            txt = otk_trunc_text_to_width (this->otk, str, w);
            odk_draw_text (this->odk, x, this->y + this->h / 2,
                           txt, this->alignment, palette);
            odk_get_text_size (this->odk, txt, &tw, NULL);
            w += x;
            x += tw + 10;
            x -= (x % 20);
            x += 20;
            w -= x;
            str = tab;
            ho_free (txt);
        } while (tab);
        ho_free (cpy);
    }

    this->need_repaint = false;
}


static void
listentry_select (otk_listentry_t * listentry, int kbd_modifier)
{
    otk_list_t *list = listentry->list;

    listentry->doubleclicktime = time (NULL);
    int was_selected = listentry->is_selected;

    /* We only care about Ctrl and Shift */
    kbd_modifier &= ControlMask | ShiftMask;

    if (list->allow_select == OTK_LIST_SELECTION_SINGLE) {
        otk_list_clear_selection ((otk_widget_t *) list);

        if (kbd_modifier & ControlMask) {
            listentry->is_selected = !was_selected;
            list->num_selected = was_selected ? 0 : 1;
        }
        else {
            listentry->is_selected = true;
            list->num_selected = 1;
        }
        list->last_single_selected = listentry;
    }

    else if (list->allow_select == OTK_LIST_SELECTION_MULTIPLE) {
        if (!config_get_bool ("misc.list_select_ctrl_shift")
            || (kbd_modifier & ControlMask)) {
            listentry->is_selected = !was_selected;
            list->num_selected += was_selected ? -1 : 1;
        }

        else if (kbd_modifier & ShiftMask) {
            bool selected;

            listentry->is_selected = true;

            selected = false;
            otk_listentry_t *entry = l_list_first (list->entries);
            while (entry && (entry != listentry)) {
                if (entry == list->last_single_selected)
                    selected = true;
                entry->is_selected = selected;
                entry = l_list_next (list->entries, entry);
            }

            selected = false;
            entry = l_list_last (list->entries);
            while (entry && (entry != listentry)) {
                if (entry == list->last_single_selected)
                    selected = true;
                entry->is_selected = selected;
                entry = l_list_prev (list->entries, entry);
            }

            list->num_selected = 0;
            entry = l_list_first (list->entries);
            while (entry) {
                if (entry->is_selected)
                    list->num_selected++;
                entry = l_list_next (list->entries, entry);
            }
        }

        else {
            otk_list_clear_selection ((otk_widget_t *) list);
            listentry->is_selected = true;
            list->num_selected = 1;
            list->last_single_selected = listentry;
        }
    }

    otk_draw (list->widget.otk);

    if (listentry->select_cb) {
        listentry->select_cb (listentry->select_cb_data);
    }
}


static void
listentry_activate (otk_listentry_t * listentry)
{
    listentry->doubleclicktime = 0;

    otk_list_clear_selection ((otk_widget_t *) listentry->list);

    if (listentry->activate_cb) {
        listentry->activate_cb (listentry->activate_cb_data);
    }
}


static void
listentry_key_handler (otk_widget_t * this, oxine_event_t * ev)
{
    otk_listentry_t *entry = (otk_listentry_t *) this;
    otk_list_t *list = entry ? entry->list : NULL;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LISTENTRY))
        return;

    oxine_key_id_t saved_key = ev->source.key;
    ev->source.key = OXINE_KEY_NULL;

    otk_widget_t *new_widget = NULL;

    switch (saved_key) {
    case OXINE_KEY_FIRST:
        list->first_visible = 0;
        listentries_adapt (list);

        new_widget = l_list_first (list->entries);
        if (new_widget && !(new_widget->is_visible)
            && !(new_widget->is_enabled)) {
            new_widget = l_list_next (list->entries, new_widget);
        }

        if (list->entry_changed_cb) {
            list->entry_changed_cb (list->entry_changed_cb_data,
                                    list->first_visible);
        }
        break;
    case OXINE_KEY_LAST:
        list->first_visible = l_list_length (list->entries);
        listentries_adapt (list);

        new_widget = l_list_last (list->entries);
        if (new_widget && !(new_widget->is_visible)
            && !(new_widget->is_enabled)) {
            new_widget = l_list_prev (list->entries, new_widget);
        }

        if (list->entry_changed_cb) {
            list->entry_changed_cb (list->entry_changed_cb_data,
                                    list->first_visible);
        }
        break;
    case OXINE_KEY_PAGE_UP:
        list->first_visible -= list->num_visible;
        listentries_adapt (list);

        new_widget = this;
        while (!((otk_listentry_t *) new_widget)->is_first) {
            new_widget = l_list_prev (list->entries, new_widget);
        }

        if (new_widget && !(new_widget->is_visible)
            && !(new_widget->is_enabled)) {
            new_widget = l_list_next (list->entries, new_widget);
        }

        if (list->entry_changed_cb) {
            list->entry_changed_cb (list->entry_changed_cb_data,
                                    list->first_visible);
        }
        break;
    case OXINE_KEY_PAGE_DOWN:
        list->first_visible += list->num_visible;
        listentries_adapt (list);

        new_widget = this;
        while (!((otk_listentry_t *) new_widget)->is_last) {
            new_widget = l_list_next (list->entries, new_widget);
        }

        if (new_widget && !(new_widget->is_visible)
            && !(new_widget->is_enabled)) {
            new_widget = l_list_prev (list->entries, new_widget);
        }

        if (list->entry_changed_cb) {
            list->entry_changed_cb (list->entry_changed_cb_data,
                                    list->first_visible);
        }
        break;
    case OXINE_KEY_UP:
    case OXINE_KEY_LEFT:
        /* If we got a OXINE_KEY_UP event and this is a selector, we jump to
         * up the prev widget. */
        if ((saved_key == OXINE_KEY_UP)
            && (list->type == OTK_LIST_TYPE_SELECTOR)) {
            new_widget = otk_widget_find_neighbour (this->otk,
                                                    OTK_DIRECTION_UP);
        }
        /* If we got a OXINE_KEY_LEFT event and this is a normal list or a
         * menu list, we jump to the next widget to the right. */
        else if ((saved_key == OXINE_KEY_LEFT)
                 && (list->type != OTK_LIST_TYPE_SELECTOR)) {
            new_widget = otk_widget_find_neighbour (this->otk,
                                                    OTK_DIRECTION_LEFT);
        }
        /* Else we search backwards in the list until we find an entry, that
         * is visible and enabled. */
        else {
            otk_listentry_t *new_entry = entry;
            do {
                if (new_entry->is_first) {
                    list->first_visible--;
                    listentries_adapt (list);
                }

                new_entry = l_list_prev (list->entries, new_entry);
            } while (new_entry && (!(new_entry->widget.is_visible)
                                   || !(new_entry->widget.is_enabled)));
            new_widget = (otk_widget_t *) new_entry;

            if (new_entry && (new_entry != entry) && list->entry_changed_cb) {
                list->entry_changed_cb (list->entry_changed_cb_data,
                                        list->first_visible);
            }
        }
        break;
    case OXINE_KEY_DOWN:
    case OXINE_KEY_RIGHT:
        /* If we got a OXINE_KEY_DOWN event and this is a selector, we jump
         * down to the next widget. */
        if ((saved_key == OXINE_KEY_DOWN)
            && (list->type == OTK_LIST_TYPE_SELECTOR)) {
            new_widget = otk_widget_find_neighbour (this->otk,
                                                    OTK_DIRECTION_DOWN);
        }
        /* If we got a OXINE_KEY_RIGHT event and this is a normal list or a
         * menu list, we jump to the next widget to the right. */
        else if ((saved_key == OXINE_KEY_RIGHT)
                 && (list->type != OTK_LIST_TYPE_SELECTOR)) {
            new_widget = otk_widget_find_neighbour (this->otk,
                                                    OTK_DIRECTION_RIGHT);
        }
        /* Else we search forewards in the list until we find an entry, that
         * is visible and enabled. */
        else {
            otk_listentry_t *new_entry = entry;
            do {
                if (new_entry->is_last) {
                    list->first_visible++;
                    listentries_adapt (list);
                }

                new_entry = l_list_next (list->entries, new_entry);
            } while (new_entry && (!(new_entry->widget.is_visible)
                                   || !(new_entry->widget.is_enabled)));
            new_widget = (otk_widget_t *) new_entry;

            if (new_entry && (new_entry != entry) && list->entry_changed_cb) {
                list->entry_changed_cb (list->entry_changed_cb_data,
                                        list->first_visible);
            }
        }
        break;
    case OXINE_KEY_SELECT:
        listentry_select (entry, ev->data.keyboard.modifier);
        break;
    case OXINE_KEY_ACTIVATE:
        listentry_activate (entry);
        break;
    case OXINE_KEY_REMOVE:
        if (entry->remove_cb) {
            entry->remove_cb (entry->remove_cb_data);
        }
        break;
    default:
        /* If we did not use (consume) this event here, we restore it for
         * others to use. */
        ev->source.key = saved_key;
        break;
    }

    if (new_widget) {
        otk_widget_set_focused (new_widget, true);
        otk_draw (this->otk);
    }
}


static void
listentry_button_handler (otk_widget_t * this, oxine_event_t * ev)
{
    otk_listentry_t *entry = (otk_listentry_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LISTENTRY))
        return;

    otk_list_t *list = entry->list;
    oxine_button_id_t saved_button = ev->source.button;
    ev->source.button = OXINE_MOUSE_BUTTON_NULL;

    switch (saved_button) {
    case OXINE_MOUSE_BUTTON_LEFT:
        if (time (NULL) - 1 < entry->doubleclicktime) {
            listentry_activate (entry);
        }
        else {
            listentry_select (entry, ev->data.mouse.button.modifier);
        }
        break;
    case OXINE_MOUSE_SCROLLWHEEL_UP:
        list->first_visible--;
        listentries_adapt (list);
        otk_draw (this->otk);

        if (list->entry_changed_cb) {
            list->entry_changed_cb (list->entry_changed_cb_data,
                                    list->first_visible);
        }
        break;
    case OXINE_MOUSE_SCROLLWHEEL_DOWN:
        list->first_visible++;
        listentries_adapt (list);
        otk_draw (this->otk);

        if (list->entry_changed_cb) {
            list->entry_changed_cb (list->entry_changed_cb_data,
                                    list->first_visible);
        }
        break;
    default:
        /* If we did not use this event we restore the event. */
        ev->source.button = saved_button;
        break;
    }
}


otk_widget_t *
otk_listentry_new (otk_widget_t * this, const char *text,
                   otk_cb_t activate_cb, void *activate_cb_data,
                   otk_cb_t select_cb, void *select_cb_data,
                   otk_cb_t remove_cb, void *remove_cb_data)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LIST))
        return NULL;

    otk_listentry_t *entry = ho_new (otk_listentry_t);

    otk_widget_constructor ((otk_widget_t *) entry, this->otk,
                            OTK_WIDGET_LISTENTRY, list->entry_x, 0,
                            list->entry_width, list->entry_height);

    entry->widget.selectable = OTK_SELECTABLE_MOUSE | OTK_SELECTABLE_KEY;

    entry->widget.draw = listentry_draw;
    entry->widget.destroy = listentry_destroy;

    entry->widget.button_handler = listentry_button_handler;
    entry->widget.key_handler = listentry_key_handler;

    entry->widget.fontsize = list->widget.fontsize;

    entry->list = list;
    entry->is_first = false;
    entry->is_last = false;
    entry->is_selected = false;
    entry->text = text ? ho_strdup (text) : NULL;
    entry->activate_cb_data = activate_cb_data;
    entry->activate_cb = activate_cb;
    entry->select_cb_data = select_cb_data;
    entry->select_cb = select_cb;
    entry->remove_cb_data = remove_cb_data;
    entry->remove_cb = remove_cb;

    l_list_append (list->entries, entry);
    list->need_adaption = true;

    return (otk_widget_t *) entry;
}


void
otk_list_clear_selection (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LIST))
        return;

    otk_listentry_t *entry = l_list_first (list->entries);
    while (entry) {
        entry->is_selected = false;
        entry = l_list_next (list->entries, entry);
    }
    list->num_selected = 0;

    otk_draw (this->otk);
}


void
otk_list_select_all (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LIST))
        return;

    otk_listentry_t *entry = l_list_first (list->entries);
    while (entry) {
        entry->is_selected = true;
        entry = l_list_next (list->entries, entry);
    }
    list->num_selected = l_list_length (list->entries);

    otk_draw (this->otk);
}


int
otk_list_get_selected_count (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LIST))
        return 0;

    return list->num_selected;
}


int *
otk_list_get_selected_pos (otk_widget_t * this, int *num_selected)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LIST))
        return NULL;

    *num_selected = list->num_selected;

    if (list->num_selected) {
        int counter = 0;
        int *selected_entries = ho_new (list->num_selected * sizeof (int));

        otk_listentry_t *entry = l_list_first (list->entries);
        int pos = 0;
        while (entry) {
            if (entry->is_selected) {
                selected_entries[counter++] = pos;
            }
            entry = l_list_next (list->entries, entry);
            pos++;
        }
        return selected_entries;
    }
    return NULL;
}


void **
otk_list_get_selected (otk_widget_t * this, int *num_selected)
{
    otk_list_t *list = (otk_list_t *) this;

    *num_selected = 0;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LIST))
        return NULL;

    *num_selected = list->num_selected;

    if (list->num_selected) {
        int counter = 0;
        void **selected_entries =
            ho_malloc (list->num_selected * sizeof (void *));

        otk_listentry_t *entry = l_list_first (list->entries);
        while (entry) {
            if (entry->is_selected) {
                selected_entries[counter++] = entry->select_cb_data;
            }
            entry = l_list_next (list->entries, entry);
        }
        return selected_entries;
    }
    return NULL;
}


otk_widget_t *
otk_listentry_get_list (otk_widget_t * this)
{
    otk_listentry_t *entry = (otk_listentry_t *) this;
    otk_list_t *list = entry ? entry->list : NULL;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LISTENTRY))
        return NULL;

    return &list->widget;
}


static void
listentries_adapt (otk_list_t * list)
{
    list->need_adaption = false;

    if (l_list_length (list->entries) == 0) {
        if (list->scrollbar) {
            otk_scrollbar_set (list->scrollbar, 0, 100);
        }
        if (list->type == OTK_LIST_TYPE_SELECTOR) {
            otk_widget_set_enabled (list->button_up, false);
            otk_widget_set_enabled (list->button_down, false);
        }
        return;
    }

    int num_entries = l_list_length (list->entries);
    if ((num_entries - list->first_visible) < list->num_visible) {
        list->first_visible = num_entries - list->num_visible;
    }
    if (list->first_visible < 0) {
        list->first_visible = 0;
    }

    int first_visible = list->first_visible;
    int last_visible = list->first_visible + list->num_visible - 1;
    if (last_visible > num_entries) {
        last_visible = num_entries - 1;
    }

    int first_y = list->entry_y;
    int entry_s = list->entry_spacing;
    if ((list->type == OTK_LIST_TYPE_MENU)
        && (num_entries < list->num_visible)) {
        first_y += ((list->num_visible * entry_s)
                    - (num_entries * entry_s)) / 2;
    }

    int pos = 0;
    otk_listentry_t *entry = l_list_first (list->entries);
    while (entry && (pos < first_visible)) {
        entry->is_first = false;
        entry->is_last = false;
        entry->widget.is_visible = false;
        entry = l_list_next (list->entries, entry);
        pos += 1;
    }
    otk_listentry_t *first = entry;
    while (entry && (pos <= last_visible)) {
        entry->is_first = (pos == first_visible);
        entry->is_last = (pos == last_visible);
        entry->widget.y = first_y + entry_s * (pos - first_visible);
        entry->widget.is_visible = true;
        entry = l_list_next (list->entries, entry);
        pos += 1;
    }
    while (entry) {
        entry->is_first = false;
        entry->is_last = false;
        entry->widget.is_visible = false;
        entry = l_list_next (list->entries, entry);
        pos += 1;
    }

    otk_widget_t *window = otk_get_current_window (list->widget.otk);
    otk_widget_t *focused = otk_window_focus_pointer_get (window);
    if (!focused || !focused->is_visible) {
        otk_widget_set_focused ((otk_widget_t *) first, true);
    }

    if (list->scrollbar) {
        double cnt = (double) num_entries;
        double beg = (double) list->first_visible;
        double num = (double) list->num_visible;
        double pos = (beg * 100.0) / cnt;
        double len = (num * 100.0) / cnt;
        /* Just to make sure rounding errors don't bite us! */
        if ((num_entries - list->num_visible) == list->first_visible) {
            pos = 100.0 - len;
        }
        otk_scrollbar_set (list->scrollbar, pos, len);
    }

    if (list->type == OTK_LIST_TYPE_SELECTOR) {
        otk_widget_t *last = l_list_last (list->entries);
        otk_widget_t *first = l_list_first (list->entries);
        otk_widget_set_enabled (list->button_down, !last->is_visible);
        otk_widget_set_enabled (list->button_up, !first->is_visible);
    }
}


static void
entries_destroy (void *data)
{
    otk_widget_t *this = (otk_widget_t *) data;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LISTENTRY))
        return;

    otk_widget_destructor (this);
}


static void
list_destroy (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LIST))
        return;

    l_list_free (list->entries, entries_destroy);

    ho_free (list);
}


static void
list_draw (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LIST)) {
        return;
    }

    if (list->need_adaption) {
        listentries_adapt (list);
    }

    if (list->type != OTK_LIST_TYPE_MENU) {
        return;
    }

    int length = l_list_length (list->entries);
    if (length < list->num_visible) {
        return;
    }

    int first_visible = list->first_visible;
    if (first_visible != 0) {
        int x1 = this->x + (this->w / 2);
        int y1 = list->entry_y - 30;
        int x2 = x1 - 20;
        int y2 = y1 + 20;
        int x3 = x1 + 20;
        int y3 = y1 + 20;

        int palette = otk_get_palette (this->otk, OTK_PALETTE_BUTTON);
        int color = palette + OSD_TEXT_PALETTE_BACKGROUND;

        odk_draw_triangle (this->odk, x1, y1, x2, y2, x3, y3, color, true);
    }

    int last_visible = list->first_visible + list->num_visible;
    if (last_visible != length) {
        int x1 = this->x + (this->w / 2);
        int y1 = list->entry_y;
        y1 += ((list->num_visible - 1) * list->entry_spacing);
        y1 += list->entry_height + 30;
        int x2 = x1 - 20;
        int y2 = y1 - 20;
        int x3 = x1 + 20;
        int y3 = y1 - 20;

        int palette = otk_get_palette (this->otk, OTK_PALETTE_BUTTON);
        int color = palette + OSD_TEXT_PALETTE_BACKGROUND;

        odk_draw_triangle (this->odk, x1, y1, x2, y2, x3, y3, color, true);
    }
}


otk_widget_t *
otk_list_new (otk_t * otk, int x, int y, int w, int h,
              int entry_height, int entry_spacing,
              bool show_border, bool show_scrollbar,
              otk_list_selection_t allow_select, void *list_cb_data)
{
    otk_list_t *list = ho_new (otk_list_t);

    otk_widget_constructor ((otk_widget_t *) list, otk,
                            OTK_WIDGET_LIST, x, y, w, h);

    list->widget.selectable = OTK_SELECTABLE_NONE;

    list->widget.draw = list_draw;
    list->widget.destroy = list_destroy;

    list->type = OTK_LIST_TYPE_NORMAL;
    list->allow_select = allow_select;

    list->entry_x = list->widget.x;
    list->entry_y = list->widget.y;
    list->entry_width = list->widget.w;
    list->entry_height = entry_height;
    list->entry_spacing = entry_spacing;

    list->entries = l_list_new ();
    list->num_selected = 0;
    list->first_visible = 0;

    list->num_visible = list->widget.h - entry_height;
    list->num_visible -= show_border ? 10 : 0;
    list->num_visible /= entry_spacing;
    list->num_visible += 1;

    list->cb_data = list_cb_data;
    list->entry_changed_cb = NULL;
    list->entry_changed_cb_data = NULL;

    list->scrollbar = NULL;
    list->button_up = NULL;
    list->button_down = NULL;

    if (show_scrollbar && (list->num_visible > 1)) {
        list->entry_width -= 35;

        int x1 = x + w - 35;
        int y1 = y + 5;
        int w1 = 30;
        int h1 = h - 10;

        list->scrollbar = otk_scrollbar_new (otk, x1, y1, w1, h1,
                                             list_set_position,
                                             list_scroll_up, list_scroll_down,
                                             list_page_up, list_page_down,
                                             list);
    }

    else if (show_scrollbar && (list->num_visible == 1)) {
        list->entry_width -= 70;

        int x1 = x + w - 35;
        int y1 = y + 5;
        int x2 = x + w - 70;
        int y2 = y + 5;

        int bw = 30;
        int bh = 30;
        int vw = 20;
        int vh = (vw / 2);

        list->button_up = otk_vector_button_new (otk, x1, y1, bw, bh,
                                                 OSD_VECTOR_ARROW_UP_SIMPLE,
                                                 vw, vh, list_scroll_up,
                                                 list);
        list->button_up->selectable = OTK_SELECTABLE_MOUSE;

        list->button_down = otk_vector_button_new (otk, x2, y2, bw, bh,
                                                   OSD_VECTOR_ARROW_DOWN_SIMPLE,
                                                   vw, vh, list_scroll_down,
                                                   list);
        list->button_down->selectable = OTK_SELECTABLE_MOUSE;
    }

    if (show_border) {
        otk_border_new (otk, x, y, w, h);

        list->entry_width -= 10;
        list->entry_x += 5;
        list->entry_y += 5;
    }

    return (otk_widget_t *) list;
}


int
otk_list_get_length (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    return l_list_length (list->entries);
}


int
otk_list_get_focus (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LIST))
        return 0;

    otk_widget_t *window = otk_get_current_window (this->otk);
    if (!window)
        return 0;

    otk_widget_t *focused = otk_window_focus_pointer_get (window);
    if (!focused)
        return 0;

    int i = 0;
    otk_listentry_t *current = l_list_first (list->entries);
    while (current) {
        if (((otk_widget_t *) current) == focused)
            return i;
        current = l_list_next (list->entries, current);
        i++;
    }

    return 0;
}


void
otk_list_set_focus (otk_widget_t * this, int pos)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LIST))
        return;

    if (pos > l_list_length (list->entries) - 1) {
        pos = l_list_length (list->entries) - 1;
    }

    int i = 0;
    otk_widget_t *current = l_list_first (list->entries);
    while (current) {
        if (i == pos) {
            while (current && !current->is_enabled) {
                current = l_list_next (list->entries, current);
            }
            if (current) {
                otk_widget_set_focused (current, true);
            }
            break;
        }
        current = l_list_next (list->entries, current);
        i++;
    }
}


void
otk_list_set_pos (otk_widget_t * this, int newpos)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LIST))
        return;

    list->first_visible = newpos;
    listentries_adapt (list);
}


int
otk_list_get_pos (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LIST))
        return -1;

    return list->first_visible;
}


void
otk_list_clear (otk_widget_t * this)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LIST))
        return;

    l_list_clear (list->entries, entries_destroy);
    list->num_selected = 0;

    otk_draw (this->otk);
}


void
otk_list_set_selected (otk_widget_t * this, int pos, bool selected)
{
    otk_list_t *list = (otk_list_t *) this;

    if (!otk_widget_is_correct (this, OTK_WIDGET_LIST))
        return;

    int i = 0;
    otk_listentry_t *current = l_list_first (list->entries);
    while (current) {
        if ((i == pos)
            && (current->widget.is_enabled)
            && (current->widget.is_visible)
            && (current->widget.selectable)) {

            if (!current->is_selected && selected)
                list->num_selected++;
            if (current->is_selected && !selected)
                list->num_selected--;

            current->is_selected = selected;
            return;
        }
        current = l_list_next (list->entries, current);
        i++;
    }
}


otk_widget_t *
otk_selector_new (otk_t * otk, int x, int y, int w,
                  otk_int_set_cb_t cb, void *cb_data)
{
    otk_list_t *list =
        (otk_list_t *) otk_list_new (otk, x, y, w, 40, 30, 30, true, true,
                                     OTK_LIST_SELECTION_NONE, NULL);

    list->entry_changed_cb = cb;
    list->entry_changed_cb_data = cb_data;

    list->type = OTK_LIST_TYPE_SELECTOR;

    return (otk_widget_t *) list;
}


otk_widget_t *
otk_selectoritem_new (otk_widget_t * selector, const char *label,
                      otk_cb_t activate_cb, void *activate_cb_data)
{
    otk_widget_t *selectoritem =
        otk_listentry_new (selector, label, activate_cb, activate_cb_data,
                           NULL, NULL, NULL, NULL);

    return selectoritem;
}


otk_widget_t *
otk_menulist_new (otk_t * otk, int x, int y, int w, int h,
                  int entry_height, int entry_spacing, void *menu_cb_data)
{
    otk_list_t *list =
        (otk_list_t *) otk_list_new (otk, x, y + 30, w, h - 60, entry_height,
                                     entry_spacing, false, false,
                                     OTK_LIST_SELECTION_NONE, menu_cb_data);

    list->type = OTK_LIST_TYPE_MENU;

    int off = (h - 60);
    off -= ((list->num_visible - 1) * entry_spacing);
    off -= entry_height;
    list->entry_y += (off / 2);

#if 0
    otk_border_new (otk, x, y, w, h);
#endif

    return (otk_widget_t *) list;
}


otk_widget_t *
otk_menuitem_new (otk_widget_t * menu, const char *text,
                  otk_cb_t activate_cb, void *activate_cb_data)
{
    otk_widget_t *menuitem =
        otk_listentry_new (menu, text, activate_cb, activate_cb_data,
                           activate_cb, activate_cb_data, NULL, NULL);

    otk_widget_set_font (menuitem, "sans", 30);
    otk_widget_set_alignment (menuitem, OTK_ALIGN_CENTER | OTK_ALIGN_VCENTER);

    return menuitem;
}
