
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: utils_playlist.c 2418 2007-07-03 16:14:01Z mschwerin $
 *
 */
#include "config.h"

#include "i18n.h"
#include "heap.h"
#include "oxine.h"
#include "odk.h"
#include "playlist.h"
#include "utils.h"

#include "menu_base.h"
#include "menu_main.h"

extern oxine_t *oxine;


void
playlist_rw_change_cb (playlist_t * playlist)
{
    assert (playlist == oxine->rw_playlist);

    if (odk_current_is_playback_mode (oxine->odk)
        && (oxine->current_playlist == oxine->rw_playlist)) {
        const char *mrl = odk_current_get_mrl (oxine->odk);

        playitem_t *playitem = playlist_first (playlist);
        while (playitem) {
            playitem->start_time = 0;
            playitem = playlist_next (playlist, playitem);
        }

        playitem_t *current = playlist_get_current (playlist);
        if (current && mrl && (strcmp (mrl, current->mrl) == 0)) {
            odk_get_pos_length (oxine->odk, NULL, &current->start_time, NULL);
        }
    }

    playlist_save (playlist, NULL);
}


static void
playitem_play (playitem_t * playitem, otk_cb_t error_cb)
{
    if (!playitem) {
        playback_ended_menu_cb (oxine);
        return;
    }

    if (starts_with (playitem->mrl, "http:/")
        || starts_with (playitem->mrl, "rtsp:/")
        || starts_with (playitem->mrl, "cdda:/")
        || starts_with (playitem->mrl, "dvd://")
        || starts_with (playitem->mrl, "vcd://")
        || starts_with (playitem->mrl, "dvb://")
        || starts_with (playitem->mrl, "v4l://")) {
        /* This is only necessary when we start oxine and start playing a
         * video at once. If we call please_wait in this situation we
         * first load the background and a very short time after that
         * start playing the video. That would not look good! */
        if (odk_current_is_logo_mode (oxine->odk)
            || odk_current_is_playback_mode (oxine->odk)) {
            otk_cb_t current_menu = get_current_menu ();
            void *current_menu_data = get_current_menu_data ();
            please_wait ();
            set_current_menu (current_menu, current_menu_data);
        }
    }

    if (!odk_play_stream (oxine->odk, playitem->title,
                          playitem->mrl, playitem->subtitle_mrl,
                          playitem->start_time, ODK_MODE_NORMAL)) {
        char *msg = NULL;
        otk_cb_t ok_cb = error_cb;
        message_dialog_type_t type = DIALOG_OK_CANCEL;

        if (starts_with (playitem->mrl, "cdda:/")
            && mrl_filename_is_device (playitem->mrl)) {
            msg = ho_strdup_printf ("%s\n%s",
                                    _("Could not play Audio CD!"),
                                    _("Please insert an Audio CD!"));
            ok_cb = play_cdda_cb;
            type = DIALOG_RETRY_CANCEL;
        }
        else if (starts_with (playitem->mrl, "vcd:/")
                 && mrl_filename_is_device (playitem->mrl)) {
            msg = ho_strdup_printf ("%s\n%s",
                                    _("Could not play Video CD!"),
                                    _("Please insert a Video CD!"));
            ok_cb = play_vcd_cb;
            type = DIALOG_RETRY_CANCEL;
        }
        else if (starts_with (playitem->mrl, "dvd:/")
                 && mrl_filename_is_device (playitem->mrl)) {
            msg = ho_strdup_printf ("%s\n%s",
                                    _("Could not play DVD!"),
                                    _("Please insert a DVD!"));
            ok_cb = play_dvd_cb;
            type = DIALOG_RETRY_CANCEL;
        }
        else if (starts_with (playitem->mrl, "v4l:/")
                 || starts_with (playitem->mrl, "dvb:/")
                 || starts_with (playitem->mrl, "vdr:/")) {
            msg = ho_strdup (_("Failed to start TV playback!"));
            type = DIALOG_OK;
        }
        else {
            msg = ho_strdup (_("Could not play title!"));
        }

        set_backto_menu (playback_ended_menu_cb, NULL);
        playlist_remove (oxine->current_playlist, playitem);
        show_message_dialog (ok_cb, NULL, playback_ended_menu_cb, NULL,
                             type, NULL, msg);
        ho_free (msg);
    }
}


void
playlist_play_first (playlist_t * playlist)
{
    assert (playlist);

    set_backto_menu (hide_user_interface, NULL);
    set_current_menu (hide_user_interface, NULL);
    oxine->current_playlist = playlist;

    playlist_lock (oxine->current_playlist);
    playitem_t *playitem = playlist_get_first (oxine->current_playlist);
    playitem_play (playitem, (otk_cb_t) playlist_play_next);
    playlist_unlock (oxine->current_playlist);
}


void
playlist_play_item (playlist_t * playlist, playitem_t * playitem)
{
    assert (playlist);

    set_backto_menu (hide_user_interface, NULL);
    set_current_menu (hide_user_interface, NULL);
    oxine->current_playlist = playlist;

    playlist_lock (oxine->current_playlist);
    playlist_set_current (oxine->current_playlist, playitem);
    playitem_play (playitem, (otk_cb_t) playlist_play_next);
    playlist_unlock (oxine->current_playlist);
}


void
playlist_play_next (void)
{
    playlist_lock (oxine->current_playlist);
    playitem_t *playitem = playlist_get_next (oxine->current_playlist);
    playitem_play (playitem, (otk_cb_t) playlist_play_next);
    playlist_unlock (oxine->current_playlist);
}


void
playlist_play_prev (void)
{
    playlist_lock (oxine->current_playlist);
    playitem_t *playitem = playlist_get_prev (oxine->current_playlist);
    playitem_play (playitem, (otk_cb_t) playlist_play_prev);
    playlist_unlock (oxine->current_playlist);
}
