
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: youtube.c 2601 2007-07-25 08:29:04Z mschwerin $
 */
#include "config.h"

#include <regex.h>
#include <sys/types.h>

#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "download.h"
#include "logger.h"
#include "oxine.h"
#include "utils.h"
#include "xmlparser.h"
#include "youtube.h"

#ifdef HAVE_YOUTUBE

/**
 * For information on the YouTube developer API see http://youtube.com/dev.
 */

extern oxine_t *oxine;

#define DEV_ID                  "oo6Z8iVYWWQ"

static char *
get_playback_mrl (filelist_t * filelist)
{
    const char *prefix = "http://www.youtube.com/get_video?";
    const char *pattern = "video_id=\\S+&.+&t=.+&f";

    char *mrl = NULL;
    regex_t reg;
    regmatch_t pm;

    int size = 0;
    char *buffer = read_entire_file (filelist->mrl, &size);
    if (!buffer) {
        goto buffer_free;
    }

    if (regcomp (&reg, pattern, REG_EXTENDED | REG_ICASE | REG_NEWLINE) != 0) {
        error (_("Failed to compile regular expression."));
        goto buffer_free;
    }

    if (regexec (&reg, buffer, 1, &pm, 0) == 0) {
        int size = 0;
        char *dst = NULL;
        const char *src = NULL;

        size = strlen (prefix) + pm.rm_eo - pm.rm_so - 1;
        mrl = ho_malloc (size);
        memset (mrl, 0, size);

        dst = mrl;
        src = prefix;
        size = strlen (prefix);
        memcpy (dst, src, size);

        dst = mrl + strlen (prefix);
        src = buffer + pm.rm_so;
        size = pm.rm_eo - pm.rm_so - 2;
        memcpy (dst, src, size);
    }
    else {
        error (_("Failed to retrieve %s playback MRL for '%s'!"),
               _("YouTube"), filelist->mrl);
    }

    if (mrl) {
        debug ("found YouTube playback MRL '%s'", mrl);
        char *redirect_mrl = NULL;
        do {
            redirect_mrl = get_redirect_url (mrl);
            if (redirect_mrl) {
                ho_free (mrl);
                mrl = redirect_mrl;
            }
        } while (redirect_mrl);
    }

    if (mrl) {
        debug ("found YouTube playback MRL '%s'", mrl);
    }

    regfree (&reg);
  buffer_free:
    ho_free (buffer);

    return mrl;
}


static bool
youtube_video_read (filelist_t * filelist)
{
    char *mrl = get_playback_mrl (filelist);

    if (mrl) {
        if (config_get_bool ("streaming.youtube.download_to_cache")) {
            char *file = ho_strdup_printf ("%s.flv", filelist->title);
            char *cache = download_to_cache (mrl, file, false);
            if (cache) {
                ho_free (mrl);
                mrl = cache;
            }
            ho_free (file);
        }

        fileitem_t *item = filelist_add (filelist, filelist->title, mrl,
                                         FILE_TYPE_REGULAR);
        if (item && filelist->thumbnail_mrl) {
            item->thumbnail_mrl = ho_strdup (filelist->thumbnail_mrl);
        }

        ho_free (mrl);
        return true;
    }

    return false;
}


static bool
youtube_videolist_read (filelist_t * filelist)
{
    bool success = false;
    const char *status = NULL;
    char *mrl = ho_strdup_printf ("%s", filelist->mrl);

    int size = 0;
    char *buffer = read_entire_file (mrl, &size);
    if (!buffer) {
        goto buffer_free;
    }

    xml_node_t *rootnode;
    xml_parser_init (buffer, size, XML_PARSER_CASE_INSENSITIVE);
    if (xml_parser_build_tree (&rootnode) < 0) {
        error (_("Parsing '%s' failed!"), filelist->mrl);
        goto buffer_free;
    }

    if (strcasecmp (rootnode->name, "ut_response") != 0) {
        error (_("Root node of '%s' must be '%s'!"),
               filelist->mrl, "ut_response");
        debug ("\n%s", buffer);
        goto parser_free;
    }

    status = xml_parser_get_property (rootnode, "status");
    if (!status || (strcasecmp (status, "ok") != 0)) {
        error (_("Retrieving video list failed!"));
        debug ("\n%s", buffer);
        goto parser_free;
    }

    xml_node_t *listnode = rootnode->child;
    if (strcasecmp (listnode->name, "video_list") != 0) {
        error (_("Retrieving video list failed!"));
        debug ("\n%s", buffer);
        goto parser_free;
    }

    xml_node_t *videonode = listnode->child;
    while (videonode) {
        if (strcasecmp (videonode->name, "video") == 0) {
            char *title = NULL;
            char *mrl = NULL;
            char *thumbnail_mrl = NULL;
            xml_node_t *node = videonode->child;
            while (node) {
                if (strcasecmp (node->name, "title") == 0) {
                    title = node->data;
                }
                else if (strcasecmp (node->name, "url") == 0) {
                    mrl = node->data;
                }
                else if (strcasecmp (node->name, "thumbnail_url") == 0) {
                    thumbnail_mrl = node->data;
                }
                node = node->next;
            }

            if (mrl && title) {
                fileitem_t *item = filelist_add (filelist, title, mrl,
                                                 FILE_TYPE_YOUTUBE_VIDEO);
                if (thumbnail_mrl) {
                    item->thumbnail_mrl = ho_strdup (thumbnail_mrl);
                }
            }
        }
        videonode = videonode->next;
    }

    success = true;
  parser_free:
    xml_parser_free_tree (rootnode);
  buffer_free:
    ho_free (buffer);
    ho_free (mrl);

    return success;
}


static bool
youtube_pagelist_read (filelist_t * filelist)
{
    bool success = false;
    const char *status = NULL;
    char *mrl = ho_strdup_printf ("%s&page=1", filelist->mrl);

    int size = 0;
    char *buffer = read_entire_file (mrl, &size);
    if (!buffer) {
        goto buffer_free;
    }

    xml_node_t *rootnode;
    xml_parser_init (buffer, size, XML_PARSER_CASE_INSENSITIVE);
    if (xml_parser_build_tree (&rootnode) < 0) {
        error (_("Parsing '%s' failed!"), filelist->mrl);
        goto buffer_free;
    }

    if (strcasecmp (rootnode->name, "ut_response") != 0) {
        error (_("Root node of '%s' must be '%s'!"),
               filelist->mrl, "ut_response");
        debug ("\n%s", buffer);
        goto parser_free;
    }

    status = xml_parser_get_property (rootnode, "status");
    if (!status || (strcasecmp (status, "ok") != 0)) {
        error (_("Retrieving video list failed!"));
        debug ("\n%s", buffer);
        goto parser_free;
    }

    xml_node_t *listnode = rootnode->child;
    if (strcasecmp (listnode->name, "video_list") != 0) {
        error (_("Retrieving video list failed!"));
        debug ("\n%s", buffer);
        goto parser_free;
    }

    int total_pages = 0;
    xml_node_t *total = listnode->child;
    if ((strcasecmp (total->name, "total") != 0) || !total->data) {
        error (_("Retrieving video list failed!"));
        debug ("\n%s", buffer);
        goto parser_free;
    }
    else {
        total_pages = atoi (total->data) / 100 + 1;
    }

    if (total_pages > 10) {
        total_pages = 10;
    }

    int i = 0;
    for (i = 0; i < total_pages; i++) {
        char *title = ho_strdup_printf ("[%s %d]", _("Page"), i + 1);
        char *mrl = ho_strdup_printf ("%s&page=%d", filelist->mrl, i + 1);
        fileitem_t *item = filelist_add (filelist, title, mrl,
                                         FILE_TYPE_YOUTUBE_VFOLDER);
        item->thumbnail_mrl = ho_strdup (OXINE_VISUALDIR "/logo_youtube.png");
        ho_free (mrl);
        ho_free (title);
    }

    success = true;
  parser_free:
    xml_parser_free_tree (rootnode);
  buffer_free:
    ho_free (buffer);
    ho_free (mrl);

    return success;
}


static bool
youtube_categorylist_read (filelist_t * filelist)
{
    struct {
        int id;
        char *name;
    } categories[] = {
        {
        1, _("Films & Animation")}, {
        2, _("Autos & Vehicles")}, {
        23, _("Comedy")}, {
        24, _("Entertainment")}, {
        10, _("Music")}, {
        25, _("News & Politics")}, {
        22, _("People & Blogs")}, {
        15, _("Pets & Animals")}, {
        26, _("How-to & DIY")}, {
        17, _("Sports")}, {
        19, _("Travel & Places")}, {
        20, _("Gadgets & Games")}, {
    0, NULL},};

    int i = 0;
    for (i = 0; categories[i].id != 0; i++) {
        char *title = ho_strdup_printf ("[%s]", categories[i].name);
        char *mrl = ho_strdup_printf ("%s&category_id=%d", filelist->mrl,
                                      categories[i].id);
        fileitem_t *item = filelist_add (filelist, title, mrl,
                                         FILE_TYPE_YOUTUBE_VFOLDER);
        item->thumbnail_mrl = ho_strdup (OXINE_VISUALDIR "/logo_youtube.png");
        ho_free (mrl);
        ho_free (title);
    }

    return true;
}


static bool
youtube_list_read (filelist_t * filelist)
{
    {
        char *title = ho_strdup_printf ("[%s]", _("Categories"));
        char *mrl = ho_strdup_printf ("http://www.youtube.com/api2_rest"
                                      "?method=youtube.videos.list_by_category_and_tag"
                                      "&dev_id=%s&per_page=100&tag=", DEV_ID);
        fileitem_t *item = filelist_add (filelist, title, mrl,
                                         FILE_TYPE_YOUTUBE_VFOLDER);
        item->thumbnail_mrl = ho_strdup (OXINE_VISUALDIR "/logo_youtube.png");
        ho_free (mrl);
        ho_free (title);
    }

    {
        char *title = ho_strdup_printf ("[%s]", _("Featured Videos"));
        char *mrl = ho_strdup_printf ("http://www.youtube.com/api2_rest"
                                      "?method=youtube.videos.list_featured"
                                      "&dev_id=%s", DEV_ID);
        fileitem_t *item = filelist_add (filelist, title, mrl,
                                         FILE_TYPE_YOUTUBE_VFOLDER);
        item->thumbnail_mrl = ho_strdup (OXINE_VISUALDIR "/logo_youtube.png");
        ho_free (mrl);
        ho_free (title);
    }

    {
        char *title = ho_strdup_printf ("[%s - %s]", _("Popular"),
                                        _("today"));
        char *mrl = ho_strdup_printf ("http://www.youtube.com/api2_rest"
                                      "?method=youtube.videos.list_popular"
                                      "&dev_id=%s&time_range=day", DEV_ID);
        fileitem_t *item = filelist_add (filelist, title, mrl,
                                         FILE_TYPE_YOUTUBE_VFOLDER);
        item->thumbnail_mrl = ho_strdup (OXINE_VISUALDIR "/logo_youtube.png");
        ho_free (mrl);
        ho_free (title);
    }

    {
        char *title = ho_strdup_printf ("[%s - %s]", _("Popular"),
                                        _("this week"));
        char *mrl = ho_strdup_printf ("http://www.youtube.com/api2_rest"
                                      "?method=youtube.videos.list_popular"
                                      "&dev_id=%s&time_range=week", DEV_ID);
        fileitem_t *item = filelist_add (filelist, title, mrl,
                                         FILE_TYPE_YOUTUBE_VFOLDER);
        item->thumbnail_mrl = ho_strdup (OXINE_VISUALDIR "/logo_youtube.png");
        ho_free (mrl);
        ho_free (title);
    }

    {
        char *title = ho_strdup_printf ("[%s - %s]", _("Popular"),
                                        _("this month"));
        char *mrl = ho_strdup_printf ("http://www.youtube.com/api2_rest"
                                      "?method=youtube.videos.list_popular"
                                      "&dev_id=%s&time_range=month", DEV_ID);
        fileitem_t *item = filelist_add (filelist, title, mrl,
                                         FILE_TYPE_YOUTUBE_VFOLDER);
        item->thumbnail_mrl = ho_strdup (OXINE_VISUALDIR "/logo_youtube.png");
        ho_free (mrl);
        ho_free (title);
    }

    {
        char *title = ho_strdup_printf ("[%s - %s]", _("Popular"),
                                        _("always"));
        char *mrl = ho_strdup_printf ("http://www.youtube.com/api2_rest"
                                      "?method=youtube.videos.list_popular"
                                      "&dev_id=%s&time_range=all", DEV_ID);
        fileitem_t *item = filelist_add (filelist, title, mrl,
                                         FILE_TYPE_YOUTUBE_VFOLDER);
        item->thumbnail_mrl = ho_strdup (OXINE_VISUALDIR "/logo_youtube.png");
        ho_free (mrl);
        ho_free (title);
    }

    return true;
}


void
youtube_read (filelist_t * filelist)
{
    bool success = false;

    switch (filelist->parent_item->type) {
    case FILE_TYPE_YOUTUBE_VFOLDER:
        if (strstr (filelist->mrl, "list_by_category_and_tag")) {
            if (strstr (filelist->mrl, "&page=")) {
                success = youtube_videolist_read (filelist);
            }
            else if (strstr (filelist->mrl, "&category_id=")) {
                success = youtube_pagelist_read (filelist);
            }
            else {
                success = youtube_categorylist_read (filelist);
            }
        }
        else if (strstr (filelist->mrl, "youtube.videos.list_featured")) {
            success = youtube_videolist_read (filelist);
        }
        else if (strstr (filelist->mrl, "youtube.videos.list_popular")) {
            success = youtube_videolist_read (filelist);
        }
        else {
            success = youtube_list_read (filelist);
        }
        break;
    case FILE_TYPE_YOUTUBE_VIDEO:
        success = youtube_video_read (filelist);
        break;
    default:
        break;
    }

    if (!success) {
        ho_free (filelist->error);
        filelist->error = ho_strdup (_("Connection failure..."));
    }
}

#endif /* HAVE_YOUTUBE */
