// This file is part of the program FRYSK.
//
// Copyright 2005, 2006, 2007, 2008, Red Hat Inc.
//
// FRYSK is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; version 2 of the License.
//
// FRYSK is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with FRYSK; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
// 
// In addition, as a special exception, Red Hat, Inc. gives You the
// additional right to link the code of FRYSK with code not covered
// under the GNU General Public License ("Non-GPL Code") and to
// distribute linked combinations including the two, subject to the
// limitations in this paragraph. Non-GPL Code permitted under this
// exception must only link to the code of FRYSK through those well
// defined interfaces identified in the file named EXCEPTION found in
// the source code files (the "Approved Interfaces"). The files of
// Non-GPL Code may instantiate templates or use macros or inline
// functions from the Approved Interfaces without causing the
// resulting work to be covered by the GNU General Public
// License. Only Red Hat, Inc. may make changes or additions to the
// list of Approved Interfaces. You must obey the GNU General Public
// License in all respects for all of the FRYSK code and other code
// used in conjunction with FRYSK except the Non-GPL Code covered by
// this exception. If you modify this file, you may extend this
// exception to your version of the file, but you are not obligated to
// do so. If you do not wish to provide this exception without
// modification, you must delete this exception statement from your
// version and license this file solely under the GPL without
// exception.

package frysk.proc.live;

import java.util.Set;
import java.util.Iterator;
import frysk.event.Event;
import frysk.sys.Signal;
import frysk.sys.WaitBuilder;
import java.util.LinkedList;
import java.util.List;
import frysk.rsl.Log;
import frysk.sys.ProcessIdentifier;
import frysk.proc.TaskAttachedObserverXXX;

/**
 * Handles wait events generated by the wait builder.
 */

class LinuxWaitBuilder implements WaitBuilder {
    private static final Log fine = Log.fine(LinuxWaitBuilder.class);

    private final LinuxPtraceHost host;
    LinuxWaitBuilder(LinuxPtraceHost host) {
	this.host = host;
    }

    /**
     * Maintain a list of fscked up kernel waitpid events - where an
     * event for a pid arrives before it has been created - so that
     * they can be re-processed when there's a fork.
     */
    private List fsckedOrderedKernelEvents;
    /**
     * Run through the list of fscked up kernel waitpid events
     * attempting delivery of each in turn.
     */
    private void attemptDeliveringFsckedKernelEvents ()
    {
	if (fsckedOrderedKernelEvents != null) {
	    Event[] pending = (Event[]) fsckedOrderedKernelEvents.toArray (new Event[0]);
	    fsckedOrderedKernelEvents = null;
	    for (int i = 0; i < pending.length; i++) {
		pending[i].execute ();
	    }
	}
    }
    /**
     * Append the fscked-up stop event (it arrived when the task
     * didn't exist) to the fscked-up list.  Will get re-processed
     * later.
     */
    private void saveFsckedOrderedKernelStoppedEvent(final ProcessIdentifier aPid,
						     final Signal aSignal) {
	if (fsckedOrderedKernelEvents == null)
	    fsckedOrderedKernelEvents = new LinkedList();
	Event rescheduled = new Event() {
		final ProcessIdentifier pid = aPid;
		final Signal signal = aSignal;
		public void execute() {
		    LinuxWaitBuilder.this.stopped(pid, signal);
		}
		public String toString () {
		    return "" + super.toString () + ",stopped,pid=" + pid;
		}
	    };
	fine.log(this, "rescheduled", rescheduled);
	fsckedOrderedKernelEvents.add(rescheduled);
    }
    
    private LinuxPtraceTask get(ProcessIdentifier pid, String why) {
	fine.log(this, why, pid);
	return host.getTask(pid);
    }
    
    private void logMissing(String what, ProcessIdentifier pid) {
	String warning = ("No task for " + what + " pid " + pid
			  + " possibly caused by earlier [test] code"
			  + " failing to clean up all childen");
	System.out.println("WARNING: " + warning);
	fine.log(warning);
    }

    public void cloneEvent(ProcessIdentifier pid,
			   ProcessIdentifier clonePid) {
        // Find the task, create its new peer, and then tell the task
        // what happened. Note that hot on the heels of this event is
        // a clone.stopped event, and the clone Task must be created
        // before that event arrives.
        LinuxPtraceTask cloningTask = get(pid, "cloneEvent");
        // Create an attached, and running, clone of TASK.
        LinuxPtraceTask clonedTask = new LinuxPtraceTask(cloningTask, clonePid);
        cloningTask.processClonedEvent(clonedTask);
	attemptDeliveringFsckedKernelEvents();
    }

    public void forkEvent(ProcessIdentifier pid, ProcessIdentifier fork) {
        // Find the task, create the new process under it (well ok the
        // task's process) and then notify the task of what
        // happened. Note that hot on the heels of this fork event is
        // the child's stop event, the fork Proc must be created
        // before that event arrives.
        LinuxPtraceTask forkingTask = get(pid, "forkEvent");
        // Create an attached and running fork of TASK.
        LinuxPtraceProc forkedProc = new LinuxPtraceProc(forkingTask, fork);
        // The forked proc's only and main task.
        LinuxPtraceTask forkedTask = new LinuxPtraceTask
	    (forkingTask, forkedProc, (TaskAttachedObserverXXX) null);
        forkingTask.processForkedEvent(forkedTask);
	attemptDeliveringFsckedKernelEvents ();
    }
    
    public void exitEvent(ProcessIdentifier pid, Signal signal,
			  int status, boolean coreDumped) {
        LinuxPtraceTask task = get(pid, "exitEvent");
	if (task == null)
	    // Stray pid from uncontrolled fork.
	    logMissing("exited", pid);
	else
	    task.processTerminatingEvent(signal, status);
    }
    
    public void execEvent(ProcessIdentifier pid) {
        LinuxPtraceTask execingTask = get(pid, "execEvent");
	// On linux an exec event implicitly kills off all of the
	// processes other tasks; need to do that explicitly; there's
	// no "terminating" event as, by this point, the tasks are
	// totally gone.
	Set tasks = ((LinuxPtraceProc)execingTask.getProc()).getAllTasks();
	tasks.remove(execingTask);
	for (Iterator i = tasks.iterator(); i.hasNext(); ) {
	    LinuxPtraceTask doa = (LinuxPtraceTask) i.next();
	    doa.processTerminatedEvent(null, 0);
	}
	// Finally notify the exec.
        execingTask.processExecedEvent();
    }
    
    public void disappeared(ProcessIdentifier pid, Throwable w) {
        LinuxPtraceTask task = get(pid, "disappeared");
        // XXX Sometimes it seems it has already disappeared and this fails
        // Catch the failure, but not sure what to do with the failure right now
        // Failures were seen here while running frysk.hpd.TestRunCommand -r 20
        try {
            task.processDisappearedEvent(w);
        } catch (Exception e) {}
    }

    public void syscallEvent(ProcessIdentifier pid) {
        LinuxPtraceTask task = get(pid, "syscallEvent");
        task.processSyscalledEvent();
    }
    
    public void stopped(ProcessIdentifier pid, Signal sig) {
        LinuxPtraceTask task = get(pid, "stopped");
	if (task == null) {
	    // If there's no Task corresponding to TID, assume that
	    // the kernel fscked up its event ordering - notifying of
	    // a child-stop before it notified us of a child-create -
	    // push the event onto a second queue where it can be
	    // processed after a fork.
	    saveFsckedOrderedKernelStoppedEvent (pid, sig);
	    return;
	}
	task.processStoppedEvent(sig);
    }
    
    public void terminated(ProcessIdentifier pid, Signal signal,
			   int status, boolean coreDumped) {
        LinuxPtraceTask task = get(pid, "terminated");
	if (task == null)
	    // Stray pid from uncontrolled fork.
	    logMissing("terminated", pid);
	else
	    task.processTerminatedEvent(signal, status);
    }
}
