// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Functions: 
//   sum             - sum all the elements in an array.
//   prod            - multiply all of the elements in an array.
//   max             - find the maximum value in an array.
//   min             - find the minimum value in an array.
//   all             - returns true if all of the array's elements are != 0.
//   any             - returns true if any of the array's elements are != 0.
//   bitOr           - does a bitwise or of all of the elements.
//   bitAnd          - does a bitwise and of all of the elements.
//-----------------------------------------------------------------------------

#ifndef POOMA_ARRAY_REDUCTIONS_H
#define POOMA_ARRAY_REDUCTIONS_H

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include "Evaluator/Reduction.h"


//-----------------------------------------------------------------------------
// Forward definitions
//-----------------------------------------------------------------------------

template<int Dim, class T, class EngineTag>
class Array;


//-----------------------------------------------------------------------------
// Specific global reduction functions.
//-----------------------------------------------------------------------------

// Sum up the elements of an array.

template<int Dim, class T, class EngineTag>
T sum(const Array<Dim, T, EngineTag> &a)
{
  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, OpAddAssign(), a);
  return ret;
}

// Compute the product of the elements of an array.

template<int Dim, class T, class EngineTag>
T prod(const Array<Dim, T, EngineTag> &a)
{
  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, OpMultiplyAssign(), a);
  return ret;
}

// Find the smallest element of an array.

template<int Dim, class T, class EngineTag>
T min(const Array<Dim, T, EngineTag> &a)
{
  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, FnMinAssign(), a);
  return ret;
}

// Find the largest element of an array.

template<int Dim, class T, class EngineTag>
T max(const Array<Dim, T, EngineTag> &a)
{
  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, FnMaxAssign(), a);
  return ret;
}

// Report if all of the elments of the array are true.

template<int Dim, class T, class EngineTag>
bool all(const Array<Dim, T, EngineTag> &a)
{
  bool ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, FnAndAssign(), a);
  return ret;
}

// Report if some of the elments of the array are true.

template<int Dim, class T, class EngineTag>
bool any(const Array<Dim, T, EngineTag> &a)
{
  bool ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, FnOrAssign(), a);
  return ret;
}

// Bitwise-or all of the elements together.

template<int Dim, class T, class EngineTag>
T bitOr(const Array<Dim, T, EngineTag> &a)
{
  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, OpBitwiseOrAssign(), a);
  return ret;
}

// Bitwise-and all of the elements together.

template<int Dim, class T, class EngineTag>
T bitAnd(const Array<Dim, T, EngineTag> &a)
{
  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, OpBitwiseAndAssign(), a);
  return ret;
}

#endif

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Reductions.h,v $   $Author: swhaney $
// $Revision: 1.13 $   $Date: 2000/07/20 15:36:24 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
