// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// IO functions to support special matrix element types including
// Vector, Tensor (Full, Antisymmetric, Symmetric, Diagonal), and
// TinyMatrix.
//-----------------------------------------------------------------------------

#ifndef POOMA_IO_MATRIX_IO_H
#define POOMA_IO_MATRIX_IO_H

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include "Tiny/Vector.h"
#include "Tiny/Tensor.h"
#include "Tiny/TinyMatrix.h"
#include "IO/CTTI.h"
#include "IO/Serializers.h"
#include <string>

#include <stdio.h>

#if POOMA_NO_STRINGSTREAM
#include <strstream>
#else
#include <sstream>
#endif

//----------------------------------------------------------------------
// I/O Functions for Vector<Dim,T,Full>
//-----------------------------------------------------------
// CTTI function for Vector<Dim,T,Full>
//-----------------------------------------------------------

template <int Dim, class T>
std::string PoomaCTTI(const Vector<Dim,T,Full>&);
template <int Dim, class T>
std::string PoomaCTTI(const Vector<Dim,T,Full>&){
  char numbuf[10];
  sprintf(numbuf,"%d,\0",Dim);
  std::string strng= "Vector<"+
    std::string(numbuf)+ PoomaCTTI(T())+ ",Full>";
  return strng;
}

//-----------------------------------------------------------
// Serializers for Vector<Dim,T,Full>
//-----------------------------------------------------------

template <class Stream, int Dim, class T>
int serialize(Stream& s, const Vector<Dim,T,Full>& vec);
template <class Stream, int Dim, class T>
int serialize(Stream& s, const Vector<Dim,T,Full>& vec){
  int nBytes=0;
  // Write the dimension of the vector.
  int dim= Dim;
  nBytes+= serialize(s,dim);
  // Get a pointer to the data.
  const T* data= &vec(0);
  // Serialize all components together.
  nBytes+= serializeN(s,data,dim);
  return nBytes;
}

template <class Stream, int Dim, class T>
int deserialize(Vector<Dim,T,Full>& vec, Stream& s);
template <class Stream, int Dim, class T>
int deserialize(Vector<Dim,T,Full>& vec, Stream& s){
  int nBytes=0;
  // Read the dimension of the vector.
  int dim;
  nBytes+= deserialize(dim,s);
  // Insist that the dimensions match.
  PInsist((dim==Dim),
  "Error: source and target Vector dimensions do not match.");

  // Get a pointer to the data.
  T* data= &vec(0);
  // Deserialize all components together.
  nBytes+= deserializeN(data,s,dim);
  return nBytes;
}


template <int Dim, class T>
int serialSizeof(const Vector<Dim,T,Full>& vec);
template <int Dim, class T>
int serialSizeof(const Vector<Dim,T,Full>& vec){
  int nBytes= sizeof(int)+ Dim*sizeof(T);
  return nBytes;
}

//----------------------------------------------------------------------
// I/O Functions for TinyMatrix<D1,D2,T,Full>
//-----------------------------------------------------------
// CTTI function for TinyMatrix<D1,D2,T,Full>
//-----------------------------------------------------------

template <int D1, int D2, class T>
std::string PoomaCTTI(const TinyMatrix<D1,D2,T,Full>&);
template <int D1, int D2, class T>
std::string PoomaCTTI(const TinyMatrix<D1,D2,T,Full>&){
  char numbuf[10];
  sprintf(numbuf,"%d,%d,\0",D1,D2);
  std::string strng= "TinyMatrix<"+
    std::string(numbuf)+ PoomaCTTI(T())+ ",Full>";
  return strng;
}

//-----------------------------------------------------------
// Serializers for TinyMatrix<D1,D2,T,Full>
//-----------------------------------------------------------

template <class Stream, int D1, int D2, class T>
int serialize(Stream& s, const TinyMatrix<D1,D2,T,Full>& mat);
template <class Stream, int D1, int D2, class T>
int serialize(Stream& s, const TinyMatrix<D1,D2,T,Full>& mat){
  int nBytes=0;
  // Write the dimensions of the matrix.
  int d1= D1, d2=D2;
  int totalSize= d1*d2;
  nBytes+= serialize(s,d1);
  nBytes+= serialize(s,d2);
  // Get a pointer to the data.
  const T* data= &mat(0,0);
  // Serialize all components together.
  nBytes+= serializeN(s,data,totalSize);
  return nBytes;
}

template <class Stream, int D1, int D2, class T>
int deserialize(TinyMatrix<D1,D2,T,Full>& mat, Stream& s);
template <class Stream, int D1, int D2, class T>
int deserialize(TinyMatrix<D1,D2,T,Full>& mat, Stream& s){
  int nBytes=0;
  // Read the dimensions of the matrix.
  int d1, d2;
  int targetSize= D1*D2;
  nBytes+= deserialize(d1,s);
  nBytes+= deserialize(d2,s);
  int sourceSize= d1*d2;
  // Insist that the total sizes match.
  PInsist((targetSize==sourceSize),
  "Error: source and target TinyMatrix sizes do not match.");
  // Get a pointer to the data.
  T* data= &mat(0,0);
  // Deserialize all components together.
  nBytes+= deserializeN(data,s,targetSize);
  return nBytes;
}

template <int D1, int D2, class T>
int serialSizeof(const TinyMatrix<D1,D2,T,Full>& mat);
template <int D1, int D2, class T>
int serialSizeof(const TinyMatrix<D1,D2,T,Full>& mat){
  return D1*D2*sizeof(T)+2*sizeof(int);
}

//----------------------------------------------------------------------
// I/O Functions for Tensor classes
//----------------------------------------------------------------------
// These definitions don't exist in Tensor.h. Only the declarations occur.
// Perhaps they should be moved to Tensor.h.
//----------------------------------------------------------------------
class Full{};
class Symmetric{};
class Antisymmetric{};
class Diagonal{};

//----------------------------------------------------------------------
// CTTI functions for Tensor EngineTag classes
//-----------------------------------------------------------

inline std::string PoomaCTTI(const Full&);
inline std::string PoomaCTTI(const Full&){
  return std::string("Full");
}

inline std::string PoomaCTTI(const Symmetric&);
inline std::string PoomaCTTI(const Symmetric&){
  return std::string("Symmetric");
}

inline std::string PoomaCTTI(const Antisymmetric&);
inline std::string PoomaCTTI(const Antisymmetric&){
  return std::string("Antisymmetric");
}

inline std::string PoomaCTTI(const Diagonal&);
inline std::string PoomaCTTI(const Diagonal&){
  return std::string("Diagonal");
}

//-----------------------------------------------------------
// CTTI function for Tensor classes
//-----------------------------------------------------------

template <int Dim, class T, class EngineTag>
std::string PoomaCTTI(const Tensor<Dim,T,EngineTag>&);
template <int Dim, class T, class EngineTag>
std::string PoomaCTTI(const Tensor<Dim,T,EngineTag>&){
  typedef Tensor<Dim,T,EngineTag>::EngineTag_t EngineTag_t;
  char numbuf[10];
  sprintf(numbuf,"%d,\0",Dim);
  std::string strng= "Tensor<"+
    std::string(numbuf)+ PoomaCTTI(T())+ ","
    + PoomaCTTI(EngineTag_t())+ ">";
  return strng;
}
//-----------------------------------------------------------
// Classes for identifying TensorEngine type.
//-----------------------------------------------------------
// This class establishes the scope for the enumerated type
// contained within it.
class TensorTypeCode{
 public:
  enum TensorType{
    Full=0,
    Symmetric=1,
    Antisymmetric=2,
    Diagonal=3
  };
};

// Unspecialized class is completely empty.
template <class EngineTag>
class TensorTypeID{};

// The specialized classes:
template <>
class TensorTypeID<Full>{
 public:
  static const TensorTypeCode::TensorType
    engineType=TensorTypeCode::Full;
};

template <>
class TensorTypeID<Symmetric>{
 public:
  static const TensorTypeCode::TensorType
    engineType=TensorTypeCode::Symmetric;
};

template <>
class TensorTypeID<Antisymmetric>{
 public:
  static const TensorTypeCode::TensorType
    engineType=TensorTypeCode::Antisymmetric;
};

template <>
class TensorTypeID<Diagonal>{
 public:
  static const TensorTypeCode::TensorType
    engineType=TensorTypeCode::Diagonal;
};


//-----------------------------------------------------------
// Serializers for Tensor<D,T,EngineTag>
//-----------------------------------------------------------
template <class Stream, int Dim, class T, class EngineTag>
int serialize(Stream& s, const Tensor<Dim,T,EngineTag>& t);
template <class Stream, int Dim, class T, class EngineTag>
int serialize(Stream& s, const Tensor<Dim,T,EngineTag>& t){
  int nBytes=0;
  // Write the dimension.
  int dim= Dim;
  nBytes+= serialize(s,dim);
  // Total data size is dependent on the engine type.
  TensorStorageSize<Dim,EngineTag> tensorStorage;
  int storageSize= tensorStorage.Size;
  TensorTypeID<EngineTag> typeID;
  // Serialize the engine code as an integer.
  int code= typeID.engineType;
  nBytes+= serialize(s,code);
  // The direct approach:
  // Get a pointer to the data through the engine.
  const Tensor<Dim,T,EngineTag>::Engine_t& engine= t.engine();
  const T* data= engine.data();
  // Serialize all components together.
  nBytes+= serializeN(s,data,storageSize);
  return nBytes;
}

template <class Stream, int Dim, class T, class EngineTag>
int deserialize(Tensor<Dim,T,EngineTag>& t, Stream& s);
template <class Stream, int Dim, class T, class EngineTag>
int deserialize(Tensor<Dim,T,EngineTag>& t, Stream& s){
  // Read the dimension.
  int nBytes=0;
  int dim;
  nBytes+= deserialize(dim,s);
  // Read the source engine code.
  int scode;
  nBytes+= deserialize(scode,s);
  TensorTypeID<EngineTag> typeID;
  TensorTypeCode::TensorType targetCode=
    typeID.engineType;
  int tcode= targetCode;
  // Insist that the dimensions and engine types match.
  PInsist(((dim==Dim) && (tcode==scode)),
  "Error: source and target Tensor objects do not match.");
  // Total data size is dependent on the engine type.
  TensorStorageSize<Dim,EngineTag> tensorStorage;
  int storageSize= tensorStorage.Size;
  // The direct approach:
  // Get a pointer to the data through the engine.
  Tensor<Dim,T,EngineTag>::Engine_t& engine= t.engine();
  T* data= engine.data();
  // Deserialize all components together.
  nBytes+= deserializeN(data,s,storageSize);
  return nBytes;
}

template <int Dim, class T, class EngineTag>
int serialSizeof(const Tensor<Dim,T,EngineTag>& t);
template <int Dim, class T, class EngineTag>
int serialSizeof(const Tensor<Dim,T,EngineTag>& t){
  int nBytes=0;
  // Include the dimension and the type code.
  nBytes+= 2*sizeof(int);
  // Total data size is dependent on the engine type.
  TensorStorageSize<Dim,EngineTag> tensorStorage;
  nBytes+= tensorStorage.Size*serialSizeof(T());
  return nBytes;
}

#endif // POOMA_IO_MATRIX_IO_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: MatrixIO.h,v $   $Author: swhaney $
// $Revision: 1.5 $   $Date: 2000/07/20 20:47:37 $
// ----------------------------------------------------------------------
// ACL:rcsinfo





