// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// These serializers are used to write an ObjectLoc instance.
//-----------------------------------------------------------------------------

template <class Stream>
int serialize(Stream& s, const ObjectLoc& object){
  // Object ID type is standardized to type long
  int nBytes= serialize(s, object.objectID);
  // File location type is standardized to long.
  nBytes+= serialize(s, object.loc);
  return nBytes;
}

template <class Stream>
int deserialize(ObjectLoc& object, Stream& s){
  // Object ID type is standardized to type long
  int nBytes= deserialize(object.objectID, s);
  // Deserialize just the file location.
  nBytes+= deserialize(object.loc, s);
  return nBytes;
}

//-----------------------------------------------------------------------------
// Global store()/retrieve routines are templated on type and are recursively
// called as a result of an initiating call by the object set.
// Signatures:
// 	FileLoc_t store(ObjectStorageSet& set, const T& t) -- will
//   write the object t at the end of the file and return the location.
// 	void retrieve(T& t, const FileLoc& loc, ObjectStorageSet& set) -- will
//   read the object t from the given location.
// These are the generic functions. Specializations must be written
//   if more complex behavior is needed for specific types.
//-----------------------------------------------------------------------------
template <class T>
FileLoc_t store(ObjectStorageSet& set, const T& t){
  // Store the object t at the end of the file.
  std::fstream* file= set.dataFile();
  // Seek to the end of the file.
  file->seekp(0,std::ios::end);
  // Record the location.
  FileLoc_t loc= file->tellp();
  // Write the object.
  serialize(*file, t);
  // return the location.
  return loc;
}

template <class T>
void retrieve(T& t, const FileLoc_t& loc, ObjectStorageSet& set){
  // Retrieve a copy of the object beginning at the given storage location
  //   and instantiate t.
  std::fstream* file= set.dataFile();
  // Seek to the specified location if different than current location.
  FileLoc_t filepos= file->tellg();
  if(filepos!=loc){
    file->seekg(loc, std::ios::beg);
  }
  deserialize(t,*file);
}

//-----------------------------------------------------------------------------
// AttributeMap member functions.
//-----------------------------------------------------------------------------
// Default constructor.
//-----------------------------------------------------------------------------
template <class T>
AttributeMap<T>::AttributeMap(){
  attributeName_m="";
  attributeType_m= "";
}
	
//-----------------------------------------------------------------------------
// Main constructor.
//-----------------------------------------------------------------------------
template <class T>
AttributeMap<T>::AttributeMap(const std::string& attributeName,
			      const std::string& attributeType){
  init(attributeName, attributeType);
}
	
//-----------------------------------------------------------------------------
// Copy constructor.
//-----------------------------------------------------------------------------
template <class T>
AttributeMap<T>::AttributeMap(const AttributeMap& amap){
  copy(amap);
}
	
//-----------------------------------------------------------------------------
// Assignment operator.
//-----------------------------------------------------------------------------
template <class T>
AttributeMap<T>& AttributeMap<T>::operator=(const AttributeMap<T>& amap){
  copy(amap);
  return *this;
}
	
//-----------------------------------------------------------------------------
// Destructor is a no-op.
//-----------------------------------------------------------------------------
template <class T>
AttributeMap<T>::~AttributeMap(){}
	
//-----------------------------------------------------------------------------
// Initializer.
//-----------------------------------------------------------------------------
template <class T>
void AttributeMap<T>::init(const std::string& attributeName,
			   const std::string& attributeType){
  attributeName_m= attributeName;
  attributeType_m= attributeType;
}

//-----------------------------------------------------------------------------
// Inserts a new attribute value pair.
//-----------------------------------------------------------------------------
template <class T>
void AttributeMap<T>::insert(const T& t, const ObjectID_t id){
  // Insert the pair int the multimap.
  map_m.insert(Value_t(t,id));
  // Assign the attribute to the inverse map.
  mapInv_m[id]= t;
}

//-----------------------------------------------------------------------------
// Creates a set of object IDs corresponding to the objects that match
// the given attribute value.
//-----------------------------------------------------------------------------
template <class T>
ObjectSubset AttributeMap<T>::selectAttribute(const T& t){
  Range_t range= equalRange(t);
  iterator iter;
  ObjectSubset s;
  // Make sure the set instance is clean.
  s.clear();
  // Place each ID in the set in the matching range.
  for(iter= range.first; iter!= range.second; iter++){
    s.insert((*iter).second);
  }
  return s;
}

//-----------------------------------------------------------------------------
// Utility to copy an attribute map.
//-----------------------------------------------------------------------------
template <class T>
void AttributeMap<T>::copy(const AttributeMap& amap){
  attributeName_m= amap.attributeName_m;
  attributeType_m= amap.attributeName_m;
  map_m= amap.map_m;
  mapInv_m= amap.mapInv_m;
}

//-----------------------------------------------------------------------------
// serialize() function for AttributeMap<T>
//-----------------------------------------------------------------------------
template <class Stream, class T>
int serialize(Stream& s, AttributeMap<T>& amap){
  int num= amap.size();
  std::string aname= amap.attributeName(),
    atype= amap.attributeType();
  int nBytes= 0;
  nBytes+= serialize(s,aname);
  nBytes+= serialize(s,atype);
  nBytes+= serialize(s,num);
  AttributeMap<T>::iterator iter;
  T attribute;
  long id;
  for(iter= amap.begin(); iter!= amap.end(); iter++){
    attribute= (*iter).first;
    id= (*iter).second;
    nBytes+= serialize(s,attribute);
    nBytes+= serialize(s,id);
  }
  return nBytes;
}

//-----------------------------------------------------------------------------
// deserialize() function for AttributeMap<T>
//-----------------------------------------------------------------------------
template <class Stream, class T>
int deserialize(AttributeMap<T>& amap, Stream& s){
  amap.clear();
  std::string aname, atype;
  int nBytes=0;
  nBytes+= deserialize(aname,s);
  nBytes+= deserialize(atype,s);
  amap.init(aname,atype);
  int num;
  T attribute;
  long id;
  nBytes+= deserialize(num,s);
  for(int i=0;i<num;i++){
    nBytes+= deserialize(attribute,s);
    nBytes+= deserialize(id,s);
    amap.insert(attribute,id);
  }
  return nBytes;
}

//-----------------------------------------------------------------------------
// serialSizeof() function for AttributeMap<T>
//-----------------------------------------------------------------------------
template <class T>
int serialSizeof(AttributeMap<T>& amap){
  int num= amap.size();
  std::string aname= amap.attributeName(),
    atype= amap.attributeType();
  int nBytes= 0;
  nBytes+= serialSizeOf(aname);
  nBytes+= serialSizeOf(atype);
  nBytes+= serialSizeOf(num);
  AttributeMap<T>::iterator iter;
  T attribute;
  for(iter= amap.begin(); iter!= amap.end(); iter++){
    attribute= (*iter).first;
    nBytes+= serialSizeOf(s,attribute);
  }
  nBytes+= num*sizeof(long);
  return nBytes;
}


//-----------------------------------------------------------------------------

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ObjectSets.cpp,v $   $Author: julianc $
// $Revision: 1.4 $   $Date: 2000/04/22 22:41:28 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
