//-----------------------------------------------------------------------------
// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <string>
#include <iostream>
#include <fstream>
#include "Pooma/Pooma.h"
#include "Utilities/Tester.h"
#include "Engine/BrickEngine.h"
#include "Engine/CompressibleBrick.h"
#include "IO/EngineIO.h"
#include "Engine/RemoteEngine.h"
#include "IO/RemoteEngineIO.h"
#include "Domain/Loc.h"
#include "Domain/Interval.h"
#include "Partition/UniformGridPartition.h"
#include "Layout/UniformGridLayout.h"
#include "Engine/MultiPatchEngine.h"
#include "Array/Array.h"
#include "Array/Reductions.h"
#include "Tiny/Vector.h"
#include "IO/ArrayIO.h"
//-----------------------------------------------------------------------------
// Test of Pooma remote engine serializers
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);
  tester.out() <<"Test remote engine serializers:"<< std::endl;

#if POOMA_CHEETAH

  // Open a file for the test.
  std::string filename;
  int context= Pooma::context();
  char numbuf[8];
  sprintf(numbuf,"%d\0",context);
  filename= "RemoteEngineSerializerTest_"+
    std::string(numbuf)+ ".dat";
  std::fstream datafile(filename.c_str(),
                        std::ios::out | std::ios::binary);

  // Create remote arrays.
  Interval<3> I3(6,6,6);
  Array<3> a0(I3), b0;
  Array<3, double, Remote<CompressibleBrick> > a1(I3), b1;
  
  Loc<3> blocks(1,1,2);
  UniformGridPartition<3> partition(blocks);   
  UniformGridLayout<3> layout(I3, partition, DistributedTag());
  
  Array<3, double, MultiPatch<UniformTag,Remote<Brick> > > 
    a2(layout), b2;
  Array<3, double, MultiPatch<UniformTag,Remote<CompressibleBrick> > > 
    a3(layout), b3;

  a0 = 0.0;
  a1 = 1.0;
  a2 = 2.0;
  a3 = 3.0;
  tester.out()<<"a0="<<a0<<std::endl;
  tester.out()<<"a1="<<a1<<std::endl;
  tester.out()<<"a2="<<a2<<std::endl;
  tester.out()<<"a3="<<a3<<std::endl;
  
  // Use Array I/O to serialize the engines.

  int ksize;
  ksize= serialize(datafile,a0);
  tester.out()<<"ksize="<<ksize<<std::endl;
  ksize= serialize(datafile,a1);
  tester.out()<<"ksize="<<ksize<<std::endl;
  ksize= serialize(datafile,a2);
  tester.out()<<"ksize="<<ksize<<std::endl;
  ksize= serialize(datafile,a3);
  tester.out()<<"ksize="<<ksize<<std::endl;

  // Close the file.
  datafile.close();

  // Reopen as another file object.
  std::fstream dfile(filename.c_str(),
                     std::ios::in | std::ios::binary);

  // Deserialize the arrays to the targets
  deserialize(b0,dfile);
  tester.out()<<"b0="<<b0<<std::endl;
  deserialize(b1,dfile);
  tester.out()<<"b1="<<b1<<std::endl;
  deserialize(b2,dfile);
  tester.out()<<"b2="<<b2<<std::endl;
  deserialize(b3,dfile);
  tester.out()<<"b3="<<b3<<std::endl;

  // Check the results.
  bool result;
  result= all(b0 == 0.0);
  tester.out()<<"result="<<result<<std::endl;
  tester.check(result);
  result= all(b1 == 1.0);
  tester.out()<<"result="<<result<<std::endl;
  tester.check(result);
  result= all(b2 == 2.0);
  tester.out()<<"result="<<result<<std::endl;
  tester.check(result);
  result= all(b3 == 3.0);
  tester.out()<<"result="<<result<<std::endl;
  tester.check(result);

  dfile.close();

#endif // POOMA_CHEETAH

  int retval = 
    tester.results("Test of remote engine serializers");
  Pooma::finalize();
  return retval;
}











