// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// RectilinearMeshData<Dim, CoordinateSystems, T> template definitions.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include files
//-----------------------------------------------------------------------------

#include "Meshes/RectilinearMesh.h"

#include <algorithm>


///////////////////////////////////////////////////////////////////////////////
//
// RectilinearMeshData<Dim, CoordinateSystem, T> members.
//
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Given a point in space, find the nearest vertex in the mesh
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
Loc<Dim>
RectilinearMeshData<Dim, CoordinateSystem, T>::
nearestVertex(const PointType_t &point) const
{
  Loc<Dim> loc;
  for (int d = 0; d < Dim; d++)
    {
      // Note: we assume the physical domain is zero-based.
      
      const T *start = &axisVertPositions_m(d)(0);
      const T *finish = start + physicalDomain()[d].length();
      
      const T *p = std::lower_bound(start, finish, point(d));

#if POOMA_BOUNDS_CHECK  
      PInsist(p != finish, 
        "Rectilinear::nearestVertex(): point is outside mesh.");
#endif

      // The lower_bound function returns the first element that is not
      // less than the point we're searching for.
      
      int i = static_cast<int>(std::distance(start, p));
      if (*p == point(d) || *p - point(d) < point(d) - *(p - 1))
        loc[d] = i;
      else
        loc[d] = i - 1;
    }
  
  return loc;
}

//-----------------------------------------------------------------------------
// Given a point in space, find the nearest vertex in the mesh with all vertex
// coordinates below the point.
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
Loc<Dim>
RectilinearMeshData<Dim, CoordinateSystem, T>::
vertexBelow(const PointType_t &point) const
{
  Loc<Dim> loc;
  for (int d = 0; d < Dim; d++)
    {
      // Note: we assume the physical domain is zero-based.
      
      const T *start = &axisVertPositions_m(d)(0);
      const T *finish = start + physicalDomain()[d].length();
      
      const T *p = std::lower_bound(start, finish, point(d));

#if POOMA_BOUNDS_CHECK  
      PInsist(p != finish, 
        "Rectilinear::vertexBelow(): point is outside mesh.");
#endif

      // The lower_bound function returns the first element that is not
      // less than the point we're searching for.
      
      int i = static_cast<int>(std::distance(start, p));
      if (*p == point(d))
        loc[d] = i;
      else
        loc[d] = i - 1;
    }
  
  return loc;
}


//-----------------------------------------------------------------------------
// Performs initialization of the compute-base arrays.
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
void 
RectilinearMeshData<Dim, CoordinateSystem, T>::
initializeFunctionArrays()
{
  cellVolumes_m.engine().setFunctor(VolumeFunctor(*this));
  vertexPositions_m.engine().setFunctor(PositionFunctor(*this));
  vertexDeltas_m.engine().setFunctor(SpacingFunctor(*this));
  
  Vector<2*Dim, PointType_t> normals;
  normals = 0.0;
  for (int d = 0; d < Dim; d++)
    {
      normals(2 * d)(d) = -1.0;
      normals(2 * d + 1)(d) = 1.0;
    }
  cellSurfaceNormals_m.engine().setConstant(normals);
}

//-----------------------------------------------------------------------------
// Sets up spacings and vertex positions.
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
template <class EngineTag>
void 
RectilinearMeshData<Dim, CoordinateSystem, T>::
initializeSpacingsPositions(const Vector<Dim, Array<1, T, EngineTag> > &
  spacings)
{
  for (int d = 0; d < Dim; d++) 
    {
      nVerts_m(d) = physicalDomain()[d].size();
      nCells_m(d) = physicalDomain()[d].size() - 1;

      // Spacings arrays are sized as number of cells.
    
      spacings_m(d).initialize(totalCellDomain()[d]);

      // Vertex positions are defined for number of verts, not cells.
    
      axisVertPositions_m(d).initialize(totalDomain()[d]);
      
      // Compute the vertex positions. It would be nice if we had
      // a partial sum function.  Also copy in the spacings.
      
      axisVertPositions_m(d)(0) = origin_m(d);
      for (int i = 0; i < nCells_m(d); ++i)
	{
	  spacings_m(d)(i) = spacings(d)(i);
	  axisVertPositions_m(d)(i + 1) = 
	    axisVertPositions_m(d)(i) + spacings(d)(i);
	}

      // Use mesh boundary conditions to fill in the guard elements.
      
      updateMeshGuardElements(2 * d);
      updateMeshGuardElements(2 * d + 1);
    }
}


//-----------------------------------------------------------------------------
// Sets up spacings and vertex positions for constant cell spacing
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
void
RectilinearMeshData<Dim, CoordinateSystem, T>::
initializeSpacingsPositions(const PointType_t &spacings)
{
  for (int d = 0; d < Dim; d++)
    {
      nVerts_m(d) = physicalDomain()[d].size();
      nCells_m(d) = physicalDomain()[d].size() - 1;

      // Spacings arrays are sized as number of cells.

      spacings_m(d).initialize(totalCellDomain()[d]);

      // Vertex positions are defined for number of verts, not cells.

      axisVertPositions_m(d).initialize(totalDomain()[d]);

      // Compute the vertex positions. It would be nice if we had
      // a partial sum function.  Also copy in the spacings.

      axisVertPositions_m(d)(0) = origin_m(d);
      for (int i = 0; i < nCells_m(d); ++i)
        {
          spacings_m(d)(i) = spacings(d);
          axisVertPositions_m(d)(i + 1) =
            axisVertPositions_m(d)(i) + spacings(d);
        }

      // Use mesh boundary conditions to fill in the guard elements.

      updateMeshGuardElements(2 * d);
      updateMeshGuardElements(2 * d + 1);
    }
}


//-----------------------------------------------------------------------------
// Helper function to update guard layer values of mesh spacings.
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
void 
RectilinearMeshData<Dim, CoordinateSystem, T>::
updateMeshGuardElements(int face)
{
  // Apply the current state of the mesh BC to fill "guard elements" of
  // spacings_m Array and axisVertPositions_m Array.

  int d = face / 2;
  int guardCell, guardVert, guardLayer;

  // The following bitwise AND logical test returns true if face is odd
  // (meaning the "high" or "right" face in the numbering convention) and 
  // returns false if face is even (meaning the "low" or "left" face in 
  // the numbering convention).
  
  if (face & 1) 
    {
      // "High" guard elements.
      
      switch (meshBCs_m(face)) 
        {
          case linearExtrapolate:
            {
              T lastSpacing = spacings_m(d)(nCells_m(d) - 1);
              for (guardCell = nCells_m(d); 
                guardCell < nCells_m(d) + guardLayers().upper(d); 
	        guardCell++) 
	        {
	          spacings_m(d)(guardCell) = lastSpacing;
	          guardVert = guardCell + 1;
	          axisVertPositions_m(d)(guardVert) = 
	            axisVertPositions_m(d)(guardVert - 1) + lastSpacing;
                }   
              break;
            }
          case periodic:
            {
              // Last vertex is identified with first.
      
              axisVertPositions_m(d)(nVerts_m(d) - 1) = 
                axisVertPositions_m(d)(0);
              for (guardCell = nCells_m(d); 
                guardCell < nCells_m(d) + guardLayers().upper(d); 
	        guardCell++) 
	        {
	          spacings_m(d)(guardCell) = 
	            spacings_m(d)(guardCell - nCells_m(d));
	          guardVert = guardCell + 1;
	          axisVertPositions_m(d)(guardVert) = 
	            axisVertPositions_m(d)(guardVert - nCells_m(d));
                }
              break;
            }
          case reflective:
            {
              for (guardCell = nCells_m(d), guardLayer = 0; 
	        guardCell < nCells_m(d) + guardLayers().upper(d), 
	        guardLayer < guardLayers().upper(d); 
	        guardCell++, guardLayer++) 
	        {
	          spacings_m(d)(guardCell) = 
	            spacings_m(d)(nCells_m(d) - guardLayer);
	          guardVert = guardCell + 1;
	          axisVertPositions_m(d)(guardVert) = 
	            axisVertPositions_m(d)(guardVert -1) + 
	            spacings_m(d)(guardCell);
                }
              break;
            }
        }
    }
  else 
    {
      // "Low" guard elements.
      
      switch (meshBCs_m(face)) 
        {
          case linearExtrapolate:
            {
              T lastSpacing = spacings_m(d)(0);
              for (guardCell = -1; guardCell >= -guardLayers().lower(d); 
                guardCell--)
                {
	          spacings_m(d)(guardCell) = lastSpacing;
	          guardVert = guardCell;
	          axisVertPositions_m(d)(guardVert) = 
	            axisVertPositions_m(d)(guardVert + 1) - lastSpacing;
                }
              break;
            }
          case periodic:
            {
              for (guardCell = -1; guardCell >= -guardLayers().lower(d); 
                guardCell--) 
                {
	          spacings_m(d)(guardCell) = 
	            spacings_m(d)(guardCell + nCells_m(d));
	          guardVert = guardCell;
	          axisVertPositions_m(d)(guardVert) = 
	            axisVertPositions_m(d)(guardVert + 
	              nCells_m(d));
                }
              break;
            }
          case reflective:
            {
              for (guardCell = -1, guardLayer = 0; 
	        guardCell >= -guardLayers().lower(d), 
	        guardLayer < guardLayers().lower(d); 
	        guardCell--, guardLayer++) 
	        {
	          spacings_m(d)(guardCell) = spacings_m(d)(guardLayer + 1);
	          guardVert = guardCell + 1;
	          axisVertPositions_m(d)(guardVert) = 
	            axisVertPositions_m(d)(guardVert + 1) -
	            spacings_m(d)(guardCell);
                }
              break;
            }
        }
    }
}

//-----------------------------------------------------------------------------
// General constructor.
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
template<class EngineTag>
RectilinearMeshData<Dim, CoordinateSystem, T>::
RectilinearMeshData(const Domain_t &pd, const PointType_t &origin,
  const Vector<Dim, Array<1, T, EngineTag> > &spacings,
  const BCTypes_t &mbc)
: initialized_m(true),
  physicalDomain_m(pd),
  guardLayers_m(guardLayerSizes(), guardLayerSizes()),
  totalDomain_m(guardLayers().addGuardLayersToDomain(physicalDomain())),
  totalCellDomain_m(
    guardLayers().addGuardLayersToDomain(physicalCellDomain())),
  cellVolumes_m(totalCellDomain()),
  vertexDeltas_m(totalCellDomain()),
  vertexPositions_m(totalDomain()),
  cellSurfaceNormals_m(totalCellDomain()),
  origin_m(origin),
  meshBCs_m(mbc)
{
  initializeFunctionArrays();
  initializeSpacingsPositions(spacings);
}

//-----------------------------------------------------------------------------
// Constant-spacings General constructor.
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
RectilinearMeshData<Dim, CoordinateSystem, T>::
RectilinearMeshData(const Domain_t &pd, const PointType_t &origin,
  const PointType_t &spacings)
: initialized_m(true),
  physicalDomain_m(pd),
  guardLayers_m(guardLayerSizes(), guardLayerSizes()),
  totalDomain_m(guardLayers().addGuardLayersToDomain(physicalDomain())),
  totalCellDomain_m(
    guardLayers().addGuardLayersToDomain(physicalCellDomain())),
  cellVolumes_m(totalCellDomain()),
  vertexDeltas_m(totalCellDomain()),
  vertexPositions_m(totalDomain()),
  cellSurfaceNormals_m(totalCellDomain()),
  origin_m(origin),
  meshBCs_m(BCTypes_t(linearExtrapolate))
{
  initializeFunctionArrays();
  initializeSpacingsPositions(spacings);
}

//-----------------------------------------------------------------------------
// Copy constructor.
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
RectilinearMeshData<Dim, CoordinateSystem, T>::
RectilinearMeshData(const This_t &model)
: initialized_m(model.initialized()),
  physicalDomain_m(model.physicalDomain()),
  guardLayers_m(model.guardLayers()),
  totalDomain_m(model.totalDomain()),
  totalCellDomain_m(model.totalCellDomain()),
  origin_m(model.origin()),
  meshBCs_m(model.meshBCs_m),
  cellVolumes_m(model.totalCellDomain()),
  vertexDeltas_m(model.totalCellDomain()),
  vertexPositions_m(model.totalDomain()),
  cellSurfaceNormals_m(model.totalCellDomain()),
  coordSys_m(model.coordinateSystem())
{
  initializeFunctionArrays();
  initializeSpacingsPositions(spacings);
}

//-----------------------------------------------------------------------------
// Copy assignment operator.
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
typename RectilinearMeshData<Dim,CoordinateSystem,T>::This_t &
RectilinearMeshData<Dim, CoordinateSystem, T>::
operator=(const This_t &rhs)
{
  if (&rhs != this)
    {
      initialized_m = rhs.initialized();
      physicalDomain_m = rhs.physicalDomain();
      guardLayers_m = rhs.guardLayers();
      totalDomain_m = rhs.totalDomain();
      totalCellDomain_m = rhs.totalCellDomain();
      origin_m = rhs.origin();
      meshBCs_m = rhs.meshBCs_m;
      cellVolumes_m = rhs.cellVolumes();
      vertexDeltas_m = rhs.vertexDeltas();
      vertexPositions_m = rhs.vertexPositions();
      cellSurfaceNormals_m = rhs.cellSurfaceNormals();
      spacings_m = rhs.spacings_m;
      axisVertPositions_m = rhs.axisVertPositions_m;
    }

  return *this;
}

//-----------------------------------------------------------------------------
// Initialize function. Can be used when default constructor was invoked to
// build the object.
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
template<class EngineTag>
void 
RectilinearMeshData<Dim, CoordinateSystem, T>::
initialize(const Domain_t &pd, const PointType_t &origin,
  const Vector<Dim, Array<1, T, EngineTag> > &spacings,
  const BCTypes_t &mbc)
{
  // Set up the data members.
    
  physicalDomain_m = pd;
  guardLayers_m.initialize(guardLayerSizes(), guardLayerSizes());
  totalDomain_m = guardLayers().addGuardLayersToDomain(physicalDomain());
  totalCellDomain_m = 
    guardLayers().addGuardLayersToDomain(physicalCellDomain());
  cellVolumes_m.setDomain(totalCellDomain());
  vertexDeltas_m.setDomain(totalCellDomain());
  vertexPositions_m.setDomain(totalDomain());
  cellSurfaceNormals_m.setDomain(totalCellDomain());
  origin_m = origin;
  meshBCs_m = mbc;

  initializeFunctionArrays();
  initializeSpacingsPositions(spacings);

  // Note that we are now initialized.

  initialized_m = true;
}      

//-----------------------------------------------------------------------------
// Initialize function. Can be used when default constructor was invoked to
// build the object.  This is for the case of constant cell spacing.
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
void
RectilinearMeshData<Dim, CoordinateSystem, T>::
initialize(const Domain_t &pd, const PointType_t &origin,
  const PointType_t &spacings)
{
  // Set up the data members.

  physicalDomain_m = pd;
  guardLayers_m.initialize(guardLayerSizes(), guardLayerSizes());
  totalDomain_m = guardLayers().addGuardLayersToDomain(physicalDomain());
  totalCellDomain_m =
    guardLayers().addGuardLayersToDomain(physicalCellDomain());
  cellVolumes_m.setDomain(totalCellDomain());
  vertexDeltas_m.setDomain(totalCellDomain());
  vertexPositions_m.setDomain(totalDomain());
  cellSurfaceNormals_m.setDomain(totalCellDomain());
  origin_m = origin;
  meshBCs_m = BCTypes_t(linearExtrapolate);

  initializeFunctionArrays();
  initializeSpacingsPositions(spacings);

  // Note that we are now initialized.

  initialized_m = true;
}


///////////////////////////////////////////////////////////////////////////////
//
// RectilinearMesh<Dim, CoordinateSystem, T> members.
//
///////////////////////////////////////////////////////////////////////////////

// Print a RectilinearMes on an output stream

template <int Dim, class CoordinateSystem, class T>
template <class Ostream>
void 
RectilinearMesh<Dim, CoordinateSystem, T>::
print(Ostream &ostr) const 
{
  ostr << "RectilinearMesh:" << '\n';
  ostr << "   Dimension: " << Dim << '\n';
  ostr << "   CoordinateSystem: " << coordinateSystem() << '\n';
  ostr << "   Physical domain: " << physicalDomain() << '\n';
  ostr << "   Total domain: " << totalDomain() << '\n';
  ostr << "   GuardLayers: " << guardLayers() << '\n';
  ostr << "   Origin: " << origin() << '\n';
  ostr << "   Axis vertex positions: " << data_m->axisVertPositions();
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: RectilinearMesh.cpp,v $   $Author: swhaney $
// $Revision: 1.15 $   $Date: 2000/03/07 13:17:41 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
