// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// RectilinearMesh test 1: Basic stuff.
//-----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

#include <iostream>

static bool OK = true;
static int checkNum = 1;

template<class T>
inline void check(const T &ans, const T &correct, Pooma::Tester &tester)
{
  bool ok = (ans == correct);
  tester.out() << "Check #" << checkNum << std::endl;
  tester.out() << "Answer:  " << ans << std::endl;
  if (!ok)
    {
      tester.out() << "Correct: " << correct << std::endl;
    }

  checkNum++;      
  OK = (OK && ok);
  tester.check(OK);
}

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc, argv);

  const int D = 2; // Dimensionality
  int d, i;

  // Create the total domain:
  Interval<D> domain, cellDomain;
  for (d = 0; d < D; d++) {
    domain[d] = Interval<1>(3 + d);
    cellDomain[d] = Interval<1>(3 + d - 1);
  }

  Vector<D,int> origin;
  for (d = 0; d < D; d++) { origin(d) = 0.0; }

  Vector<D, Array<1,int> > spacings;
  for (d = 0; d < D; d++) {
    spacings(d).initialize(cellDomain[d]);
    for (i = 0; i < cellDomain[d].size(); i++) {
      spacings(d)(i) = (i+1)*10;
    }
  }

  RectilinearMesh<D, Cartesian<D>, int> mesh(domain, origin, spacings);

  Interval<2> pd(3, 4), td(Interval<1>(-1,3), Interval<1>(-2,5));
  Interval<2> pcd(2, 3), tcd(Interval<1>(-1,2), Interval<1>(-2,4));
  check(mesh.physicalDomain(), pd, tester); 
  check(mesh.totalDomain(), td, tester);
  check(mesh.physicalCellDomain(), pcd, tester);
  check(mesh.totalCellDomain(), tcd, tester);

  check(mesh.totalVolumeOfCells(), 1800, tester);
  check(mesh.totalVolumeOfCells(mesh.totalCellDomain()), 8400, tester);

  check(mesh.vertexPositions().read(0,0), origin, tester);
  check(mesh.vertexPositions().read(3,5), Vector<2,int>(50,120), tester);
  check(mesh.cellSurfaceNormals().read(0,0)(2), Vector<2,int>(0,-1), tester);
  
  check(mesh.cellContaining(origin), Loc<2>(0,0), tester);
  check(mesh.cellContaining(Vector<2,int>(11,12)), Loc<2>(1,1), tester);
  check(mesh.vertexBelow(Vector<2,int>(11,12)), Loc<2>(1,1), tester);
  check(mesh.nearestVertex(Vector<2,int>(11,12)), Loc<2>(1,1), tester);
  check(mesh.cellContaining(Vector<2,int>(19,31)), Loc<2>(1,2), tester);
  check(mesh.nearestVertex(Vector<2,int>(19,31)), Loc<2>(1,2), tester);
  
  int ret = tester.results("rm_test1");
  Pooma::finalize();
    
  return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: rm_test1.cpp,v $   $Author: julianc $
// $Revision: 1.8 $   $Date: 2000/07/20 23:38:23 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
