// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_THREADS_POOMAMUTEX_H
#define POOMA_THREADS_POOMAMUTEX_H

//-----------------------------------------------------------------------------
// Class:
//   Pooma::Mutex_t
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview: 
// Pooma::Mutex_t is a typedef for the appropriate type of mutex
// object to be used when running in parallel. 
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/Configuration.h"
#include "Utilities/PAssert.h"

//-----------------------------------------------------------------------------
//
// Full Description:
//
// Pooma uses Mutex_t objects to protect various pieces of data from
// simultaneous access by multiple threads. The actual type of the
// mutex is determined here. Usually this will just be a smarts
// mutex. However, if we're using a different threading system, we may 
// need to include a wrapper for that system's mutex. Pooma assumes
// that the mutex has lock() and unlock() member functions, so any
// type with this interface can be typedef'd to Pooma::Mutex_t below.
//
// In the serial case, Mutex_t is a typedef for the dummy Mutex class
// defined below.
//
//-----------------------------------------------------------------------------

#if POOMA_THREADS && !POOMA_SMARTS_SCHEDULER_SERIALASYNC

// NOTE: This is probably not correct if real PThreads are being used,
// unless we've adjusted the path or something to pick up a version of
// Mutex.h that wraps a PThread mutex.

#include "Mutex.h"

// NOTE: This is probably not correct if real PThreads are being used,
// unless we've adjusted the path or something to pick up a version of
// Mutex.h that wraps a PThread mutex.

namespace Pooma {
  typedef Smarts::Mutex Mutex_t;
}

#else

// Dummy Mutex class

namespace Pooma {

  class DummyMutex
  {
  public:

#if defined(NOPAssert)

    // Constructors and copy assignment.
    // Hopefully these will nix warnings from Purify.

    inline DummyMutex() { }
    inline DummyMutex(const DummyMutex &) { }
    inline DummyMutex &operator=(const DummyMutex &) { return *this; }

    // Mutex interface:
    // Optimized version is a noop.

    inline void lock() { }
    inline void unlock() { }

#else

    // Testing version. Checks for lock/unlock calls
    // that are not in sync.

    // Constructors and copy assignment.
    // Hopefully these will nix warnings from Purify.

    inline DummyMutex() 
      : locked_m(false) 
    { }

    inline DummyMutex(const DummyMutex &model) 
      : locked_m(model.locked_m) 
    { }

    inline DummyMutex &operator=(const DummyMutex &model) 
    { 
      locked_m = model.locked_m;
      return *this; 
    }

    inline void lock() 
    {
      PAssert(!locked_m);
      locked_m = true;
    }

    inline void unlock() 
    {
      PAssert(locked_m);
      locked_m = false;
    }

  private:

    bool locked_m;

#endif

  };

  typedef DummyMutex Mutex_t;

}

#endif

// } // namespace POOMA

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_THREADS_POOMAMUTEX_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: PoomaMutex.h,v $   $Author: sa_smith $
// $Revision: 1.12 $   $Date: 2000/07/10 20:19:55 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
